/*-
 * Copyright (C)2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: unrmufs2.c,v 1.16 2024/11/24 18:46:24 babolo Exp $"

#define BLIN_COMPAT      4
#define Bpars_COMPAT     4
#define MIFE_COMPAT      5

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <strings.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <ufs/ffs/fs.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <mife.h>
#include "unrmufs2.h"

#define FSBUFSZ 512

static char const *fsfl[] =
{ "UNCLEAN"  , "DOSOFTDEP" , "NEEDSFSCK", "SUJ"
, "ACLS"     , "MULTILABEL", "GJOURNAL" , "FLAGS_UPDATED"
, "NFS4ACLS" , "METACKHASH", "TRIM"     ,            NULL
,        NULL,         NULL,        NULL,            NULL
,        NULL,         NULL,        NULL,            NULL
,        NULL,         NULL,        NULL,            NULL
, "INDEXDIRS", "VARBLKSIZE", "COOLOPT1" , "COOLOPT2"
, "COOLOPT3" , "COOLOPT4"  , "COOLOPT5" , "COOLOPT6"
};

__attribute__((__noreturn__)) static void
usage(int ex) {
    fprintf( stderr
           , " @BABOLO V.M "VERS"  "DATE" http://www.babolo.ru/\n"
             "    -d path   - devise or image of file system\n"
             "    -D        - print direct block numbers\n"
             "    -f path   - config file\n"
             "    -G        - print main FS parameters\n"
             "    -i num    - min CG to look at\n"
             "    -I        - print indirect block numbers\n"
             "    -m num    - max CG to look at\n"
             "    -M num    - max inode to look at in every CG\n"
             "    -q        - quiet\n"
             "    -v        - verbose\n"
             "    -# text   - NOOP for comments\n"
             "    -?        - this text\n"
           )
    ;
    exit(ex);
}

static int
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_parsin(unrmufs2_cf *cf, int argc, char *argv[]) {                                         /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    const char  *flar = "d:Df:Gi:Im:M:qrv#:?=:";
    babolo_opts *bos  = NULL;
    int          ex   = EX_OK;
    const char  *o    ;
    int          c    ;

    ifBLIN_QX2("+ %d", argc);
    if  (!(bos = babolo_openopts(BLIN_BIT0 | Bpars_NONU | Bpars_SYME | Bpars_ESYM, 'Z'))) {
        ifBLIN_QW0("babolo_openopts");
        ex = EX_SOFTWARE;
        goto out;
    }
    if  (babolo_setopts(bos, (UNRMUFS2_CFILE & cf->flags) ? 0 : Bpars_NONU, argc, argv, flar)) {
        ifBLIN_QW0("babolo_setopts");
        ex = EX_SOFTWARE;
        goto out;
    }
    while (0 < (c = babolo_getopts(bos))) {
        switch (c) {
        case 'd': if  (!!cf->ifs) {
                ;     ifBLIN_QX1("Can't redefine FS device or image");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->ifs = o;
                ; }
                ; break;
        case 'D': cf->flags |= UNRMUFS2_DIRCT;
                ; break;
        case 'f': if  (!!cf->conf) {
                ;     ifBLIN_QX1("Can't redefine conf");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->conf = o;
                ; }
                ; break;
        case 'G': cf->flags |= UNRMUFS2_GEOM;
                ; break;
        case 'i': if  (!!~cf->cgmin) {
                ;     ifBLIN_QX1("Can't redefine cgmin");
                ; } else {
                ;     cf->cgmin = (uint32_t)babolo_getoptunum(bos, 0);
                ;     if  (!!errno) {
                ;         ifBLIN_QW0("Err -%c arg", bos->c);
                ;         ex = EX_USAGE;
                ;         goto out;
                ; }   }
                ; break;
        case 'I': cf->flags |= UNRMUFS2_INDIR;
                ; break;
        case 'm': if  (!!~cf->cgmax) {
                ;     ifBLIN_QX1("Can't redefine cgmax");
                ; } else {
                ;     cf->cgmax = (uint32_t)babolo_getoptunum(bos, 0);
                ;     if  (!!errno) {
                ;         ifBLIN_QW0("Err -%c arg", bos->c);
                ;         ex = EX_USAGE;
                ;         goto out;
                ; }   }
                ; break;
        case 'M': if  (!!~cf->dimax) {
                ;     ifBLIN_QX1("Can't redefine dimax");
                ; } else {
                ;     cf->dimax = (uint32_t)babolo_getoptunum(bos, 0);
                ;     if  (!!errno) {
                ;         ifBLIN_QW0("Err -%c arg", bos->c);
                ;         ex = EX_USAGE;
                ;         goto out;
                ; }   }
                ; break;
        case 'q': BLIN_QUIET(cf->flags);
                ; break;
        case 'r': cf->flags |= UNRMUFS2_RTRAC;
                ; break;
        case 'v': BLIN_VERBOSE(cf->flags);
                ; break;
        case '#': break;
        case '?': usage(EX_OK);
        case '=': if  (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; }
                ; switch(o[0]) {
                  case '0': blin_ctl(BLIN_CTL_DUMP);           exit(EX_OK);
                  case '1': babolo_optest(flar);               exit(EX_OK);
                  case '2': babolo_optext(bos);                exit(EX_OK);
                  case '3': babolo_dumpopts(bos);              exit(EX_OK);
                  default :                                usage(EX_USAGE);
                  }
        default : ifBLIN_QW0("Ill flag -%c", bos->c);
                ; usage(EX_USAGE);
    }   }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
main(int argc, char *argv[]) {
    off_t              *cglist ;        /* [0] -  CG,  -   CG */
    babolo_parm        *parms  ;
    char               *buf    ;
    char               *cfmap  ;        /*                             */
    struct cg          *cg     = NULL;
    int                 ex     = EX_OK;
    unrmufs2_cf        *cf     ;
    char               *q      ;
    ssize_t             p      ;
    int                 c      ;

    if  (!(cf = calloc(1, sizeof(unrmufs2_cf)))) {
        ifBLIN_QW0("No mem %"BLIN_X, sizeof(unrmufs2_cf));
        ex = EX_OSERR;
        goto out;
    }
    blin_ctl(BLIN_CTL_FLAG | BLIN_CTL_FSET, (cf->flags = BLIN_BIT0));
#   define blin_internal_flags (BLIN_MASK & cf->flags)
    cf->cgmin = 0xFFFFFFFFU;
    cf->cgmax = 0xFFFFFFFFU;
    cf->dimax = 0xFFFFFFFFU;
    if  (!!(ex = unrmufs2_parsin(cf, argc, argv))) {
        ifBLIN_QW0("parsin");
        ex = EX_USAGE;
        goto out;
    }
    if  (!cf->conf) {
    } else if (0 > (cf->confd = open(cf->conf, O_RDONLY | O_CLOEXEC))) {
        ifBLIN_QW0("Conf file error");
    } else {
        cf->flags |= UNRMUFS2_CFILE;
        if  (!(cf->md = mife_init(MIFE_FULL | 1))) {
            ifBLIN_QW0("mife_init");
            ex = EX_OSERR;
            goto out;
        }
        if  (0 > mife_ctlfdsc(cf->md, cf->confd)) {
            ifBLIN_QW0("mife_ctlfdsc");
            ex = EX_IOERR;
            goto out;
        }
        if  (!(cfmap = mife_window(cf->md, 0, 0))) {
            ifBLIN_QW0("mife_window");
            ex = EX_IOERR;
            goto out;
        }
        if  (!(parms = babolo_getparm(Bpars_NOEN | Bpars_NOAL, &cfmap, NULL, 0))) {
            ifBLIN_QW0("babolo_getparm");
            ex = EX_IOERR;
            goto out;
        }
        if  (!!(ex = unrmufs2_parsin(cf, parms->argc, parms->argv))) {
            ifBLIN_QW0("parsin cf");
            ex = EX_IOERR;
            goto out;
    }   }
    if  (!cf->ifs) {
        ifBLIN_QX0("File system device or image is not defined");
        errno = ENOENT;
        ex = EX_USAGE;
        goto out;
    }
    if  (0 > (cf->ifd = open(cf->ifs, O_RDONLY))) {
        ifBLIN_QW0("Can't open %s~", cf->ifs);
        ex = EX_IOERR;
        goto out;
    }
    if  (!(cf->fs = malloc(SBLOCKSIZE))) {
        ifBLIN_QW0("No mem SBLOCKSIZE");
        ex = EX_OSERR;
        goto out;
    }
    if  (0 > (ex = unrmufs2_sbmain(cf))) {
        ifBLIN_QW0("Superblock not found");
        ex = -ex;
    } else {
        unrmufs2_cgsize(cf);
    }
    if  (!!ex) ifBLIN_QW0("Superblock with errors");
    if  (!(cglist = unrmufs2_sbcg(cf))) {
        ifBLIN_QW0("Copies of superblock are erroneous");
        ex = EX_DATAERR;
        goto out;
    }
    if  (BLIN_BIT1 & cf->flags) {
        ifBLIN_QX1( "fs sblkno   =%d  cblkno=%d iblkno   =%d  dblkno =%d old_cgoffset=%d"
                  , cf->fs->fs_sblkno
                  , cf->fs->fs_cblkno
                  , cf->fs->fs_iblkno
                  , cf->fs->fs_dblkno
                  , cf->fs->fs_old_cgoffset
                  );
        ifBLIN_QX1( "fs sbsize   =%d cgsize=%d old_dsize=%d    ncg    =%d   inopb       =%u"
                  , cf->fs->fs_sbsize
                  , cf->fs->fs_cgsize
                  , cf->fs->fs_old_dsize
                  , cf->fs->fs_ncg
                  , cf->fs->fs_inopb
                  );
        ifBLIN_QX1( "fs fragshift=%d    nindir=%d old_nspf =%d    fsbtodb=%d   metaspace   =%ld"
                  , cf->fs->fs_fragshift
                  , cf->fs->fs_nindir
                  , cf->fs->fs_old_nspf
                  , cf->fs->fs_fsbtodb
                  , cf->fs->fs_metaspace
                  );
        ifBLIN_QX1( "fs ipg      =%u  fpg   =%d  bsize    =%d fsize  =%d frag        =%d"
                  , cf->fs->fs_fpg
                  , cf->fs->fs_ipg
                  , cf->fs->fs_bsize
                  , cf->fs->fs_fsize
                  , cf->fs->fs_frag
                  );
    } else if (UNRMUFS2_GEOM & cf->flags) {
        printf( "fs sblkno   =%d  cblkno=%d iblkno   =%d  dblkno =%d old_cgoffset=%d\n"
              , cf->fs->fs_sblkno
              , cf->fs->fs_cblkno
              , cf->fs->fs_iblkno
              , cf->fs->fs_dblkno
              , cf->fs->fs_old_cgoffset
              );
        printf( "fs sbsize   =%d cgsize=%d old_dsize=%d    ncg    =%d   inopb       =%u\n"
              , cf->fs->fs_sbsize
              , cf->fs->fs_cgsize
              , cf->fs->fs_old_dsize
              , cf->fs->fs_ncg
              , cf->fs->fs_inopb
              );
        printf( "fs fragshift=%d    nindir=%d old_nspf =%d    fsbtodb=%d   metaspace   =%ld\n"
              , cf->fs->fs_fragshift
              , cf->fs->fs_nindir
              , cf->fs->fs_old_nspf
              , cf->fs->fs_fsbtodb
              , cf->fs->fs_metaspace
              );
        printf( "fs ipg      =%u  fpg   =%d  bsize    =%d fsize  =%d frag        =%d\n"
              , cf->fs->fs_fpg
              , cf->fs->fs_ipg
              , cf->fs->fs_bsize
              , cf->fs->fs_fsize
              , cf->fs->fs_frag
              );
    }
    if  (!(buf = malloc(FSBUFSZ))) {
        ifBLIN_QW0("No mem FSBUFSZ");
        ex = EX_OSERR;
        goto out;
    }
    q = buf;
    p = FSBUFSZ;
    c = '<';
    if  (0 > (ex = snprintf(q, (size_t)p, "fs_flags=%08X", cf->fs->fs_flags))) {
        ifBLIN_QW0("snprintf");
        ex = EX_IOERR;
        goto out;
    }
    if  (0 >= (p -= ex)) {
        ifBLIN_QX0("buffer overload");
        ex = EX_CONFIG;
        goto out;
    }
    q += ex;
    if  (!!cf->fs->fs_flags) {
        for (uint32_t k = 1, j = 0; !!k; k <<= 1, j++) {
            if  ((uint32_t)cf->fs->fs_flags & k) {
                if  (0 > (ex = snprintf(q, (size_t)p, "%c%s", c, !fsfl[j] ? "?" : fsfl[j]))) {
                    ifBLIN_QW0("snprintf");
                    ex = EX_IOERR;
                    goto out;
                }
                if  (0 >= (p -= ex)) {
                   ifBLIN_QX0("buffer overload");
                   ex = EX_CONFIG;
                   goto out;
                }
                q += ex;
                c = ',';
        }   }
        *q++ = '>';
        if  (0 >= (p -= 1)) {
            ifBLIN_QX0("buffer overload");
            ex = EX_CONFIG;
            goto out;
    }   }
    ex = EX_OK;
    *q = '\0';
    if  (BLIN_BIT1 & cf->flags) {
        ifBLIN_QX0("%s CGSIZE=%zu SBLOCKSIZE=%zu", buf, cf->cgsize, (size_t)SBLOCKSIZE);
    } else if (UNRMUFS2_GEOM & cf->flags) {
        printf("%s CGSIZE=%zu SBLOCKSIZE=%zu\n", buf, cf->cgsize, (size_t)SBLOCKSIZE);
    }
    for (off_t i = (0xFFFFFFFFU == cf->cgmin) ? 1 : cf->cgmin + 1; i <= cglist[0]; i++) {
        cg = unrmufs2_cgtest(cf, cglist[i]);
        if  (!cg) {
            ifBLIN_QX0("unrmufs2_cgtest %lu", i - 1);
            ex = EX_OSFILE;
            goto out;
        } else if (!!(ex = unrmufs2_readinog(cf, cg, cglist[i], i - 1))) {
            ifBLIN_QX0("unrmufs2_readinog %lu", i - 1);
            goto out;
        }
        free(cg);
        cg = NULL;
    }
out:
    free(cg);
    exit(ex);
#   undef blin_internal_flags
}
