/*-
 * Copyright (C)2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: sbtest.c,v 1.4 2024/11/10 12:48:25 babolo Exp $"

#define BLIN_COMPAT      4
#define MIFE_COMPAT      5

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <stdio.h>
#include <errno.h>
#include <ufs/ffs/fs.h>
#include <babolo/BLINflag.h>
#include <mife.h>
#include "unrmufs2.h"

#ifndef UFS_NDADDR
# define UFS_NDADDR 12 /* Direct addresses in inode.   */
#endif
#ifndef UFS_NIADDR
# define UFS_NIADDR 3  /* Indirect addresses in inode. */
#endif

#define FCHK(lhs, op, rhs, fmt)                    \
    if  (lhs op rhs) {                             \
        ifBLIN_QX1( "%s(" #fmt ") %s %s(" #fmt ")" \
                  , #lhs                           \
                  , (intmax_t)lhs                  \
                  , #op                            \
                  , #rhs                           \
                  , (intmax_t)rhs                  \
                  );                               \
        ex++;                                      \
    }
#define WCHK(lhs, op, rhs, fmt)                    \
    if  (lhs op rhs) {                             \
        ifBLIN_QX1( "%s(" #fmt ") %s %s(" #fmt ")" \
                  , #lhs                           \
                  , (intmax_t)lhs                  \
                  , #op                            \
                  , #rhs                           \
                  , (intmax_t)rhs                  \
                  );                               \
    }

static const uint32_t crc32t[256] =
{ 0x00000000, 0xF26B8303, 0xE13B70F7, 0x1350F3F4, 0xC79A971F, 0x35F1141C, 0x26A1E7E8, 0xD4CA64EB
, 0x8AD958CF, 0x78B2DBCC, 0x6BE22838, 0x9989AB3B, 0x4D43CFD0, 0xBF284CD3, 0xAC78BF27, 0x5E133C24
, 0x105EC76F, 0xE235446C, 0xF165B798, 0x030E349B, 0xD7C45070, 0x25AFD373, 0x36FF2087, 0xC494A384
, 0x9A879FA0, 0x68EC1CA3, 0x7BBCEF57, 0x89D76C54, 0x5D1D08BF, 0xAF768BBC, 0xBC267848, 0x4E4DFB4B
, 0x20BD8EDE, 0xD2D60DDD, 0xC186FE29, 0x33ED7D2A, 0xE72719C1, 0x154C9AC2, 0x061C6936, 0xF477EA35
, 0xAA64D611, 0x580F5512, 0x4B5FA6E6, 0xB93425E5, 0x6DFE410E, 0x9F95C20D, 0x8CC531F9, 0x7EAEB2FA
, 0x30E349B1, 0xC288CAB2, 0xD1D83946, 0x23B3BA45, 0xF779DEAE, 0x05125DAD, 0x1642AE59, 0xE4292D5A
, 0xBA3A117E, 0x4851927D, 0x5B016189, 0xA96AE28A, 0x7DA08661, 0x8FCB0562, 0x9C9BF696, 0x6EF07595
, 0x417B1DBC, 0xB3109EBF, 0xA0406D4B, 0x522BEE48, 0x86E18AA3, 0x748A09A0, 0x67DAFA54, 0x95B17957
, 0xCBA24573, 0x39C9C670, 0x2A993584, 0xD8F2B687, 0x0C38D26C, 0xFE53516F, 0xED03A29B, 0x1F682198
, 0x5125DAD3, 0xA34E59D0, 0xB01EAA24, 0x42752927, 0x96BF4DCC, 0x64D4CECF, 0x77843D3B, 0x85EFBE38
, 0xDBFC821C, 0x2997011F, 0x3AC7F2EB, 0xC8AC71E8, 0x1C661503, 0xEE0D9600, 0xFD5D65F4, 0x0F36E6F7
, 0x61C69362, 0x93AD1061, 0x80FDE395, 0x72966096, 0xA65C047D, 0x5437877E, 0x4767748A, 0xB50CF789
, 0xEB1FCBAD, 0x197448AE, 0x0A24BB5A, 0xF84F3859, 0x2C855CB2, 0xDEEEDFB1, 0xCDBE2C45, 0x3FD5AF46
, 0x7198540D, 0x83F3D70E, 0x90A324FA, 0x62C8A7F9, 0xB602C312, 0x44694011, 0x5739B3E5, 0xA55230E6
, 0xFB410CC2, 0x092A8FC1, 0x1A7A7C35, 0xE811FF36, 0x3CDB9BDD, 0xCEB018DE, 0xDDE0EB2A, 0x2F8B6829
, 0x82F63B78, 0x709DB87B, 0x63CD4B8F, 0x91A6C88C, 0x456CAC67, 0xB7072F64, 0xA457DC90, 0x563C5F93
, 0x082F63B7, 0xFA44E0B4, 0xE9141340, 0x1B7F9043, 0xCFB5F4A8, 0x3DDE77AB, 0x2E8E845F, 0xDCE5075C
, 0x92A8FC17, 0x60C37F14, 0x73938CE0, 0x81F80FE3, 0x55326B08, 0xA759E80B, 0xB4091BFF, 0x466298FC
, 0x1871A4D8, 0xEA1A27DB, 0xF94AD42F, 0x0B21572C, 0xDFEB33C7, 0x2D80B0C4, 0x3ED04330, 0xCCBBC033
, 0xA24BB5A6, 0x502036A5, 0x4370C551, 0xB11B4652, 0x65D122B9, 0x97BAA1BA, 0x84EA524E, 0x7681D14D
, 0x2892ED69, 0xDAF96E6A, 0xC9A99D9E, 0x3BC21E9D, 0xEF087A76, 0x1D63F975, 0x0E330A81, 0xFC588982
, 0xB21572C9, 0x407EF1CA, 0x532E023E, 0xA145813D, 0x758FE5D6, 0x87E466D5, 0x94B49521, 0x66DF1622
, 0x38CC2A06, 0xCAA7A905, 0xD9F75AF1, 0x2B9CD9F2, 0xFF56BD19, 0x0D3D3E1A, 0x1E6DCDEE, 0xEC064EED
, 0xC38D26C4, 0x31E6A5C7, 0x22B65633, 0xD0DDD530, 0x0417B1DB, 0xF67C32D8, 0xE52CC12C, 0x1747422F
, 0x49547E0B, 0xBB3FFD08, 0xA86F0EFC, 0x5A048DFF, 0x8ECEE914, 0x7CA56A17, 0x6FF599E3, 0x9D9E1AE0
, 0xD3D3E1AB, 0x21B862A8, 0x32E8915C, 0xC083125F, 0x144976B4, 0xE622F5B7, 0xF5720643, 0x07198540
, 0x590AB964, 0xAB613A67, 0xB831C993, 0x4A5A4A90, 0x9E902E7B, 0x6CFBAD78, 0x7FAB5E8C, 0x8DC0DD8F
, 0xE330A81A, 0x115B2B19, 0x020BD8ED, 0xF0605BEE, 0x24AA3F05, 0xD6C1BC06, 0xC5914FF2, 0x37FACCF1
, 0x69E9F0D5, 0x9B8273D6, 0x88D28022, 0x7AB90321, 0xAE7367CA, 0x5C18E4C9, 0x4F48173D, 0xBD23943E
, 0xF36E6F75, 0x0105EC76, 0x12551F82, 0xE03E9C81, 0x34F4F86A, 0xC69F7B69, 0xD5CF889D, 0x27A40B9E
, 0x79B737BA, 0x8BDCB4B9, 0x988C474D, 0x6AE7C44E, 0xBE2DA0A5, 0x4C4623A6, 0x5F16D052, 0xAD7D5351
}; 
                                                                             
static __inline __pure2 int
ilog2(u_int mask) {
    return(mask == 0 ? 0 : 8 * (int)sizeof(mask) - __builtin_clz(mask) - 1);
}

uint32_t
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_crc32t(unrmufs2_cf *cf, uint8_t *b, int32_t s) {                                          /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    uint32_t crc = ~0U;

    (void)cf;
    for (int32_t j = 0; s > j; j++) {
	crc = crc32t[(crc ^ b[j]) & 0x0FF] ^ (crc >> 8);
    }
    return(crc);
#   undef blin_internal_flags
}

uint32_t
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_sbtest(unrmufs2_cf *cf, struct fs *fs) {                                                  /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
#ifndef FS_OKAY
    uint32_t           save_ckhash;
#endif
    uint64_t           maxfilesize;
    uint32_t           sectorsize = DEV_BSIZE;
    uint64_t           sizepb     ;
    uint32_t           crc        = 0;
    int                ex         = 0;

    if  (SBLOCKSIZE < fs->fs_sbsize) {
        ifBLIN_QX1("Size %d > SBLOCKSIZE(%d)", fs->fs_sbsize, SBLOCKSIZE);
        errno = ENOSPC;
        goto out;
    }
#ifndef FS_OKAY
    save_ckhash = fs->fs_ckhash;
    fs->fs_ckhash = 0;
#endif
    crc = unrmufs2_crc32t(cf, (uint8_t*)fs, fs->fs_sbsize);
#ifndef FS_OKAY
    fs->fs_ckhash = save_ckhash;
#endif
    errno = 0;
    switch(fs->fs_magic) {
    case FS_UFS2_MAGIC: 					                       break;
    case FS_UFS1_MAGIC: ifBLIN_QX0("Bad magick: UFS1"		   ); errno = EBADMSG; break;
    case FS_BAD_MAGIC : ifBLIN_QX0("Incomplete magick"  	   ); errno = EBADMSG; break;
    case FS_42INODEFMT: ifBLIN_QX0("Bad magick: FS_42"  	   ); errno = EBADMSG; break;
    case FS_44INODEFMT: ifBLIN_QX0("Bad magick: FS_44"  	   ); errno = EBADMSG; break;
    default	      : ifBLIN_QX0("Bad magick: %08X", fs->fs_magic);                  break;
    }
    if  (!!errno) goto out;
    FCHK(fs->fs_sblockloc, !=, SBLOCK_UFS2, %#jX);
    FCHK(fs->fs_frag, <, 1, %jd);
    FCHK(fs->fs_frag, >, MAXFRAG, %jd);
    FCHK(fs->fs_bsize, <, MINBSIZE, %jd);
    FCHK(fs->fs_bsize, >, MAXBSIZE, %jd);
    FCHK(fs->fs_bsize, <, (int32_t)roundup(sizeof(struct fs), DEV_BSIZE), %jd);
    FCHK(fs->fs_fsize, <, (int32_t)sectorsize, %jd);
    FCHK(fs->fs_fsize * fs->fs_frag, !=, fs->fs_bsize, %jd);
    FCHK(powerof2(fs->fs_fsize), ==, 0, %jd);
    FCHK(fs->fs_sbsize, >, SBLOCKSIZE, %jd);
    FCHK(fs->fs_sbsize, <, (signed)sizeof(struct fs), %jd);
    FCHK((uint32_t)fs->fs_sbsize % sectorsize, !=, 0, %jd);
    FCHK(fs->fs_fpg, <, 3 * fs->fs_frag, %jd);
    FCHK(fs->fs_ncg, <, 1, %jd);
    FCHK(fs->fs_fsbtodb, !=, ilog2((u_int)fs->fs_fsize / sectorsize), %jd);
    FCHK(fs->fs_old_cgoffset, <, 0, %jd);
    FCHK(fs->fs_old_cgoffset * (~fs->fs_old_cgmask), >, fs->fs_fpg, %jd);
    FCHK((int32_t)CGSIZE(fs), >, fs->fs_bsize, %jd);
    FCHK((uint64_t)fs->fs_ipg * fs->fs_ncg, >, (((int64_t)(1)) << 32) - fs->fs_inopb, %jd);
    FCHK(fs->fs_cstotal.cs_nifree, <, 0, %jd);
    FCHK(fs->fs_cstotal.cs_nifree, >, (int64_t)fs->fs_ipg * fs->fs_ncg, %jd);
    FCHK(fs->fs_cstotal.cs_ndir, >, ((int64_t)fs->fs_ipg * fs->fs_ncg) - fs->fs_cstotal.cs_nifree, %jd);
    FCHK(fs->fs_size, <, 8 * fs->fs_frag, %jd);
    FCHK(fs->fs_size, <=, ((int64_t)fs->fs_ncg - 1) * fs->fs_fpg, %jd);
    FCHK(fs->fs_size, >, (int64_t)fs->fs_ncg * fs->fs_fpg, %jd);
    FCHK(fs->fs_csaddr, <, 0, %jd);
    FCHK(fs->fs_cssize, !=, fragroundup(fs, fs->fs_ncg * sizeof(struct csum)), %jd);
    if  ((fs->fs_old_cgoffset > 0) && (~fs->fs_old_cgmask < 0)) {
        ifBLIN_QX1( "fs->fs_old_cgoffset(%d) > 0 and ~fs->fs_old_cgmask(%d) < 0"
                  , fs->fs_old_cgoffset
                  , ~fs->fs_old_cgmask
                  );
        ex++;
    }
    FCHK(fs->fs_sblockloc, !=, SBLOCK_UFS2, %#jx);
    FCHK(fs->fs_maxsymlinklen, !=, ((UFS_NDADDR + UFS_NIADDR) * sizeof(ufs2_daddr_t)), %jd);
    FCHK(fs->fs_nindir, !=, fs->fs_bsize / (int32_t)sizeof(ufs2_daddr_t), %jd);
    FCHK(fs->fs_inopb, !=, (uint32_t)fs->fs_bsize / (uint32_t)sizeof(struct ufs2_dinode), %jd);
    FCHK(fs->fs_bsize, <, MINBSIZE, %jd);
    FCHK(fs->fs_bsize, >, MAXBSIZE, %jd);
    FCHK(fs->fs_bsize, <, (int32_t)roundup(sizeof(struct fs), DEV_BSIZE), %jd);
    FCHK(powerof2(fs->fs_bsize), ==, 0, %jd);
    FCHK(fs->fs_frag, <, 1, %jd);
    FCHK(fs->fs_frag, >, MAXFRAG, %jd);
    FCHK(fs->fs_frag, !=, numfrags(fs, fs->fs_bsize), %jd);
    FCHK(fs->fs_fsize, <, (int32_t)sectorsize, %jd);
    FCHK(fs->fs_fsize * fs->fs_frag, !=, fs->fs_bsize, %jd);
    FCHK(powerof2(fs->fs_fsize), ==, 0, %jd);
    FCHK(fs->fs_fpg, <, 3 * fs->fs_frag, %jd);
    FCHK(fs->fs_ncg, <, 1, %jd);
    FCHK(fs->fs_ipg, <, fs->fs_inopb, %jd);
    FCHK((uint64_t)fs->fs_ipg * fs->fs_ncg, >, (((int64_t)(1)) << 32) - fs->fs_inopb, %jd);
    FCHK(fs->fs_cstotal.cs_nifree, <, 0, %jd);
    FCHK(fs->fs_cstotal.cs_nifree, >, (int64_t)fs->fs_ipg * fs->fs_ncg, %jd);
    FCHK(fs->fs_cstotal.cs_ndir, <, 0, %jd);
    FCHK(fs->fs_cstotal.cs_ndir, >, ((int64_t)fs->fs_ipg * fs->fs_ncg) - (int64_t)fs->fs_cstotal.cs_nifree, %jd);
    FCHK(fs->fs_sbsize, >, SBLOCKSIZE, %jd);
    FCHK(fs->fs_sbsize, <, (signed)sizeof(struct fs), %jd);
    if  (fs->fs_maxbsize == 0) {
        ifBLIN_QX1("Misconfigured filesystem fs_maxbsize == 0");
    } else {
	FCHK(fs->fs_maxbsize, <, fs->fs_bsize, %jd);
    }
    FCHK(powerof2(fs->fs_maxbsize), ==, 0, %jd);
    FCHK(fs->fs_maxbsize, >, FS_MAXCONTIG * fs->fs_bsize, %jd);
    FCHK(fs->fs_bmask, !=, ~(fs->fs_bsize - 1), %#jx);
    FCHK(fs->fs_fmask, !=, ~(fs->fs_fsize - 1), %#jx);
    FCHK(fs->fs_qbmask, !=, ~fs->fs_bmask, %#jx);
    FCHK(fs->fs_qfmask, !=, ~fs->fs_fmask, %#jx);
    FCHK(fs->fs_bshift, !=, ilog2((u_int)fs->fs_bsize), %jd);
    FCHK(fs->fs_fshift, !=, ilog2((u_int)fs->fs_fsize), %jd);
    FCHK(fs->fs_fragshift, !=, ilog2((u_int)fs->fs_frag), %jd);
    FCHK(fs->fs_fsbtodb, !=, ilog2((u_int)fs->fs_fsize / sectorsize), %jd);
    FCHK(fs->fs_old_cgoffset, <, 0, %jd);
    if  ((fs->fs_old_cgoffset > 0) && (~fs->fs_old_cgmask < 0)) {
        ifBLIN_QX1( "fs->fs_old_cgoffset(%d) > 0 and ~fs->fs_old_cgmask(%d) < 0"
                  , fs->fs_old_cgoffset
                  , ~fs->fs_old_cgmask
                  );
        ex++;
    }
    FCHK(fs->fs_old_cgoffset * (~fs->fs_old_cgmask), >, fs->fs_fpg, %jd);
    FCHK(CGSIZE(fs), >, (unsigned long)fs->fs_bsize, %jd);
    if  (!!ex) {          /* If anything has failed up to this point, it is usafe to proceed      *
                           * as checks below may divide by zero or make other fatal calculations. */
        ifBLIN_QX0("Superblock is not valid, %d errors", ex);
        errno = EFTYPE;
        goto out;
    }
    FCHK(fs->fs_sblkno, !=, roundup(howmany(fs->fs_sblockloc + SBLOCKSIZE, fs->fs_fsize), fs->fs_frag), %jd);
    FCHK(fs->fs_csaddr + howmany(fs->fs_cssize, fs->fs_fsize), >, fs->fs_size, %jd);
    FCHK(fs->fs_csaddr, <, cgdmin(fs, dtog(fs, fs->fs_csaddr)), %jd);
    FCHK(dtog(fs, fs->fs_csaddr + howmany(fs->fs_cssize, fs->fs_fsize)), >, dtog(fs, fs->fs_csaddr), %jd);
    FCHK(fs->fs_sbsize % (int32_t)sectorsize, !=, 0, %jd);
    FCHK(fs->fs_ipg % fs->fs_inopb, !=, 0, %jd);
    FCHK(fs->fs_sblkno, !=, roundup( howmany(fs->fs_sblockloc + SBLOCKSIZE, fs->fs_fsize), fs->fs_frag), %jd);
    FCHK(fs->fs_cblkno, !=, fs->fs_sblkno + roundup(howmany(SBLOCKSIZE, fs->fs_fsize), fs->fs_frag), %jd);
    FCHK(fs->fs_iblkno, !=, fs->fs_cblkno + fs->fs_frag, %jd);
    FCHK(fs->fs_dblkno, !=, fs->fs_iblkno + (int32_t)(fs->fs_ipg / INOPF(fs)), %jd);
    FCHK(fs->fs_cgsize, >, fs->fs_bsize, %jd);
    FCHK(fs->fs_cgsize, <, fs->fs_fsize, %jd);
    FCHK(fs->fs_cgsize % fs->fs_fsize, !=, 0, %jd);
    FCHK(fs->fs_csaddr, <, 0, %jd);
    FCHK(fs->fs_cssize, !=, fragroundup(fs, fs->fs_ncg * sizeof(struct csum)), %jd);
    FCHK(fs->fs_csaddr + howmany(fs->fs_cssize, fs->fs_fsize), >, fs->fs_size, %jd);
    FCHK(fs->fs_csaddr, <, cgdmin(fs, dtog(fs, fs->fs_csaddr)), %jd);
    FCHK(dtog(fs, fs->fs_csaddr + howmany(fs->fs_cssize, fs->fs_fsize)), >, dtog(fs, fs->fs_csaddr), %jd);
    if  ((fs->fs_maxcontig == 0) && (fs->fs_contigsumsize != 0)) {
        ifBLIN_QX1( "fs->fs_maxcontig(%d) == 0 and fs->fs_contigsumsize(%d) != 0"
                  , fs->fs_maxcontig
                  , fs->fs_contigsumsize
                  );
        ex++;
    }
    if  ((fs->fs_maxcontig > 1) && (fs->fs_contigsumsize != MIN(fs->fs_maxcontig, FS_MAXCONTIG))) {
        ifBLIN_QX1( "fs->fs_maxcontig(%d) > 1 and "
                    "fs->fs_contigsumsize (%d)!= MIN(fs->fs_maxcontig, FS_MAXCONTIG)(%d)"
                  , fs->fs_maxcontig
                  , fs->fs_contigsumsize
                  , MIN(fs->fs_maxcontig, FS_MAXCONTIG)
                  );
        ex++;
    }
       /* These values have a tight interaction with each other that        *
        * makes it hard to tightly bound them. So we can only check         *
        * that they are within a broader possible range.                    *
        *                                                                   *
        * The size cannot always be accurately determined, but ensure       *
        * that it is consistent with the number of cylinder groups (fs_ncg) *
        * and the number of fragments per cylinder group (fs_fpg). Ensure   *
        * that the summary information size is correct and that it starts   *
        * and ends in the data area of the same cylinder group.             */
    FCHK(fs->fs_size, <, 8 * fs->fs_frag, %jd);
    FCHK(fs->fs_size, <=, ((int64_t)fs->fs_ncg - 1) * fs->fs_fpg, %jd);
    FCHK(fs->fs_size, >, (int64_t)fs->fs_ncg * fs->fs_fpg, %jd);
    WCHK( fs->fs_dsize
        , !=                             /* older versions of growfs failed */
        ,   fs->fs_size                  /* to correctly update fs_dsize    */
          - fs->fs_sblkno
          - (int64_t)fs->fs_ncg * (fs->fs_dblkno - fs->fs_sblkno)
          - howmany(fs->fs_cssize, fs->fs_fsize)
        , %jd
        );
    WCHK(fs->fs_metaspace, <, 0, %jd);
    WCHK(fs->fs_metaspace, >, fs->fs_fpg / 2, %jd);
    WCHK(fs->fs_minfree, >, 99, %jd%%);
    WCHK(fs->fs_maxcontig, <, 0, %jd);
    WCHK(fs->fs_maxcontig, >, MAX(256, MAXPHYS / fs->fs_bsize), %jd);
    maxfilesize = (uint64_t)fs->fs_bsize * UFS_NDADDR - 1;
    sizepb = (uint64_t)fs->fs_bsize;
    for (int i = 0; i < UFS_NIADDR; i++) {
        sizepb *= (uint64_t)NINDIR(fs);
        maxfilesize += sizepb;
    }
    WCHK(fs->fs_maxfilesize, !=, maxfilesize, %jd);
    if  (!!ex) {
        ifBLIN_QX0("Superblock isn't valid, %d errors", ex);
        errno = ENOENT;
        goto out;
    }
    errno = 0;
out:
    return(crc);
#   undef blin_internal_flags
}
