/*-
 * Copyright (C)2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: sbfind.c,v 1.11 2024/11/23 23:37:54 babolo Exp $"

#define BLIN_COMPAT      4
#define MIFE_COMPAT      5

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <strings.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <ufs/ffs/fs.h>
#include <babolo/BLINflag.h>
#include <mife.h>
#include "unrmufs2.h"

static const off_t sbp[] = SBLOCKSEARCH;
static const char *sbn[] = {"UFS2  ", "UFS1  ", "FLOPPY", "PIGGY "};

int
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_sbmain(unrmufs2_cf *cf) {                                                                 /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    uint32_t           ckhash = 0;
    int                ex     = 0;

    if  (!(cf->fs = malloc(SBLOCKSIZE))) {
        ifBLIN_QW0("No mem SBLOCKSIZE");
        ex = -EX_OSERR;
        goto out;
    }
    for (int i = 0; 0 <= sbp[i]; i++) {
        if  (UNRMUFS2_RTRAC & cf->flags) {
            ifBLIN_QX0( "Read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                      , (size_t)SBLOCKSIZE
                      , (size_t)SBLOCKSIZE
                      , sbp[i]
                      , sbp[i]
                      , sbp[i] / DEV_BSIZE
                      , sbp[i] / DEV_BSIZE
                      );
        }
        if  (0 > pread(cf->ifd, cf->fs, SBLOCKSIZE, sbp[i])) {
            ifBLIN_QW0( "Can't read SB %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                      , (size_t)SBLOCKSIZE
                      , (size_t)SBLOCKSIZE
                      , sbp[i]
                      , sbp[i]
                      , sbp[i] / DEV_BSIZE
                      , sbp[i] / DEV_BSIZE
                      );
            ex = -EX_IOERR;
            goto out;
        }
        ckhash = unrmufs2_sbtest(cf, cf->fs);
        if  (ENOSPC == errno) continue;
        ifBLIN_QX1("%s crc %08X %s %08X%s"
                  , sbn[i]
#ifndef FS_OKAY
                  , cf->fs->fs_ckhash
                  , (cf->fs->fs_ckhash == ckhash) ? "==" : "<>"
#else
                  , 0
                  , ""
#endif
                  , ckhash
                  , (!errno) ? " OK" : ""
                  );
        if  (!errno) break;
    }
    if  (!!errno) ex = 1;
out:
#ifndef FS_OKAY
    if  (!ex && (FS_METACKHASH & cf->fs->fs_flags) && (cf->fs->fs_ckhash != ckhash)) ex = 2;
#endif
    return(ex);
#   undef blin_internal_flags
}

off_t *
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_sbcg(unrmufs2_cf *cf) {                                                                   /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    struct fs         *protofs;
    void              *fsrbuf ;
    uint32_t           ckhash ;
    off_t             *cglist = NULL;
    u_int32_t          ioerr  ;
    u_int32_t          nospc  ;
    u_int32_t          bdmsg  ;
    u_int32_t          noent  ;
    u_int32_t          nocrc  ;
    u_int32_t          crcer  ;
    u_int32_t          error  ;
    off_t              sblk   ;
    struct fsrecovery *fsr    ;
    struct fs         *fs     ;
    summary           *s      ;

    if  (!(fsrbuf = malloc(SBLOCKSIZE))) {
    	ifBLIN_QW0("No mem SBLOCKSIZE");
    	goto out;
    }
    if  (!(protofs = malloc(SBLOCKSIZE))) {
    	ifBLIN_QW0("No mem SBLOCKSIZE");
    	goto out;
    }
    if  (!(fs = malloc(SBLOCKSIZE))) {
        ifBLIN_QW0("No mem SBLOCKSIZE");
        goto out;
    }
    for (size_t secsize = DEV_BSIZE; SBLOCKSIZE >= secsize; secsize *= 2) {
        if  (UNRMUFS2_RTRAC & cf->flags) {
            ifBLIN_QX0( "Read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                      , secsize
                      , secsize
                      , SBLOCK_UFS2 - (off_t)secsize
                      , SBLOCK_UFS2 - (off_t)secsize
                      , (SBLOCK_UFS2 - (off_t)secsize) / DEV_BSIZE
                      , (SBLOCK_UFS2 - (off_t)secsize) / DEV_BSIZE
                      );
        }
        if  (0 > pread(cf->ifd, fsrbuf, secsize, (SBLOCK_UFS2 - (off_t)secsize))) {
            if  (EINVAL == errno) continue;
            ifBLIN_QW0("Can't read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                      , secsize
                      , secsize
                      , SBLOCK_UFS2 - (off_t)secsize
                      , SBLOCK_UFS2 - (off_t)secsize
                      , (SBLOCK_UFS2 - (off_t)secsize) / DEV_BSIZE
                      , (SBLOCK_UFS2 - (off_t)secsize) / DEV_BSIZE
                      );
            goto out;
        }
        fsr = (struct fsrecovery *)&fsrbuf[secsize - sizeof(*fsr)];
        if  (FS_UFS2_MAGIC == fsr->fsr_magic) {
            bzero(protofs, sizeof(struct fs));
            protofs->fs_fpg = fsr->fsr_fpg;
            protofs->fs_fsbtodb = fsr->fsr_fsbtodb;
            protofs->fs_sblkno = fsr->fsr_sblkno;
            protofs->fs_magic = fsr->fsr_magic;
            protofs->fs_ncg = fsr->fsr_ncg;
            if  (!(s = calloc(protofs->fs_ncg, sizeof(summary)))) {
                ifBLIN_QW0("No mem summary %u", protofs->fs_ncg);
                goto out;
            }
            if  (cf->cgmax > protofs->fs_ncg) cf->cgmax = protofs->fs_ncg;
            if  (!(cglist = calloc(cf->cgmax + 1, sizeof(off_t)))) {
                ifBLIN_QW0("No mem cglist%u", cf->cgmax);
                goto out;
            }
            cglist[0] = 0;
            for (uint32_t cg = 0; cg < cf->cgmax; cg++) {
                sblk = ((int64_t)protofs->fs_fpg * cg + protofs->fs_sblkno) << protofs->fs_fsbtodb;
                cglist[cg + 1] = sblk;
                if  (UNRMUFS2_RTRAC & cf->flags) {
                    ifBLIN_QX0( "Read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                              , (size_t)SBLOCKSIZE
                              , (size_t)SBLOCKSIZE
                              , sblk << DEV_BSHIFT
                              , sblk << DEV_BSHIFT
                              , sblk
                              , sblk
                              );
                }
                if  (0 > pread(cf->ifd, fs, SBLOCKSIZE, sblk << DEV_BSHIFT)) {
                    s->ioerr = 1;
                    ifBLIN_QW0( "Can't read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                              , (size_t)SBLOCKSIZE
                              , (size_t)SBLOCKSIZE
                              , sblk << DEV_BSHIFT
                              , sblk << DEV_BSHIFT
                              , sblk
                              , sblk
                              );
                    cglist[0] = 0;
                    continue;
                }
                ckhash = unrmufs2_sbtest(cf, fs);
                switch(errno) {
                case ENOSPC:
                    s->nospc = 1;
                    continue;
                case EBADMSG:
                    s->bdmsg = 1;
                    continue;
                case ENOENT:
                    s->noent = 1;
                    continue;
                case 0:
                    if  (!cf->cgsize) {
                        cf->fs = fs;
                        unrmufs2_cgsize(cf);
                    }
                    break;
                default:
                    s->error = 1;
                }
#ifndef FS_OKAY
                if  (!fs->fs_ckhash) {
                    s->nocrc = 1;
                } else if (fs->fs_ckhash != ckhash) {
                    s->crcer = 1;
                }
#endif
                if  ((0xFFFFFFFFU != cf->cgmin) && (cg >= cf->cgmin)) {
                    ifBLIN_QX1("%u             crc %08X %s %08X%s"
                              , cg
#ifndef FS_OKAY
                              , fs->fs_ckhash
                              , (fs->fs_ckhash == ckhash) ? "==" : "<>"
#else
                              , 0
                              , ""
#endif
                              , ckhash
                              , (!errno) ? " OK" : ""
                              );
                    unrmufs2_sbdif(cf, cg, cf->fs, fs);
            }   }
            ioerr = nospc = bdmsg = noent = nocrc = crcer = error = 0;
            for (long cg = 0; cg < cf->cgmax; cg++) {
                if  (s[cg].ioerr) ioerr++;
                if  (s[cg].nospc) nospc++;
                if  (s[cg].bdmsg) bdmsg++;
                if  (s[cg].noent) noent++;
                if  (s[cg].nocrc) nocrc++;
                if  (s[cg].crcer) crcer++;
                if  (s[cg].error) error++;
            }
            ifBLIN_QX1( "%u->%u..%u cg, %u ioerr, %u nospc, %u bdmsg, %u noent, %u nocrc, %u crcer, %u error"
                      , protofs->fs_ncg
                      , (0xFFFFFFFFU != cf->cgmin) ? (cf->cgmin - 1) : 0
                      , cf->cgmax
                      , ioerr
                      , nospc
                      , bdmsg
                      , noent
                      , nocrc
                      , crcer
                      , error
                      );
        }
        if  (!!cglist) cglist[0] = cf->cgmax;
        break;
    }
    if  (!!cglist && !cglist[0]) {
        free(cglist);
        cglist = NULL;
    }
out:
    return(cglist);
#   undef blin_internal_flags
}
