/*-
 * Copyright (C)2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: cgtest.c,v 1.19 2024/11/24 18:46:24 babolo Exp $"

#define BLIN_COMPAT      4
#define MIFE_COMPAT      5

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <ufs/ffs/fs.h>
#include <ufs/ufs/dir.h>
#include <babolo/BLINflag.h>
#include <mife.h>
#include "unrmufs2.h"

#ifndef UFS_NDADDR
# define UFS_NDADDR NDADDR
#endif

#ifndef UFS_NIADDR
# define UFS_NIADDR NIADDR
#endif

void
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_cgsize(unrmufs2_cf *cf) {                                                                 /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    cf->cgsize = sizeof(struct cg)                                                    /* base cg     */
               + (size_t)cf->fs->fs_old_cpg * sizeof(int32_t)                         /* old btotoff */
               + (size_t)cf->fs->fs_old_cpg * sizeof(uint16_t)                        /* old boff    */
               + howmany(cf->fs->fs_ipg, NBBY)                                        /* inode map   */
               + howmany((size_t)cf->fs->fs_fpg, NBBY) + sizeof(int32_t)              /* block map   */
    ;
    if  (0 < cf->fs->fs_contigsumsize) {
        cf->cgsize += (size_t)cf->fs->fs_contigsumsize * sizeof(int32_t)              /* cluster sum */
                    + howmany(((size_t)cf->fs->fs_fpg >> cf->fs->fs_fragshift), NBBY) /* cluster map */
        ;
    }
#   undef blin_internal_flags
}

struct cg *
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_cgtest(unrmufs2_cf *cf, off_t cgoff) {                                                    /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
#ifndef FS_OKAY
    uint32_t           save_ckhash;
#endif
    size_t             cgsize     ;
    off_t              cgofb      ;
    uint32_t           crc        = 0;
    struct cg         *cg         ;
    int                ex         = 0;

    ifBLIN_QX3("+ %"BLIN_O"d", cgoff);
    cgsize = (cf->cgsize + SBLOCKSIZE - 1) & ~((size_t)SBLOCKSIZE - 1);
    if  (!(cg = malloc(cgsize))) {
    	ifBLIN_QW0("No mem %zu", cgsize);
        ex = EX_OSERR;
    	goto out;
    }
    cgofb = (cgoff + ((cf->fs->fs_cblkno - cf->fs->fs_sblkno) << cf->fs->fs_fsbtodb)) << DEV_BSHIFT;
    if  (UNRMUFS2_RTRAC & cf->flags) {
        ifBLIN_QX0( "Read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                  , cgsize
                  , cgsize
                  , cgofb
                  , cgofb
                  , cgofb / DEV_BSIZE
                  , cgofb / DEV_BSIZE
                  );
    }
    if  (0 > pread(cf->ifd, cg, cgsize, cgofb)) {
        ifBLIN_QW0( "Can't read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                  , cgsize
                  , cgsize
                  , cgofb
                  , cgofb
                  , cgofb / DEV_BSIZE
                  , cgofb / DEV_BSIZE
                  );
        ex = EX_IOERR;
        goto out;
    }
#ifndef FS_OKAY
    save_ckhash = cg->cg_ckhash;
    cg->cg_ckhash = 0;
#endif
    crc = unrmufs2_crc32t(cf, (uint8_t*)cg, cf->fs->fs_cgsize);
#ifndef FS_OKAY
    cg->cg_ckhash = save_ckhash;
#endif
    if  (CG_MAGIC != cg->cg_magic) {
        ex = EX_PROTOCOL;
        ifBLIN_QX0("%d %08X %08X", ex, CG_MAGIC, cg->cg_magic);
    }
#ifndef FS_OKAY
    if  (!ex && (FS_METACKHASH & cf->fs->fs_flags) && (cg->cg_ckhash != crc)) ex = 2;
#endif
    if  (!ex) {
        ifBLIN_QX1( "magic %08X, crc %08X %08X OK"
                  , cg->cg_magic
#ifndef FS_OKAY
                  , cg->cg_ckhash
#else
                  , 0
#endif
                  , crc
                  );
    } else {
        ifBLIN_QX0( "magic %08X, crc %08X %08X"
                  , cg->cg_magic
#ifndef FS_OKAY
                  , cg->cg_ckhash
#else
                  , 0
#endif
                  , crc
                  );
    }
out:
    ifBLIN_QX3("- %"BLIN_X, BLIN_I(cg));
    return(cg);
#   undef blin_internal_flags
}

static void
inout(unrmufs2_cf *cf, struct direct *dentry, uint32_t loc, int t) {
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    if  (BLIN_BIT1 & cf->flags) {
        ifBLIN_QX1( "ino=%u %c%08X  reclen=%u type=%02o namlen=%u=%.*s"
                  , dentry->d_ino
                  , !t ? '@' : '#'
                  , loc
                  , dentry->d_reclen
                  , dentry->d_type
                  , dentry->d_namlen
                  , (int)dentry->d_namlen, dentry->d_name
                  );
    } else {
        printf( "ino=%u %c%08X  reclen=%u type=%02o namlen=%u=%.*s\n"
              , dentry->d_ino
              , !t ? '@' : '#'
              , loc
              , dentry->d_reclen
              , dentry->d_type
              , dentry->d_namlen
              , (int)dentry->d_namlen, dentry->d_name
              );
    }
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_dir(unrmufs2_cf *cf, struct ufs2_dinode *ino) {                                           /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    struct direct      *dentry     ;
    u_int16_t           reclen     ;
    char                buf        [cf->fs->fs_bsize];
    int                 ex         = EX_OK;
    char               *p          ;
    char               *q          ;
    uint                l          ;
    uint8_t             c          ;

    ifBLIN_QX3("+ %"BLIN_X, BLIN_I(ino));
    for (int64_t k = 0; k < UFS_NDADDR; k++) {
        if  (!!ino->di_db[k]) {
            if  (UNRMUFS2_RTRAC & cf->flags) {
                ifBLIN_QX0( "%ld Read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                          , k
                          , (size_t)cf->fs->fs_bsize
                          , (size_t)cf->fs->fs_bsize
                          , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                          , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                          , ino->di_db[k]
                          , ino->di_db[k]
                          );
            }
            ex = (int)pread( cf->ifd
                           , buf
                           , (size_t)cf->fs->fs_bsize
                           , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                           );
            if  (0 > ex) {
                ifBLIN_QW0( "Can't %ld read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                          , k
                          , (size_t)cf->fs->fs_bsize
                          , (size_t)cf->fs->fs_bsize
                          , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                          , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                          , ino->di_db[k]
                          , ino->di_db[k]
                          );
                continue;
            }
            if (0 < ex) ex = EX_OK;
            p = buf;
            for ( dentry = (struct direct *)p
                ; ino->di_size > (uint64_t)(p - buf)
                ; dentry = (struct direct *)(p += reclen)
                ) {
                reclen = dentry->d_reclen;
                inout(cf, dentry, (uint32_t)(p - buf), 0);
                if  (!reclen) break;
                l = (uint)strnlen(dentry->d_name, dentry->d_namlen);
                if  (20 < ((int)reclen - (int)l)) {
                    l = ((8 + l) & 0xFFFFFFFC) + 4;
                    q = p + l;
                    c = 0;
                    for (uint i = 0; i < (reclen - l); i++) c |= q[i];
                    if  (!!c) {
                        blin_dumb(0, q, reclen - l);
                        for ( dentry = (struct direct *)q
                            ; q < (p + reclen)
                            ; dentry = (struct direct *)(q += dentry->d_reclen)
                            ) {
                            inout(cf, dentry, (uint32_t)(q - buf), 1);
                            if  (!dentry->d_reclen) break;
    }   }   }   }   }   }
    for (int64_t k = 0; k < UFS_NIADDR; k++) {
        if  (!!ino->di_ib[k]) {
            if  (UNRMUFS2_RTRAC & cf->flags) {
                ifBLIN_QX0( "%ld Read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                          , k
                          , (size_t)cf->fs->fs_bsize
                          , (size_t)cf->fs->fs_bsize
                          , ino->di_ib[k] << DEV_BSHIFT
                          , ino->di_ib[k] << DEV_BSHIFT
                          , ino->di_ib[k]
                          , ino->di_ib[k]
                          );
            }
            if  (0 > pread(cf->ifd, buf, (size_t)cf->fs->fs_bsize, ino->di_ib[k] << DEV_BSHIFT)) {
                ifBLIN_QW0( "Can't %ld read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                          , k
                          , (size_t)cf->fs->fs_bsize
                          , (size_t)cf->fs->fs_bsize
                          , ino->di_ib[k] << DEV_BSHIFT
                          , ino->di_ib[k] << DEV_BSHIFT
                          , ino->di_ib[k]
                          , ino->di_ib[k]
                          );
                continue;
    }   }   }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_doino(unrmufs2_cf *cf, struct ufs2_dinode *ino) {                                         /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    char                buf        [cf->fs->fs_bsize];
    int                 ex         = EX_OK;
    uint                l          ;

    ifBLIN_QX3("+ %"BLIN_X, BLIN_I(ino));
    if  (IFDIR == (ino->di_mode & IFMT)) {
        if  (!!(ex = unrmufs2_dir(cf, ino))) {
            ifBLIN_QW0("unrmufs2_dir");
            ex = EX_IOERR;
            goto out;
        }
    } else if ((IFLNK == (ino->di_mode & IFMT)) && !ino->di_blocks) {
        l = (uint)strnlen((char*)ino->di_db, (UFS_NDADDR + UFS_NIADDR) * sizeof(ufs2_daddr_t));
        if  (BLIN_BIT1 & cf->flags) {
            ifBLIN_QX1("(%u) -> %.*s", l, l, (char*)ino->di_db);
        } else {
            printf("(%u) -> %.*s\n", l, l, (char*)ino->di_db);
        }
    } else {
        for (int k = 0; (UNRMUFS2_DIRCT & cf->flags) && (k < UFS_NDADDR); k++) {
            if  (!!ino->di_db[k]) {
                if  (BLIN_BIT1 & cf->flags) {
                    ifBLIN_QX1("D %d %ld", k, ino->di_db[k]);
                } else {
                    printf("D %d %ld\n", k, ino->di_db[k]);
                }
                if  (UNRMUFS2_RTRAC & cf->flags) {
                    ifBLIN_QX0( "%d Read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                              , k
                              , (size_t)cf->fs->fs_bsize
                              , (size_t)cf->fs->fs_bsize
                              , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_db[k]
                              , ino->di_db[k]
                              );
                }
                ex = (int)pread( cf->ifd
                               , buf
                               , (size_t)cf->fs->fs_bsize
                               , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                               );
                if  (0 > ex) {
                    ifBLIN_QW0( "Can't %d read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                              , k
                              , (size_t)cf->fs->fs_bsize
                              , (size_t)cf->fs->fs_bsize
                              , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_db[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_db[k]
                              , ino->di_db[k]
                              );
                    continue;
        }   }   }
        for (int k = 0; (UNRMUFS2_INDIR & cf->flags) && (k < UFS_NIADDR); k++) {
            if  (!!ino->di_ib[k]) {
                if  (BLIN_BIT1 & cf->flags) {
                    ifBLIN_QX1("D %d %ld", k, ino->di_ib[k]);
                } else {
                    printf("D %d %ld\n", k, ino->di_ib[k]);
                }
                if  (UNRMUFS2_RTRAC & cf->flags) {
                    ifBLIN_QX0( "%d Read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                              , k
                              , (size_t)cf->fs->fs_bsize
                              , (size_t)cf->fs->fs_bsize
                              , ino->di_ib[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_ib[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_ib[k]
                              , ino->di_ib[k]
                              );
                }
                ex = (int)pread( cf->ifd
                               , buf
                               , (size_t)cf->fs->fs_bsize
                               , ino->di_ib[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                               );
                if  (0 > ex) {
                    ifBLIN_QW0( "Can't %d read %06zX(%zu) @%09lX(%lu)@%lX(%lu)"
                              , k
                              , (size_t)cf->fs->fs_bsize
                              , (size_t)cf->fs->fs_bsize
                              , ino->di_ib[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_ib[k] << (DEV_BSHIFT + cf->fs->fs_fsbtodb)
                              , ino->di_ib[k]
                              , ino->di_ib[k]
                              );
                    continue;
    }   }   }   }
    if  (0 < ex) ex = EX_OK;
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_inod(unrmufs2_cf *cf, struct ufs2_dinode *ig, off_t cgoff, off_t cgn, size_t sz) {        /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
#ifndef FS_OKAY
    uint32_t            save_ighash;
#endif
    uint32_t            crc        = 0;
    int                 ex         = EX_OK;
    uint        d = 0;
    uint        i = 0;
    const char *s [] = {"", "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "**"};

    for (uint k = 0; k < UFS_NDADDR; k++) {
        if  (k > 12) {
            d = 12;
            break;
        } else if (!!ig[sz].di_db[k]) {
            d = k;
        } else {
            break;
    }   }
    for (uint k = 0; k < UFS_NIADDR; k++) {
        if  (k > 12) {
            i = 12;
            break;
        } else if (!!ig[sz].di_ib[k]) {
            i = k;
        } else {
            break;
    }   }
    ifBLIN_QX3("+ %"BLIN_O"d %zu", cgoff, sz);
#ifndef FS_OKAY
    save_ighash = ig[sz].di_ckhash;
    ig[sz].di_ckhash = 0;
#endif
    crc = unrmufs2_crc32t(cf, (uint8_t*)&ig[sz], sizeof(struct ufs2_dinode));
#ifndef FS_OKAY
    ig[sz].di_ckhash = save_ighash;
#endif
    if  (BLIN_BIT1 & cf->flags) {
        ifBLIN_QX1( "%zu di mode=%06o nlink=%u blksize=%u size=%lu blocks=%lu ckhash=%08X%s%08X%s%s%s%s"
                  , (size_t)cgn * cf->fs->fs_ipg + sz
                  , ig[sz].di_mode
                  , ig[sz].di_nlink
                  , ig[sz].di_blksize
                  , ig[sz].di_size
                  , ig[sz].di_blocks
#ifndef FS_OKAY
                  , ig[sz].di_ckhash
#else
                  , 0
#endif
#ifndef FS_OKAY
                  ,   (FS_METACKHASH & cf->fs->fs_flags)
                    ? ((ig[sz].di_ckhash == crc) ? "==" : "<>")
                    : (!ig[sz].di_ckhash ? "==" : "~~")
#else
                  , "~~"
#endif
                  , crc
                  , (!(UNRMUFS2_DIRCT & cf->flags) && !!d) ? " D" : ""
                  , !(UNRMUFS2_DIRCT & cf->flags) ? s[d] : ""
                  , (!(UNRMUFS2_INDIR & cf->flags) && !!i) ? " I" : ""
                  , !(UNRMUFS2_INDIR & cf->flags) ? s[i] : ""
                  );
    } else if (!!ig[sz].di_mode || !!ig[sz].di_nlink || !!ig[sz].di_size || !!ig[sz].di_blocks) {
        printf( "%zu di mode=%06o nlink=%u blksize=%u size=%lu blocks=%lu ckhash=%08X%s%08X%s%s%s%s\n"
              , (size_t)cgn * cf->fs->fs_ipg + sz
              , ig[sz].di_mode
              , ig[sz].di_nlink
              , ig[sz].di_blksize
              , ig[sz].di_size
              , ig[sz].di_blocks
#ifndef FS_OKAY
              , ig[sz].di_ckhash
#else
              , 0
#endif
#ifndef FS_OKAY
              ,   (FS_METACKHASH & cf->fs->fs_flags)
                ? ((ig[sz].di_ckhash == crc) ? "==" : "<>")
                : (!ig[sz].di_ckhash ? "==" : "~~")
#else
              , "~~"
#endif
              , crc
              , (!(UNRMUFS2_DIRCT & cf->flags) && !!d) ? " D" : ""
              , !(UNRMUFS2_DIRCT & cf->flags) ? s[d] : ""
              , (!(UNRMUFS2_INDIR & cf->flags) && !!i) ? " I" : ""
              , !(UNRMUFS2_INDIR & cf->flags) ? s[i] : ""
              );
    }
    if  (!!(ex = unrmufs2_doino(cf, &ig[sz]))) {
        ifBLIN_QW0("unrmufs2_doino");
        goto out;
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 **                                                                                                 **/
unrmufs2_readinog(unrmufs2_cf *cf, struct cg *cg, off_t cgoff, off_t cgn) {                        /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    size_t              igsize ;
    off_t               igofb  ;
    struct ufs2_dinode *ig     ;
    int                 ex     = EX_OK;

    ifBLIN_QX3("+ %"BLIN_O"d", cgoff);
    igsize = (size_t)cf->fs->fs_ipg * sizeof(struct ufs2_dinode);
    if  (!(ig = malloc(igsize))) {
            ifBLIN_QW0("No mem %zu", igsize);
            ex = EX_OSERR;
            goto out;
    }
    igofb = (cgoff + ((cf->fs->fs_iblkno - cf->fs->fs_sblkno) << cf->fs->fs_fsbtodb)) << DEV_BSHIFT;
    if  (UNRMUFS2_RTRAC & cf->flags) {
        ifBLIN_QX0( "Read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                  , igsize
                  , igsize
                  , igofb
                  , igofb
                  , igofb / DEV_BSIZE
                  , igofb / DEV_BSIZE
                  );
    }
    if  (0 > pread(cf->ifd, ig, igsize, igofb)) {
        ifBLIN_QW0( "Can't read %06zX(%zu) @%09"BLIN_O"X(%"BLIN_O"u)@%"BLIN_O"X(%"BLIN_O"u)"
                  , igsize
                  , igsize
                  , igofb
                  , igofb
                  , igofb / DEV_BSIZE
                  , igofb / DEV_BSIZE
                  );
        ex = EX_IOERR;
        goto out;
    }
    for ( size_t i = 0
        ; (i < (size_t)cf->fs->fs_ipg) && (i < cf->dimax) && (i <= cg->cg_initediblk)
        ; i++
        ) {
        ex |= unrmufs2_inod(cf, ig, cgoff, cgn, i);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}
