/*-
 * Copyright Sergey Kosyakov ks@itp.ac.ru 1999
 * Copyright (C) @BABOLO  2002 http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] = "\
@(#)Copyright Sergey Kosyakov ks@itp.ac.ru 1999\n\
@(#)Copyright (C) @BABOLO  2002 http://www.babolo.ru/\n\
@(#)All rights reserved.\n";
static const char rcsid[] = "$Id: utils.c,v 1.2 2010/10/18 16:51:36 babolo Exp $";
#endif /* not lint */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/file.h>
#include <unistd.h>
#include <errno.h>
#include "tund.h"

#define SEQ_INISIZE	16

extern int debug;
extern int control_port;
extern Seq *stun;

/*****************************************************************************/
/* Sequence support                                                          */
/*****************************************************************************/

Seq *
seq_new() {
    Seq *ret;

    ret = (Seq *)malloc(sizeof(Seq));
    bzero(ret, sizeof(Seq));
    ret->buf = malloc(SEQ_INISIZE * sizeof(void *));
    bzero(ret->buf, SEQ_INISIZE * sizeof(void *));
    ret->maxlen = SEQ_INISIZE;
    ret->length = 0;
    return(ret);
}

int
seq_append(Seq *s, void *item) {
    void **buf;
    int len;

    s->length++;
    if  (s->length >= s->maxlen) {
        len = s->maxlen * 2;
        buf = malloc(len * sizeof(void *));
        bzero(buf, len*sizeof(void *));
        bcopy(s->buf, buf, s->maxlen * sizeof(void *));
        free(s->buf);
        s->buf = buf;
        s->maxlen = len;
    }
    s->buf[s->length - 1] = item;
    return(s->length - 1);
}

/*****************************************************************************/
/* Log and errors                                                            */
/*****************************************************************************/

#define LOG_BUFS	1024
static char log_buf[LOG_BUFS];
static int log_init=0;

void
Log(char *fmt, ...) {
    va_list ap;
    int ret;

    if  (!log_init) {
        openlog("tund", LOG_CONS | LOG_NDELAY, LOG_DAEMON);
        log_init = 1;
    }
    bzero(log_buf, LOG_BUFS);
    va_start(ap, fmt);
    ret = vsnprintf(log_buf, LOG_BUFS, fmt, ap);
    va_end(ap);
    if(debug) printf("%s\n", log_buf); else syslog(LOG_WARNING, log_buf);
}

void
Error(char *fmt, ...) {
    va_list ap;
    int ret;

    if  (!log_init) {
        openlog("tund", LOG_CONS | LOG_NDELAY, LOG_DAEMON);
        log_init = 1;
    }
    bzero(log_buf, LOG_BUFS);
    va_start(ap, fmt);
    ret = vsnprintf(log_buf, LOG_BUFS, fmt, ap);
    va_end(ap);
    if  (debug) printf("%s\n", log_buf); else syslog(LOG_ERR, log_buf);
    exit(1);
}

/*****************************************************************************/
/* Parsing config file                                                       */
/*****************************************************************************/

static int
parse_host_name(char *name, struct in_addr *addr) { 
    int n1, n2, n3, n4;
    char buf[32];
    char dummy;
    int ret;  

    ret = sscanf(name, "%d.%d.%d.%d%c", &n1, &n2, &n3, &n4, &dummy);
    if  (ret != 4) return(0);

    if  ((n1 < 0) || (n1 > 255)) return(0);
    if  ((n2 < 0) || (n2 > 255)) return(0);
    if  ((n3 < 0) || (n3 > 255)) return(0);
    if  ((n4 < 0) || (n4 > 255)) return(0);
    snprintf(buf, 32, "%d.%d.%d.%d", n1, n2, n3, n4);
    addr->s_addr = inet_addr(buf);
    return(1);
}

static int
find_services() {
    struct servent *sn;

    sn = getservbyname("tund", "udp");                                  /* Set up controlling socket */
    if  (!sn) Error("Can not locate service tund/udp at /etc/services");
    control_port = ntohs(sn->s_port);
    Log("Service tund/udp found at port %d", control_port);
    return(0);
}

static int
parse_config_options(Tunnel *tun, char *str) {
    struct in_addr addr;
    struct hostent *hs;

    while (*str == ' ' || *str == '\n' || *str == '\t') str++;
    if  (!(*str)) return(0);
    if  (strncmp(str, "rule=", 5) == 0) {
        tun->tag=atoi(str+5);
        return(0);
    }
    if  (strncmp(str, "divert=", 7) == 0) {
        tun->dp = atoi(str + 7);
        return(0);
    }
    if  (strncmp(str, "host=", 5) == 0) {
        bzero(&(tun->to), sizeof(struct sockaddr_in));
        tun->to.sin_port = htons((unsigned short)control_port);
        tun->to.sin_family = AF_INET;
        if  (parse_host_name(str + 5, &addr)) {
            bcopy(&addr, &(tun->to.sin_addr), sizeof(struct in_addr));
        } else {
            hs = gethostbyname(str + 5);
            if  (!hs) {
                printf("Can not find host %s\n", str + 5);
                return(-1);
            }
            bcopy(hs->h_addr_list[0], &addr, sizeof(addr));
            bcopy(&addr, &(tun->to.sin_addr), sizeof(struct in_addr));
        }
        tun->remoteName = strdup(str + 5);
        return(0);
    }
    if  (strncmp(str, "encrypt=", 8) == 0) {
        tun->flags |= TUNF_ENC;
        if  (strcmp(str + 8, "blowfish") == 0) {
            tun->flags |= TUNF_CIPHER_BF;
        } else if(strcmp(str + 8, "idea") == 0) {
            tun->flags |= TUNF_CIPHER_IDEA;
        } else if(strcmp(str + 8, "rc5") == 0) {
            tun->flags |= TUNF_CIPHER_RC5;
        } else {
            Error("Unknown cipher [%s] for tunnel [%s]", str + 8, tun->label);
        }
        return(0);
    }
    return(1);
}

int
parse_config() {
    char str[1024];
    Tunnel *tunnel;
    int line=0;
    char *tmp;
    char *p;
    FILE *f;

    if  (find_services()) return(-1);
    f = fopen(get_real_path(CONFIG_FILE), "r");
    if  (!f) {
        printf("Can not open config file %s\n", get_real_path(CONFIG_FILE));
        return(-1);
    }
    while (fgets(str, 1023, f)) {
        p=str;
        line++;
        while(*p==' ') p++; /*skip blanks*/
        if  (*p == '#' || *p == '\n' || *p == '\r' || *p == 0) continue;
        if  (strncmp(p, "tunnel", 6) == 0) {
            p += 6;
            while (*p == ' ') p++;
            tunnel = (Tunnel *)malloc(sizeof(Tunnel));
            bzero(tunnel, sizeof(Tunnel));
            tunnel->tun_desc = strdup(p);  /*Tunnel line*/
            tmp = strsep(&p, " \t");  /*label*/
            if  (!tmp) Error("Error in config file %s at line %d\n", get_real_path(CONFIG_FILE), line);
            tunnel->label = strdup(tmp);
            tunnel->tunAddr = atoi(tmp);
            while ((tmp = strsep(&p, " \t\n"))) { /* options */
                if  (parse_config_options(tunnel, tmp)) {
                    Error("Illegal option [%s]", tmp);
            }   }
            register_tunnel(tunnel->tunAddr, seq_append(stun, tunnel));
            if  (debug)
                Log( "Tunnel [%s]: tag=%d dp=%d host=[%s] flags=0x%x a=%05hu"
                   , tunnel->label
                   , tunnel->tag
                   , tunnel->dp
                   , tunnel->remoteName
                   , tunnel->flags
                   , tunnel->tunAddr
                   );
    }   } 
    fclose(f);
    if  (control_port && stun->length > 0) return(0);
    return(-1);
}

/*****************************************************************************/
/* RNG                                                                       */
/*****************************************************************************/

static int32_t *rng_F;
static int32_t rng_S;
static int32_t rng_RL = 9689;
static int32_t rng_D = 471;
static u_int32_t *rng_R;
static u_int32_t rng_msk = 0x7FFFFFFF;

int
get_rand() {
    int tmp;
    int n;

    tmp = rng_R[*rng_F] ^ rng_R[rng_S];
    tmp &= rng_msk;
    rng_R[*rng_F] = tmp;
    (*rng_F)++;
    rng_S++;
    if  (*rng_F != rng_RL) {
        if  (rng_S == rng_RL) rng_S = 0;
    } else {
        *rng_F = 0;
    }
    if  (debug) {
        n = *rng_F - rng_D;
        if  (n < 0) n = n + rng_RL;
        if  (n != rng_S) {
            printf("n=%d, rng_S=%d\n", n, rng_S);
            exit(1);
    }   }
    /* msync(rng_F, rng_RL * 4 + 4, MS_SYNC); */
    return(tmp);
}

int
init_rng() {
    int fd;
    struct stat sb;
    int ret;

    fd = open(get_real_path(SEED_FILE), O_RDWR);
    if  (fd < 0) {
        Error("Can not open file %s", get_real_path(SEED_FILE));
        exit(1);
    }
    ret = fstat(fd, &sb);
    if  (ret < 0) {
        perror("Can not get stat of seed file");
        exit(1);
    }
    if  (sb.st_size != rng_RL * 4 + 4) {
        printf("Illegal size %d of seed file\n", (int)sb.st_size);
        exit(1);
    } 
    rng_F = mmap(NULL, rng_RL * 4 + 4, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    if  (rng_F == MAP_FAILED) {
        perror("Can not mmap seed file");
        exit(1);
    }
    if  (*rng_F >= rng_RL || *rng_F < 0) {
        printf("Corrupted RNG seed file\n");
        exit(1);
    }
    rng_R = (u_int32_t*)rng_F + 1;
    rng_S = *rng_F - rng_D;
    if  (rng_S < 0) rng_S = rng_RL + rng_S;
    if  (debug) printf("F=%d, S=%d\n", *rng_F, rng_S);
    /* for (ret = 0; ret < 200000; ret++) printf("%d\n", get_rand()); */
    return(0);
}

/*****************************************************************************/
/* File pathes                                                               */
/*****************************************************************************/

char *
get_real_path(char *path) {
    static char buf[1024];

    sprintf(buf, "%s/etc/%s", INST_DIR, path);
    return(buf);
}

/*****************************************************************************/
/* Lock for ensure for single running copy of tund                           */
/*****************************************************************************/

static int fd_lock;

void
tund_lock() {
    FILE *pidf;
    int ret;

    fd_lock = open(LOCK_FILE, O_CREAT | O_RDWR, 0600);
    if  (fd_lock < 0) Error("Can not open/create %s: error %s", LOCK_FILE, strerror(errno));
    ret = flock(fd_lock, LOCK_EX | LOCK_NB);
    if  (ret < 0) Error("Can not flock %s: error %s", LOCK_FILE, strerror(errno));
    pidf = fdopen(fd_lock, "w+");
    if  (pidf == NULL) Error("Can not fdopen %s: error %s", LOCK_FILE, strerror(errno));
    fprintf(pidf, "%d\n", getpid());
    fflush(pidf);
    if  (debug) Log("TUND is locked");
}

void
tund_unlock() {
    flock(fd_lock, LOCK_UN);
    close(fd_lock);
}
