/*-
 * Copyright Sergey Kosyakov ks@itp.ac.ru 1999
 * Copyright (C) @BABOLO  2002 http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * $Id: tund.h,v 1.2 2010/10/18 16:51:36 babolo Exp $
 */

#ifndef __TUND_H_
#define __TUND_H_

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <openssl/rsa.h>

#define INST_DIR	"/usr/local"
#define CONFIG_FILE	"tund.conf"
#define SEED_FILE	"tund.seed.rng"
#define RSAKEY_FILE	"tund.rsa.key"
#define RSAPUB_DB	"tund.hosts.key"
#define RSAPUBKEY_FILE	"rsa_pub.key"
#define LOCK_FILE	"/var/run/tund.pid"
#define BUF_SIZE        9200

/* Tunnel parameters */
typedef struct {
    int            tag;                /* ipfw divert rule number                    */
    int            dp;                 /* divert port                                */
    int            socketDivert;       /* divert socket                              */
    time_t         timer;              /* operation expiration timer                 */
    unsigned short state;
    unsigned short tunAddr;            /* tunnel unique number                       */
    unsigned int   flags;
    void         (*f_exp)(int);        /* expire callback, arg=tun_ind               */
    RSA           *r_key;              /* remote RSA public key                      */
    void          *localCipherKey;     /* symmetric cipher specific key              */
    void          *remoteCipherKey;    /* symmetric cipher specific key              */
    char          *tun_desc;           /* tunnel description as defined in tund.conf */
    char          *label;              /* tunnel unique label                        */
    char          *remoteName;         /* remote host name                           */
    struct         sockaddr_in l_addr; /* local IP address                           */
    struct         sockaddr_in to;     /* remote host+port address                   */
    unsigned char  local_key[16];      /* local symmetric encryption key             */
    unsigned char  remote_key[16];     /* peer's symmetric encryption key            */
} Tunnel;

/*Tunnel's flags*/
#define TUNF_ENC          1            /* encryption                                 */
#define TUNF_HAS_RPWD	  2            /* tunnel already has remote symmetric passwd */
#define TUNF_CIPHER_BF	 (1<<16)       /* BlowFish cipher                            */
#define TUNF_CIPHER_IDEA (1<<17)       /* IDEA cipher                                */
#define TUNF_CIPHER_RC5	 (1<<18)       /* RC5 cipher                                 */

/* Tunnel's states*/
#define TUNS_INI          0            /* initial state                              */
#define TUNS_RESET_AW     2            /* waiting for reset acknolegement            */
#define TUNS_SPAW         4            /* symmetric password acknolegment waiting    */
#define TUNS_PRC         16            /* tunnel in operation state                  */

typedef struct {                       /* Common use sequence type                   */
    int    length;
    int    maxlen;
    void **buf;
} Seq;

typedef struct {                       /* POLL callbacks                             */
    int    tun_ind;                    /* tunnel number in extern Seq *stun          */
    void (*f)(int, int);               /* tun_ind, socket                            */
} CBF;

/***Protocol packets***/

typedef struct {                       /* Common packet header                       */
    unsigned char ver;	               /* protocol version                           */
    unsigned char flags;               /* packet flags                               */
    unsigned short tunAddr;            /* tunnel unique number                       */
} PacketHead;

typedef struct {                       /* Control packet header                      */
    unsigned int rn;                   /* random number                              */
    unsigned short mt;                 /* message type                               */
    unsigned short body_len;           /* lenght of body                             */
} ControlHead;

/*Message types*/
#define CMT_SPWD          1            /* new symmetric password                     */
#define CMT_SPWD_ACK      2            /* CMT_SPWD receiving acknolegment            */
#define CMT_RESET         3            /* start tunnel ini procedure on peer         */
#define CMT_RESET_ACK     4            /* CMT_RESET acknolegement                    */

typedef struct {                       /* Control packet structure                   */
    PacketHead    h;
    ControlHead   c;
    unsigned char body[BUF_SIZE];
} CP;

typedef struct {                       /* Data packet structure                      */
    PacketHead    h;
    unsigned char body[BUF_SIZE];
} DP;

typedef struct {                       /* Encrypted data packet structure            */
    PacketHead     h;
    unsigned short rn;                 /* random number                              */
    unsigned short body_len;           /* body length                                */
    unsigned char  body[BUF_SIZE];
} EDP;

#define PACKET_MAX_SIZE	(sizeof(CP) + 16)

/*Packet's flags*/
#define PPF_DATA	  1            /* data packet                                */
#define PPF_CNT		  2            /* control packet                             */
#define PPF_PROTO_VER	  1            /* protocol version=1                         */

void add_expired_operation   (int tun_ind, int t, void (*f_exp)(int));
void remove_expired_operation(int tun_ind);
Seq *seq_new                 ();
int  seq_append              (Seq *s, void *item);

/*Log*/
void Log  (char *fmt, ...);
void Error(char *fmt, ...);

/*Socket procedures*/
void read_packet (int tun_ind, int s);
void input_divert(int tun_ind, int s);

/*Security functions*/
void           init_alarm        ();
void           init_secur        ();
void           start_secur_tunnel(int tun_ind);
int            data_encrypt      (unsigned char *buf, int len, int tun_ind);
void           data_decrypt      (unsigned char *buf, int len, int tun_ind);
unsigned char *get_md5_hash      (unsigned char *data, int len);
void           proceed_cp        (int tun_ind, int len);
void           set_cipher_key    (int tun_ind, int remote);
int            cipher_encrypt    (unsigned char *in, unsigned char *out, int len, int tun_ind);
void           cipher_decrypt    (unsigned char *in, unsigned char *out, int len, int tun_ind);

#define TUN_CIPHER_BF	  1
#define TUN_CIPHER_IDEA	  2
#define TUN_CIPHER_RC5	  4

/*Utils*/
void  tund_lock       ();
void  tund_unlock     ();
int   get_rand        ();
int   init_rng        ();
int   parse_config    ();
void  do_it           ();
char *get_real_path   (char *path);
void  init_search     ();
void  register_tunnel (unsigned short tunAddr, int tun_ind);
int   find_tunnel_from(struct sockaddr_in *from, unsigned short tunAddr);
void  ks_sendto       ( int tun_ind
                      , int s
                      , void *msg
                      , size_t len
                      , int flags
                      , struct sockaddr *to
                      , int tolen
                      );

#endif /* __TUND_H_ */
