/*-
 * Copyright Sergey Kosyakov ks@itp.ac.ru 1999
 * Copyright (C) @BABOLO  2002 http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] = "\
@(#)Copyright Sergey Kosyakov ks@itp.ac.ru 1999\n\
@(#)Copyright (C) @BABOLO  2002 http://www.babolo.ru/\n\
@(#)All rights reserved.\n";
static const char rcsid[] = "$Id: scb.c,v 1.2 2010/10/18 16:51:36 babolo Exp $";
#endif /* not lint */

#include <sys/socket.h>
#include <sys/types.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include "tund.h"

unsigned char packetBuf[PACKET_MAX_SIZE];
extern int socket_control;
extern int control_port;
extern int debug;
extern Seq *stun;

static void input_packet(int tun_ind,int s,int len);
static void input_control(int tun_ind,int s,int len);

static int
ks_recvfrom(int s, void *buf, size_t len, int flags, struct sockaddr *from, socklen_t *fromlen) {
    int ret;

    ret = recvfrom(s, buf, len, flags, from, fromlen);
#ifdef DEBUG
    /* if  (debug) Log("Recvfrom packet on socket %d len=%d", s, ret); */
#endif /*DEBUG*/
    if  (ret < 0) Error("Error in ks_recvfrom(s=%d) while recvfrom: %s", s, strerror(errno));
    return(ret);
}

void
ks_sendto(int tun_ind, int s, void *msg, size_t len, int flags, struct sockaddr *to, int tolen) {
    int ret;

    ret=sendto(s,msg,len,flags,to,tolen);
    if  (ret < 0) {
        if  (errno == EINVAL) {
            Log("Dropped illegal IP packet (t=%d,so=%d,l=%d)", tun_ind, s, len);
            return;
        } else if(errno == EHOSTDOWN) {
            Log("Error %d while sendto(t=%d,s=%d): %s", errno, tun_ind, s, strerror(errno));
            return;
        } else {
            Error("Error %d while sendto(t=%d,s=%d): %s", errno, tun_ind, s, strerror(errno));
    }   }
    if  (ret != len) Error("Error while sendto(t=%d,s=%d): %d!=%d", tun_ind, s, ret,len);
#ifdef DEBUG
    /* if  (debug) Log("Sended %d bytes to socket %d", ret, s); */
#endif /*DEBUG*/
}

void
read_packet(int tun_ind, int s) {
    struct sockaddr_in from;
    PacketHead *phead;
    int len;
    socklen_t fromlen;

    fromlen = sizeof(from);
    len = ks_recvfrom(s, packetBuf, PACKET_MAX_SIZE, 0, (struct sockaddr *)&from, &fromlen);
    phead = (PacketHead *)packetBuf;
    if  (len < sizeof(PacketHead) + 1 || phead->ver != PPF_PROTO_VER) {
        Log( "Dropped packet (illegal packet) l=%d from %s:%d lh4=0x%x f=0x%x v=%d"
           , len
           , inet_ntoa(from.sin_addr)
           , (int)ntohs(from.sin_port)
           , phead->tunAddr
           , phead->flags
           , phead->ver
           );
        return;
    }
    tun_ind = find_tunnel_from(&from, phead->tunAddr);
    if  (tun_ind < 0) {
        Log( "Dropped packet (unknown tunnel) l=%d from %s:%d lh4=0x%x f=0x%x v=%d"
           , len
           , inet_ntoa(from.sin_addr)
           , (int)ntohs(from.sin_port)
           , phead->tunAddr
           , phead->flags
           , phead->ver
           );
        return;
    }
    switch(phead->flags & 0x0003) {
    case PPF_DATA:
        input_packet(tun_ind,s,len);
        break;
    case PPF_CNT :
        input_control(tun_ind,s,len);
        break;
    default      :
        Log( "Dropped packet len=%d from %s:%d lh4=0x%x flags=0x%x"
           , len
           , inet_ntoa(from.sin_addr)
           , (int)ntohs(from.sin_port)
           , phead->tunAddr
           , phead->flags
           );
        return;
}   }

static void
input_packet(int tun_ind, int s, int buf_len) {
    Tunnel *tun;
    DP *data;
    EDP *en_data;
    int body_len;
    unsigned char *body;

    tun = (Tunnel *)stun->buf[tun_ind];
    if  (tun->flags & (TUNF_ENC | TUNF_HAS_RPWD)) {
        en_data = (EDP *)packetBuf;
        data_decrypt(packetBuf + sizeof(PacketHead), buf_len-sizeof(PacketHead), tun_ind);
        body = en_data->body;
        body_len = en_data->body_len;
    } else {
        data = (DP *)packetBuf;
        body_len = buf_len-sizeof(PacketHead);
        body = data->body;
    }
#ifdef DEBUG
    if  (debug) Log("Received %d bytes from tunnel [%s]", body_len, tun->label);
#endif /*DEBUG*/
    if  (tun->state & TUNS_PRC) {
        ks_sendto( tun_ind
                 , tun->socketDivert
                 , body
                 , body_len
                 , 0
                 , (struct sockaddr*)&(tun->l_addr)
                 , sizeof(struct sockaddr_in)
                 );
#ifdef DEBUG
    } else if(debug) {
        Log("Dropped %d bytes on tunnel [%s]", body_len, tun->label);
#endif /*DEBUG*/
}   }

static void
input_control(int tun_ind, int s, int buf_len) {
    Tunnel *tun;

    tun = (Tunnel *)stun->buf[tun_ind];
#ifdef DEBUG
    if  (debug) Log("Received %d bytes from control of tunnel [%s]", buf_len, tun->label);
#endif /*DEBUG*/
    proceed_cp(tun_ind, buf_len);
}

void
input_divert(int tun_ind, int s) {
    struct sockaddr from;
    socklen_t fromlen = sizeof(from);
    EDP *en_data;
    int buf_len;
    Tunnel *tun;
    DP *data;
    int len;

    tun = (Tunnel *)stun->buf[tun_ind];
#ifdef DEBUG
    if  (debug) Log("Input packet on divert socket %d", s);
#endif /*DEBUG*/
    if  (tun->flags & TUNF_ENC) {
        en_data = (EDP *)packetBuf;
        len = ks_recvfrom(s, en_data->body, BUF_SIZE, 0, &from, &fromlen);
        en_data->h.tunAddr = tun->tunAddr;
        en_data->h.flags = PPF_DATA;
        en_data->h.ver = PPF_PROTO_VER;
        en_data->rn = get_rand();
        en_data->body_len = (unsigned short)len;
        if  ((buf_len = data_encrypt(packetBuf + sizeof(PacketHead), len + 4, tun_ind)) < 0) {
            Log("No valid encription");
            return;
        }
        buf_len += sizeof(PacketHead);
    } else {
        data = (DP *)packetBuf;
        len = ks_recvfrom(s, data->body, BUF_SIZE, 0, &from, &fromlen);
        buf_len = len+sizeof(PacketHead);
        data->h.tunAddr = tun->tunAddr;
        data->h.flags = PPF_DATA;
        data->h.ver = PPF_PROTO_VER;
    }
    if  (tun->state & TUNS_PRC) {
        ks_sendto( tun_ind
                 , socket_control
                 , packetBuf
                 , buf_len
                 , 0
                 , (struct sockaddr*)(&(tun->to))
                 , sizeof(struct sockaddr_in)
                 );
#ifdef DEBUG
    } else {
        if  (debug) Log("Dropped %d bytes on tunnel [%s]", len, tun->label);
#endif /*DEBUG*/
}   }
