/*-
 * Copyright Sergey Kosyakov ks@itp.ac.ru 1999
 * Copyright (C) @BABOLO  2002 http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] = "\
@(#)Copyright Sergey Kosyakov ks@itp.ac.ru 1999\n\
@(#)Copyright (C) @BABOLO  2002 http://www.babolo.ru/\n\
@(#)All rights reserved.\n";
static const char rcsid[] = "$Id: alarm.c,v 1.1 2010/10/15 11:32:04 babolo Exp $";
#endif /* not lint */

#include <stdlib.h>
#include <sys/time.h>
#include <signal.h>
#include <errno.h>
#include "tund.h"

#define	TIMER_GRAN	10	                                    /* alarm timer granularity, sec. */

extern Seq *stun;
extern int debug;
static int has_exp_oper=0;

static void
h_sigalarm(int sig, int code, struct sigcontext *sc) {
    Tunnel *tun;
    int i;

#ifdef DEBUG
    if  (debug) Log("h_sigalarm(%d,%d,0x%x)", sig, code, sc);
#endif /* DEBUG */
    if  (!has_exp_oper) return;
    if  (debug) Log("Has %d expired operations", has_exp_oper);
    has_exp_oper = 0;
    for (i = 0; i < stun->length; i++) {
        tun = (Tunnel *)stun->buf[i];
        if  (tun->f_exp) {
            has_exp_oper++;
            if  ((--(tun->timer)) <= 0) {
#ifdef DEBUG
              if  (debug) Log("Call 0x%x on tunnel #%d", tun->f_exp, i);
#endif /* DEBUG */
              tun->f_exp(i);
              has_exp_oper--;
}   }   }   }

void
init_alarm() {
    struct itimerval ival;
    struct sigaction sa;
    int ret;

    sa.sa_handler = h_sigalarm;
    sigemptyset(&sa.sa_mask);
    sigaddset(&sa.sa_mask, SIGALRM);
    sa.sa_flags = SA_RESTART;
    ret = sigaction(SIGALRM, &sa, NULL);
    if  (ret) Error("Can not set sigaction(SIGALRM): %s", strerror(errno));
    ival.it_interval.tv_sec = TIMER_GRAN;
    ival.it_interval.tv_usec = 0;
    gettimeofday(&(ival.it_value), NULL);
    ival.it_value.tv_sec =+ TIMER_GRAN; /* XXXX += ??? */
    ival.it_value.tv_usec = 0;
    ret = setitimer(ITIMER_REAL, &ival, NULL);
    if  (ret) Error("Can not setitimer");
}

void
add_expired_operation(int tun_ind, int t, void (*f_exp)(int)) {
    Tunnel *tun = (Tunnel *)stun->buf[tun_ind];
    sigset_t sset;

    sigemptyset(&sset);
    sigaddset(&sset, SIGALRM);
    sigprocmask(SIG_BLOCK, &sset, NULL);
    tun->timer = t / TIMER_GRAN;
    tun->f_exp = f_exp;
    has_exp_oper++;
    if  (debug) Log("Set up expiration callback 0x%x for %d sec on tunnel [%s]", f_exp, t, tun->label);
    sigprocmask(SIG_UNBLOCK, &sset, NULL);
}

void
remove_expired_operation(int tun_ind) {
    Tunnel *tun = (Tunnel *)stun->buf[tun_ind];
    sigset_t sset;

    sigemptyset(&sset);
    sigaddset(&sset, SIGALRM);
    sigprocmask(SIG_BLOCK, &sset, NULL);
    if  (!tun->f_exp) goto L_end;
    tun->timer = 0;
    tun->f_exp = NULL;
    has_exp_oper--;
    if  (debug) Log("Remove expiration callback tunnel [%s]", tun->label);
L_end:
    sigprocmask(SIG_UNBLOCK, &sset, NULL);
}
