/*-
 * Copyright (C)2020..2025 @BABOLO http://www.babolo.ru/
 * Inspired by su2-1.3.
 * This version written from scratch and FreeBSD's /usr/bin/su

 * Copyright (c) 2002, 2005 Networks Associates Technologies, Inc.
 * All rights reserved.
 *
 * Portions of this software were developed for the FreeBSD Project by
 * ThinkSec AS and NAI Labs, the Security Research Division of Network
 * Associates, Inc.  under DARPA/SPAWAR contract N66001-01-C-8035
 * ("CBOSS"), as part of the DARPA CHATS research program.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Copyright (c) 1988, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2020..2025 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: prep.c,v 1.26 2025/03/15 16:31:42 babolo Exp $"

#define BLIN_COMPAT  3
#define Bpars_COMPAT 4
#define MIFE_COMPAT  5

#include <sys/resource.h>
#include <sys/types.h>

#ifndef BLIN_OSLinux
# include <login_cap.h>
#endif

#include <sys/event.h>
#include <sys/stat.h>
#include <sysexits.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <paths.h>
#include <pwd.h>
#include <grp.h>
#include <err.h>

#ifndef BLIN_OSLinux
# include <security/pam_appl.h>
#endif

#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <mife.h>
#include "su2.h"
#include "su2_env.h"

extern char      **environ;
static char       *cleanenv = NULL;
static char const *su_env[] =
{ "MM_CHARSET", "WINDOWPATH", "ENV"  , "PWD"
, "XAUTHORITY", "INFOPATH"  , "USER" , "HOME"
, "LOGNAME"   , "MANPATH"   , "LANG" , "PATH"
, "TMPDIR"    , "LC_ALL"    , "SHELL", "GROUP"
, "SHLVL"     ,         NULL
};

static su2_pass *
/*****************************************************************************************************
 ****    passwd ( == su2_pass)                                                   ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_copw(su2_cf *cf, su2_pass *pw) {                                                             /****
 ****                                                                                             ****
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    su2_pass  *tpw = NULL;
    size_t     stp = 0;
    char      *p;

    ifBLIN_QX3("+");
    if  (!pw) {
        errno = 0;
        goto out;
    }
    if  (!!pw->pw_name)   stp += strlen(pw->pw_name)   + 1;
    if  (!!pw->pw_passwd) stp += strlen(pw->pw_passwd) + 1;
#ifndef BLIN_OSLinux
    if  (!!pw->pw_class)  stp += strlen(pw->pw_class)  + 1;
#endif
    if  (!!pw->pw_gecos)  stp += strlen(pw->pw_gecos)  + 1;
    if  (!!pw->pw_dir)    stp += strlen(pw->pw_dir)    + 1;
    if  (!!pw->pw_shell)  stp += strlen(pw->pw_shell)  + 1;
    if  (!(tpw = malloc(sizeof(su2_pass) + stp))) {
        ifBLIN_QW0("Malloc failed #3");
        goto out;
    }
    bcopy(pw, tpw, sizeof(su2_pass));
    p = (char*)(void*)tpw + sizeof(su2_pass);
    if  (!!pw->pw_name) {
        tpw->pw_name = p;
        p = stpcpy(p, pw->pw_name) + 1;
    }
    if  (!!pw->pw_passwd) {
        tpw->pw_passwd = p;
        p = stpcpy(p, pw->pw_passwd) + 1;
    }
#ifndef BLIN_OSLinux
    if  (!!pw->pw_class) {
        tpw->pw_class = p;
        p = stpcpy(p, pw->pw_class) + 1;
    }
#endif
    if  (!!pw->pw_gecos) {
        tpw->pw_gecos = p;
        p = stpcpy(p, pw->pw_gecos) + 1;
    }
    if  (!!pw->pw_dir) {
        tpw->pw_dir = p;
        p = stpcpy(p, pw->pw_dir) + 1;
    }
    if  (!!pw->pw_shell) {
        tpw->pw_shell = p;
        p = stpcpy(p, pw->pw_shell) + 1;
    }
out:
    ifBLIN_QX3("- %"BLIN_X, BLIN_I(tpw));
    return(tpw);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****    pw  passwd ( == su2_pass),  uname ( ,           ****
 ****   uid ), ,  uname == NULL,   uid.  c == ':',  UID           ****
 ****   uname   uid,    pw,   uid.              ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_iniud(su2_cf *cf, char c, const char *uname, su2_pass **pw, uid_t uid) {                     /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    int         ex = EX_OK;
    char       *tail;
    su2_pass   *tpw;

    ifBLIN_QX3("+ -%c %s %d", c, !uname ? "" : uname, uid);
    *pw = NULL;
    if  (!!*pw) {
        ifBLIN_QX0("UID redefinition");
        errno = EUSERS;
        ex = EX_USAGE;
        goto out;
    } else if (!!uname) {
        errno = 0;
        tpw = getpwnam(uname);
        if  (!!errno) {
            ifBLIN_QW0("getpwnam %s", uname);
            ex = EX_NOPERM;
            goto out;
        } else if (!!tpw) {
            if  ((c == ':') && (tpw->pw_uid != uid)) goto next;
            goto ready;
        }
        errno = 0;
        uid = (uid_t)strtoul(uname, &tail, 0);
        if  (errno) {
            ifBLIN_QW0("-%c strtoul=%s~", c, uname);
            ex = EX_USAGE;
            goto out;
        } else if (*tail) {
            ifBLIN_QX0("-%c text=%s~", c, uname);
            errno = EDOM;
            ex = EX_USAGE;
            goto out;
    }   }
next:
    errno = 0;
    tpw = getpwuid(uid);
    if  (!!errno) {
        ifBLIN_QW0("getpwuid %d", uid);
        ex = EX_NOPERM;
        goto out;
    } else if (!tpw) {
        ifBLIN_QW0("No %d UID", uid);
        ex = EX_USAGE;
        goto out;
    }
ready:
    if  (!(*pw = malloc(sizeof(su2_pass)))) {
        ifBLIN_QW0("Malloc failed #2");
        ex = EX_OSERR;
        goto out;
    } else if (!(*pw = su2_copw(cf, tpw))) {
        ifBLIN_QW0("copw");
        ex = EX_OSERR;
        goto out;
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   nice                                                   ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_nice(su2_cf *cf) {                                                                           /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    int ex = EX_OK;

    ifBLIN_QX3("+ nice before %d", cf->nice);
    if  (!!cf->topw->pw_uid && !!cf->mypw->pw_uid && (0 > cf->nice)) {
        ifBLIN_QX1("Not allowed nice %d", cf->nice);
        cf->nice = 0;
    }
    errno = 0;
    cf->nice += getpriority(PRIO_PROCESS, 0);
    if  (!!errno) {
        ifBLIN_QW0("getpriority");
        ex = EX_OSERR;
        goto out;
    }
    if  (-20 > cf->nice) {
        ifBLIN_QX1("nice corrected %d->%d", cf->nice, -20);
        cf->nice = -20;
    }
    if  (20 < cf->nice) {
        ifBLIN_QX1("nice corrected %d->%d", cf->nice, 20);
        cf->nice = 20;
    }
    if  (!!setpriority(PRIO_PROCESS, 0, cf->nice)) ifBLIN_QW1("setpriority");
out:
    ifBLIN_QX3("- %d nice after %d", ex, cf->nice);
    return(ex);
#   undef blin_internal_flags
}

void
/*****************************************************************************************************
 *****************************************************************************************************
 ****                                                                        ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_dump(su2_cf *cf) {                                                                           /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    ifBLIN_QX0( "flags %08X %u:%u tty=[%d]%s nice=%d sh=%s\n"
                "from=[%d]%s to=[%d]%s by=[%d]%s(%d:%d) arg[%d..%d]\n"
                "command=%s%c"
              , cf->flags
              , cf->uid
              , cf->gid
              , cf->ttyfd
              , !!cf->tty ? cf->tty : ""
              , cf->nice
              , cf->shell
              , cf->frompw->pw_uid
              , cf->frompw->pw_name
              , cf->topw->pw_uid
              , cf->topw->pw_name
              , cf->mypw->pw_uid
              , cf->mypw->pw_name
              , cf->uid
              , cf->gid
              , cf->argp
              , cf->amax
              , !!cf->command ? cf->command : ""
              , !!cf->command ? '~' : 0
              );
    for (uint i = 0; (i < cf->amax) || !!cf->args[i]; ++i) {
        ifBLIN_QX0( "%c[%3u]%s%s~"
                  , (i == cf->argp) ? ((i == cf->amax) ? '#' : '*') : ((i == cf->amax) ? '+' : ' ')
                  , i
                  , cf->args[i] ? "=" : ""
                  , cf->args[i] ? cf->args[i] : "");
    }
    return;
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****      tty                                                                 ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_tty(su2_cf *cf) {                                                                            /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    int           ex   = EX_OK;
    const char   *o;

    ifBLIN_QX3("+");
    for (int i = 2; i >= 0; --i) {
        if  (!!isatty(i) && !!(o = ttyname(i))) {
            cf->ttyfd = i;
            if  (!(cf->tty = strdup(o))) {
                ifBLIN_QW0("strdup=%s~", o);
                ex = EX_OSERR;
                goto out;
            }
            break;
    }   }
    if  (!cf->tty) ifBLIN_QX1("No tty");
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****                                                                               ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_shell(su2_cf *cf) {                                                                          /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    int           ex   = EX_OK;
    const char   *o;

    ifBLIN_QX3("+");
    if  (!cf->xfile) {
        setusershell();
        if  (!!cf->topw->pw_uid && !!cf->mypw->pw_uid) {
            for (;;) {
                if  (!(o = getusershell())) {
                    cf->shell = _PATH_BSHELL;
                    break;
                }
                if  (!strcmp((SU2_LOGIN & cf->flags) ? cf->topw->pw_shell : cf->mypw->pw_shell, o)) {
                    cf->shell = (SU2_LOGIN & cf->flags) ? cf->topw->pw_shell : cf->mypw->pw_shell;
                    break;
            }   }
            endusershell();
        } else {
            cf->shell = (SU2_LOGIN & cf->flags) ? cf->topw->pw_shell : cf->mypw->pw_shell;
        }
        if  (!(SU2_DASH & cf->flags)) {
            if  (!!cf->command) {
                cf->args[cf->argp] = cf->command;
                cf->amax++;
                cf->args[--cf->argp] = "-c";
            }
            o = strrchr(cf->shell, '/');
            if  (o) ++o; else o = cf->shell;
            if  (SU2_LOGIN & cf->flags) {
                cf->args[--cf->argp] = "-su2";
            } else if (strcmp(o, "csh") || strcmp(o, "tcsh")) {
                cf->args[--cf->argp] = "-m";
                cf->args[--cf->argp] = "_su2";
            } else {
                cf->args[--cf->argp] = "su2";
        }   }
    } else {
        cf->args[cf->argp] = NULL;
        cf->args[--cf->argp] = cf->xfile;
        cf->args[--cf->argp] = "/bin/sh";
        cf->shell = _PATH_BSHELL;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static su2_found
/*****************************************************************************************************
 ****                                                ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_finduser(su2_cf *cf, char const *path) {                                                     /****
 ****                                                                                             ****
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    su2_found        ex = SU2_FNDNONE;
    char            *window;
    char            *p;

    ifBLIN_QX3("+ %s %s", path, cf->frompw->pw_name);
    if  (!(cf->md = mife_init(MIFE_FULL | MIFE_BUFX | 0x0F01))) {
        ifBLIN_QW0("mife_init");
        ex = SU2_FNDFAIL;
        goto out;
    }
    if  (0 > mife_ctlfile(cf->md, path)) {
        ifBLIN_QW0("mife_ctlfile %s", path);
        ex = SU2_FNDFAIL;
        goto out;
    }
    if  (!(window = mife_window(cf->md, 0, 0))) {
        ifBLIN_QW0("mife_window %s", path);
        ex = SU2_FNDFAIL;
        goto out;
    }
    for (p = strstr(window, "+*"); !!p; p = strstr(p, "+*")) {
        ifBLIN_QX4("full %"BLIN_D, BLIN_I(p - window));
        if  (' ' < p[2]) continue;
        if  ((p == window) || (' ' >= *(p - 1))) {
            ifBLIN_QX4("FULL %"BLIN_D, BLIN_I(p - window));
            ex = SU2_FNDFULL;
            goto out;
    }   }
    for (p = strchr(window, '*'); !!p; p = strchr(p, '*')) {
        ifBLIN_QX4("star %"BLIN_D, BLIN_I(p - window));
        if  (' ' < p[1]) continue;
        if  ((p == window) || (' ' >= *(p - 1))) {
            ifBLIN_QX4("STAR %"BLIN_D, BLIN_I(p - window));
            ex = SU2_FNDSTAR;
            goto out;
    }   }
    for (p = strstr(window, cf->frompw->pw_name); !!p; p = strstr(p, cf->frompw->pw_name)) {
        ifBLIN_QX4("user %"BLIN_D, BLIN_I(p - window));
        if  (' ' < p[strlen(cf->frompw->pw_name)]) continue;
        if  (p == window) {
            ifBLIN_QX4("USER %"BLIN_D, BLIN_I(p - window));
            ex = SU2_FNDUSER;
            goto out;
        }
        if  ('+' == *(p - 1)) {
            ifBLIN_QX4("PLUS %"BLIN_D, BLIN_I(p - window));
            ex = SU2_FNDPLUS;
            goto out;
        }
        if  (' ' >= *(p - 1)) {
            ifBLIN_QX4("USER %"BLIN_D, BLIN_I(p - window));
            ex = SU2_FNDUSER;
            goto out;
    }   }
out:
    ifBLIN_QX3("- %02X", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****                                                                  ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_rc(su2_cf *cf) {                                                                             /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    int           ex   = EX_OK;
    struct stat   sb;
    char         *p;

    ifBLIN_QX3("+ %"BLIN_X, BLIN_I(cf));
    /* cf->cond = SU2_FNDNONE; from calloc() */
    /* cf->next = SU2_FNDNONE; from calloc() */
    if  (!cf->mypw->pw_uid) {
        cf->cond = SU2_FNDFULL;
        goto out;
    }
    if  (cf->mypw->pw_uid == cf->topw->pw_uid) {
        cf->next = SU2_FNDFULL;
        goto out;
    }
    if  (!!stat(SYSRC, &sb)) {
        ifBLIN_QW1("stat %s", SYSRC);
    } else if ((sb.st_uid != 0) || ((S_IWUSR | S_IWGRP | S_IWOTH) & sb.st_mode)) {
        ifBLIN_QW1("Illegal rights %s", SYSRC);
    } else if (SU2_FNDFAIL == (cf->cond = su2_finduser(cf, SYSRC))) {
        ifBLIN_QW1("IO fail %s", SYSRC);
    } else if (SU2_NOPASS & ~cf->cond) {
        if  (0 > asprintf(&p, "%s/%s", cf->topw->pw_dir, USERC)) {
            ex = EX_OSERR;
            goto out;
        }
        if  (!!stat(p, &sb)) {
            ifBLIN_QW1("stat %s", p);
        } else if (  !(S_ISUID & sb.st_mode)
                  || ((S_IWGRP | S_IWOTH) & sb.st_mode)
                  || ((sb.st_uid != cf->topw->pw_uid) && (sb.st_uid != 0))
                  || (sb.st_nlink != 1)
                  ) {
            ifBLIN_QX0("Illegal rights %s, must S------ &! -w--w- & link == 1 & uid", p);
        } else if (SU2_FNDFAIL == (cf->next = su2_finduser(cf, p))) {
            ifBLIN_QW1("IO fail %s", p);
    }   }
    ifBLIN_QX3("cond %02X next %02X", cf->cond, cf->next);
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****     fork()                                                           ****
 *****************************************************************************************************
 ****                                                                                             ****/
su2_posenv( su2_cf *cf                                                                           /****/
#ifndef BLIN_OSLinux
          , login_cap_t *lc                                                                      /****/
#endif
          ) {                                                                                    /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    char         **envpam;
    struct group  *grent;
    int            ex = EX_OK;
    const char    *p;
    char         **e;
    char          *v;

    ifBLIN_QX3("+ %"BLIN_X, BLIN_I(cf));
#ifdef BLIN_OSLinux
    if  (!!setgid(cf->topw->pw_gid)) {
        ifBLIN_QW0("setgid %u", cf->topw->pw_gid);
        ex = EX_NOPERM;
        goto out;
    }
    if  (!!setuid(cf->topw->pw_uid)) {
        ifBLIN_QW0("setuid %u", cf->topw->pw_uid);
        ex = EX_NOPERM;
        goto out;
    }
#else
    if  (0 > setusercontext( lc
                           , cf->topw
                           , cf->topw->pw_uid
                           , LOGIN_SETUSER | LOGIN_SETCPUMASK | LOGIN_SETLOGINCLASS
        )                  ) {
        ifBLIN_QW0("setusercontext(SETUSER SETCPUMASK SETLOGINCLASS)");
        ex = EX_OSERR;
        goto out;
    }
#endif
    ifBLIN_QX6("uid %d", getuid());
    if  (SU2_LOGIN & cf->flags) {
        p = getenv("TERM");
        environ = &cleanenv;
        if  (!!setenv("HOME", cf->topw->pw_dir, 1)) {
            ifBLIN_QW1("setenv(HOME, %s, 1)", cf->topw->pw_dir);
        }
        if  (!!setenv("SHELL", cf->shell, 1)) ifBLIN_QW1("setenv(SHELL, %s, 1)", cf->shell);
#ifndef BLIN_OSLinux
        if  (!!(envpam = pam_getenvlist(cf->pamh))) {
            for (e = envpam; !!*e; ++e) {
                if  (1024 < strlen(*e)) continue;
                if  (!strncmp(*e, "LD_", 3)) continue;
                if  (!(v = strchr(*e, '='))) continue;
                *(v++) = '\0';
                if  (1 == babolo_testword(&env, *e)) continue;
                if  (!!setenv(*e, v, 1)) ifBLIN_QW1("setenv(setenv(%s, %s, 1)", *e, v);
        }   }
        if  (0 > setusercontext( lc
                               , cf->topw
                               , cf->topw->pw_uid
                               , LOGIN_SETPATH | LOGIN_SETUMASK | LOGIN_SETENV | LOGIN_SETRESOURCES
            )                  ) {
            ifBLIN_QW0("setusercontext(LOGIN_SETPATH LOGIN_SETUMASK LOGIN_SETENV)");
            ex = EX_OSERR;
            goto out;
        }
#endif
        if  (!!p && !!setenv("TERM", p, 1)) ifBLIN_QW1("setenv(TERM, %s, 1)", p);
        p = NULL;
#ifndef BLIN_OSLinux
        p = pam_getenv(cf->pamh, "HOME");
#endif
        if  (!!chdir(p ? p : cf->topw->pw_dir)) {
            ifBLIN_QW0("chdir %s", p ? p : cf->topw->pw_dir);
            ex = EX_UNAVAILABLE;
            goto out;
    }   }
    for (int i = 0; !!su_env[i]; ++i) {
        if  (!(p = getenv(su_env[i]))) {
            ifBLIN_QW1("getenv %s", su_env[i]);
        } else {
            asprintf(&v, "SU2_%s", su_env[i]);
            if  (!v) {
                ifBLIN_QW1("asprintf %s", su_env[i]);
            } else {
                if  (!!setenv(v, p, 1)) ifBLIN_QW1("setenv(%s, %s, 1)", v, p);
                free(v);
    }   }   }
    asprintf(&v, "%u", cf->frompw->pw_uid);
    if  (!v) {
        ifBLIN_QW1("asprintf frompw->pw_uid");
    } else {
        if  (!!setenv("SU2_REAL_UID", v, 1)) ifBLIN_QW1("setenv(SU2_REAL_UID, %s, 1)", v);
        free(v);
    }
    asprintf(&v, "%u", cf->frompw->pw_gid);
    if  (!v) {
        ifBLIN_QW1("asprintf frompw->pw_gid");
    } else {
        if  (!!setenv("SU2_REAL_GID", v, 1)) ifBLIN_QW1("setenv(SU2_REAL_GID, %s, 1)", v);
        free(v);
    }
    if  (!!setenv("SU2_REAL_UNAME", cf->frompw->pw_name, 1)) {
        ifBLIN_QW1("setenv(SU2_REAL_UNAME, %s, 1)", cf->frompw->pw_name);
    }
    errno = 0;
    if  (!(grent = getgrgid(cf->frompw->pw_gid))) {
        if  (!errno) {
            ifBLIN_QX1("No group %u", cf->frompw->pw_gid);
        } else {
            ifBLIN_QW1("getgrgid %u", cf->frompw->pw_gid);
        }
    } else if (!!setenv("SU2_REAL_GNAME", grent->gr_name, 1)) {
        ifBLIN_QW1("setenv(SU2_REAL_GNAME, %s, 1)", grent->gr_name);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}
