/*-
 * Copyright (C)2020..2021 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2020..2021 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: util.c,v 1.24 2022/08/28 00:56:28 babolo Exp $"

#define MIFE_COMPAT     5
#define BLIN_COMPAT     4
#define RECOBE_COMPAT   VMAJOR
#define RECOBE_INTERNAL 1

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <malloc_np.h>
#include <babolo/BLINflag.h>
#include <mife.h>
#include "recobe.h"

static recobe_chunk *
/*****************************************************************************************************
 ****  convertor--puller,      mife                            ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_getmife(recobe_chain *chain) {                                                            /****
                                                                                                  ****
 *****************************************************************************************************/
    recobe_chunk    *chunk = NULL;
    off_t            toff = 0;
    ssize_t          insz;
    u_char          *inin;
    mife_descriptor *md;

#   define blin_internal_flags (chain->flags)
    RECOBE_CHAIN(chain);
    if  (!chain->converdata) {
        ifBLIN_QX0("No converdata");
        errno = EFAULT;
        goto out;
    }
    md = chain->converdata;
    ifBLIN_QX3("+ %08X %p of=%lld ob=%u", chain->flags, chain, md->ofpoint, chain->outblock);
    if  (chain->flags & RECOBE_USED) toff = md->ofpoint + chain->outblock;
    ifBLIN_QX3("  toff=%"BLIN_O"d", toff);
    if  (!(inin = mife_window(md, toff, (ssize_t)chain->inblock))) {
        ifBLIN_QW0("mife_window(,%"BLIN_D",%"BLIN_O"d)", toff, (ssize_t)chain->inblock);
        goto out;
    }
    if  (0 > (insz = mife_ctlsize(md))) {
        ifBLIN_QW0("mife_ctlsize");
        goto out;
    } else if (insz > (ssize_t)chain->inblock) {
        insz = (ssize_t)chain->inblock;
    }
    chain->outblock = insz;
    chain->flags |= RECOBE_USED | RECOBE_DOEND;
    if  (!(chunk = recobe_chunknew(chain, insz, insz, !insz ? NULL : inin))) {
        ifBLIN_QW0("recobe_chunknew %d", insz);
        goto out;
    }
out:
    RECOBE_CHUNK(chunk);
    ifBLIN_QX3("- %"BLIN_X, chunk);
    return(chunk);
#   undef blin_internal_flags
}

recobe_chain *
/*****************************************************************************************************
 *****************************************************************************************************
 ****  chain     mife                                                      ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_openmd(recobe_conf *cf, mife_descriptor *md, u_int32_t chunksize) {                       /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    recobe_chain *chain = NULL;

    ifBLIN_QX3("+ [%d]", chunksize);
    if  (!(chain = malloc(sizeof(recobe_chain)))) {
        ifBLIN_QW0("malloc(%d)", sizeof(recobe_chain));
        goto out;
    }
    chain->converdata = md;
    chain->cf = cf;
    chain->flags = (cf->flags & ~(RECOBE_USED | RECOBE_DOEND)) | RECOBE_CHAINER;
    chain->tail = NULL;
    chain->next = NULL;
    chain->outblock = 0;
    chain->inblock = chunksize;
    chain->puller = recobe_getmife;
    chain->cnv = (recobe_conv){0, 0, 1, 0, 1};
    chain->bsubst = cf->bsubst;
    chain->esubst = cf->esubst;
    chain->bcode = cf->bcode;
    chain->ecode = cf->ecode;
out:
    RECOBE_CHAIN(chain);
    ifBLIN_QX3("- %"BLIN_X, chain);
    return(chain);
#   undef blin_internal_flags
}

recobe_chain *
/*****************************************************************************************************
 *****************************************************************************************************
 ****  chain     filename                                                   ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_openfile(recobe_conf *cf, const char *filename, u_int32_t chunksize) {                    /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    recobe_chain    *chain = NULL;
    mife_descriptor *md = NULL;

    ifBLIN_QX3("+ %s[%d]", filename, chunksize);
    if  (!(md = mife_init(((cf->flags >> 1) & BLIN_GEN1) | MIFE_SLID))) {
        ifBLIN_QW0("mife_init");
        goto out;
    }
    if  (!!(filename)) {
        if  (0 > mife_ctlfile(md, filename)) {
            ifBLIN_QW0("mife_ctlfile(,%s)", filename);
            goto out;
        }
    } else {
        if  (!!fflush(stdin)) ifBLIN_QW1("fflush");
        if  (0 > mife_ctlfdsc(md, fileno(stdin))) {
            ifBLIN_QW0("mife_ctlfdsc(,%d)", fileno(stdin));
            goto out;
    }   }
    chain = recobe_openmd(cf, md, chunksize);
out:
    ifBLIN_QX3("- %"BLIN_X, chain);
    return(chain);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****    chain,  recobe_openfile()  recobe_openmd()               ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_close(recobe_chain *chain) {                                                              /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    int ex = 0;

    if  (chain->next) ex = recobe_close(chain->next);
    chain->next = NULL;
    if  (RECOBE_CNVDFREE & chain->flags) {
        free(chain->converdata);
        chain->converdata = NULL;
    }
    free(chain->tail);
    chain->tail = NULL;
    free(chain);
    return(ex);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   wchr   chunk,                                    ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_inword(recobe_chain *chain, recobe_chunk *chunk, u_int64_t *wchr) {                       /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    int64_t       l = (chain->inwlen > 8) ? 8 : chain->inwlen;
    const u_char *c = recobe_chunktext(chunk);
    int           ex = 0;

    ifBLIN_QX3( "+ %08X %"BLIN_X" %"BLIN_X" %u+%u %u"
              , chain->flags
              , BLIN_I(chain)
              , BLIN_I(chunk)
              , chunk->position
              , l
              , chunk->length
              );
    if  (!recobe_chunksize(chunk)) goto out;
    if  (chunk->position + l > chunk->length) {
        ifBLIN_QX2("Long %u+%u %u", chunk->position, l, chunk->length);
        ex = -(int)recobe_chunksize(chunk);
        errno = ENOSPC;
        goto out;
    }
    if  (!(chain->flags & RECOBE_ORDER)) {
        *wchr = recobe_load(&c[ex], l);
        ex += l;
        chunk->position += l;
    }
    if  (8 < chain->inwlen) {
        chunk->position += 1;
        if  (!!c[ex++]) {
            ex = -EX_DATAERR;
            errno = EINVAL;
            goto out;
    }   }
    if  (9 < chain->inwlen) {
        chunk->position += 1;
        if  (!!c[ex++]) {
            ex = -EX_DATAERR;
            errno = EINVAL;
            goto out;
    }   }
    if  (!!(chain->flags & RECOBE_ORDER)) {
        *wchr = recobe_load(&c[ex], -l);
        ex += l;
        chunk->position += l;
    }
out:
    ifBLIN_QX3("- %d %u %u", ex, chunk->position, chunk->length);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   wchr   chunk   ,               ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_outword(recobe_chain *chain, recobe_chunk *chunk, u_int64_t wchr) {                       /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    int64_t l = chain->outwlen;
    int     ex = EX_OK;

    ifBLIN_QX3( "+ %08X %"BLIN_X" %"BLIN_X" %u+%u %u"
              , chain->flags
              , BLIN_I(chain)
              , BLIN_I(chunk)
              , chunk->position
              , chunk->length
              , chunk->size
              );
    if  (chunk->size < chunk->length + l) {
        ifBLIN_QX0("size=%u < length=%u + outwlen=%u", chunk->size, chunk->length, l);
        errno = ENOSPC;
        ex = -EX_UNAVAILABLE;
    } else if ((8 > chain->outwlen) && ((1ULL << (chain->outwlen * 8)) <= wchr)) {
        ifBLIN_QX1("length %016llX > %u", wchr, chain->outwlen);
        errno = ERANGE;
        ex = -EX_CANTCREAT;
    } else {
        recobe_store( &chunk->chunk[chunk->length]
                    , (chain->flags & RECOBE_ORDER) ? -l : l
                    , wchr
                    )
        ;
        chunk->length += (ex = l);
    }
    ifBLIN_QX3("- %d %u+%u %u", ex, chunk->position, chunk->length, chunk->size);
    return(ex);
#   undef blin_internal_flags
}

u_int32_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****     chunk                                                      ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chunksize(recobe_chunk *chunk) {                                                          /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t size = 0;

    if  (!!chunk && (chunk->length > chunk->position) && !(RECOBE_EOT == chunk->position)) {
        size = chunk->length - chunk->position;
    }
    return(size);
}

u_int32_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****     chunk                                             ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chunkempty(recobe_chunk *chunk) {                                                         /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t size = 0;

    if  (!!chunk && (chunk->size > chunk->length) && !(RECOBE_EOT == chunk->position)) {
        size = chunk->size - chunk->length;
    }
    return(size);
}

const u_char *
/*****************************************************************************************************
 *****************************************************************************************************
 ****        chunk                                        ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chunktext(recobe_chunk *chunk) {                                                          /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    const u_char *txt = NULL;

    if  (!!chunk && !(RECOBE_EOT == chunk->position)) txt = &chunk->chunk[chunk->position];
    return(txt);
}

recobe_chunk *
/*****************************************************************************************************
 *****************************************************************************************************
 ****  recobe_chunk    sz,  ..    in  ln.         ****
 ****  (!in && !sz && !ln),    .                                          ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chunknew(recobe_chain *chain, u_int32_t sz, u_int32_t ln, const u_char *in) {             /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    recobe_chunk *newchunk;

#   define blin_internal_flags (chain->flags)
    if  (!(newchunk = malloc(sz + sizeof(recobe_chunk)))) {
        ifBLIN_QW0("No mem %d", sz + sizeof(recobe_chunk));
        goto out;
    }
    newchunk->size = malloc_usable_size(newchunk) - sizeof(recobe_chunk);
    newchunk->length = (ln > sz) ? 0 : ln;
    newchunk->position = 0;
    if  (!in && !sz && !ln) newchunk->position = RECOBE_EOT;
    if  (!!in && !!ln) bcopy(in, newchunk->chunk, ln);
out:
    return(newchunk);
#   undef blin_internal_flags
}

recobe_chunk *
/*****************************************************************************************************
 *****************************************************************************************************
 ****     chain->tail                                                          ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chunkopy(recobe_chain *chain) {                                                           /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    recobe_chunk *newchunk;

#   define blin_internal_flags (chain->flags)
    if  (!(newchunk = malloc(sizeof(recobe_chunk) + recobe_chunksize(chain->tail)))) {
        ifBLIN_QW0("malloc(%"BLIN_D")", sizeof(recobe_chunk) + recobe_chunksize(chain->tail));
    }
    bcopy(chain->tail, newchunk, sizeof(recobe_chunk) + recobe_chunksize(chain->tail));
    return(newchunk);
#   undef blin_internal_flags
}

recobe_chunk *
/*****************************************************************************************************
 *****************************************************************************************************
 ****        in0  in1                                 ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chunkat(recobe_chain *chain, recobe_chunk *in0, recobe_chunk *in1) {                      /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    recobe_chunk *newchunk = NULL;
    u_int32_t     len0     = 0;
    u_int32_t     len1     = 0;

#   define blin_internal_flags (chain->flags)
    len0 = recobe_chunksize(in0);
    len1 = recobe_chunksize(in1);
    if  (!(newchunk = recobe_chunknew(chain, len0 + len1, len0, recobe_chunktext(in0)))) {
        ifBLIN_QW0("recobe_chunknew %u", len0 + len1);
        goto out;
    }
    newchunk->length += len1;
    if  (!!len1) memcpy(&newchunk->chunk[len0], recobe_chunktext(in1), len1);
out:
    return(newchunk);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****      chain    source.                               ****
 **** : < 0 -                                                                      ****
 ****           == 0 -   source                                                    ****
 ****           == 1 - source                                                                 ****
 ****           == 2 - source  chain                                                        ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chunksum(recobe_chain *chain, recobe_chunk *source) {                                     /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    recobe_chunk *newchunk;
    u_int32_t     tailen = 0;
    u_int32_t     srclen = 0;
    int           ex     = EX_OK;

#   define blin_internal_flags (chain->flags)
    if  (!recobe_chunksize(source)) {
        if  (RECOBE_EOT == source->position) ex = (!recobe_chunksize(chain->tail)) ? 2 : 1;
        goto out;
    }
    if  (!(newchunk = recobe_chunkat(chain, chain->tail, source))) {
        ifBLIN_QW0("recobe_chunkat %u", tailen + srclen);
        goto out;
    }
    free(chain->tail);
    chain->tail = newchunk;
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/*****************************************************************************************************
 *****************************************************************************************************
 ****  --  ( -c)                                                   ****
 *****************************************************************************************************
                                                                                                  ****/
convertor(recobe_chain *chain, recobe_chunk *chunk) {                                            /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    u_char       *table = (u_char*)chain->converdata;
    u_int64_t     word;
    int           ex;

    ifBLIN_QX3("+ %08X", chain->flags);
    if  (0 > (ex = recobe_inword(chain, chain->tail, &word))) {
        ifBLIN_QW1("recobe_inword");
    } else if (0 > (ex = recobe_outword(chain, chunk, table[word & 0x0FF]))) {
        chain->tail->position--;
        ex = 0;
        ifBLIN_QW1("recobe_outword");
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

recobe_chain *
/*****************************************************************************************************
 *****************************************************************************************************
 ****  chain c -c (--)  p,  domain  0,    ,     ****
 ****    p   .                                                            ****
 ****    p   --                          ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_openc(recobe_conf *cf, const char *p, int domain, recobe_chain *next) {                   /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    recobe_chain *chain = NULL;

    ifBLIN_QX3("+ d%d:%s", domain, p);
    if  (!(chain = malloc(sizeof(recobe_chain)))) {
        ifBLIN_QW0("malloc(%d)", sizeof(recobe_chain));
        goto out;
    }
    chain->cf = cf;
    chain->flags = cf->flags & ~(RECOBE_USED | RECOBE_CHAINER);
    chain->tail = NULL;
    chain->next = next;
    if  (!!domain) {
        chain->cnv = recobe_collection.conv(p, domain, 0xFFFF);
    } else {
        chain->cnv = recobe_collection.conv(p, 0, RECOBE_C);
    }
    chain->convertor = convertor;
    if  (  !chain->cnv.domain
        || !recobe_collection.domains[chain->cnv.domain]->reconv
        || !(chain->converdata = recobe_collection.domains[chain->cnv.domain]->reconv(chain->cnv))
        ) {
        ifBLIN_QX0("Illegal reconv -c %s", p);
        free(chain);
        chain = NULL;
        errno = ENOSYS;
        goto out;
    }
    chain->bsubst = cf->bsubst;
    chain->esubst = cf->esubst;
    chain->bcode = cf->bcode;
    chain->ecode = cf->ecode;
    chain->inwlen = 1;
    chain->outwlen = 1;
    chain->inwnum = 1;
    chain->outwnum = 1;
out:
    RECOBE_CHAIN(chain);
    ifBLIN_QX3("- %"BLIN_X, chain);
    return(chain);
#   undef blin_internal_flags
}

recobe_chunk *
/*****************************************************************************************************
 *****************************************************************************************************
 ****      in  inlen  chain                         ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_chainarr(recobe_chain *chain, size_t inlen, void *in) {                                   /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    recobe_chunk    *toout = NULL;
    recobe_chain    *tail;
    mife_descriptor *md    = NULL;
    recobe_conf     *cf;

#   define blin_internal_flags (chain->flags)
    if  (!(cf = calloc(1, sizeof(recobe_conf)))) {
        ifBLIN_QW0("calloc %d", sizeof(recobe_conf));
        goto out;
    }
    cf->flags = chain->flags;
    if  (!(md = mife_init(((blin_internal_flags >> 1) & BLIN_GEN1) | MIFE_SLID))) {
        ifBLIN_QW0("mife_init");
        goto out;
    }
    if  (0 > mife_ctlbuff(md, in, inlen)) {
        ifBLIN_QW0("mife_ctlfile");
        goto out;
    }
    for (tail = chain; ; tail = tail->next) {
        if  (!tail->next) {
            if  (!(tail->next = recobe_openmd(cf, md, RECOBE_PREFLEN))) {
                ifBLIN_QW0("recobe_openmd");
                goto out;
            }
            break;
    }   }
    if  (!(toout = recobe_chainbuf(chain))) ifBLIN_QW0("recobe_chainbuf");
    recobe_close(tail->next);
    tail->next = NULL;
out:
    return(toout);
#   undef blin_internal_flags
}

recobe_chain *
/*****************************************************************************************************
 *****************************************************************************************************
 ****  chain c -k  p.                                                            ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_openk(recobe_conf *cf, const char *p, int domain, recobe_chain *next) {                   /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    recobe_chain *chain = NULL;

    ifBLIN_QX3("+ w:%s %"BLIN_X, p, BLIN_I(next));
    if  (!(chain = malloc(sizeof(recobe_chain)))) {
        ifBLIN_QW0("malloc(%d)", sizeof(recobe_chain));
        goto out;
    }
    if  (!!domain) {
        chain->cnv = recobe_collection.conv(p, domain, 0xFFFF);
    } else {
        chain->cnv = recobe_collection.conv(p, 0, RECOBE_K);
    }
    if  (!chain->cnv.domain || (!!chain->cnv.nonesrc && !!chain->cnv.nonedst)) {
        ifBLIN_QX0("No %s recode", p);
        free(chain);
        chain = NULL;
        errno = ENOENT;
        goto out;
    }
    chain->next = next;
    chain->converdata = cf->mod;
    chain->flags |= RECOBE_CNVDFREE;
    chain->cf = cf;
    chain->flags = cf->flags & ~(RECOBE_USED | RECOBE_CHAINER | RECOBE_DOEND);
    chain->tail = NULL;
    chain->bsubst = cf->bsubst;
    chain->esubst = cf->esubst;
    chain->bcode = cf->bcode;
    chain->ecode = cf->ecode;
    if  (!!recobe_collection.domains[chain->cnv.domain]->param(chain)) {
        ifBLIN_QW0("param");
        free(chain);
        chain = NULL;
        goto out;
    }
    if  (!chain->convertor) {
        recobe_cnverr(__FILE__, __func__, __LINE__, "No recode", chain->cnv);
        free(chain);
        chain = NULL;
        errno = ENOSYS;
        goto out;
    }
    if  (!!chain->bsubst) chain->outwnum += strlen(chain->bsubst);
    if  (!!chain->esubst) chain->outwnum += strlen(chain->esubst);
    if  (RECOBE_ERHEXPRT & chain->flags) chain->outwnum += 2 * chain->inwnum;
out:
    RECOBE_CHAIN(chain);
    ifBLIN_QX3("- %"BLIN_X, chain);
    return(chain);
#   undef blin_internal_flags
}
