/*-
 * Copyright (C)2005..2019 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2005..2019 @BABOLO http://www.babolo.ru/\n"
#ident "@(#) $Id: popushd.c,v 1.17 2019/09/02 17:12:24 babolo Exp $\n"

#define BLIN_COMPAT  2
#define MIFE_COMPAT  3
#define Bpars_COMPAT 4
#define MULAR_COMPAT 0

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <stdarg.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <err.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "popushd.h"

u_int32_t state;

void
mesg(u_int32_t flags, const char *fmt, const char *fmt1, ...) {
    va_list ap;

    if (flags & (POPUSH_CRI | BLIN_VER1)) {
        va_start(ap, fmt1);
        if  (flags & POPUSH_DMN) {
            if  (flags & POPUSH_ERX) vwarnx(fmt, ap);
            else vwarn(fmt, ap);
        } else {
            if  (flags & POPUSH_ERX) vsyslog((flags & POPUSH_CRI) ? LOG_CRIT : LOG_ERR, fmt, ap);
            else vsyslog((flags & POPUSH_CRI) ? LOG_CRIT : LOG_ERR, fmt1, ap);
        }
        va_end(ap);
}   }

static void
sighup(int sig) {
    state |= F_SIGN;
}

static void
sigint(int sig) {
    state |= F_STOP;
}

static void
usage(int ex) {
    fprintf( stderr
           , "popushd @BABOLO V.M "VERS"  "DATE" http://www.babolo.ru/\n"
             "Usage:\n"
             "popushd [-qv ][-f conf][-p pid]\n"
             "    -. envname - for cwd\n"
             "    -@ envname - for PID\n"
             "    -a 0xXX - absolute quote\n"
             "    -d - not daeminize\n"
             "    -f configfile\n"
             "    -p PIDfile\n"
             "    -q - quiet\n"
             "    -v[v[v...]] - verbose\n"
             )
    ;
    exit(ex);
}

int
main(int argc, char *argv[], char **envp) {
    int ex = EX_OK;
    int c;
    popush_main rr;
    popush_main *r = &rr;
    popush_cnf *e;
    struct kevent evnt;
    pid_t mypid;
    char myspid[8];
    char *emyspid = NULL;
    FILE *pidfile;

    r->actionbit = NULL;
    bzero(r, sizeof(popush_main));
    r->flags = BLIN_VER1;
    r->cnfname = RCFILE;
    r->pidname = PIDFILE;
    if  (!(r->wdpath = getcwd(NULL, 0))) {
        ifBLIN_QV1(r->flags) warn("getcwd failed");
    }
    while ((c = getopt(argc, argv, ".:@:a:df:p:qvV")) != EOF) {
        switch (c) {
        case '.':
           if  (setenv(optarg, r->wdpath, 1) < 0) {
               ifBLIN_QV1(r->flags) warn("setenv %s=%s", optarg, r->wdpath);
           }
           break;
        case '@': emyspid = optarg;                  break;
        case 'a': r->absq = strtol(optarg, NULL, 0); break;
        case 'd': r->flags |= POPUSH_DMN;            break;
        case 'f': r->cnfname = optarg;               break;
        case 'p': r->pidname = optarg;               break;
        case 'q': r->flags &= ~BLIN_VERMASK;         break;
        case 'v': BLIN_VERBOSE(r->flags);            break;
        case 'V': r->flags |= POPUSH_ANA;            break;
        case '?': usage(EX_OK);
        default:  usage(EX_USAGE);
    }   }
    argc -= optind;
    argv += optind;
    if  (r->wdpath && r->cnfname && (r->cnfname[0] !=  '/')) {
        if  (asprintf(&(r->cnfname), "%s/%s", r->wdpath, r->cnfname) < 0) {
            ifBLIN_QV1(r->flags) warn("No mem #1");
            exit(EX_OSERR);
        }
        ifBLIN_QV3(r->flags) fprintf(stderr, "cnfname=%s~\n", r->cnfname);
    }
    if  (r->wdpath && r->pidname && (r->pidname[0] !=  '/')) {
        if  (asprintf(&(r->pidname), "%s/%s", r->wdpath, r->pidname) < 0) {
            ifBLIN_QV1(r->flags) warn("No mem #2");
            exit(EX_OSERR);
        }
        ifBLIN_QV3(r->flags) fprintf(stderr, "pidname=%s~\n", r->pidname);
    }
    if  (~r->flags & POPUSH_DMN && daemon(0, 0)) {
        mesg(r->flags | POPUSH_CRI, DE("daemon"));
        exit(EX_UNAVAILABLE);
    }
    mypid = getpid();
    if (emyspid) {
        if  (snprintf(myspid, 8, "%d", mypid) > 7) {
            ifBLIN_QV1(r->flags) warn("Too high pid %d", mypid);
        } else if (setenv(emyspid, myspid, 1) < 0) {
            ifBLIN_QV1(r->flags) warn("setenv %s=%s", emyspid, myspid);
        }
    }
    openlog("popushd", LOG_PID, LOG_DAEMON);
    if  ((r->kq = kqueue()) < 0) {
        mesg(r->flags | POPUSH_CRI, DE("kqueue"));
        ex = EX_OSERR;
        goto out;
    }
    state = F_STRT;
    if  (signal(SIGHUP   , sighup) == SIG_ERR) ifBLIN_QV1(r->flags) warn("SIGHUP");
    if  (signal(SIGINT   , sigint) == SIG_ERR) ifBLIN_QV1(r->flags) warn("SIGINT");
    if  (signal(SIGTERM  , sigint) == SIG_ERR) ifBLIN_QV1(r->flags) warn("SIGTERM");
    if  (signal(SIGXCPU  , sigint) == SIG_ERR) ifBLIN_QV1(r->flags) warn("SIGXCPU");
    if  (signal(SIGXFSZ  , sigint) == SIG_ERR) ifBLIN_QV1(r->flags) warn("SIGXFSZ");
    if  (signal(SIGVTALRM, sigint) == SIG_ERR) ifBLIN_QV1(r->flags) warn("SIGVTALRM");
    while (1) {
        ifBLIN_QV2(r->flags) fprintf(stderr, "Loop state %08X\n", state);
        if  ((state & F_STOP) && (unlink(r->pidname) < 0)) {
            mesg(r->flags, DE("PID file delete %s failed"), r->pidname);
        }
        if  (state & (F_STRT | F_SIGN)) {
            if  ((pidfile = fopen(r->pidname, "w")) == NULL) {
                mesg(r->flags, DE("PID file open %s failed"), r->pidname);
            } else {
                if (fprintf(pidfile, "%d\n", mypid) < 0) {
                    mesg(r->flags, DE("Can't write PID"));
                }
                if  (fclose(pidfile)) {
                    mesg(r->flags, DE("PID file not closed"));
        }   }   }
        pidfile = NULL;
        if  (state & (F_STRT | F_SIGN | F_SELF)) {
            if  (!(e = syntan(r))) {
                mesg(r->flags | POPUSH_ERX, DE("Syntax error"));
            } else if (r->flags & POPUSH_ANA) {
                break;
            } else if (diffmerge(r, e)) {
                /* mesg(r->flags | POPUSH_ERX, DE("Configuration error")); */
            } else {
                cnfree(r->cnf);
                r->cnf = e;
                free(r->actionbit);
                r->actionbit = NULL;
        }   }
        if  (!r->cnf) {
            mesg(r->flags | POPUSH_ERX, DE("No config %s"), r->cnfname);
            ex = EX_TEMPFAIL;
            goto out;
        }
        if  (!r->actionbit && !(r->actionbit = createbit(MULAR_NEXT(r->cnf->cluster)))) {
            mesg(r->flags, DE("createbit"));
            ex = EX_TEMPFAIL;
            goto out;
        }
        if  (state & F_STRT) setact(r, r->cnf->strt);
        if  (state & F_STOP) setact(r, r->cnf->stop);
        if  (state & F_SIGN) setact(r, r->cnf->sign);
        if  (state & F_SELF) setaction(r, r->cnf->self);
        state &= F_STOP;
        if  ((ex = setevent(r))) {
            mesg(r->flags | POPUSH_ERX, DE("setevent"));
            goto out;
        }
        execevent(r);
        if  (state & F_STOP) {
            mesg(r->flags | POPUSH_CRI | POPUSH_ERX, DE("killed"));
            goto out;
        }
        if  (state) continue;
        bzero(&evnt, sizeof(evnt));
        c = kevent(r->kq, NULL, 0, &evnt, 1, NULL);
        if  (c > 0) setkact(r, &evnt);
    }
out:if  (!(state & F_STOP)) {
        /* XXXX restart */
    }
    closelog();
    exit(ex);
}
