/*-
 * Copyright (C)2008..2022 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2008..2022 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: rx_sqlite3.c,v 1.3 2022/12/17 17:02:42 babolo Exp $"

#define BLIN_COMPAT      4
#define Bpars_COMPAT     4
#define MULAR_COMPAT     0
#define MIFE_COMPAT      5
#define PGOBLIN_COMPAT   4
#define PGOBLIN_INTERNAL 1

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <regex.h>
#include <errno.h>
#include <stdio.h>
#include <err.h>
#include <sqlite3.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"
#include "pgoblin3sqlite3.h"

/* expansions to SQLite 3 */

#define NOMEM   9999
#define BACKREF 10

typedef struct chunk {
    u_int32_t         length;     /*                   */
    u_int32_t         size;       /*                              */
    u_int32_t         position;   /*        */
    char              chunk[0];   /* Byte array                                     */
} chunk;

static chunk *
chunk_init(BLIN_flag flags, const char *in, ssize_t len) {
    chunk   *str;
    ssize_t  l;

    if  ((len < 0) && !!in) len = strlen(in);
    l = 512;
    while(l < (len + 1 + (ssize_t)sizeof(chunk))) l *= 2;
    str = malloc(l);
    str->size = l - sizeof(chunk);
    if  (!in) {
        str->chunk[0] = '\0';
        str->length = 0;
    } else {
        strncpy(str->chunk, in, len + 1);
        str->length = len;
    }
    return(str);
}

static chunk *
qexpand(chunk *str, u_int32_t need) {
    u_int32_t  l = str->size;

    need += str->length + 1 + sizeof(chunk);
    while(need > l) l *= 2;
    if  (str->size < l) {
        str = realloc(str, l);
        str->size = l - sizeof(chunk);
    }
    return(str);
}

static chunk *
appchr(chunk *str, char c) {
    str = qexpand(str, 1);
    str->chunk[str->length++] = c;
    str->chunk[str->length] = '\0';
    return(str);
}

static chunk *
append(chunk *str, const char *add, ssize_t len) {
    if  (0 > len) len = strlen(add);
    str = qexpand(str, len);
    memcpy(str->chunk + str->length, add, len);
    str->length += len;
    str->chunk[str->length] = '\0';
    return(str);
}

static chunk *
combine( chunk      *str
       , const char *rply
       , regmatch_t *pmatch
       , const char *in
       , int         pos
       ) {
    const char *p = rply;

    for (;;) {
        const char *rfrom;
        int         so;
        int         eo;

        rfrom = p;
        p = index(p, '\\');
        if  (!p) {
            str = append(str, rfrom, -1);
            break;
        }
        if  (p > rfrom) str = append(str, rfrom, p - rfrom);
        p++;
        if  (!*p) {  /* \<EOT> */
            str = appchr(str, '\\');
            break;
        }
        if  (*p == '\\') {
            str = appchr(str, '\\');
            p++;
            continue;
        } else if (*p == '&') {
            so = pmatch[0].rm_so;
            eo = pmatch[0].rm_eo;
            p++;
        } else if (*p >= '1' && *p <= '9') {
            int idx = *p - '0';

            so = pmatch[idx].rm_so;
            eo = pmatch[idx].rm_eo;
            p++;
        } else {  /*  XXXX ,  \t, \n, \r  .. */
            continue;
        }
        if  ((so >= 0) && (eo >= 0)) {
            assert(so >= pos);
            in += so - pos;
            str = append(str, in, eo - so);
    }   }
    return(str);
}

static int
rep_regexp(chunk **o, const char *in, regex_t *preg, const char *rply, BLIN_flag flags) {
    regmatch_t  pmatch[BACKREF];
    int         eflags;
    ssize_t     search;      /*     in,   0   */
    ssize_t     pos;         /*     in                                    */
    ssize_t     lin;         /*   in                                                   */
    int         esc   = 0;
    int         ex    = 0;

    if  (!(*o = chunk_init(flags, NULL, 0))) {
        ex = NOMEM;
    } else {
        if  (index(rply, '\\')) esc = 1;
        lin = strlen(in);
        eflags = REG_STARTEND;
        if  (flags & REG_PEND) eflags |= REG_NOTEOL;
        for (pos = 0, search = 0; search <= lin; ) {
            pmatch[0].rm_so = pos;
            pmatch[0].rm_eo = lin;
            ex = regexec(preg, in, BACKREF, pmatch, eflags);
            if  (ex == REG_NOMATCH) {
                ex = 0;
                break;
            }
            if  (!!ex) break;
            if  (pmatch[0].rm_so > pos) { /*     */
                int l;

                l = pmatch[0].rm_so - pos;
                *o = append(*o, in, l);
                in += l;
                pos = pmatch[0].rm_so;
            }
            if  (esc) {
                *o = combine(*o, rply, pmatch, in, pos);
            } else {
                *o = append(*o, rply, -1);
            }
            pos = pmatch[0].rm_eo;
            if  (!(flags & SQLITE_GLOB)) break;
            search = pos;
            if  (pmatch[0].rm_so == pmatch[0].rm_eo) search++;
            eflags |= REG_NOTBOL;
        }
        if  (pos < lin) {
            *o = append(*o, in, -1);
    }   }
    return(ex);
}

char *
/*****************************************************
 *****************************************************
 **                                                 **/
_pgoblin_sqlite3_rerr(int ex, regex_t *preg) {     /**
 **                                                 **
 *****************************************************
 *****************************************************/
    static char *er;
    size_t       l;

    if  (!!(er = malloc((l = regerror(ex, preg, NULL, 0))))) regerror(ex, preg, er, l);
    return(er);
}

void
/*************************************************************************************
 *************************************************************************************
 **                                                                                 **/
_pgoblin_sqlite3_regexrepl(sqlite3_context *ctx, int argc, sqlite3_value **argv) { /**
 **                                                                                 **
 *************************************************************************************
 *************************************************************************************/
#   define ARF (REG_BASIC | REG_EXTENDED | REG_ICASE | REG_NEWLINE | REG_NOSPEC | REG_PEND)
    BLIN_flag   flags;
    const char *rply;
    regex_t    *preg;
    const char *rex;
    const char *flx = NULL;
    int         ex  = 0;
    const char *in;
    char       *o;
    chunk      *t;

    flags = ((BLIN_flag)BLIN_I(sqlite3_user_data(ctx)) & BLIN_MASK) | REG_EXTENDED;
#   define blin_internal_flags flags
    if  (  (argc < 3)
        || !(in = (const char *)sqlite3_value_text(argv[0]))
        || !(rex = (const char *)sqlite3_value_text(argv[1]))
        || !(rply = (const char *)sqlite3_value_text(argv[2]))
        ) {
        sqlite3_result_null(ctx);
    } else {
        if  (argc > 3) {
            flx = (const char *)sqlite3_value_text(argv[3]);
            for ( ; !!*flx; ++flx) {
                switch(*flx) {
                case 'b': /* BRE */
                    flags &= ~REG_EXTENDED;
                case 'g':
                    flags |= SQLITE_GLOB;
                    break;
                case 'i':
                    flags |= REG_ICASE;
                    break;
                case 'n': /* \n affects ^ $ . [^ */
                    flags |= REG_NEWLINE;
                    break;
                case 'q': /* literal string */
                    flags |= REG_NOSPEC;
                    flags &= ~REG_EXTENDED;
                break;
                    case 'z': /* NUL as ordinary character */
                    flags |= REG_PEND;
                    break;
                default:
                    ifBLIN_QX0("regex_replace: Ill flag %c(%02X)", *flx, *flx);
                    sqlite3_result_error(ctx, "Ill flag", -1);
                    return;
        }   }   }
        if  (!(preg = malloc(sizeof(regex_t)))) {
            ifBLIN_QX0("regex_replace: No mem");
            sqlite3_result_error_nomem(ctx);
        } else if ((ex = regcomp(preg, rex, flags & ARF))) {
            if  (!(o = pgoblin_sqlite3_rerr(ex, preg))) {
                ifBLIN_QX0("regex_replace: regcomp: No mem");
                sqlite3_result_error_nomem(ctx);
            } else {
                ifBLIN_QX0("regex_replace: regcomp: %s", o);
                sqlite3_result_error(ctx, o, -1);
            }
            free(o);
        } else if (!!(ex = rep_regexp(&t, in, preg, rply, flags))) {
            if  (ex == NOMEM) {
                ifBLIN_QX0("regex_replace: rep_regexp: No mem");
                sqlite3_result_error_nomem(ctx);
            } else {
                if  (!(o = pgoblin_sqlite3_rerr(ex, preg))) {
                    ifBLIN_QX0("regex_replace: rerr: No mem");
                    sqlite3_result_error_nomem(ctx);
                } else {
                    ifBLIN_QX0("regex_replace: rep_regexp: %s", o);
                    sqlite3_result_error(ctx, o, -1);
                }
                free(o);
            }
            regfree(preg);
            free(t);
        } else {
            if  (!(o = strndup(t->chunk, t->length))) {
                ifBLIN_QX0("regex_replace: strndup: No mem");
                sqlite3_result_error_nomem(ctx);
            } else {
                sqlite3_result_text(ctx, o, -1, free);
    }   }   }
#   undef blin_internal_flags
}

void
/*************************************************************************************
 *************************************************************************************
 **                                                                                 **/
_pgoblin_sqlite3_upandown(sqlite3_context *ctx, int argc, sqlite3_value **argv) {  /**
 **                                                                                 **
 *************************************************************************************
 *************************************************************************************/
#   define ARF (REG_BASIC | REG_EXTENDED | REG_ICASE | REG_NEWLINE | REG_NOSPEC | REG_PEND)
    BLIN_flag   flags;
    const char *in;
    char       *o;

    flags = (BLIN_flag)BLIN_I(sqlite3_user_data(ctx)) & (BLIN_MASK | 1);
#   define blin_internal_flags flags
    if  ((argc < 1) || !(in = (const char *)sqlite3_value_text(argv[0]))) {
        sqlite3_result_null(ctx);
    } else if (!(o = malloc(strlen(in) + 1))) {
        ifBLIN_QX0("regex_replace: strdup: No mem");
        sqlite3_result_error_nomem(ctx);
    } else {
        size_t i;
        char   c = 1;

        for (i = 0; !!c; ++i) {
            c = in[i];
            if  (flags & 1) {
                if  ((c >= 'a') && (c <= 'z')) c = (c & 0xDF);
                if  ((c >= '') && (c <= '')) c = (c | 0x20);
                if  (c == '') c = '';
            } else {
                if  ((c >= 'A') && (c <= 'Z')) c = (c | 0x20);
                if  ((c >= '') && (c <= '')) c = (c & 0xDF);
                if  (c == '') c = '';
            }
            o[i] = c;
        }
        sqlite3_result_text(ctx, o, -1, free);
    }
#   undef blin_internal_flags
}

__weak_reference(_pgoblin_sqlite3_regexrepl, pgoblin_sqlite3_regexrepl);
__weak_reference(_pgoblin_sqlite3_upandown, pgoblin_sqlite3_upandown);
__weak_reference(_pgoblin_sqlite3_rerr, pgoblin_sqlite3_rerr);
