/*-
 * Copyright (C)2008..2022 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2008..2022 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: db_sqlite3.c,v 1.118 2022/11/13 20:49:25 babolo Exp $"

#define BLIN_COMPAT      4
#define Bpars_COMPAT     4
#define MULAR_COMPAT     0
#define MIFE_COMPAT      5
#define PGOBLIN_COMPAT   4
#define PGOBLIN_INTERNAL 1
#define PGOBLIN_MODULE   pgoblin_module

#include <sys/limits.h>
#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sysexits.h>
#include <sys/time.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdint.h>
#include <regex.h>
#include <errno.h>
#include <stdio.h>
#include <err.h>
#include <sqlite3.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"
#include "pgoblin3sqlite3.h"

/* SQLite 3 interface to pgoblin */

#define INDX(R, C) (R * rio->pq->ncol + C)

#define ERRMSG(W, X) {                                                                                \
    if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {                            \
        ifBLIN_QX0(W " %d: %s", ex, sqlite3_errmsg(conn->odf));                                       \
        ERROUT((X), oxdecode(ex));                                                                    \
}   }
#define TYPEMAX 6
static char *types[TYPEMAX] = {NULL, "integer", "float", "text", "blob", "null"};
static const char statenames[][3] =
{ "bg", "bq", "bz", "be", "b1", "b2", "bb", "bd", "bc", "bt", "bn", "bx"
};

static size_t mdp[] = {1024, 512, 512};
pgoblin_dbases PGOBLIN_MODULE;

static u_char sqle[] =
{ 0            /* SQLITE_OK         Successful result                          */
, ENOTSUP      /* SQLITE_ERROR      SQL error or missing database              */
, ENOEXEC      /* SQLITE_INTERNAL   An internal logic error in SQLite          */
, EPERM        /* SQLITE_PERM       Access permission denied                   */
, ECANCELED    /* SQLITE_ABORT      Callback routine requested an abort        */
, ETXTBSY      /* SQLITE_BUSY       The database file is locked                */
, EDEADLK      /* SQLITE_LOCKED     A table in the database is locked          */
, ENOMEM       /* SQLITE_NOMEM      A malloc() failed                          */
, EROFS        /* SQLITE_READONLY   Attempt to write a readonly database       */
, EINTR        /* SQLITE_INTERRUPT  Operation terminated by sqlite3_interrupt()*/
, EIO          /* SQLITE_IOERR      Some kind of disk I/O error occurred       */
, EILSEQ       /* SQLITE_CORRUPT    The database disk image is malformed       */
, EFAULT       /* SQLITE_NOTFOUND   (Internal Only) Table or record not found  */
, EOVERFLOW    /* SQLITE_FULL       Insertion failed because database is full  */
, ENOATTR      /* SQLITE_CANTOPEN   Unable to open the database file           */
, ENOLCK       /* SQLITE_PROTOCOL   Database lock protocol error               */
, ENOENT       /* SQLITE_EMPTY      (Internal Only) Database table is empty    */
, EIDRM        /* SQLITE_SCHEMA     The database schema changed                */
, EMSGSIZE     /* SQLITE_TOOBIG     Too much data for one row of a table       */
, EEXIST       /* SQLITE_CONSTRAINT Abort due to contraint violation           */
, EPROCUNAVAIL /* SQLITE_MISMATCH   Data type mismatch                         */
, EINVAL       /* SQLITE_MISUSE     Library used incorrectly                   */
, ENOSYS       /* SQLITE_NOLFS      Uses OS features not supported on host     */
, EAUTH        /* SQLITE_AUTH       Authorization denied                       */
, EFTYPE       /* SQLITE_FORMAT     Auxiliary database format error            */
, EDOM         /* SQLITE_RANGE      2nd parameter to sqlite3_bind out of range */
, EBADF        /* SQLITE_NOTADB     File opened that is not a database file    */
#ifdef SQLITE_NOTICE
, 0            /* SQLITE_NOTICE     Notifications from sqlite3_log()           */
#endif
#ifdef SQLITE_WARNING
, 0            /* SQLITE_WARNING    Warnings from sqlite3_log()                */
#endif
};

static int
/**********************************************************************
 **                                                                  **/
oxdecode(u_int i) {                                                 /**
 **                                                                  **
 **********************************************************************/
    int e;
    if      (i == SQLITE_ROW)                e = EAGAIN;
/*  else if (i == SQLITE_DONE) */
    else if ((i > sizeof(sqle)))             e = ENOMSG;
    else                                     e = sqle[i];
    return(e);
}

static const char *
/**********************************************************************
 **                                                                  **/
exdecode(int ex) {                                                  /**
 **                                                                  **
 **********************************************************************/
    const char *c;
    switch (ex) {
    case SQLITE_OK                     : c = "SQLITE_OK";                      break;
    case SQLITE_ERROR                  : c = "SQLITE_ERROR";                   break;
    case SQLITE_INTERNAL               : c = "SQLITE_INTERNAL";                break;
    case SQLITE_PERM                   : c = "SQLITE_PERM";                    break;
    case SQLITE_ABORT                  : c = "SQLITE_ABORT";                   break;
    case SQLITE_BUSY                   : c = "SQLITE_BUSY";                    break;
    case SQLITE_LOCKED                 : c = "SQLITE_LOCKED";                  break;
    case SQLITE_NOMEM                  : c = "SQLITE_NOMEM";                   break;
    case SQLITE_READONLY               : c = "SQLITE_READONLY";                break;
    case SQLITE_INTERRUPT              : c = "SQLITE_INTERRUPT";               break;
    case SQLITE_IOERR                  : c = "SQLITE_IOERR";                   break;
    case SQLITE_CORRUPT                : c = "SQLITE_CORRUPT";                 break;
    case SQLITE_NOTFOUND               : c = "SQLITE_NOTFOUND";                break;
    case SQLITE_FULL                   : c = "SQLITE_FULL";                    break;
    case SQLITE_CANTOPEN               : c = "SQLITE_CANTOPEN";                break;
    case SQLITE_PROTOCOL               : c = "SQLITE_PROTOCOL";                break;
    case SQLITE_EMPTY                  : c = "SQLITE_EMPTY";                   break;
    case SQLITE_SCHEMA                 : c = "SQLITE_SCHEMA";                  break;
    case SQLITE_TOOBIG                 : c = "SQLITE_TOOBIG";                  break;
    case SQLITE_CONSTRAINT             : c = "SQLITE_CONSTRAINT";              break;
    case SQLITE_MISMATCH               : c = "SQLITE_MISMATCH";                break;
    case SQLITE_MISUSE                 : c = "SQLITE_MISUSE";                  break;
    case SQLITE_NOLFS                  : c = "SQLITE_NOLFS";                   break;
    case SQLITE_AUTH                   : c = "SQLITE_AUTH";                    break;
    case SQLITE_FORMAT                 : c = "SQLITE_FORMAT";                  break;
    case SQLITE_RANGE                  : c = "SQLITE_RANGE";                   break;
    case SQLITE_NOTADB                 : c = "SQLITE_NOTADB";                  break;
#ifdef SQLITE_NOTICE
    case SQLITE_NOTICE                 : c = "SQLITE_NOTICE";                  break;
#endif
#ifdef SQLITE_WARNING
    case SQLITE_WARNING                : c = "SQLITE_WARNING";                 break;
#endif
    case SQLITE_ROW                    : c = "SQLITE_ROW";                     break;
    case SQLITE_DONE                   : c = "SQLITE_DONE";                    break;
#ifdef SQLITE_ERROR_MISSING_COLLSEQ
    case SQLITE_ERROR_MISSING_COLLSEQ  : c = "SQLITE_ERROR_MISSING_COLLSEQ";   break;
#endif
#ifdef SQLITE_ERROR_RETRY
    case SQLITE_ERROR_RETRY            : c = "SQLITE_ERROR_RETRY";             break;
#endif
    case SQLITE_IOERR_READ             : c = "SQLITE_IOERR_READ";              break;
    case SQLITE_IOERR_SHORT_READ       : c = "SQLITE_IOERR_SHORT_READ";        break;
    case SQLITE_IOERR_WRITE            : c = "SQLITE_IOERR_WRITE";             break;
    case SQLITE_IOERR_FSYNC            : c = "SQLITE_IOERR_FSYNC";             break;
    case SQLITE_IOERR_DIR_FSYNC        : c = "SQLITE_IOERR_DIR_FSYNC";         break;
    case SQLITE_IOERR_TRUNCATE         : c = "SQLITE_IOERR_TRUNCATE";          break;
    case SQLITE_IOERR_FSTAT            : c = "SQLITE_IOERR_FSTAT";             break;
    case SQLITE_IOERR_UNLOCK           : c = "SQLITE_IOERR_UNLOCK";            break;
    case SQLITE_IOERR_RDLOCK           : c = "SQLITE_IOERR_RDLOCK";            break;
    case SQLITE_IOERR_DELETE           : c = "SQLITE_IOERR_DELETE";            break;
    case SQLITE_IOERR_BLOCKED          : c = "SQLITE_IOERR_BLOCKED";           break;
    case SQLITE_IOERR_NOMEM            : c = "SQLITE_IOERR_NOMEM";             break;
#ifdef SQLITE_IOERR_ACCESS  /* SQLite 3.6.0 and up */
    case SQLITE_IOERR_ACCESS           : c = "SQLITE_IOERR_ACCESS";            break;
    case SQLITE_IOERR_CHECKRESERVEDLOCK: c = "SQLITE_IOERR_CHECKRESERVEDLOCK"; break;
    case SQLITE_IOERR_LOCK             : c = "SQLITE_IOERR_LOCK";              break;
    case SQLITE_IOERR_CLOSE            : c = "SQLITE_IOERR_CLOSE";             break;
    case SQLITE_IOERR_DIR_CLOSE        : c = "SQLITE_IOERR_DIR_CLOSE";         break;
#endif
#ifdef SQLITE_IOERR_SHMOPEN
    case SQLITE_IOERR_SHMOPEN          : c = "SQLITE_IOERR_SHMOPEN";           break;
#endif
#ifdef SQLITE_IOERR_SHMSIZE
    case SQLITE_IOERR_SHMSIZE          : c = "SQLITE_IOERR_SHMSIZE";           break;
#endif
#ifdef SQLITE_IOERR_SHMLOCK
    case SQLITE_IOERR_SHMLOCK          : c = "SQLITE_IOERR_SHMLOCK";           break;
#endif
#ifdef SQLITE_IOERR_SHMMAP
    case SQLITE_IOERR_SHMMAP           : c = "SQLITE_IOERR_SHMMAP";            break;
#endif
#ifdef SQLITE_IOERR_SEEK
    case SQLITE_IOERR_SEEK             : c = "SQLITE_IOERR_SEEK";              break;
#endif
#ifdef SQLITE_IOERR_DELETE_NOENT
    case SQLITE_IOERR_DELETE_NOENT     : c = "SQLITE_IOERR_DELETE_NOENT";      break;
#endif
#ifdef SQLITE_IOERR_MMAP
    case SQLITE_IOERR_MMAP             : c = "SQLITE_IOERR_MMAP";              break;
#endif
#ifdef SQLITE_IOERR_GETTEMPPATH
    case SQLITE_IOERR_GETTEMPPATH      : c = "SQLITE_IOERR_GETTEMPPATH";       break;
#endif
#ifdef SQLITE_IOERR_CONVPATH
    case SQLITE_IOERR_CONVPATH         : c = "SQLITE_IOERR_CONVPATH";          break;
#endif
#ifdef SQLITE_IOERR_VNODE
    case SQLITE_IOERR_VNODE            : c = "SQLITE_IOERR_VNODE";             break;
#endif
#ifdef SQLITE_IOERR_AUTH
    case SQLITE_IOERR_AUTH             : c = "SQLITE_IOERR_AUTH";              break;
#endif
#ifdef SQLITE_IOERR_BEGIN_ATOMIC
    case SQLITE_IOERR_BEGIN_ATOMIC     : c = "SQLITE_IOERR_BEGIN_ATOMIC";      break;
#endif
#ifdef SQLITE_IOERR_COMMIT_ATOMIC
    case SQLITE_IOERR_COMMIT_ATOMIC    : c = "SQLITE_IOERR_COMMIT_ATOMIC";     break;
#endif
#ifdef SQLITE_IOERR_ROLLBACK_ATOMIC
    case SQLITE_IOERR_ROLLBACK_ATOMIC  : c = "SQLITE_IOERR_ROLLBACK_ATOMIC";   break;
#endif
#ifdef SQLITE_LOCKED_SHAREDCACHE
    case SQLITE_LOCKED_SHAREDCACHE     : c = "SQLITE_LOCKED_SHAREDCACHE";      break;
#endif
#ifdef SQLITE_BUSY_RECOVERY
    case SQLITE_BUSY_RECOVERY          : c = "SQLITE_BUSY_RECOVERY";           break;
#endif
#ifdef SQLITE_BUSY_SNAPSHOT
    case SQLITE_BUSY_SNAPSHOT          : c = "SQLITE_BUSY_SNAPSHOT";           break;
#endif
#ifdef SQLITE_CANTOPEN_NOTEMPDIR
    case SQLITE_CANTOPEN_NOTEMPDIR     : c = "SQLITE_CANTOPEN_NOTEMPDIR";      break;
#endif
#ifdef SQLITE_CANTOPEN_ISDIR
    case SQLITE_CANTOPEN_ISDIR         : c = "SQLITE_CANTOPEN_ISDIR";          break;
#endif
#ifdef SQLITE_CANTOPEN_FULLPATH
    case SQLITE_CANTOPEN_FULLPATH      : c = "SQLITE_CANTOPEN_FULLPATH";       break;
#endif
#ifdef SQLITE_CANTOPEN_CONVPATH
    case SQLITE_CANTOPEN_CONVPATH      : c = "SQLITE_CANTOPEN_CONVPATH";       break;
#endif
#ifdef SQLITE_CORRUPT_VTAB
    case SQLITE_CORRUPT_VTAB           : c = "SQLITE_CORRUPT_VTAB";            break;
#endif
#ifdef SQLITE_READONLY_RECOVERY
    case SQLITE_READONLY_RECOVERY      : c = "SQLITE_READONLY_RECOVERY";       break;
#endif
#ifdef SQLITE_READONLY_CANTLOCK
    case SQLITE_READONLY_CANTLOCK      : c = "SQLITE_READONLY_CANTLOCK";       break;
#endif
#ifdef SQLITE_READONLY_ROLLBACK
    case SQLITE_READONLY_ROLLBACK      : c = "SQLITE_READONLY_ROLLBACK";       break;
#endif
#ifdef SQLITE_READONLY_DBMOVED
    case SQLITE_READONLY_DBMOVED       : c = "SQLITE_READONLY_DBMOVED";        break;
#endif
#ifdef SQLITE_READONLY_CANTINIT
    case SQLITE_READONLY_CANTINIT      : c = "SQLITE_READONLY_CANTINIT";       break;
#endif
#ifdef SQLITE_READONLY_DIRECTORY
    case SQLITE_READONLY_DIRECTORY     : c = "SQLITE_READONLY_DIRECTORY";      break;
#endif
#ifdef SQLITE_ABORT_ROLLBACK
    case SQLITE_ABORT_ROLLBACK         : c = "SQLITE_ABORT_ROLLBACK";          break;
#endif
#ifdef SQLITE_CONSTRAINT_CHECK
    case SQLITE_CONSTRAINT_CHECK       : c = "SQLITE_CONSTRAINT_CHECK";        break;
#endif
#ifdef SQLITE_CONSTRAINT_COMMITHOOK
    case SQLITE_CONSTRAINT_COMMITHOOK  : c = "SQLITE_CONSTRAINT_COMMITHOOK";   break;
#endif
#ifdef SQLITE_CONSTRAINT_FOREIGNKEY
    case SQLITE_CONSTRAINT_FOREIGNKEY  : c = "SQLITE_CONSTRAINT_FOREIGNKEY";   break;
#endif
#ifdef SQLITE_CONSTRAINT_FUNCTION
    case SQLITE_CONSTRAINT_FUNCTION    : c = "SQLITE_CONSTRAINT_FUNCTION";     break;
#endif
#ifdef SQLITE_CONSTRAINT_NOTNULL
    case SQLITE_CONSTRAINT_NOTNULL     : c = "SQLITE_CONSTRAINT_NOTNULL";      break;
#endif
#ifdef SQLITE_CONSTRAINT_PRIMARYKEY
    case SQLITE_CONSTRAINT_PRIMARYKEY  : c = "SQLITE_CONSTRAINT_PRIMARYKEY";   break;
#endif
#ifdef SQLITE_CONSTRAINT_TRIGGER
    case SQLITE_CONSTRAINT_TRIGGER     : c = "SQLITE_CONSTRAINT_TRIGGER";      break;
#endif
#ifdef SQLITE_CONSTRAINT_UNIQUE
    case SQLITE_CONSTRAINT_UNIQUE      : c = "SQLITE_CONSTRAINT_UNIQUE";       break;
#endif
#ifdef SQLITE_CONSTRAINT_VTAB
    case SQLITE_CONSTRAINT_VTAB        : c = "SQLITE_CONSTRAINT_VTAB";         break;
#endif
#ifdef SQLITE_CONSTRAINT_ROWID
    case SQLITE_CONSTRAINT_ROWID       : c = "SQLITE_CONSTRAINT_ROWID";        break;
#endif
#ifdef SQLITE_NOTICE_RECOVER_WAL
    case SQLITE_NOTICE_RECOVER_WAL     : c = "SQLITE_NOTICE_RECOVER_WAL";      break;
#endif
#ifdef SQLITE_NOTICE_RECOVER_ROLLBACK
    case SQLITE_NOTICE_RECOVER_ROLLBACK: c = "SQLITE_NOTICE_RECOVER_ROLLBACK"; break;
#endif
#ifdef SQLITE_WARNING_AUTOINDEX
    case SQLITE_WARNING_AUTOINDEX      : c = "SQLITE_WARNING_AUTOINDEX";       break;
#endif
#ifdef SQLITE_AUTH_USER
    case SQLITE_AUTH_USER              : c = "SQLITE_AUTH_USER";               break;
#endif
#ifdef SQLITE_OK_LOAD_PERMANENTLY
    case SQLITE_OK_LOAD_PERMANENTLY    : c = "SQLITE_OK_LOAD_PERMANENTLY";     break;
#endif
    default                            :
        if  (ex & ~0xFF) c = exdecode(ex & 0xFF); else c = "UNKNOWN";
    }
    return(c);
}

static pgoblin_realpq *
/**********************************************************************
 **                                                                  **/
resv(pgoblin_rdb *rdb) {                                            /**
 **                                                                  **
 **********************************************************************/
    pgoblin_realpq *rsv = NULL;

    if  (!(rsv = calloc(1, sizeof(pgoblin_realpq)))) {
        ifBLIN_QW0("No mem");
    } else {
        rsv->flags = rdb->flags & (BLIN_MASK | PGOBLIN_DB_TYPE);
        rsv->db = rdb;
    }
    return(rsv);
}

static int
/**********************************************************************
 **                                                                  **/
busy_handler(void *b, int cnt) {                                    /**
 **                                                                  **
 **********************************************************************/
    float t = 1.001;

    (void)b;
    if  (cnt < 10090) {
        for(; cnt >= 0; --cnt) t *= 1.001;
        usleep((useconds_t)t);
    }
    return(cnt < 0);
}

static int
/**********************************************************************
 **                                                                  **/
collect(sqlite3_stmt *vm, pgoblin_realpq *res) {                    /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

    if  (!!res) {
#       define blin_internal_flags (res->flags & BLIN_MASK)
        ifBLIN_QX3("+");
        if  (!(res->flags & PGOBLIN_TINIT)) {
            res->ncol = sqlite3_column_count(vm);
            res->nrow = 0;
            res->serv = NULL;
            res->flags |= PGOBLIN_TINIT;
            if  (0 < res->ncol) {
                const char *column_name;
                ssize_t     l = 0;
                char       *p;

                for (int y = 0; y < res->ncol; ++y) l += strlen(sqlite3_column_name(vm, y)) + 1;
                ++l;
                if  (!(res->serv = malloc(res->ncol * sizeof(char *) + l))) {
                    ifBLIN_QW0("No mem %"BLIN_D, res->ncol * sizeof(char *) + l);
                    ERROUT(-EX_SOFTWARE, EINVAL);
                }
                p = (char *)&res->serv[res->ncol];
                for (int y = 0; y < res->ncol; ++y) {
                    if  (0 >= l) {
                        ifBLIN_QX0("strings %"BLIN_X" overlow on %d of %d", l, y, res->ncol);
                        break;
                    }
                    column_name = sqlite3_column_name(vm, y);
                    if  (!!column_name) {
                        ifBLIN_QX5("%d[%"BLIN_D"]=%s", y, l, column_name);
                        res->serv[y] = p;
                        p = memccpy(p, column_name, 0, l);
                        l -= p - res->serv[y];
                    } else {
                        res->serv[y] = NULL;
    }   }   }   }   }
out:;
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
clear(pgoblin_exenv *exenv, pgoblin_nr nio) {                       /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio    *rio;
    pgoblin_realpq *rsv;
    int             ex = EX_OK;

    GET_RIO(rio, exenv->options, nio);
    if  (!!rio && !!rio->pq) {
        rsv = rio->pq;
        if  ((rsv->flags & PGOBLIN_MULAR) && !!rsv->md) mular_destroy(rsv->md);
        if  ((rsv->flags & PGOBLIN_VM) && !!rsv->vm) {
            ex = sqlite3_finalize(rsv->vm);
            ifBLIN_QX4("sqlite3_finalize %d %s", ex, exdecode(ex));
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                ifBLIN_QX0("sqlite3_finalize %d: %s", ex, exdecode(ex));
                errno = oxdecode(ex);
                ex = EX_SOFTWARE;
        }   }
        if  (!!rsv->serv) free(rsv->serv);
        if  (!!rsv->typs) free(rsv->typs);
        free(rio->pq);
        rio->pq = NULL;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static ssize_t
/******************************************************************************
 **                                                                          **/
valinfo(pgoblin_exenv *exenv, pgoblin_nr nio, int kind, int row, int col) { /**
 **                                                                          **
 ******************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio  *rio;
    ssize_t       ex = PGOBLIN_SSIZE_MIN;

    GET_RIO(rio, exenv->options, nio);
    ifBLIN_QX2("+ %08X %d x %d (%d)", rio->pq->flags, row, col, kind);
    switch (kind) {
    case PGOBLIN_IsNull:
        if  (rio->pq) {
            if  (rio->pq->flags & PGOBLIN_VM) {
                ifBLIN_QX1("VM instead of MULAR");
                errno = EDOOFUS;
            } else if (!(rio->pq->flags & PGOBLIN_MULAR)) {
                ifBLIN_QX0("NOTHING instead of MULAR");
                errno = EDOOFUS;
            } else if ((row >= 0) && (row < rio->pq->nrow) && (col >= 0) && (col < rio->pq->ncol)) {
                ex = !*(char**)mular_getix(rio->pq->md, INDX(row, col));
            } else {
                ifBLIN_QX1( "+ row %d %s in [0..%d], col %d %s in [0..%d]"
                          , row, (row >= 0 && row < rio->pq->nrow) ? "" : "NOT", rio->pq->nrow
                          , col, (col >= 0 && col < rio->pq->ncol) ? "" : "NOT", rio->pq->ncol
                          );
                errno = EDOOFUS;
            }
        } else if (row == 0 && col == 0) {
            ex = 1;
        } else {
            ifBLIN_QX1("row %d col %d in emty", row, col);
            errno = EDOOFUS;
        }
        break;
    case PGOBLIN_Length:
        switch (valinfo(exenv, nio, PGOBLIN_IsNull, row, col)) {
        case  1:
            ex = 0;
            break;
        case  0:
            if  (rio->pq->flags & PGOBLIN_MULAR) {
                ex = strlen(*(char**)mular_getix(rio->pq->md, INDX(row, col)));
            } else if (rio->pq->flags & PGOBLIN_VM) {
                ifBLIN_QX0("VM instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            } else {
                ifBLIN_QX0("NOTHING instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            }
            break;
        default:
            errno = EDOOFUS;
        }
        break;
    case PGOBLIN_TypeOid:
        if  (!!rio->pq->typs && (rio->pq->ncol > col)) {
            ex = rio->pq->typs[col];
        } else {
            errno = 0;
        }
        break;
    default:
        errno = ENOTSUP;
    }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
init(pgoblin_main *options, int slot) {                             /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (options->flags & BLIN_MASK)
#ifdef SQLITE_CONFIG_SINGLETHREAD
    if  (sqlite3_threadsafe()) {
        ex = sqlite3_config(SQLITE_CONFIG_SINGLETHREAD);
        if  (ex != SQLITE_OK) {
            ifBLIN_QX0("sqlite3_config SINGLETHREAD %d: %s", ex, exdecode(ex));
            ERROUT(EX_UNAVAILABLE, oxdecode(ex));
    }   }
#endif
#ifdef SQLITE_CONFIG_MEMSTATUS
    ex = sqlite3_config(SQLITE_CONFIG_MEMSTATUS, 0);
    if  (ex != SQLITE_OK) {
        ifBLIN_QX0("sqlite3_config MEMSTATUS %d: %s", ex, exdecode(ex));
        ERROUT(EX_UNAVAILABLE, oxdecode(ex));
    }
#endif
#if defined(SQLITE_VERSION_NUMBER) && (SQLITE_VERSION_NUMBER >= 3006000)
    ex = sqlite3_initialize();
    if  (ex != SQLITE_OK) {
        ifBLIN_QX0("sqlite3_initialize %d: %s", ex, exdecode(ex));
        ERROUT(EX_UNAVAILABLE, oxdecode(ex));
    }
#endif
#if defined(SQLITE_CONFIG_SINGLETHREAD)                                   \
 || defined(SQLITE_CONFIG_MEMSTATUS)                                      \
 || (defined(SQLITE_VERSION_NUMBER) && (SQLITE_VERSION_NUMBER >= 3006000))
out:
#endif
    if  (!ex) {
        PGOBLIN_MODULE.flags = (PGOBLIN_MODULE.flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE))
                             | (options->flags & BLIN_MASK)
                             | (slot & PGOBLIN_DB_TYPE)
        ;
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
fini(pgoblin_main *options) {                                       /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (PGOBLIN_MODULE.flags & BLIN_MASK)
#if defined(SQLITE_VERSION_NUMBER) && (SQLITE_VERSION_NUMBER >= 3006000)
    sqlite3_shutdown();
    if  (ex != SQLITE_OK) {
        ifBLIN_QX0("sqlite3_shutdown %d: %s", ex, exdecode(ex));
        ERROUT(EX_UNAVAILABLE, oxdecode(ex));
    }
out:
#endif
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
shurecon(pgoblin_exenv *exenv, pgoblin_nr ndb) {                    /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *conn = NULL;
    pgoblin_rdb     *rdb;
    int              ex   = EX_OK;

/* XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX *
   X sqlite3_config(SQLITE_CONFIG_MMAP_SIZE, 268435456, 268435456)  X
   XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX */

    GET_CON(rdb, exenv->options, ndb);
    conn = rdb->conn;
    if  (!!conn) goto out;
    ifBLIN_QX4("d%"BLIN_X" H version=%s lib version=%s", rdb, SQLITE_VERSION, sqlite3_version);
    if  (!(conn = calloc(1, sizeof(pgoblin_db_conn)))) {
        ifBLIN_QX0("no MEM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    conn->flags = rdb->flags & (BLIN_MASK | PGOBLIN_DB_TYPE);
    ex = sqlite3_open_v2( rdb->dbname ? rdb->dbname : ":memory:"
                        , &conn->odf
                        , SQLITE_OPEN_READWRITE
                        | SQLITE_OPEN_CREATE
                        | SQLITE_OPEN_PRIVATECACHE
#ifdef SQLITE_OPEN_EXRESCODE
                        | SQLITE_OPEN_EXRESCODE
#endif
                        , NULL
                        );
    if  (!conn->odf) {
        ifBLIN_QX0("aborted");
        ERROUT(EX_UNAVAILABLE, ENODEV);
    }
    ERRMSG("sqlite3_open", EX_UNAVAILABLE);
    MARK_R_CONN_GO(ndb);;
    ;;  rdb->conn = conn;
    ;;  rdb->flags = PGOBLIN_CLOSECONN
 /* ;; */          | (rdb->flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE))
 /* ;; */          | (PGOBLIN_MODULE.flags & (BLIN_MASK | PGOBLIN_DB_TYPE))
    ;;  ;
    MARK_R_CONN_WENT(ndb);;
    ex = sqlite3_busy_handler(conn->odf, busy_handler, NULL);
    ERRMSG("sqlite3_busy_handler", EX_UNAVAILABLE);
    ex = sqlite3_create_function( conn->odf
                                , "regexp"
                                , 2
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)rdb->flags
                                , pgoblin_sqlite3_regexp
                                , NULL
                                , NULL
                                );
    ERRMSG("regexp creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "quote_literal"
                                , 1
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)((rdb->flags & BLIN_MASK) | SQLITE_SQ)
                                , pgoblin_sqlite3_quote
                                , NULL
                                , NULL
                                );
    ERRMSG("quote_literal creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "quote_ident"
                                , 1
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)
                                         ( (rdb->flags & BLIN_MASK)
                                         | SQLITE_DQ | SQLITE_SF | SQLITE_KY
                                         )
                                , pgoblin_sqlite3_quote
                                , NULL
                                , NULL
                                );
    ERRMSG("quote_ident creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "quote_ident_always"
                                , 1
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)((rdb->flags & BLIN_MASK) | SQLITE_DQ)
                                , pgoblin_sqlite3_quote
                                , NULL
                                , NULL
                                );
    ERRMSG("quote_ident_always creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "quote_html"
                                , 1
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)(rdb->flags & BLIN_MASK)
                                , pgoblin_sqlite3_quote_html
                                , NULL
                                , NULL
                                );
    ERRMSG("quote_html creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "gettimeofday"
                                , 0
                                , SQLITE_ANY
                                , (void*)(long int)(rdb->flags & BLIN_MASK)
                                , pgoblin_sqlite3_gettimeofday
                                , NULL
                                , NULL
                                );
    ERRMSG("gettimeofday creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "regexp_replace"
                                , 4
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)(rdb->flags & BLIN_MASK)
                                , pgoblin_sqlite3_regexrepl
                                , NULL
                                , NULL
                                );
    ERRMSG("regexp_replace 4 creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "regexp_replace"
                                , 3
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)(rdb->flags & BLIN_MASK)
                                , pgoblin_sqlite3_regexrepl
                                , NULL
                                , NULL
                                );
    ERRMSG("regexp_replace 3 creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "koilower"
                                , 1
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)(rdb->flags & BLIN_MASK)
                                , pgoblin_sqlite3_upandown
                                , NULL
                                , NULL
                                );
    ERRMSG("koilower creation", EX_DATAERR);
    ex = sqlite3_create_function( conn->odf
                                , "koiupper"
                                , 1
                                , SQLITE_UTF8 | SQLITE_DETERMINISTIC
                                , (void*)(long int)((rdb->flags & BLIN_MASK) | 1)
                                , pgoblin_sqlite3_upandown
                                , NULL
                                , NULL
                                );
    ERRMSG("koiupper creation", EX_DATAERR);
out:
    ifBLIN_QX4("- %d d%"BLIN_X"->c%"BLIN_X, ex, BLIN_I(rdb), BLIN_I(conn));
    return(ex);
#   undef blin_internal_flags
}

static int
/********************************************************************************
 **                                                                            **/
query(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nct) { /**
 **                                                                            **
 ********************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    BLIN_flag        cond;
    pgoblin_db_conn *conn;
    int              ncol;
    pgoblin_realpq **rsv = NULL;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rou;
    pgoblin_rio     *rct;
    char            *qry;
    int              idx;
    int              ex  = 0;
    const char      *q;
    char            *p;
    char            *c;
    int              i;
    static const babolo_lexor copy = {(u_char*)
    "\000..." "...."    "  | "    "  .."    "...."    "...."    "...."    "...."
    " .."    "..."    "...."    ".-.."    "IIII"    "IIII"    "II.."    "...."
    ".EEC"    "EEEE"    "EEEE"    "EEEO"    "PEEE"    "EEEE"    "EYE."    "...E"
    ".EEC"    "EEEE"    "EEEE"    "EEEO"    "PEEE"    "EEEE"    "EYE."    "...."
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    , NULL, Bpars_AABS | Bpars_CEND, 0x1C, 0x1C, 1
    ,{/*00*/ 3, ' ', 0x00, '|', 0x00, '-', 0x05, 'C', 0x09, 0xFF
     ,/*05*/ 0, '-', 0x07, 0xFF
     ,/*07*/ 0, '|', 0x00, 0x07
     ,/*09*/ 0, 'O', 0x0B, 0xFF
     ,/*0B*/ 0, 'P', 0x0D, 0xFF
     ,/*0D*/ 0, 'Y', 0x0F, 0xFF
     ,/*0F*/ 8, '', 0xFE, ' ', 0x0F, '|', 0x0F, '-', 0x19, 'C', 0xFD
              , 'O', 0xFD, 'P', 0xFD, 'Y', 0xFD, 'E', 0xFD, 0xFF
     ,/*19*/ 0, '-', 0x1A, 0xFF
     ,/*1A*/ 0, '|', 0x0F, 0x1A
    }};
    /*****************
        * 0 ...
        *   ' --\n'...
        * 1 'COPY "'
        * 1 'COPY --\n "'
        * 1 "COPY '"
        * 1 'COPY\n"'
        * 2 'COPY x'
        * 2 'COPY --\n x'
    *****************/

    GET_CON(rdb, exenv->options, ndb);
    GET_RIO(rou, exenv->options, nou);
    GET_RIO(rct, exenv->options, nct);
    qry = rct->text;
    cond = rdb->flags & PGOBLIN_CR_MASK;
    conn = rdb->conn;
    ifBLIN_QX4("+ d%"BLIN_X"->c%"BLIN_X" %d=%s~", BLIN_I(rdb), BLIN_I(conn), cond, qry);
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    rsv = &rou->pq;
    *rsv = NULL;
    if  (qry && *qry) conn->query = qry; else goto out;
    q = conn->query;
    switch (cond) {
    case PGOBLIN_CR_IN:
        conn->coln = 0;
        conn->state = bg;
        conn->semi = NULL;
        switch (babolo_goword(&copy, &q)) {
        case 0:
            ifBLIN_QX4("sqlite3_prepare_v2#1 %d=%s~", cond, conn->query);
            ex = sqlite3_prepare_v2(conn->odf, conn->query, -1, &conn->vm, &conn->query);
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_prepare_v2#1 %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            conn->flags |= PGOBLIN_COPYIN;
            break;
        case 1:
            if  (!*q) ERROUT(EX_DATAERR, EINVAL);
            q--;
            if  (*q == '\'') {
                for (idx = 1; ;) {
                    if (q[idx++] == '\'' && q[idx] != '\'') break;
                    if (!q[idx]) ERROUT(EX_DATAERR, EINVAL);
                }
            } else if (*q == '\"') {
                for (idx = 1; ;) {
                    if (q[idx++] == '\"' && q[idx] != '\"') break;
                    if (!q[idx]) ERROUT(EX_DATAERR, EINVAL);
                }
            } else ERROUT(EX_DATAERR, EINVAL);
            idx++;
            ifBLIN_QX4("goword quoted %d", idx);
            goto p2;
        case 2:
            q--;
            for (idx = 1; ;) if ((copy.retable[q[idx++] & 0xFF] & 0xC0) != 0x40) break;
            ifBLIN_QX4("goword direct %d", idx);
        p2: if  (!(p = strndup(q, --idx))) {
                ifBLIN_QX0("no mem #0");
                ERROUT(EX_OSERR, ENOMEM);
            }
            conn->query = p;
            if  (asprintf(&c, "SELECT * FROM %s", conn->query) < 0) {
                ifBLIN_QX0("no mem #1");
                ERROUT(EX_OSERR, ENOMEM);
            }
            ifBLIN_QX4("sqlite3_prepare_v2#2 %d=%s~", cond, c);
            ex = sqlite3_prepare_v2(conn->odf, c, -1, &conn->vm, NULL);
            free(c);
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_prepare_v2#2 %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            if  (!conn->vm) {
                ifBLIN_QX0("no VM #0");
                ERROUT(EX_SOFTWARE, ENOEXEC);
            }
            ex = sqlite3_step(conn->vm);
            ifBLIN_QX4("sqlite3_step %d %s", ex, exdecode(ex));
            if  (ex == SQLITE_ROW || ex == SQLITE_DONE) ex = 0;
            if  (ex) {
                ifBLIN_QX0("sqlite3_step: %d: %s", ex, sqlite3_errmsg(conn->odf));
                if  (conn->vm) {
                    ex = sqlite3_finalize(conn->vm);
                    if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                        if  (!*rsv && !(*rsv = resv(rdb))) {
                            ifBLIN_QX0("no RAM");
                            ERROUT(EX_OSERR, ENOMEM);
                        }
                        ifBLIN_QX0("sqlite3_finalize %d: %s", ex, sqlite3_errmsg(conn->odf));
                        ERROUT(EX_SOFTWARE, oxdecode(ex));
                }   }
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            ncol = sqlite3_column_count(conn->vm);
            ex = sqlite3_finalize(conn->vm);
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_finalize %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            conn->coln = ncol;
            if  (!(p = malloc(2 * ncol * sizeof(char)))) {
                ifBLIN_QX0("no mem #2");
                ERROUT(EX_OSERR, ENOMEM);
            }
            for (i = 0; i < ncol; i++) {
                p[2 * i] = '?';
                p[2 * i + 1] = ',';
            }
            p[2 * ncol - 1] = '\0';
            if  (asprintf(&c, "INSERT INTO %s VALUES(%s)", conn->query, p) < 0) {
                ifBLIN_QX0("no mem #3");
                ERROUT(EX_OSERR, ENOMEM);
            }
            free(p);
            ifBLIN_QX4("sqlite3_prepare_v2#3 %d=%s~", cond, c);
            ex = sqlite3_prepare_v2(conn->odf, c, -1, &conn->vm, NULL);
            free(c);
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_prepare_v2#3 %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            if  (!conn->vm) {
                ifBLIN_QX0("no VM #1");
                ERROUT(EX_SOFTWARE, ENOEXEC);
        }   }
        conn->ncol = 0;
        conn->flags |= PGOBLIN_COPYIN;
        break;
    case PGOBLIN_CR_OUT:
        switch (babolo_goword(&copy, &q)) {
        case 1:
            if  (!*q) {
                ERROUT(EX_DATAERR, EINVAL);
            }
            q--;
            if  (*q == '\'') {
                for (idx = 1; ;) {
                    if  (q[idx++] == '\'' && q[idx] != '\'') break;
                    if  (!q[idx]) {
                        ERROUT(EX_DATAERR, EINVAL);
                }   }
            } else if (*q == '\"') {
                for (idx = 1; ;) {
                    if  (q[idx++] == '\"' && q[idx] != '\"') break;
                    if  (!q[idx]) {
                        ERROUT(EX_DATAERR, EINVAL);
                }   }
            } else {
                ERROUT(EX_DATAERR, EINVAL);
            }
            idx++;
            ifBLIN_QX4("goword quoted %d", idx);
            goto p3;
        case 2:
            q--;
            for (idx = 1; ;) if ((copy.retable[q[idx++] & 0xFF] & 0xC0) != 0x40) break;
            ifBLIN_QX4("goword direct %d", idx);
        p3: if  (!(p = strndup(q, --idx))) {
                ifBLIN_QX0("no mem #5");
                ERROUT(EX_OSERR, ENOMEM);
            }
            conn->query = p;
            if  (asprintf(&c, "SELECT * FROM %s", conn->query) < 0) {
                ifBLIN_QX0("no mem #6");
                ERROUT(EX_OSERR, ENOMEM);
            }
            ifBLIN_QX4("sqlite3_prepare_v2#8 %d=%s~", cond, c);
            ex = sqlite3_prepare_v2(conn->odf, c, -1, &conn->vm, NULL);
            free(c);
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_prepare_v2#8 %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            if  (!conn->vm) {
                ifBLIN_QX0("no VM #2");
                ERROUT(EX_SOFTWARE, ENOEXEC);
            }
            conn->flags |= PGOBLIN_STREAMI;
            goto out;
        case 0:
            ;
        }   /* FALLTHRU */
    case PGOBLIN_CR_SSEL:
        ifBLIN_QX4("sqlite3_prepare_v2#6 %d=%s~", cond, conn->query);
        ex = sqlite3_prepare_v2(conn->odf, conn->query, -1, &conn->vm, &conn->query);
        if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
            if  (!*rsv && !(*rsv = resv(rdb))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            ifBLIN_QX0("sqlite3_prepare_v2#6 %d: %s", ex, sqlite3_errmsg(conn->odf));
            ERROUT(EX_SOFTWARE, oxdecode(ex));
        }
        if  (!conn->vm) {
            ifBLIN_QX0("no VM #3");
            ERROUT(EX_SOFTWARE, ENOEXEC);
        }
        conn->flags |= PGOBLIN_STREAMI;
        break;
    case PGOBLIN_CR_OK:
        while (!ex && !!conn->query && !!*conn->query) {
            ifBLIN_QX4("sqlite3_prepare_v2#4 %d=%s~", cond, conn->query);
#ifdef SQLITE_BUZYREPEAT
            for (int g = 0; g < SQLITE_BUZYREPEAT; g++) {
                ex = sqlite3_prepare_v2(conn->odf, conn->query, -1, &conn->vm, &conn->query);
                if  (SQLITE_BUSY != (ex & 0xFF)) break;
                ifBLIN_QX1("SQLITE_BUSY %d #%d", ex, g);
                usleep(SQLITE_BUZYWAIT);
            }
#else
            ex = sqlite3_prepare_v2(conn->odf, conn->query, -1, &conn->vm, &conn->query);
#endif
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_prepare_v2#4 %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            if  (!conn->vm) {
                ifBLIN_QX4("sqlite3_prepare_v2#4 no VM");
                break;
            }
            ncol = sqlite3_column_count(conn->vm);
            if  (!!ncol) {
                ifBLIN_QX1("wait for PGOBLIN_CR_OK but received %d cols", ncol);
            }
            for (; ;) {
                ex = sqlite3_step(conn->vm);
                if  (ex == SQLITE_ROW) continue;
                if  (ex == SQLITE_DONE) ex = 0;
                if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                    if  (!*rsv && !(*rsv = resv(rdb))) {
                        ifBLIN_QX0("no RAM");
                        ERROUT(EX_OSERR, ENOMEM);
                    }
                    ifBLIN_QX0("sqlite3_step#2 %d: %s", ex, sqlite3_errmsg(conn->odf));
                    ERROUT(EX_SOFTWARE, oxdecode(ex));
                }
                break;
            }
            if  (!!ex) {
                ifBLIN_QX0("sqlite3_step: %d: %s", ex, sqlite3_errmsg(conn->odf));
                if  (conn->vm) {
                    ex = sqlite3_finalize(conn->vm);
                    if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                        if  (!*rsv && !(*rsv = resv(rdb))) {
                            ifBLIN_QX0("no RAM");
                            ERROUT(EX_OSERR, ENOMEM);
                        }
                        ifBLIN_QX0("sqlite3_finalize %d: %s", ex, sqlite3_errmsg(conn->odf));
                        ERROUT(EX_SOFTWARE, oxdecode(ex));
                }   }
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            ex = sqlite3_finalize(conn->vm);
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_finalize %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
        }   }
        break;
    case PGOBLIN_CR_TUPL:
        while (!ex && !!conn->query && !!*(conn->query)) {
            ifBLIN_QX4("sqlite3_prepare_v2#5 %d=%s~", cond, conn->query);
            ex = sqlite3_prepare_v2(conn->odf, conn->query, -1, &conn->vm, &conn->query);
            if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
                if  (!*rsv && !(*rsv = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0( "sqlite3_prepare_v2=%s~ %d: %s"
                          , conn->query
                          , ex
                          , sqlite3_errmsg(conn->odf)
                          );
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            if  (!conn->vm) {
                ifBLIN_QX4("sqlite3_prepare_v2#5 no VM");
                break;
            }
            if  (!!*rsv) clear(exenv, nou);
            if  (!(*rsv = resv(rdb))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            if  (0 > (collect(conn->vm, *rsv))) {
                ifBLIN_QX0("collect");
                goto out;
            }
            for (int y = 0; ; ++y) {
                ex = sqlite3_step(conn->vm);
                ERRMSG("sqlite3_step#3", EX_SOFTWARE);
                ifBLIN_QX4("sqlite3_step %d %d %s", y, ex, exdecode(ex));
                if  (ex == SQLITE_ROW) {
                    if  (!(*rsv)->typs) {
                        if  (!((*rsv)->typs = calloc(1, (*rsv)->ncol))) {
                            ifBLIN_QW0("No mem %"BLIN_D, (*rsv)->ncol);
                            ERROUT(-EX_SOFTWARE, EINVAL);
                        }
                        for (int y = 0; y < (*rsv)->ncol; ++y) {
                            (*rsv)->typs[y] = sqlite3_column_type(conn->vm, y);
                    }   }
                    ex = 0;
                    if  (!((*rsv)->flags & PGOBLIN_MULAR)) {
                        (*rsv)->md = mular_create( MULAR_CHAR | MULAR_ZERO
                                                 | MULAR_PFRE | MULAR_STRI | MULAR_UPPE
                                                 , 3, sizeof(char*), mdp
                                                 );
                        (*rsv)->md->tofree = free;
                        (*rsv)->flags |= PGOBLIN_MULAR;
                    }
                    for (ncol = 0; ncol < (*rsv)->ncol; ncol++) {
                        char   *tt;
                        char  **m;

                        if  (!(m = mular_add((*rsv)->md))) {
                            ifBLIN_QX0("mular_add %d x %d", y, ncol);
                            ERROUT(EX_SOFTWARE, EDOOFUS);
                        }
                        if  (sqlite3_column_type(conn->vm, ncol) == SQLITE_NULL) {
                            *m = NULL;
                        } else {
                            ifBLIN_QX7( "[%"BLIN_D"]=%s~"
                                      , strlen((const char *)sqlite3_column_text(conn->vm, ncol))
                                      , sqlite3_column_text(conn->vm, ncol)
                                      );
                            if  (!sqlite3_column_text(conn->vm, ncol)) {
                                ifBLIN_QX0("no sqlite3_column_text");
                                ERROUT(EX_OSERR, ENOMEM);
                            }
                            if  (!(tt = strdup((const char *)sqlite3_column_text(conn->vm, ncol)))) {
                                ifBLIN_QX0("no mem #7");
                                ERROUT(EX_OSERR, ENOMEM);
                            }
                            *m = tt;
                    }   }
                    (*rsv)->nrow++;
                    continue;
                }
                if  (ex == SQLITE_DONE) ex = 0;
                break;
            }
            if  (!!ex) {
                ifBLIN_QX0("sqlite3_step: %d: %s", ex, sqlite3_errmsg(conn->odf));
                if  (conn->vm) {
                    ex = sqlite3_finalize(conn->vm);
                    ERRMSG("sqlite3_finalize", EX_SOFTWARE);
                }
                ERROUT(EX_SOFTWARE, oxdecode(ex));
            }
            ex = sqlite3_finalize(conn->vm);
            ERRMSG("sqlite3_finalize", EX_SOFTWARE);
        }
        if  (!!*rsv && !(*rsv)->ncol) {
            ifBLIN_QX0("sqlite3_get_table waits for PGOBLIN_CR_TUPL but received no tuples");
            ERROUT(EX_SOFTWARE, ENODEV);
        }
        break;
    case PGOBLIN_CR_BSEL:
    case PGOBLIN_CR_BTUPL:
    default:
        ifBLIN_QX0("illegal cond=%d", cond);
        ex = EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
out:
    ifBLIN_QX4("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************************
 **                                                                              **/
prepare(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nct) { /**
 **                                                                              **
 **********************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *conn;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rou;
    pgoblin_rio     *rct;
    pgoblin_realpq **rsv = NULL;
    char            *qry;
    int              ex   = 0;

    GET_CON(rdb, exenv->options, ndb);
    GET_RIO(rou, exenv->options, nou);
    GET_RIO(rct, exenv->options, nct);
    qry = rct->text;
    conn = rdb->conn;
    ifBLIN_QX4("+ =%s~", qry);
    rsv = &rou->pq;
    if  (!(*rsv = resv(rdb))) {
        ifBLIN_QX0("no RAM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    if  (!qry || !*qry) goto out;
    ifBLIN_QX4("sqlite3_prepare_v2#4 =%s~", qry);
    ex = sqlite3_prepare_v2(conn->odf, qry, -1, &(*rsv)->vm, NULL);
    ERRMSG("sqlite3_prepare_v2#4", EX_SOFTWARE);
out:
    ifBLIN_QX4("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/*****************************************************************************************************
 **                                                                                                 **/
execute(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nct, pgoblin_nr nin) {    /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    BLIN_flag        cond;
    pgoblin_db_conn *conn;
    int              ncol;
    int              imdx;
    pgoblin_realpq **rsv  = NULL;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rin;
    pgoblin_rio     *rct;
    pgoblin_rio     *rou;
    pgoblin_realpq  *res = NULL;
    char            *qry;
    int              col;
    int              row;
    int              ex   = 0;
    sqlite3_stmt    *vm;

    GET_CON(rdb, exenv->options, ndb);
    cond = rdb->flags & PGOBLIN_ACR_MASK;
    conn = rdb->conn;
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    if  ((PGOBLIN_CR_OK != cond) && (PGOBLIN_CR_TUPL != cond)) {
        ifBLIN_QX0("illegal cond=%d", cond);
        ERROUT(EX_UNAVAILABLE, ENOTSUP);
    }
    GET_RIO(rct, exenv->options, nct);
    qry = rct->text;
    if  (!qry || !*qry) goto out;
    ifBLIN_QX4("sqlite3_prepare_v2#4 %d=%s~", cond, qry);
    ex = sqlite3_prepare_v2(conn->odf, qry, -1, &vm, NULL);
    if  ((ex != SQLITE_OK) && (ex != SQLITE_ROW) && (ex != SQLITE_DONE)) {
        if  (!res && !(res = resv(rdb))) {
            ifBLIN_QX0("no RAM");
            ERROUT(EX_OSERR, ENOMEM);
        }
        ifBLIN_QX0("sqlite3_prepare_v2#4 %d: %s", ex, sqlite3_errmsg(conn->odf));
        ERROUT(EX_SOFTWARE, oxdecode(ex));
    }
    vm = conn->vm;
    if  (!vm) goto out;
    GET_RIO(rou, exenv->options, nou);
    rsv = &rou->pq;
    res = NULL;
    GET_RIO(rin, exenv->options, nin);
    col = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Nfields);
    row = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Ntuples);
    imdx = sqlite3_bind_parameter_count(vm);
    for (int i = 0; i < row; i++) {
        for (int j = 0; (j < col) && (j < imdx); j++) {
            ifBLIN_QX4( "sqlite3_bind_text#3 [%d]%ld=%s~"
                      , j
                      , (long)pgoblin_db_valinfo(exenv, nin, PGOBLIN_Length, i, j)
                      , pgoblin_db_getvalue(exenv, nin, i, j)
                      );
            ex = sqlite3_bind_text( vm
                                  , j + 1
                                  , pgoblin_db_getvalue(exenv, nin, i, j)
                                  , (int)pgoblin_db_valinfo(exenv, nin, PGOBLIN_Length, i, j)
                                  , SQLITE_STATIC
                                  );
            if  (SQLITE_RANGE == ex) {
                ifBLIN_QX1("No index %d", j + 1);
            } else if ((SQLITE_OK != ex) && (SQLITE_ROW != ex) && (SQLITE_DONE != ex)) {
                if  (!res && !(res = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_bind_text#3 %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
        }   }
        res = NULL;
        if  (PGOBLIN_CR_OK == cond) {
            ncol = sqlite3_column_count(vm);
            if  (ncol) {
                ifBLIN_QX1("wait for PGOBLIN_CR_OK but received %d cols", ncol);
            }
        } else if (i + 1 >= row) {
            if  (!(res = resv(rdb))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            collect(vm, res);
        }
        for (; ;) {
            ex = sqlite3_step(vm);
            ERRMSG("sqlite3_step#3", EX_SOFTWARE);
            if  (!!res && !(res->flags & PGOBLIN_TINIT)) {
                res->nrow = 0;
                res->flags |= PGOBLIN_TINIT;
            }
            if  (ex == SQLITE_ROW) {
                if  (!!res) {
                    ex = 0;
                    if  (!(res->flags & PGOBLIN_MULAR)) {
                        res->md = mular_create( MULAR_CHAR
                                              | MULAR_ZERO
                                              | MULAR_PFRE
                                              | MULAR_STRI
                                              | MULAR_UPPE
                                              , 3
                                              , sizeof(char*)
                                              , mdp
                                              );
                        res->md->tofree = free;
                        res->flags |= PGOBLIN_MULAR;
                    }
                    for (int j = 0; j < res->ncol; j++) {
                        const u_char *t;
                        char *tt;

                        if  (sqlite3_column_type(vm, j) == SQLITE_NULL) {
                            *(char**)(mular_add(res->md)) = NULL;
                        } else if ((t = sqlite3_column_text(vm, j)) && (tt = strdup((char*)t))) {
                            *(char**)(mular_add(res->md)) = tt;
                        } else {
                            ifBLIN_QX0("no mem #7");
                            ERROUT(EX_OSERR, ENOMEM);
                    }   }
                    res->nrow++;
                }
                continue;
            }
            if  (ex == SQLITE_DONE) ex = 0;
            break;
        }
        if  (!!ex) {
            ifBLIN_QX0("sqlite3_step: %d: %s", ex, sqlite3_errmsg(conn->odf));
            if  (vm) {
                ex = sqlite3_finalize(vm);
                ERRMSG("sqlite3_finalize", EX_SOFTWARE);
            }
            ERROUT(EX_SOFTWARE, oxdecode(ex));
        }
        ex = sqlite3_finalize(vm);
        ERRMSG("sqlite3_finalize", EX_SOFTWARE);
        if  (!!res && !res->ncol) {
            ifBLIN_QX0("sqlite3_get_table waits for PGOBLIN_CR_TUPL but received no tuples");
            ERROUT(EX_SOFTWARE, ENODEV);
    }   }
out:
    if  (!!rsv) *rsv = res; else if (!!res) clear(exenv, nou);
    ifBLIN_QX4("- %d %"BLIN_X, ex, BLIN_I(!rsv ? NULL : *rsv));
    return(ex);
#   undef blin_internal_flags
}

static int64_t
/**********************************************************************
 **                                                                  **/
resinfo(pgoblin_exenv *exenv, pgoblin_nr nio, int kind) {           /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_realpq *rsv;
    pgoblin_rio    *rio;
    int64_t         ex = 0;

    GET_RIO(rio, exenv->options, nio);
    rsv = rio->pq;

    ifBLIN_QX2("+ %d", kind);
    if  (!!rsv) {
        switch (kind) {
        case PGOBLIN_Ntuples:
            ex = rsv->nrow;
            break;
        case PGOBLIN_Nfields:
            ex = rsv->ncol;
            break;
        case PGOBLIN_LastOid:
            if  ((PGOBLIN_VM & rsv->flags) && !!rsv->db && !!rsv->db->conn) {
                ex = sqlite3_last_insert_rowid(((pgoblin_db_conn *)rsv->db->conn)->odf);
            }
            break;
        case PGOBLIN_Nchange:
            if  ((PGOBLIN_VM & rsv->flags) && !!rsv->db && !!rsv->db->conn) {
                ex = sqlite3_changes(((pgoblin_db_conn *)rsv->db->conn)->odf);
            }
            break;
        default             :
            ex = -EX_UNAVAILABLE;
            errno = ENOTSUP;
    }   }
out:
    ifBLIN_QX2("- %ld", ex);
    return(ex);
#   undef blin_internal_flags
}

static const char *
/**************************************************************************
 **                                                                      **/
subinfo(pgoblin_exenv *exenv, pgoblin_nr nio, int kind, int64_t sub) {  /**
 **                                                                      **
 **************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio  *rio;
    const char   *cx = NULL;
    int64_t       ex;

    GET_RIO(rio, exenv->options, nio);
    ifBLIN_QX2("+ %d %d", kind, sub);
    switch (kind) {
    case PGOBLIN_ColName:
        if  (!!rio->pq->serv && (rio->pq->ncol > sub)) cx = rio->pq->serv[sub];
        break;
    case PGOBLIN_TypeNm:
        if  (!!rio->pq->serv && (rio->pq->ncol > sub)) {
            ex = valinfo(exenv, nio, PGOBLIN_TypeOid, 0, sub);
            cx = types[(ex < TYPEMAX) ? ex : 0];
        }
        break;
    default:;
    }
out:
    ifBLIN_QX2("- %s", cx);
    return(cx);
#   undef blin_internal_flags
}

static char *
/**********************************************************************
 **                                                                  **/
erinfo(pgoblin_exenv *exenv, pgoblin_nr nio) {                      /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *conn;
    pgoblin_rio     *rio;
    const char      *cx = NULL;

    GET_RIO(rio, exenv->options, nio);
    conn = rio->pq->db->conn;
    cx = sqlite3_errmsg(conn->odf);
out:
    return(!cx ? NULL : strdup(cx));
}

static char *
/**********************************************************************
 **                                                                  **/
getvalue(pgoblin_exenv *exenv, pgoblin_nr nio, int row, int col) {  /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio  *rio;
    char         *c = NULL;

    GET_RIO(rio, exenv->options, nio);
    ifBLIN_QX2("+ %08X %d x %d", rio->pq->flags, row, col);
    if  (!!rio->pq && !valinfo(exenv, nio, PGOBLIN_IsNull, row, col)) {
        if  (rio->pq->flags & PGOBLIN_VM) {
            ifBLIN_QX0("VM instead of MULAR");
            errno = EDOOFUS;
        } else if (rio->pq->flags & PGOBLIN_MULAR) {
            c = *(char**)mular_getix(rio->pq->md, INDX(row, col));
            ifBLIN_QX4("mular=%s~", c);
        } else {
            ifBLIN_QX0("NOTHING instead of MULAR");
            errno = EDOOFUS;
    }   }
    ifBLIN_QX2("- %s", !c ? "" : c);
out:
    return(c);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
get(pgoblin_rdb *rdb, char ***row, pgoblin_realpq *rsv) {           /**
 **                                                                  **
 **********************************************************************/
    pgoblin_db_conn *conn = rdb->conn;
    int              ex   = 0;

#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    if  (!(conn->flags & PGOBLIN_STREAMI)) {
        ifBLIN_QX0("not COPYout state");
        ERROUT(-EX_DATAERR, ENODEV);
    }
    for (rsv->ncol = 0; !rsv->ncol;) {
        if  (!conn->vm) {
            conn->flags |= PGOBLIN_STREAME;
            break;
        }
        while (!rsv->ncol) {
            collect(conn->vm, rsv);
            ex = sqlite3_step(conn->vm);
            ERRMSG("sqlite3_step#4", -EX_SOFTWARE);
            if  (rsv->ncol) conn->flags |= PGOBLIN_STREAMT;
            if  (ex == SQLITE_ROW) {
                ex = 0;
                if  (!(*row = malloc(sizeof(char**) * rsv->ncol))) {
                    ifBLIN_QX0("no mem #1");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                for (int i = 0; i < rsv->ncol; i++) {
                    const u_char *t;

                    if  (sqlite3_column_type(conn->vm, i) == SQLITE_NULL) {
                        (*row)[i] = NULL;
                    } else if (  !(t = sqlite3_column_text(conn->vm, i))
                              || !((*row)[i] = strdup((char*)t))
                              ) {
                        ifBLIN_QX0("no mem #2");
                        ERROUT(EX_OSERR, ENOMEM);
                }   }
                continue;
            }
            if  (ex == SQLITE_DONE) {
                if  (conn->flags & PGOBLIN_STREAMT) conn->flags |= PGOBLIN_STREAME;
                ex = 0;
            }
            break;
        }
        if  (conn->flags & PGOBLIN_STREAME) break;
        if  (ex) {
            ifBLIN_QX0("sqlite3_step: %d: %s", ex, sqlite3_errmsg(conn->odf));
            if  (conn->vm) {
                ex = sqlite3_finalize(conn->vm);
                ERRMSG("sqlite3_finalize", -EX_SOFTWARE);
            }
            ERROUT(-EX_SOFTWARE, oxdecode(ex));
        }
        if  (!rsv->ncol && conn->query && *(conn->query)) {
            ex = sqlite3_finalize(conn->vm);
            ERRMSG("sqlite3_finalize", -EX_SOFTWARE);
            ifBLIN_QX4("sqlite3_prepare_v2#7 =%s~", conn->query);
            ex = sqlite3_prepare_v2(conn->odf, conn->query, -1, &conn->vm, &conn->query);
            ERRMSG("sqlite3_prepare_v2#7", -EX_SOFTWARE);
    }   }
    if  (!row) conn->flags |= PGOBLIN_STREAME;
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/***********************************************************************************
 **                                                                               **/
getstream(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, int quant) {     /**
 **                                                                               **
 ***********************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *conn;
    pgoblin_realpq **rsv;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rou;
    char           **row;
    int              ex   = EX_OK;
    int              i;

    GET_CON(rdb, exenv->options, ndb);
    conn = rdb->conn;
    GET_RIO(rou, exenv->options, nou);
    if  (!(rsv = &rou->pq)) {
        ifBLIN_QX0("no res");
        ERROUT(EX_USAGE, EINVAL);
    }
    if  (!!*rsv) clear(exenv, nou);
    if  (!(*rsv = resv(rdb))) {
        ifBLIN_QX0("no RAM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    if  (!!(ex = get(rdb, &row, *rsv))) goto out;
    (*rsv)->nrow = 0;
    if  (conn->flags & PGOBLIN_STREAME) {
        ex = 0;
    } else {
        (*rsv)->md = mular_create( MULAR_CHAR | MULAR_ZERO | MULAR_PFRE | MULAR_STRI | MULAR_UPPE
                                 , 3
                                 , sizeof(u_char*)
                                 , mdp
                                 );
        (*rsv)->md->tofree = free;
        (*rsv)->flags |= PGOBLIN_MULAR;
        for (i = 0; i < (*rsv)->ncol; ++i) {
            if (!row[i]) {
               *(char**)(mular_add((*rsv)->md)) = NULL;
            } else if (!(*(char**)(mular_add((*rsv)->md)) = row[i])) {
                ifBLIN_QX0("no mem");
                ERROUT(EX_OSERR, ENOMEM);
        }   }
        (*rsv)->nrow = 1;
        free(row);
        row = NULL;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/*****************************************************************************
 **                                                                         **/
getcopy(pgoblin_exenv *exenv, pgoblin_nr ndb, void **buf) {                /**
 **                                                                         **
 *****************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *conn;
    pgoblin_realpq  *res;
    pgoblin_rdb     *rdb;
    char           **row;
    int              ex   = 0;

    GET_CON(rdb, exenv->options, ndb);
    conn = rdb->conn;
    res  = resv(rdb);
    if  ((ex = get(rdb, &row, res))) goto out;
    /* XXXX         */
    if  (conn->flags & PGOBLIN_STREAME) {
        *buf = NULL;
        ex = 0;
    } else {
        int   l;
        int   k;

        l = 2;
        for (int ncol = 0; ncol < res->ncol; ncol++) {
            if (row[ncol]) l += strlen(row[ncol]) * 2 + 1; else l += 3;
        }
        if  (!(*buf = malloc(l))) {
            ifBLIN_QW0("no RAM");
            ex = -EX_OSERR;
            goto out;
        }
        k = 0;
        for (int ncol = 0; ncol < res->ncol; ncol++) {
            ex = pgoblin_onecopy(&(*buf)[k], row[ncol], -1, (ncol < (res->ncol - 1)) ? -1 : 0);
            if  (0 > ex) {
                ifBLIN_QW0("COPY sqlite3");
                goto out;
            }
            k += ex;
            (*(char**)buf)[k] = 0;
        }
        free(row);
        ex = k;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static const char clasnames[][3] =
{ "Lx", "LX", "LX", "LX",  "LX", "LX", "LX", "LX",  "LX", "LX", "LX", "LX",  "LX", "LX", "LX", "LX"
, "Lz", "LZ", "LZ", "LZ",  "Lz", "LZ", "LZ", "LZ",  "Lz", "LZ", "LZ", "LZ",  "Lz", "LZ", "LZ", "LZ"
, "LL", "LL", "LL", "LL",  "LL", "LL", "LL", "LL",  "Lb", "Lt", "Ln", "Lv",  "Lf", "Lr", "LL", "LL"
, "L0", "L1", "L2", "L3",  "L4", "L5", "L6", "L7",  "LN", "LN", "LN", "LN",  "LN", "LN", "LN", "LN"
, "Lc", "LC", "LC", "LC",  "Lc", "LC", "LC", "LC",  "Lc", "LC", "LC", "LC",  "Lc", "LC", "LC", "LC"
, "LS", "LS", "LS", "LS",  "LS", "LS", "LS", "LS",  "LS", "LS", "LS", "LS",  "Ls", "LS", "LS", "LS"
, "Lw", "LW", "LW", "LW",  "LW", "LW", "LW", "LW",  "LW", "LW", "LW", "LW",  "LW", "LW", "LW", "LW"
, "Le", "LE", "LE", "LE",  "LE", "LE", "LE", "LE",  "LE", "LE", "LE", "LE",  "LE", "LE", "LE", "LE"
, "Lu", "LU", "LU", "LU",  "LU", "LU", "LU", "LU",  "LU", "LU", "LU", "LU",  "LU", "LU", "LU", "LU"
, "Ld", "LD", "LD", "LD",  "LD", "LD", "LD", "LD",  "LD", "LD", "LD", "LD",  "LD", "LD", "LD", "LD"
};

static const u_char class[256] =
{ Lz, Le, Le, Le,  Le, Le, Le, Le,  Le, Lc, Lw, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Ld, Le
, L0, L1, L2, L3,  L4, L5, L6, L7,  Le, Le, Le, Le,  Le, Le, Le, Le

, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Lu, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Ls, Le, Le, Le
, Le, Le, Lb, Le,  Le, Le, Lf, Le,  Le, Le, Le, Le,  Le, Le, Ln, Le
, Le, Le, Lr, Le,  Lt, Le, Lv, Le,  Le, Le, Le, Le,  Le, Le, Le, Le

, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le

, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
};

static const char pn[][3] = 
{ "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- "
, "Co", "Cb", "Cr", "Cq", "Cg", "Cp", "Cn", "Cc"
, "Cl", "Cs", "Cx", "Ck", "Cu", "- ", "- ", "- "
};

static u_int32_t automa[bx][LX >> 4] =
/* 0 Lx  1 \0 Lz     2 LL     3 LN  4 \t Lc 5  \ Ls     6 \n Lw  7  * Le  8  N Lu  9  . Ld  */
{{Cu|bg,Co|Cc|bz,Co|Cp|bc,Co|Cp|bc,Co|Cc|bb,Co|  bq,Co|Cc|Cl|bg,Co|Cp|bc,Co|Cp|bc,Co|Cp|bc}/* bg */
,{Cu|bq,      bz,   Cg|bc,Cb|   b1,   Cc|bb,     bb,   Cc|Cl|bg,   Cp|bc,   Cn|bt,      bx}/* bq */
,{Cu|bz,      bz,      bz,      bz,   Cc|bb,     bz,   Cc|Cl|bg,      bz,      bz,      bz}/* bz */
,{Ck|be,      bz,   Cg|bc,Cb|   b1,   Cc|bb,  Cp|bc,   Cc|Cl|bg,   Cp|bc,   Cp|bc,   Cp|bc}/* be */
,{Ck|b1,Cq|   bz,Cq|Cp|bc,Cr|   b2,Cq|Cc|bb,Cq|  be,Cq|Cc|Cl|bg,Cq|Cp|bc,Cq|Cp|bc,Cq|Cp|bc}/* b1 */
,{Ck|b2,Cq|   bz,Cq|Cp|bc,Cr|Cq|bc,Cq|Cc|bb,Cq|  be,Cq|Cc|Cl|bg,Cq|Cp|bc,Cq|Cp|bc,Cq|Cp|bc}/* b2 */
,{Cu|bb,Co|Cc|bz,Co|Cp|bc,Co|Cp|bc,Co|Cc|bb,Co|  bd,Co|Cc|Cl|bg,Co|Cp|bc,Co|Cp|bc,Co|Cp|bc}/* bb */
,{Cu|bb,      bz,   Cg|bc,Cb|   b1,   Cc|bb,  Cp|bc,   Cc|Cl|bg,   Cp|bc,   Cn|bn,   Cp|bc}/* bd */
,{Ck|bc,      bz,   Cp|bc,   Cp|bc,   Cc|bb,     be,   Cc|Cl|bg,   Cp|bc,   Cp|bc,   Cp|bc}/* bc */
,{Cu|bt,      bz,   Cx|bx,   Cx|bx,      bb,  Cx|bx,      Cl|bg,   Cx|bx,   Cx|bx,   Cx|bx}/* bt */
,{Cu|bt,      bz,   Cx|bx,   Cx|bx,      bb,  Cx|bx,      Cl|bg,   Cx|bx,   Cx|bx,   Cx|bx}/* bn */
};

static int
/****************************************************************************
 **                                                                        **/
putcopy(pgoblin_exenv *exenv, pgoblin_nr ndb, const void *buf, int nby) { /**
 **                                                                        **
 ****************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    /* sqlite3_bind_text() should not be used for any XXXX */
    u_int32_t        control;
    pgoblin_db_conn *conn;
    pgoblin_rdb     *rdb;
    int              ex = 0;
    u_char           cl;
    int              i;
    u_char          *c;
    u_char          *p;
    u_char          *q;
    u_char          *b;

    GET_CON(rdb, exenv->options, ndb);
    conn = rdb->conn;
    if  (!(conn->flags & (PGOBLIN_COPYIN | PGOBLIN_COPYEN))) { /* XXXX COPYEN   */
        ifBLIN_QX0("not COPYin state");
        ERROUT(-EX_DATAERR, ENODEV);
    }
    if  (!(b = malloc(nby))) {
        ifBLIN_QX0("no mem #0");
        ERROUT(-EX_OSERR, ENOMEM);
    }
    bcopy(buf, b, nby);
    c = p = q = b;
    blin_stateheader(BLIN_4STX24G, pn);
    for (; conn->state < bx; q++) {
        cl = (q < (b + nby)) ? class[*q] : Lx;
        control = automa[conn->state][cl >> 4];
        blin_statebody( BLIN_4STX24G
                      , pn
                      , statenames
                      , clasnames[cl]
                      , (char*)q
                      , b + nby - q
                      , control
                      , conn->state
                      , BLIN_I(q)
                      , p - c
                      );
        if  (control & Co) c = p = q;
        if  (control & Cb) conn->reg = class[*q] & 0x07;
        if  (control & Cr) conn->reg = (conn->reg << 3) | (class[*q] & 0x07);
        if  (control & Cq) *(p++) = conn->reg & 0xFF;
        if  (control & Cg) *(p++) = class[*q] & 0x0F;
        if  (control & Cp) {
            if  (p == q) p++; else *(p++) = *q;
        }
        if  ((control & (Cn | Cc | Cl)) && (conn->flags & PGOBLIN_COPYST)) {
            if  (!conn->vm) {
                ifBLIN_QX0("no VM");
                ERROUT(-EX_SOFTWARE, ENOEXEC);
            }
            ex = sqlite3_reset(conn->vm);
            ERRMSG("sqlite3_reset", -EX_SOFTWARE);
            conn->flags &= ~PGOBLIN_COPYST;
        }
        if  (control & Cn) {
            free(conn->semi);
            conn->semi = NULL;
            ifBLIN_QX4("sqlite3_bind_text#1 [%d]NULL", conn->ncol);
            ex = sqlite3_bind_text(conn->vm, ++(conn->ncol), NULL, 0, NULL);
            ERRMSG("sqlite3_bind_text#1", -EX_SOFTWARE);
        }
        if  (control & Cc) { /*   */
            *p = '\0';
            if  (conn->semi) {
                if  (!(p = malloc(strlen((char*)conn->semi) + strlen((char*)c) + 1))) {
                    ifBLIN_QX0("no mem #1");
                    ERROUT(-EX_OSERR, ENOMEM);
                }
                strcpy((char*)p, (char*)conn->semi);
                strcat((char*)p, (char*)c);
                ifBLIN_QX4("sqlite3_bind_text#2 [%d]%d=%s~", conn->ncol, -1, p);
                /* SQLITE_STATIC should be used in most cases instead SQLITE_TRANSIENT XXXX */
                ex = sqlite3_bind_text(conn->vm, ++(conn->ncol), (char*)p, -1, SQLITE_TRANSIENT);
                ERRMSG("sqlite3_bind_text#2", -EX_SOFTWARE);
                free(p);
                free(conn->semi);
                conn->semi = NULL;
            } else {
                ifBLIN_QX4("sqlite3_bind_text#3 [%d]%d=%s~", conn->ncol, -1, c);
                /* SQLITE_STATIC should be used in most cases instead SQLITE_TRANSIENT XXXX */
                ex = sqlite3_bind_text(conn->vm, ++(conn->ncol), (char*)c, -1, SQLITE_TRANSIENT);
                ERRMSG("sqlite3_bind_text#3", -EX_SOFTWARE);
        }   }
        if  (control & Cl) { /*   */
            conn->flags |= PGOBLIN_COPYST;
            ex = sqlite3_step(conn->vm);
            ERRMSG("sqlite3_step#1", -EX_SOFTWARE);
            conn->ncol = 0;
        }
        if  (control & Cs) {
            conn->flags |= PGOBLIN_COPYEN;
            conn->ncol = 0;
        }
        if  (control & Ck) {
            i = p - c;
            if  (!i) break;
            if  (conn->semi) {
                if  (!(p = malloc(strlen((char*)conn->semi) + i + 1))) {
                    ifBLIN_QX0("no mem #2");
                    ERROUT(-EX_OSERR, ENOMEM);
                }
                strcpy((char*)p, (char*)conn->semi);
                bcopy(c, index((char*)p, '\0'), i);
                p[strlen((char*)conn->semi) + i] = '\0';
                free(conn->semi);
                conn->semi = p;
            } else {
                if  (!(conn->semi = malloc(i + 1))) {
                    ifBLIN_QX0("no mem #3");
                    ERROUT(-EX_OSERR, ENOMEM);
                }
                bcopy(c, conn->semi, i);
                conn->semi[i] = '\0';
        }   }
        if  (control & (Ck | Cu)) break;
        conn->state = control & C_state;
    }
    free(b);
out:
    return(ex ? ex : 1);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
endstream(pgoblin_exenv *exenv, pgoblin_nr ndb) {                   /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    BLIN_flag        cond;
    pgoblin_db_conn *conn = NULL;
    pgoblin_rdb     *rdb;
    int              ex   = 0;

    GET_CON(rdb, exenv->options, ndb);
    cond = rdb->flags & PGOBLIN_CR_MASK;
    conn = rdb->conn;
    switch (cond) {
    case PGOBLIN_CR_SSEL:
    case PGOBLIN_CR_OUT:
        if  (!(conn->flags & PGOBLIN_STREAMI)) {
            ifBLIN_QX0("not stream state");
            ERROUT(-EX_DATAERR, ENODEV);
        }
        goto crin;
    case PGOBLIN_CR_IN:
        if  (!(conn->flags & PGOBLIN_COPYIN)) {
            ifBLIN_QX0("not COPYin state");
            ERROUT(-EX_DATAERR, ENODEV);
        }
    crin:
        ex = sqlite3_finalize(conn->vm);
        conn->flags &= ~PGOBLIN_ALLIN;
        ERRMSG("sqlite3_finalize", EX_SOFTWARE);
        break;
    default:
        ifBLIN_QX0("- %d", cond);
        ex = -EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
out:
    if  (!!conn) conn->flags = 0;
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
finish(pgoblin_exenv *exenv, pgoblin_nr ndb) {                      /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *conn;
    pgoblin_rdb     *rdb;
    int              ex = EX_OK;

    GET_CON(rdb, exenv->options, ndb);
    conn = rdb->conn;
    if  (!!conn) {
        if  (!!conn->odf) {
            sqlite3_close_v2(conn->odf);
            conn->odf = NULL;
        }
        free(rdb->conn);
        rdb->conn = NULL;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/*****************************************************************************************************
 **                                                                                                 **/
prexec(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nio, pgoblin_nr nin) {     /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    BLIN_flag        cond;
    pgoblin_db_conn *conn;
    int              ncol;
    int              imdx;
    pgoblin_realpq **rsv  = NULL;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rin;
    pgoblin_rio     *rou;
    pgoblin_rio     *rio;
    pgoblin_realpq  *res;
    int              col;
    int              row;
    int              ex   = 0;
    sqlite3_stmt    *vm;

    GET_CON(rdb, exenv->options, ndb);
    cond = rdb->flags & PGOBLIN_ACR_MASK;
    conn = rdb->conn;
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    if  ((PGOBLIN_CR_OK != cond) && (PGOBLIN_CR_TUPL != cond)) {
        ifBLIN_QX0("illegal cond=%d", cond);
        ERROUT(EX_UNAVAILABLE, ENOTSUP);
    }
    ifBLIN_QX4(" %d", cond);
    GET_RIO(rio, exenv->options, nio);
    vm = rio->pq->vm;
    if  (!vm) goto out;
    GET_RIO(rou, exenv->options, nou);
    rsv = &rou->pq;
    res = NULL;
    GET_RIO(rin, exenv->options, nin);
    col = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Nfields);
    row = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Ntuples);
    imdx = sqlite3_bind_parameter_count(vm);
    for (int i = 0; i < row; i++) {
        for (int j = 0; (j < col) && (j < imdx); j++) {
            ifBLIN_QX4( "sqlite3_bind_text#3 [%d]%ld=%s~"
                      , j
                      , (long)pgoblin_db_valinfo(exenv, nin, PGOBLIN_Length, i, j)
                      , pgoblin_db_getvalue(exenv, nin, i, j)
                      );
            ex = sqlite3_bind_text( vm
                                  , j + 1
                                  , pgoblin_db_getvalue(exenv, nin, i, j)
                                  , (int)pgoblin_db_valinfo(exenv, nin, PGOBLIN_Length, i, j)
                                  , SQLITE_STATIC
                                  );
            if  (SQLITE_RANGE == ex) {
                ifBLIN_QX1("No index %d", j + 1);
            } else if ((SQLITE_OK != ex) && (SQLITE_ROW != ex) && (SQLITE_DONE != ex)) {
                if  (!res && !(res = resv(rdb))) {
                    ifBLIN_QX0("no RAM");
                    ERROUT(EX_OSERR, ENOMEM);
                }
                ifBLIN_QX0("sqlite3_bind_text#3 %d: %s", ex, sqlite3_errmsg(conn->odf));
                ERROUT(EX_SOFTWARE, oxdecode(ex));
        }   }
        res = NULL;
        if  (PGOBLIN_CR_OK == cond) {
            ncol = sqlite3_column_count(vm);
            if  (ncol) {
                ifBLIN_QX1("wait for PGOBLIN_CR_OK but received %d cols", ncol);
            }
        } else if (i + 1 >= row) {
            if  (!(res = resv(rdb))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            collect(vm, res);
        }
        for (; ;) {
#ifdef SQLITE_BUZYREPEAT
            for (int g = 0; g < SQLITE_BUZYREPEAT; g++) {
                ex = sqlite3_step(vm);
                if  (SQLITE_BUSY != (ex & 0xFF)) break;
                ifBLIN_QX1("SQLITE_BUSY %d #%d", ex, g);
                usleep(SQLITE_BUZYWAIT);
                sqlite3_reset(vm);
            }
#else
            ex = sqlite3_step(vm);
#endif
            ERRMSG("sqlite3_step#3", EX_SOFTWARE);
            if  (!!res && !(res->flags & PGOBLIN_TINIT)) {
                res->nrow = 0;
                res->flags |= PGOBLIN_TINIT;
            }
            if  (ex == SQLITE_ROW) {
                if  (!!res) {
                    ex = 0;
                    if  (!(res->flags & PGOBLIN_MULAR)) {
                        res->md = mular_create( MULAR_CHAR
                                              | MULAR_ZERO
                                              | MULAR_PFRE
                                              | MULAR_STRI
                                              | MULAR_UPPE
                                              , 3
                                              , sizeof(char*)
                                              , mdp
                                              );
                        res->md->tofree = free;
                        res->flags |= PGOBLIN_MULAR;
                    }
                    for (int j = 0; j < res->ncol; j++) {
                        const u_char *t;
                        char *tt;

                        if  (sqlite3_column_type(vm, j) == SQLITE_NULL) {
                            *(char**)(mular_add(res->md)) = NULL;
                        } else if ((t = sqlite3_column_text(vm, j)) && (tt = strdup((char*)t))) {
                            *(char**)(mular_add(res->md)) = tt;
                        } else {
                            ifBLIN_QX0("no mem #7");
                            ERROUT(EX_OSERR, ENOMEM);
                    }   }
                    res->nrow++;
                }
                continue;
            }
            if  (ex == SQLITE_DONE) ex = 0;
            break;
        }
        if  (!!ex) {
            ifBLIN_QX0("sqlite3_step: %d: %s", ex, sqlite3_errmsg(conn->odf));
            if  (vm) {
                ex = sqlite3_finalize(vm);
                ERRMSG("sqlite3_finalize", EX_SOFTWARE);
            }
            ERROUT(EX_SOFTWARE, oxdecode(ex));
        }
        ex = sqlite3_reset(vm);
        ERRMSG("sqlite3_reset", EX_SOFTWARE);
        if  (!!res && !res->ncol) {
            ifBLIN_QX0("sqlite3_get_table waits for PGOBLIN_CR_TUPL but received no tuples");
            ERROUT(EX_SOFTWARE, ENODEV);
    }   }
out:
    if  (!!rsv) *rsv = res; else if (!!res) clear(exenv, nou);
    ifBLIN_QX4("- %d %"BLIN_X, ex, BLIN_I(!rsv ? NULL : *rsv));
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
denote(pgoblin_exenv *exenv, pgoblin_nr ndb, void **note) {         /**
 **                                                                  **
 **********************************************************************/
    return(EX_OK);
}

pgoblin_dbases PGOBLIN_MODULE =
{ (PGOBLIN_DB_PREPARE | PGOBLIN_DB_COPYIN | PGOBLIN_DB_COPYOUT)
, "3pGoblin-" VERS
, "sqlite3\0" VERS
, 0
, init
, fini
, shurecon
, query
, prepare
, execute
, resinfo
, subinfo
, erinfo
, valinfo
, getvalue
, getstream
, getcopy
, putcopy
, endstream
, NULL /* tarry    */
, NULL /* notifies */
, finish
, clear
, NULL /* erconn   */
, prexec
, denote
, NULL /* dummy1   */
, NULL /* dummy2   */
};
