/*-
 * Copyright (C)2008..2021 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2008..2021 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: db_sqlite2.c,v 1.105 2021/09/18 17:58:28 babolo Exp $"

#define BLIN_COMPAT      3
#define Bpars_COMPAT     3
#define MULAR_COMPAT     0
#define MIFE_COMPAT      4
#define PGOBLIN_COMPAT   2
#define PGOBLIN_INTERNAL 1
#define PGOBLIN_MODULE   pgoblin_module

#include <sys/limits.h>
#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sysexits.h>
#include <sys/time.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <err.h>
#include <sqlite.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"
#include "pgoblin3sqlite2.h"

/* SQLite 2 interface to pgoblin */

#define INDX(R, C) (R * rsv->ncol + C)

static const char statenames[][3] =
{ "bg", "bq", "bz", "be", "b1", "b2", "bb", "bd", "bc", "bt", "bn", "bx"
};

struct pgoblin_mode {
    BLIN_flag         flags;                                                        /* Must be first */
/*                    PGOBLIN_DB_TYPE                                                                */
#   define            PGOBLIN_TINIT   0x010000
#   define            PGOBLIN_MULAR   0x020000
#   define            PGOBLIN_VM      0x040000
    int               ncol;
    union {
        mular_descriptor *md;
        sqlite_vm        *vm;
    };
    char            **serv;
    int               nrow;
};

static size_t mdp[] = {1024, 512, 512};
pgoblin_dbases PGOBLIN_MODULE;

static u_char sqle[] =
{ 0            /* SQLITE_OK         Successful result                          */
, ENOTSUP      /* SQLITE_ERROR      SQL error or missing database              */
, ENOEXEC      /* SQLITE_INTERNAL   An internal logic error in SQLite          */
, EPERM        /* SQLITE_PERM       Access permission denied                   */
, ECANCELED    /* SQLITE_ABORT      Callback routine requested an abort        */
, ETXTBSY      /* SQLITE_BUSY       The database file is locked                */
, EDEADLK      /* SQLITE_LOCKED     A table in the database is locked          */
, ENOMEM       /* SQLITE_NOMEM      A malloc() failed                          */
, EROFS        /* SQLITE_READONLY   Attempt to write a readonly database       */
, EINTR        /* SQLITE_INTERRUPT  Operation terminated by sqlite_interrupt() */
, EIO          /* SQLITE_IOERR      Some kind of disk I/O error occurred       */
, EILSEQ       /* SQLITE_CORRUPT    The database disk image is malformed       */
, EFAULT       /* SQLITE_NOTFOUND   (Internal Only) Table or record not found  */
, EOVERFLOW    /* SQLITE_FULL       Insertion failed because database is full  */
, ENOATTR      /* SQLITE_CANTOPEN   Unable to open the database file           */
, ENOLCK       /* SQLITE_PROTOCOL   Database lock protocol error               */
, ENOENT       /* SQLITE_EMPTY      (Internal Only) Database table is empty    */
, EIDRM        /* SQLITE_SCHEMA     The database schema changed                */
, EMSGSIZE     /* SQLITE_TOOBIG     Too much data for one row of a table       */
, EEXIST       /* SQLITE_CONSTRAINT Abort due to contraint violation           */
, EPROCUNAVAIL /* SQLITE_MISMATCH   Data type mismatch                         */
, EINVAL       /* SQLITE_MISUSE     Library used incorrectly                   */
, ENOSYS       /* SQLITE_NOLFS      Uses OS features not supported on host     */
, EAUTH        /* SQLITE_AUTH       Authorization denied                       */
, EFTYPE       /* SQLITE_FORMAT     Auxiliary database format error            */
, EDOM         /* SQLITE_RANGE      2nd parameter to sqlite_bind out of range  */
, EBADF        /* SQLITE_NOTADB     File opened that is not a database file    */
};

static int
/**********************************************************************
 **                                                                  **/
oxdecode(int i) {                                                   /**
 **                                                                  **
 **********************************************************************/
    int e;
    if      (i == SQLITE_ROW)                e = EAGAIN;
/*  else if (i == SQLITE_DONE) */
    else if ((i < 0) || (i > SQLITE_NOTADB)) e = ENOMSG;
    else                                     e = sqle[i];
    return(e);
}

static const char *
/**********************************************************************
 **                                                                  **/
exdecode(int ex) {                                                  /**
 **                                                                  **
 **********************************************************************/
    const char *c;
    switch (ex) {
    case SQLITE_OK                     : c = "SQLITE_OK";                      break;
    case SQLITE_ROW                    : c = "SQLITE_ROW";                     break;
    case SQLITE_DONE                   : c = "SQLITE_DONE";                    break;
    default                            : c = sqlite_error_string(ex);
    }
    return(c);
}

static pgoblin_mode *
/**********************************************************************
 **                                                                  **/
resv(pgoblin_conn *db) {                                            /**
 **                                                                  **
 **********************************************************************/
    pgoblin_mode *rsv = NULL;

    if  (!(rsv = calloc(1, sizeof(pgoblin_mode)))) {
        ifBLIN_QW0("No mem");
    } else {
        rsv->flags = db->flags & (BLIN_MASK | PGOBLIN_DB_TYPE);
    }
    return(rsv);
}

const static char rrr[8] = {0x00, 0x00, 0x20, 0x80, 0x10, 0x40, 0x10, 0x40};

static int
/**********************************************************************
 **                                                                  **/
finalize(BLIN_flag flags, sqlite_vm *vm, u_char mode) {             /**
 **                                                                  **
 **********************************************************************/
    char *errmsg = NULL;
    int   ex = EX_OK;
    int   e;

#   define blin_internal_flags (flags & BLIN_MASK)
    if  (!!vm || !!mode) {
        ex = sqlite_finalize(vm, &errmsg);
        e = rrr[mode + (ex ? 2 : 0) + (errmsg ? 4 : 0)];
        if  (e & 0x80) ifBLIN_QX0("sqlite_finalize: %d %s", ex, exdecode(ex));
        if  (e & 0x40) ifBLIN_QX0("sqlite_finalize: %d %s: %s", ex, exdecode(ex), errmsg);
        if  (e & 0x20) ifBLIN_QX1("sqlite_finalize: %d %s", ex, exdecode(ex));
        if  (e & 0x10) ifBLIN_QX1("sqlite_finalize: %d %s: %s", ex, exdecode(ex), errmsg);
        if  (e & 0x50) sqlite_freemem(errmsg);
        errmsg = NULL;
        errno = oxdecode(ex);
        if  (e & 0xF0) ex = -EX_SOFTWARE;
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
reset(BLIN_flag flags, sqlite_vm *vm) {                             /**
 **                                                                  **
 **********************************************************************/
    char *errmsg = NULL;
    int   ex = EX_OK;
    int   e;

#   define blin_internal_flags (flags & BLIN_MASK)
    if  (!!(ex = sqlite_reset(vm, &errmsg)) || !!errmsg) {
        e = rrr[(ex ? 2 : 0) + (errmsg ? 4 : 0)];
        if  (e & 0x20) ifBLIN_QX0("sqlite_reset: %d %s: %s", ex, exdecode(ex));
        if  (e & 0x10) ifBLIN_QX0("sqlite_reset: %d %s: %s", ex, exdecode(ex), errmsg);
        if  (e & 0x50) sqlite_freemem(errmsg);
        errmsg = NULL;
        errno = oxdecode(ex);
        if  (e & 0xF0) ex = -EX_SOFTWARE;
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
compile(pgoblin_conn *db, char *in, char **out, BLIN_flag cond) {   /**
 **                                                                  **
 **********************************************************************/
    char            *errmsg = NULL;
    pgoblin_db_conn *conn   = db->conn;
    int              ex     = EX_OK;
    int              e;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    ifBLIN_QX4("sqlite_compile %d=%s~", cond, in);
    ex = sqlite_compile(conn->odf, in, (const char **)out, &conn->vm, &errmsg);
    if  (cond & PGOBLIN_ALTERQ) free(in);
    if  (!!ex || !!errmsg) {
        if  (!errmsg) {
            ifBLIN_QX0("sqlite_compile: %d %s", ex, exdecode(ex));
        } else {
            ifBLIN_QX0("sqlite_compile: %d %s: %s", ex, exdecode(ex), errmsg);
            sqlite_freemem(errmsg);
        }
        if  (!!(e = finalize(db->flags, conn->vm, 0))) ifBLIN_QX0("err %d", e);
        if  (!ex) ex = e;
    }
    if  (!ex && !conn->vm) {
        if  (cond & PGOBLIN_ALTERV) {
            ifBLIN_QX0("sqlite2 no VM");
            errno = ENODEV;
            ex = EX_SOFTWARE;
        } else {
            ifBLIN_QX4("sqlite2 no VM");
    }   }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
collect(int ncol, int ex, const char **serv, pgoblin_mode *res) {   /**
 **                                                                  **
 **********************************************************************/
    if  (!!res) {
#       define blin_internal_flags (res->flags & BLIN_MASK)
        if  (!(res->flags & PGOBLIN_TINIT)) {
            res->ncol = ncol;
            res->nrow = 0;
            res->serv = NULL;
            res->flags |= PGOBLIN_TINIT;
            if  ((ex == SQLITE_ROW) && !!serv && (0 < ncol)) {
                ssize_t  l = 0;
                char    *p;

                for (int y = 0; y < 2 * ncol; ++y) if (!!serv[y]) l += strlen(serv[y]) + 1;
                ++l;
                if  (!(res->serv = malloc(2 * ncol * sizeof(char *) + l))) {
                    ifBLIN_QX0("No mem %"BLIN_D, 2 * ncol * sizeof(char *) + l);
                    ERROUT(-EX_SOFTWARE, EINVAL);
                }
                p = (char *)&res->serv[2 * ncol];
                for (int y = 0; y < 2 * ncol; ++y) {
                    if  (0 >= l) {
                        ifBLIN_QX0("strings %"BLIN_X" overlow on %d of %d", l, y, 2 * ncol);
                        break;
                    }
                    if  (!!serv[y]) {
                        res->serv[y] = p;
                        p = memccpy(p, serv[y], 0, l);
                        l -= p - res->serv[y];
                    } else {
                        res->serv[y] = NULL;
        }   }   }   }
        if  (res->ncol != ncol) {
            ifBLIN_QX0("sqlite_step: Ncol %d <> %d", res->ncol, ncol);
            ERROUT(-EX_SOFTWARE, EINVAL);
    }   }
out:;
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
init(BLIN_flag flags) {                                             /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (flags & BLIN_MASK)
    PGOBLIN_MODULE.flags = (PGOBLIN_MODULE.flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE))
                             | (flags & (BLIN_MASK | PGOBLIN_DB_TYPE))
    ;
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
fini() {                                                            /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (pgoblin_db_sqlite3.flags & BLIN_MASK)
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
shurecon(pgoblin_conn *db) {                                        /**
 **                                                                  **
 **********************************************************************/
    char            *errmsg = NULL;
    pgoblin_db_conn *conn;
    int              ex     = EX_OK;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    if  (db->conn != NULL) goto out;
    ifBLIN_QX4("lib version=%s\nlib encoding=%s", sqlite_version, sqlite_encoding);
    if  (!(conn = calloc(1, sizeof(pgoblin_db_conn)))) {
        ifBLIN_QX0("no MEM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    conn->flags = db->flags & (BLIN_MASK | PGOBLIN_DB_TYPE);
    conn->odf = sqlite_open(db->dbname ? db->dbname : ":memory:", 0, &errmsg);
    if  (errmsg) {
        ifBLIN_QX0("sqlite_open: %s", errmsg);
        sqlite_freemem(errmsg);
        errmsg = NULL;
        ERROUT(EX_UNAVAILABLE, ENODEV);
    } else if (!conn->odf) {
        ifBLIN_QX0("aborted");
        ERROUT(EX_UNAVAILABLE, ENODEV);
    }
    MARK_R_CONN_GO(db);;
    ;;  db->conn = conn;
    ;;  db->flags = PGOBLIN_CLOSECONN
 /* ;; */         | (db->flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE))
 /* ;; */         | (PGOBLIN_MODULE.flags & (BLIN_MASK | PGOBLIN_DB_TYPE))
    ;;  ;
    MARK_R_CONN_WENT(db);;
    ex = sqlite_create_function( conn->odf
                               , "regexp"
                               , 2
                               , pgoblin_sqlite2_regexp
                               , (void*)(long int)db->flags
                               );
    if  (!!ex) {
        ifBLIN_QX0("regexp creation %d %s", ex, exdecode(ex));
        ERROUT(EX_DATAERR, oxdecode(ex));
    }
    ex = sqlite_create_function( conn->odf
                               , "quote_literal"
                               , 1
                               , pgoblin_sqlite2_quote
                               , (void*)(long int)((db->flags & BLIN_MASK) | SQLITE_SQ)
                               );
    if  (!!ex) {
        ifBLIN_QX0("quote_literal creation %d %s", ex, exdecode(ex));
        ERROUT(EX_DATAERR, oxdecode(ex));
    }
    ex = sqlite_create_function( conn->odf
                               , "quote_ident"
                               , 1
                               , pgoblin_sqlite2_quote
                               , (void*)(long int)
                                        ( (db->flags & BLIN_MASK)
                                        | SQLITE_DQ | SQLITE_SF | SQLITE_KY
                               )        );
    if  (!!ex) {
        ifBLIN_QX0("quote_ident creation %d %s", ex, exdecode(ex));
        ERROUT(EX_DATAERR, oxdecode(ex));
    }
    ex = sqlite_create_function( conn->odf
                               , "quote_ident_always"
                               , 1
                               , pgoblin_sqlite2_quote
                               , (void*)(long int)((db->flags & BLIN_MASK) | SQLITE_DQ)
                               );
    if  (!!ex) {
        ifBLIN_QX0("quote_ident_always creation %d %s", ex, exdecode(ex));
        ERROUT(EX_DATAERR, oxdecode(ex));
    }
    ex = sqlite_create_function( conn->odf
                               , "quote_html"
                               , 1
                               , pgoblin_sqlite2_quote_html
                               , (void*)(long int)(db->flags & BLIN_MASK)
                               );
    if  (!!ex) {
        ifBLIN_QX0("quote_html creation %d %s", ex, exdecode(ex));
        ERROUT(EX_DATAERR, oxdecode(ex));
    }
    ex = sqlite_create_function( conn->odf
                               , "gettimeofday"
                               , 0
                               , pgoblin_sqlite2_gettimeofday
                               , (void*)(long int)(db->flags & BLIN_MASK)
                               );
    if  (!!ex) {
        ifBLIN_QX0("gettimeofday creation %d %s", ex, exdecode(ex));
        ERROUT(EX_DATAERR, oxdecode(ex));
    }
    ex = sqlite_create_function( conn->odf
                               , "instr"
                               , 2
                               , pgoblin_sqlite2_instr
                               , (void*)(long int)(db->flags & BLIN_MASK)
                               );
    if  (!!ex) {
        ifBLIN_QX0("instr creation %d %s", ex, exdecode(ex));
        ERROUT(EX_DATAERR, oxdecode(ex));
    }
out:
    return(ex);
#   undef blin_internal_flags
}

/*      0x01..0x0F */
/*   ASCII    */
#define Lx         0x00
#define Lz         0x10
#define Lb ('\b' | 0x20)
#define Lf ('\f' | 0x20)
#define Ln ('\n' | 0x20)
#define Lr ('\r' | 0x20)
#define Lt ('\t' | 0x20)
#define Lv ('\v' | 0x20)
#define L0 ('0') /*0x30*/
#define L1 ('1') /*0x31*/
#define L2 ('2') /*0x32*/
#define L3 ('3') /*0x33*/
#define L4 ('4') /*0x34*/
#define L5 ('5') /*0x35*/
#define L6 ('6') /*0x36*/
#define L7 ('7') /*0x37*/
#define Lc         0x40  /* \t */
#define Ls ('\\')/*0x5C*/
#define Lw         0x60  /* \n */
#define Le         0x70  /*  * */
#define Lu         0x80  /* \N */
#define Ld         0x90  /* \. */
#define LX         0xA0

static const char clasnames[][3] =
{ "Lx", "LX", "LX", "LX",  "LX", "LX", "LX", "LX",  "LX", "LX", "LX", "LX",  "LX", "LX", "LX", "LX"
, "Lz", "LZ", "LZ", "LZ",  "Lz", "LZ", "LZ", "LZ",  "Lz", "LZ", "LZ", "LZ",  "Lz", "LZ", "LZ", "LZ"
, "LL", "LL", "LL", "LL",  "LL", "LL", "LL", "LL",  "Lb", "Lt", "Ln", "Lv",  "Lf", "Lr", "LL", "LL"
, "L0", "L1", "L2", "L3",  "L4", "L5", "L6", "L7",  "LN", "LN", "LN", "LN",  "LN", "LN", "LN", "LN"
, "Lc", "LC", "LC", "LC",  "Lc", "LC", "LC", "LC",  "Lc", "LC", "LC", "LC",  "Lc", "LC", "LC", "LC"
, "LS", "LS", "LS", "LS",  "LS", "LS", "LS", "LS",  "LS", "LS", "LS", "LS",  "Ls", "LS", "LS", "LS"
, "Lw", "LW", "LW", "LW",  "LW", "LW", "LW", "LW",  "LW", "LW", "LW", "LW",  "LW", "LW", "LW", "LW"
, "Le", "LE", "LE", "LE",  "LE", "LE", "LE", "LE",  "LE", "LE", "LE", "LE",  "LE", "LE", "LE", "LE"
, "Lu", "LU", "LU", "LU",  "LU", "LU", "LU", "LU",  "LU", "LU", "LU", "LU",  "LU", "LU", "LU", "LU"
, "Ld", "LD", "LD", "LD",  "LD", "LD", "LD", "LD",  "LD", "LD", "LD", "LD",  "LD", "LD", "LD", "LD"
};

static const u_char class[256] =
{ Lz, Le, Le, Le,  Le, Le, Le, Le,  Le, Lc, Lw, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Ld, Le
, L0, L1, L2, L3,  L4, L5, L6, L7,  Le, Le, Le, Le,  Le, Le, Le, Le

, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Lu, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Ls, Le, Le, Le
, Le, Le, Lb, Le,  Le, Le, Lf, Le,  Le, Le, Le, Le,  Le, Le, Ln, Le
, Le, Le, Lr, Le,  Lt, Le, Lv, Le,  Le, Le, Le, Le,  Le, Le, Le, Le

, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le

, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
, Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le,  Le, Le, Le, Le
};

static const char pn[][3] = 
{ "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- "
, "Co", "Cb", "Cr", "Cq", "Cg", "Cp", "Cn", "Cc"
, "Cl", "Cs", "Cx", "Ck", "Cu", "- ", "- ", "- "
};

#define Co 0x00800000 /*              */
#define Cb 0x00400000 /*             */
#define Cr 0x00200000 /*           */
#define Cq 0x00100000 /*                 */
#define Cg 0x00080000 /*   */
#define Cp 0x00040000 /*          */
#define Cn 0x00020000 /* NULL        */
#define Cc 0x00010000 /*             */
#define Cl 0x00008000 /*              */
#define Cs 0x00004000 /* COPY                 */          
#define Cx 0x00002000 /*                     */
#define Ck 0x00001000 /*             */
#define Cu 0x00000800 /*        */
#define C_state  0xFF /*         */

static u_int32_t automa[bx][LX >> 4] =
/* 0 Lx  1 \0 Lz     2 LL     3 LN  4 \t Lc 5  \ Ls     6 \n Lw  7  * Le  8  N Lu  9  . Ld  */
{{Cu|bg,Co|Cc|bz,Co|Cp|bc,Co|Cp|bc,Co|Cc|bb,Co|  bq,Co|Cc|Cl|bg,Co|Cp|bc,Co|Cp|bc,Co|Cp|bc}/* bg */
,{Cu|bq,      bz,   Cg|bc,Cb|   b1,   Cc|bb,     bb,   Cc|Cl|bg,   Cp|bc,   Cn|bt,      bx}/* bq */
,{Cu|bz,      bz,      bz,      bz,   Cc|bb,     bz,   Cc|Cl|bg,      bz,      bz,      bz}/* bz */
,{Ck|be,      bz,   Cg|bc,Cb|   b1,   Cc|bb,  Cp|bc,   Cc|Cl|bg,   Cp|bc,   Cp|bc,   Cp|bc}/* be */
,{Ck|b1,Cq|   bz,Cq|Cp|bc,Cr|   b2,Cq|Cc|bb,Cq|  be,Cq|Cc|Cl|bg,Cq|Cp|bc,Cq|Cp|bc,Cq|Cp|bc}/* b1 */
,{Ck|b2,Cq|   bz,Cq|Cp|bc,Cr|Cq|bc,Cq|Cc|bb,Cq|  be,Cq|Cc|Cl|bg,Cq|Cp|bc,Cq|Cp|bc,Cq|Cp|bc}/* b2 */
,{Cu|bb,Co|Cc|bz,Co|Cp|bc,Co|Cp|bc,Co|Cc|bb,Co|  bd,Co|Cc|Cl|bg,Co|Cp|bc,Co|Cp|bc,Co|Cp|bc}/* bb */
,{Cu|bb,      bz,   Cg|bc,Cb|   b1,   Cc|bb,  Cp|bc,   Cc|Cl|bg,   Cp|bc,   Cn|bn,   Cp|bc}/* bd */
,{Ck|bc,      bz,   Cp|bc,   Cp|bc,   Cc|bb,     be,   Cc|Cl|bg,   Cp|bc,   Cp|bc,   Cp|bc}/* bc */
,{Cu|bt,      bz,   Cx|bx,   Cx|bx,      bb,  Cx|bx,      Cl|bg,   Cx|bx,   Cx|bx,   Cx|bx}/* bt */
,{Cu|bt,      bz,   Cx|bx,   Cx|bx,      bb,  Cx|bx,      Cl|bg,   Cx|bx,   Cx|bx,   Cx|bx}/* bn */
};

static int
/**********************************************************************
 **                                                                  **/
putcopy(pgoblin_conn *db, const void *buf, int nby) {               /**
 **                                                                  **
 **********************************************************************/
    u_int32_t        control;
    pgoblin_db_conn *conn = db->conn;
    u_char           cl;
    int              ex   = 0;
    int              i;
    u_char          *c;
    u_char          *p;
    u_char          *q;
    u_char          *b;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    if  (!(conn->flags & (PGOBLIN_COPYIN | PGOBLIN_COPYEN))) { /* XXXX COPYEN   */
        ifBLIN_QX0("not COPYin state");
        ERROUT(-EX_DATAERR, ENODEV);
    }
    if  (!(b = malloc(nby))) {
        ifBLIN_QX0("no mem #0");
        ERROUT(-EX_OSERR, ENOMEM);
    }
    bcopy(buf, b, nby);
    c = p = q = b;
    blin_stateheader(BLIN_4STX24G, pn);
    for (; conn->state < bx; q++) {
        cl = (q < (b + nby)) ? class[*q] : Lx;
        control = automa[conn->state][cl >> 4];
        blin_statebody( BLIN_4STX24G
                      , pn
                      , statenames
                      , clasnames[class[*q]]
                      , (char*)q
                      , b + nby - q
                      , control
                      , conn->state
                      , BLIN_I(q)
                      , p - c
                      );
        if  (control & Co) c = p = q;
        if  (control & Cb) conn->reg = class[*q] & 0x07;
        if  (control & Cr) conn->reg = (conn->reg << 3) | (class[*q] & 0x07);
        if  (control & Cq) *(p++) = conn->reg & 0xFF;
        if  (control & Cg) *(p++) = class[*q] & 0x0F;
        if  (control & Cp) {
            if  (p == q) p++; else *(p++) = *q;
        }
        if  ((control & (Cn | Cc | Cl)) && (conn->flags & PGOBLIN_COPYST)) {
            if  (!conn->vm) {
                ifBLIN_QX0("no VM");
                ERROUT(-EX_SOFTWARE, ENOEXEC);
            }
            if  (!!(ex = reset(db->flags, conn->vm))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            conn->flags &= ~PGOBLIN_COPYST;
        }
        if  (control & Cn) {
            free(conn->semi);
            conn->semi = NULL;
            ifBLIN_QX4("sqlite_bind#1 [%d]NULL", conn->ncol);
            if  (!!(ex = sqlite_bind(conn->vm, ++(conn->ncol), NULL, 0, 0))) {
                ifBLIN_QX0("sqlite_bind#1 %d %s", ex, exdecode(ex));
                ERROUT(-EX_SOFTWARE, oxdecode(ex));
        }   }
        if  (control & Cc) { /*   */
            *p = '\0';
            if  (conn->semi) {
                if  (!(p = malloc(strlen((char*)conn->semi) + strlen((char*)c) + 1))) {
                    ifBLIN_QX0("no mem #1");
                    ERROUT(-EX_OSERR, ENOMEM);
                }
                strcpy((char*)p, (char*)conn->semi);
                strcat((char*)p, (char*)c);
                ifBLIN_QX4("sqlite_bind#2 [%d]%d=%s~", conn->ncol, -1, p);
                ex = sqlite_bind(conn->vm, ++(conn->ncol), (char*)p, -1, 1);
                free(p);
                free(conn->semi);
                conn->semi = NULL;
                if  (!!ex) {
                    ifBLIN_QX0("sqlite_bind#2 %d %s", ex, exdecode(ex));
                    ERROUT(-EX_SOFTWARE, oxdecode(ex));
                }
            } else {
                ifBLIN_QX4("sqlite_bind#3 [%d]%d=%s~", conn->ncol, -1, c);
                if  (!!(ex = sqlite_bind(conn->vm, ++(conn->ncol), (char*)c, -1, 1))) {
                    ifBLIN_QX0("sqlite_bind#3 %d %s", ex, exdecode(ex));
                    ERROUT(-EX_SOFTWARE, oxdecode(ex));
        }   }   }
        if  (control & Cl) { /*   */
            conn->flags |= PGOBLIN_COPYST;
            ex = sqlite_step(conn->vm, NULL, NULL, NULL);
            ifBLIN_QX4("sqlite_step#1 %d %s", ex, exdecode(ex));
            conn->ncol = 0;
        }
        if  (control & Cs) {
            conn->flags |= PGOBLIN_COPYEN;
            conn->ncol = 0;
        }
        if  (control & Ck) {
            i = p - c;
            if  (!i) break;
            if  (conn->semi) {
                if  (!(p = malloc(strlen((char*)conn->semi) + i + 1))) {
                    ifBLIN_QX0("no mem #2");
                    ERROUT(-EX_OSERR, ENOMEM);
                }
                strcpy((char*)p, (char*)conn->semi);
                bcopy(c, index((char*)p, '\0'), i);
                p[strlen((char*)conn->semi) + i] = '\0';
                free(conn->semi);
                conn->semi = p;
            } else {
                if  (!(conn->semi = malloc(i + 1))) {
                    ifBLIN_QX0("no mem #3");
                    ERROUT(-EX_OSERR, ENOMEM);
                }
                bcopy(c, conn->semi, i);
                conn->semi[i] = '\0';
        }   }
        if  (control & (Ck | Cu)) break;
        conn->state = control & C_state;
    }
    free(b);
out:
    return(ex ? ex : 1);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
clear(pgoblin_mode **rsv) {                                         /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags ((*rsv)->flags & BLIN_MASK)
    if  (!!rsv && *rsv) {
        if  (((*rsv)->flags & PGOBLIN_MULAR) && !!(*rsv)->md) mular_destroy((*rsv)->md);
        if  (((*rsv)->flags & PGOBLIN_VM) && !!(ex = finalize((*rsv)->flags, (*rsv)->vm, 0))) {
            ifBLIN_QX0("err %d", ex);
        }
        if  (!!(*rsv)->serv) free((*rsv)->serv);
        free(*rsv);
        *rsv = NULL;
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
finish(u_int32_t flags, int kind, void **entity) {                  /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (flags & BLIN_MASK)
    switch (kind) {
    case PGOBLIN_Connect: {
            pgoblin_db_conn **conn = (pgoblin_db_conn **)entity;

            if  (conn && *conn) {
                if  (!!(*conn)->odf) {
                    sqlite_close((*conn)->odf);
                    (*conn)->odf = NULL;
                }
                free(*conn);
                *conn = NULL;
        }   }
        break;
    case PGOBLIN_FrCOPY:
        if  (entity && *entity) {
            free(*entity);
            *entity = NULL;
        }
        break;
    case PGOBLIN_FrNote:
        break;
    default:
        ex = EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
query(pgoblin_conn *db, pgoblin_mode **rsv, char *qry) {            /**
 **                                                                  **
 **********************************************************************/
    BLIN_flag        cond = db->flags & PGOBLIN_CR_MASK;
    pgoblin_db_conn *conn = db->conn;
    int              ncol;
    int              idx;
    pgoblin_mode    *res;
    const char     **row;
    int              ex   = 0;
    const char      *q;
    char            *p;
    char            *c;
    int              i;
    static const babolo_lexor copy = {(u_char*)
    "\000..." "...."    "  | "    "  .."    "...."    "...."    "...."    "...."
    " .."    "..."    "...."    ".-.."    "IIII"    "IIII"    "II.."    "...."
    ".EEC"    "EEEE"    "EEEE"    "EEEO"    "PEEE"    "EEEE"    "EYE."    "...E"
    ".EEC"    "EEEE"    "EEEE"    "EEEO"    "PEEE"    "EEEE"    "EYE."    "...."
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    , NULL, Bpars_AABS | Bpars_CEND, 0x1C, 0x1C, 1
    ,{/*00*/ 3, ' ', 0x00, '|', 0x00, '-', 0x05, 'C', 0x09, 0xFF
     ,/*05*/ 0, '-', 0x07, 0xFF
     ,/*07*/ 0, '|', 0x00, 0x07
     ,/*09*/ 0, 'O', 0x0B, 0xFF
     ,/*0B*/ 0, 'P', 0x0D, 0xFF
     ,/*0D*/ 0, 'Y', 0x0F, 0xFF
     ,/*0F*/ 8, '', 0xFE, ' ', 0x0F, '|', 0x0F, '-', 0x19, 'C', 0xFD
              , 'O', 0xFD, 'P', 0xFD, 'Y', 0xFD, 'E', 0xFD, 0xFF
     ,/*19*/ 0, '-', 0x1A, 0xFF
     ,/*1A*/ 0, '|', 0x0F, 0x1A
    }};
    /*****************
        * 0 ...
        *   ' --\n'...
        * 1 'COPY "'
        * 1 'COPY --\n "'
        * 1 "COPY '"
        * 1 'COPY\n"'
        * 2 'COPY x'
        * 2 'COPY --\n x'
    *****************/

#   define blin_internal_flags (db->flags & BLIN_MASK)
    ifBLIN_QX4("+ %d=%s~", cond, qry);
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    res = NULL;
    if  (qry && *qry) conn->query = qry; else goto out;
    q = conn->query;
    switch (cond) {
    case PGOBLIN_CR_IN:
        conn->coln = 0;
        conn->state = bg;
        conn->semi = NULL;
        switch (babolo_goword(&copy, &q)) {
        case 0:
            if  (!!(ex = compile(db, conn->query, &conn->query, cond))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            conn->flags |= PGOBLIN_COPYIN;
            break;
        case 1:
            if  (!*q) ERROUT(EX_DATAERR, EINVAL);
            q--;
            if  (*q == '\'') {
                for (idx = 1; ;) {
                    if (q[idx++] == '\'' && q[idx] != '\'') break;
                    if (!q[idx]) ERROUT(EX_DATAERR, EINVAL);
                }
            } else if (*q == '\"') {
                for (idx = 1; ;) {
                    if (q[idx++] == '\"' && q[idx] != '\"') break;
                    if (!q[idx]) ERROUT(EX_DATAERR, EINVAL);
                }
            } else {
                ERROUT(EX_DATAERR, EINVAL);
            }
            idx++;
            ifBLIN_QX4("goword quoted %d", idx);
            goto p2;
        case 2:
            q--;
            for (idx = 1; ;) if ((copy.retable[q[idx++] & 0xFF] & 0xC0) != 0x40) break;
            ifBLIN_QX4("goword direct %d", idx);
        p2: if  (!(p = strndup(q, --idx))) {
                ifBLIN_QX0("no mem #0");
                ERROUT(EX_OSERR, ENOMEM);
            }
            conn->query = p;
            if  (asprintf(&c, "SELECT * FROM %s", conn->query) < 0) {
                ifBLIN_QX0("no mem #1");
                ERROUT(EX_OSERR, ENOMEM);
            }
            if  (!!(ex = compile(db, c, NULL, cond | PGOBLIN_ALTERQ | PGOBLIN_ALTERV))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            ex = sqlite_step(conn->vm, &ncol, NULL, NULL);
            ifBLIN_QX4("sqlite_step %d %s", ex, exdecode(ex));
            if  (ex == SQLITE_ROW || ex == SQLITE_DONE) ex = 0;
            if  (!!ex) {
                ifBLIN_QX0("sqlite_step: %d %s", ex, exdecode(ex));
                if  (!!finalize(db->flags, conn->vm, 0)) {
                    ifBLIN_QX0("");
                }
                goto out;
            }
            if  (!!(ex = finalize(db->flags, conn->vm, 1))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            conn->coln = ncol;
            if  (!(p = malloc(2 * ncol * sizeof(char)))) {
                ifBLIN_QX0("no mem #2");
                ERROUT(EX_OSERR, ENOMEM);
            }
            for (i = 0; i < ncol; i++) {
                p[2 * i] = '?';
                p[2 * i + 1] = ',';
            }
            p[2 * ncol - 1] = '\0';
            if  (asprintf(&c, "INSERT INTO %s VALUES(%s)", conn->query, p) < 0) {
                ifBLIN_QX0("no mem #3");
                ERROUT(EX_OSERR, ENOMEM);
            }
            free(p);
            if  (!!(ex = compile(db, c, NULL, cond | PGOBLIN_ALTERQ | PGOBLIN_ALTERV))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
        }   }
        conn->ncol = 0;
        conn->flags |= PGOBLIN_COPYIN;
        break;
    case PGOBLIN_CR_OUT:
        switch (babolo_goword(&copy, &q)) {
        case 1:
            if  (!*q) {
                ERROUT(EX_DATAERR, EINVAL);
            }
            q--;
            if  (*q == '\'') {
                for (idx = 1; ;) {
                    if  (q[idx++] == '\'' && q[idx] != '\'') break;
                    if  (!q[idx]) {
                        ERROUT(EX_DATAERR, EINVAL);
                }   }
            } else if (*q == '\"') {
                for (idx = 1; ;) {
                    if  (q[idx++] == '\"' && q[idx] != '\"') break;
                    if  (!q[idx]) {
                        ERROUT(EX_DATAERR, EINVAL);
                }   }
            } else {
                ERROUT(EX_DATAERR, EINVAL);
            }
            idx++;
            ifBLIN_QX4("goword quoted %d", idx);
            goto p3;
        case 2:
            q--;
            for (idx = 1; ;) if ((copy.retable[q[idx++] & 0xFF] & 0xC0) != 0x40) break;
            ifBLIN_QX4("goword direct %d", idx);
        p3: if  (!(p = strndup(q, --idx))) {
                ifBLIN_QX0("no mem #5");
                ERROUT(EX_OSERR, ENOMEM);
            }
            conn->query = p;
            if  (asprintf(&c, "SELECT * FROM %s", conn->query) < 0) {
                ifBLIN_QX0("no mem #6");
                ERROUT(EX_OSERR, ENOMEM);
            }
            if  (!!(ex = compile(db, c, NULL, cond | PGOBLIN_ALTERQ | PGOBLIN_ALTERV))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            conn->flags |= PGOBLIN_STREAMI;
            goto out;
        case 0:
            ;
        }   /* FALLTHRU */
    case PGOBLIN_CR_SSEL:
        if  (!!(ex = compile(db, conn->query, &conn->query, cond | PGOBLIN_ALTERV))) {
            ifBLIN_QX0("err %d", ex);
            goto out;
        }
        conn->flags |= PGOBLIN_STREAMI;
        break;
    case PGOBLIN_CR_OK:
        while (!ex && !!conn->query && !!*conn->query) {
            if  (!!(ex = compile(db, conn->query, &conn->query, cond))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            if  (!conn->vm) break;
            for (; ;) {
                ex = sqlite_step(conn->vm, &ncol, NULL, NULL);
                ifBLIN_QX4("sqlite_step %d %s", ex, exdecode(ex));
                if  (!!ncol) {
                    ifBLIN_QX1("wait for PGOBLIN_CR_OK but received %d cols", ncol);
                }
                if  (ex == SQLITE_ROW) continue;
                if  (ex == SQLITE_DONE) ex = 0;
                break;
            }
            if  (!!ex) {
                ifBLIN_QX0("sqlite_step: %d %s", ex, exdecode(ex));
                if  (!!finalize(db->flags, conn->vm, 0)) ifBLIN_QX0("");
                goto out;
            }
            if  (!!(ex = finalize(db->flags, conn->vm, 1))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
        }   }
        break;
    case PGOBLIN_CR_TUPL:
        while (!ex && !!conn->query && !!*conn->query) {
            const char **serv;

            if  (!!(ex = compile(db, conn->query, &conn->query, cond))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            if  (!conn->vm) break;
            if  (!!res) clear(&res);
            if  (!(res = resv(db))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            for (; ;) {
                ex = sqlite_step(conn->vm, &ncol, &row, &serv);
                ifBLIN_QX4("sqlite_step %d %s", ex, exdecode(ex));
                if  (0 > (ex = collect(ncol, ex, serv, res))) {
                    ifBLIN_QX0("collect");
                    goto out;
                }
                if  (ex == SQLITE_ROW) {
                    ex = 0;
                    if  (!(res->flags & PGOBLIN_MULAR)) {
                        res->md = mular_create( MULAR_CHAR | MULAR_ZERO
                                              | MULAR_PFRE | MULAR_STRI | MULAR_UPPE
                                              , 3, sizeof(char*), mdp
                                              );
                        res->md->tofree = free;
                        res->flags |= PGOBLIN_MULAR;
                    }
                    while (!!row && ncol--) {
                        char *t;

                        t = NULL;
                        if  (!!*row && !(t = strdup(*row))) {
                            ifBLIN_QX0("no mem #7");
                            ERROUT(EX_OSERR, ENOMEM);
                        }
                        *(char**)(mular_add(res->md)) = t;
                        row++;
                    }
                    res->nrow++;
                    continue;
                }
                if  (ex == SQLITE_DONE) ex = 0;
                break;
            }
            if  (!!ex) {
                ifBLIN_QX0("sqlite_step: %d %s", ex, exdecode(ex));
                if  (!!finalize(db->flags, conn->vm, 0)) ifBLIN_QX0("finalize in err");
                goto out;
            }
            if  (!!(ex = finalize(db->flags, conn->vm, 1))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
        }   }
        if  (!res->ncol) {
            ifBLIN_QX0("sqlite_get_table waits for PGOBLIN_CR_TUPL but received no tuples");
            ERROUT(EX_SOFTWARE, ENODEV);
        }
        break;
    case PGOBLIN_CR_BSEL:
    case PGOBLIN_CR_BTUPL:
    default:
        ifBLIN_QX0("illegal cond=%d", cond);
        ex = EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
out:
    if  (!!rsv) *rsv = res; else if (!!res) clear(&res);
    ifBLIN_QX4("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
prepare(pgoblin_conn *db, pgoblin_mode **rsv, char *qry) {          /**
 **                                                                  **
 **********************************************************************/
    pgoblin_db_conn *conn = db->conn;
    int              ex   = 0;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    ifBLIN_QX4("+ =%s~", qry);
    if  (!(*rsv = resv(db))) {
        ifBLIN_QX0("no RAM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    if  (!qry || !*qry) goto out;
    if  (!!(ex = compile(db, qry, NULL, 0))) {
        ifBLIN_QX0("err %d", ex);
        goto out;
    }
    (*rsv)->vm = conn->vm;
out:
    ifBLIN_QX4("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/*************************************************************************************
 **                                                                                 **/
execute(pgoblin_conn *db, pgoblin_mode **rsv, const void *qp, pgoblin_mode *qin) { /**
 **                                                                                 **
 *************************************************************************************/
    const char     **tuple;
    BLIN_flag        cond = db->flags & PGOBLIN_ACR_MASK;
    pgoblin_db_conn *conn = db->conn;
    int              ncol;
    pgoblin_mode    *res;
    char            *qry;
    int              col;
    int              row;
    int              ex   = 0;
    sqlite_vm       *vm;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    if  (  (PGOBLIN_CR_OK != (cond & ~PGOBLIN_ALTERQ))
        && (PGOBLIN_CR_TUPL != (cond & ~PGOBLIN_ALTERQ))
        ) {
        ifBLIN_QX0("illegal cond=%d", cond);
        ERROUT(EX_UNAVAILABLE, ENOTSUP);
    }
    if  (cond & PGOBLIN_ALTERQ) {
        ifBLIN_QX4(" %d", cond);
        vm = ((pgoblin_mode *)qp)->vm;
    } else {
        qry = (char *)qp;
        if  (!qry || !*qry) goto out;
        if  (!!(ex = compile(db, qry, NULL, cond))) {
            ifBLIN_QX0("err %d", ex);
            goto out;
        }
        vm = conn->vm;
    }
    if  (!vm) goto out;
    res = NULL;
    col = pgoblin_db_resinfo(PGOBLIN_Nfields, qin);
    row = pgoblin_db_resinfo(PGOBLIN_Ntuples, qin);
    for (int i = 0; i < row; i++) {
        for (int j = 0; j < col; j++) {
            ifBLIN_QX4( "sqlite_bind#4 [%d]%d=%s~"
                      , j
                      , pgoblin_db_valinfo(PGOBLIN_Length, qin, i, j)
                      , pgoblin_db_getvalue(qin, i, j)
                      );
            ex = sqlite_bind( vm
                            , j + 1
                            , pgoblin_db_getvalue(qin, i, j)
                            , pgoblin_db_valinfo(PGOBLIN_Length, qin, i, j)
                            + (pgoblin_db_getvalue(qin, i, j) ? 1 : 0)
                            , 1
                            );
            if  (SQLITE_RANGE == ex) {
                ifBLIN_QX1("No index %d", j + 1);
            } else if (!!ex) {
                ifBLIN_QX0("sqlite_bind#4 %d %s", ex, exdecode(ex));
                ERROUT(-EX_SOFTWARE, oxdecode(ex));
        }   }
        res = NULL;
        if  ((PGOBLIN_CR_TUPL == (cond & ~PGOBLIN_ALTERQ)) && (i + 1 >= row) && !(res = resv(db))) {
            ifBLIN_QX0("no RAM");
            ERROUT(EX_OSERR, ENOMEM);
        }
        for (; ;) {
            const char **serv;

            if  (!res) {
                ex = sqlite_step(vm, &ncol, NULL, NULL);
            } else {
                ex = sqlite_step(vm, &ncol, &tuple, &serv);
                if  (0 > (ex = collect(ncol, ex, serv, res))) {
                    ifBLIN_QX0("collect");
                    goto out;
            }   }
            ifBLIN_QX4("sqlite_step %d %s", ex, exdecode(ex));
            if  ((PGOBLIN_CR_OK == (cond & ~PGOBLIN_ALTERQ)) & !!ncol) {
                ifBLIN_QX1("wait for PGOBLIN_CR_OK but received %d cols", ncol);
            }
            if  (ex == SQLITE_ROW) {
                if  (!!res) {
                    ex = 0;
                    if  (!(res->flags & PGOBLIN_MULAR)) {
                        res->md = mular_create( MULAR_CHAR
                                              | MULAR_ZERO
                                              | MULAR_PFRE
                                              | MULAR_STRI
                                              | MULAR_UPPE
                                              , 3
                                              , sizeof(char*)
                                              , mdp
                                              );
                        res->md->tofree = free;
                        res->flags |= PGOBLIN_MULAR;
                    }
                    while (tuple && ncol--) {
                        char *t;

                        t = NULL;
                        if  (*tuple && !(t = strdup(*tuple))) {
                            ifBLIN_QX0("no mem #7");
                            ERROUT(EX_OSERR, ENOMEM);
                        }
                        *(char**)(mular_add(res->md)) = t;
                        tuple++;
                    }
                    res->nrow++;
                }
                continue;
            }
            if  (ex == SQLITE_DONE) ex = 0;
            break;
        }
        if  (ex) {
            ifBLIN_QX0("sqlite_step: %d %s", ex, exdecode(ex));
            if  (!!finalize(db->flags, vm, 0)) ifBLIN_QX0("");
            goto out;
        }
        if  (cond & PGOBLIN_ALTERQ) {
            if  (!!(ex = reset(db->flags, vm))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
        } else {
            if  (!!(ex = finalize(db->flags, vm, 1))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
        }   }
        if  (!!res && !res->ncol) {
            ifBLIN_QX0("sqlite_get_table waits for PGOBLIN_CR_TUPL but received no tuples");
            ERROUT(EX_SOFTWARE, ENODEV);
    }   }
out:
    if  (!!rsv) *rsv = res; else if (!!res) clear(&res);
    ifBLIN_QX4("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
get(pgoblin_conn *db, const char ***row, pgoblin_mode *rsv) {       /**
 **                                                                  **
 **********************************************************************/
    pgoblin_db_conn *conn = db->conn;
    const char     **serv;
    int              ncol;
    int              ex   = 0;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    if  (!(conn->flags & PGOBLIN_STREAMI)) {
        ifBLIN_QX0("not COPYout state");
        ERROUT(-EX_DATAERR, ENODEV);
    }
    for (ncol = 0; !ncol;) {
        if  (!conn->vm) {
            conn->flags |= PGOBLIN_STREAME;
            break;
        }
        while (!ncol) {
            ex = sqlite_step(conn->vm, &ncol, row, &serv);
            ifBLIN_QX4("sqlite_step=%d %d %s", ncol, ex, exdecode(ex));
            if  (0 > (ex = collect(ncol, ex, serv, rsv))) {
                ifBLIN_QX0("collect");
                ex = -ex;
                goto out;
            }
            if  (!!ncol) conn->flags |= PGOBLIN_STREAMT;
            if  (ex == SQLITE_ROW) {
                ex = 0;
                continue;
            }
            if  (ex == SQLITE_DONE) {
                if  (conn->flags & PGOBLIN_STREAMT) conn->flags |= PGOBLIN_STREAME;
                ex = 0;
            }
            break;
        }
        if  (conn->flags & PGOBLIN_STREAME) break;
        if  (ex) {
            ifBLIN_QX0("sqlite_step: %d %s", ex, exdecode(ex));
            if  (!!finalize(db->flags, conn->vm, 0)) ifBLIN_QX0("");
            goto out;
        }
        if  (!ncol && !!conn->query && !!*(conn->query)) {
            if  (!!(ex = finalize(db->flags, conn->vm, 1))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
            }
            if  (!!(ex = compile(db, conn->query, &conn->query, 0))) {
                ifBLIN_QX0("err %d", ex);
                goto out;
    }   }   }
    if  (!row) conn->flags |= PGOBLIN_STREAME;
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
getstream(pgoblin_conn *db, pgoblin_mode **rsv, int quant) {        /**
 **                                                                  **
 **********************************************************************/
    pgoblin_db_conn *conn = db->conn;
    const char     **row;
    int              ex   = EX_OK;
    int              i;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    if  (!rsv) {
        ifBLIN_QX0("no rsv");
        ERROUT(EX_OSERR, ENOMEM);
    }
    if  (!!*rsv) clear(rsv);
    if  (!(*rsv = resv(db))) {
        ifBLIN_QX0("no RAM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    if  (!!(ex = get(db, &row, *rsv))) goto out;
    (*rsv)->nrow = 0;
    if  (conn->flags & PGOBLIN_STREAME) {
        ex = 0;
    } else if (!ex) {
        (*rsv)->md = mular_create( MULAR_CHAR | MULAR_ZERO | MULAR_PFRE | MULAR_STRI | MULAR_UPPE
                                 , 3
                                 , sizeof(u_char*)
                                 , mdp
                                 );
        (*rsv)->md->tofree = free;
        (*rsv)->flags |= PGOBLIN_MULAR;
        for (i = 0; i < (*rsv)->ncol; ++i) {
            if (!row[i]) {
               *(u_char**)(mular_add((*rsv)->md)) = NULL;
            } else if (!(*(u_char**)(mular_add((*rsv)->md)) = (u_char*)strdup(row[i]))) {
                ifBLIN_QX0("no mem");
                ERROUT(EX_OSERR, ENOMEM);
        }   }
        (*rsv)->nrow = 1;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
getcopy(pgoblin_conn *db, void **buf) {                             /**
 **                                                                  **
 **********************************************************************/
    pgoblin_db_conn *conn = db->conn;
    pgoblin_mode    *res  = resv(db);
    char           **row;
    int              ex   = 0;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    if  ((ex = get(db, (const char ***)&row, res))) goto out;
    /* XXXX         */
    if  (conn->flags & PGOBLIN_STREAME) {
        *buf = NULL;
        ex = 0;
    } else {
        int   l;
        int   k;

        l = 2;
        for (int ncol = 0; ncol < res->ncol; ncol++) {
            if (row[ncol]) l += strlen(row[ncol]) * 2 + 1; else l += 3;
        }
        if  (!(*buf = malloc(l))) {
            ifBLIN_QW0("no RAM");
            ex = -EX_OSERR;
            goto out;
        }
        k = 0;
        for (int ncol = 0; ncol < res->ncol; ncol++) {
            ex = pgoblin_onecopy(&(*buf)[k], row[ncol], -1, (ncol < (res->ncol - 1)) ? -1 : 0);
            if  (0 > ex) {
                ifBLIN_QW0("COPY sqlite2");
                goto out;
            }
            k += ex;
            (*(char**)buf)[k] = 0;
        }
        ex = k;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
endstream(pgoblin_conn *db) {                                       /**
 **                                                                  **
 **********************************************************************/
    BLIN_flag        cond = db->flags & PGOBLIN_CR_MASK;
    pgoblin_db_conn *conn = db->conn;
    int              ex   = 0;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    switch (cond) {
    case PGOBLIN_CR_SSEL:
    case PGOBLIN_CR_OUT:
        if  (!(conn->flags & PGOBLIN_STREAMI)) {
            ifBLIN_QX0("not stream state");
            ERROUT(-EX_DATAERR, ENODEV);
        }
        conn->flags &= ~PGOBLIN_ALLOUT;
        if  (!!(ex = finalize(db->flags, conn->vm, 1))) {
            ifBLIN_QX0("err %d", ex);
            goto out;
        }
        break;
    case PGOBLIN_CR_IN:
        if  (!(conn->flags & PGOBLIN_COPYIN)) {
            ifBLIN_QX0("not COPYin state");
            ERROUT(-EX_DATAERR, ENODEV);
        }
        conn->flags &= ~PGOBLIN_ALLIN;
        if  (!!(ex = finalize(db->flags, conn->vm, 1))) {
            ifBLIN_QX0("err %d", ex);
            goto out;
        }
        break;
    default:
        ifBLIN_QX0("- %d", cond);
        ex = -EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
out:
    conn->flags = 0;
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
resinfo(int kind, pgoblin_mode *rsv) {                              /**
 **                                                                  **
 **********************************************************************/
    int ex;

    switch (kind) {
    case PGOBLIN_Ntuples: ex = rsv ? rsv->nrow : 0; break;
    case PGOBLIN_Nfields: ex = rsv ? rsv->ncol : 0; break;
    default             :
        ex = -EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
    return(ex);
}

static const char *
/**********************************************************************
 **                                                                  **/
subinfo(int kind, pgoblin_mode *rsv, int sub) {                     /**
 **                                                                  **
 **********************************************************************/
    const char *cx = NULL;

    switch (kind) {
    case PGOBLIN_ColName:
        if  (!!rsv->serv && (rsv->ncol > sub)) cx = rsv->serv[sub];
        break;
    case PGOBLIN_TypeNm:
        if  (!!rsv->serv && (rsv->ncol > sub)) cx = rsv->serv[sub + rsv->ncol];
        break;
    default:;
    }
    return(cx);
}

static char *
/**********************************************************************
 **                                                                  **/
erinfo(pgoblin_mode *rsv) {                                         /**
 **                                                                  **
 **********************************************************************/
    char *cx = NULL;

    return(cx);
}

static int
/**********************************************************************
 **                                                                  **/
valinfo(int kind, pgoblin_mode *rsv, int row, int col) {            /**
 **                                                                  **
 **********************************************************************/
    int ex;

#   define blin_internal_flags (rsv->flags & BLIN_MASK)
    ifBLIN_QX2("+ %08X %d x %d (%d)", rsv->flags, row, col, kind);
    switch (kind) {
    case PGOBLIN_IsNull:
        if  (rsv) {
            if  (rsv->flags & PGOBLIN_VM) {
                ifBLIN_QX1("VM instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            } else if (!(rsv->flags & PGOBLIN_MULAR)) {
                ifBLIN_QX0("NOTHING instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            } else if ((row >= 0) && (row < rsv->nrow) && (col >= 0) && (col < rsv->ncol)) {
                ex = !*(char**)mular_getix(rsv->md, INDX(row, col));
            } else {
                ifBLIN_QX1( "+ row %d %s in [0..%d], col %d %s in [0..%d]"
                          , row, (row >= 0 && row < rsv->nrow) ? "" : "NOT", rsv->nrow
                          , col, (col >= 0 && col < rsv->ncol) ? "" : "NOT", rsv->ncol
                          );
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            }
        } else if (row == 0 && col == 0) {
            ex = 1;
        } else {
            ifBLIN_QX1("row %d col %d in emty", row, col);
            ex = -EX_SOFTWARE;
            errno = EDOOFUS;
        }
        break;
    case PGOBLIN_Length:
        switch (valinfo(PGOBLIN_IsNull, rsv, row, col)) {
        case  1:
            ex = 0;
            break;
        case  0:
            if  (rsv->flags & PGOBLIN_MULAR) {
                ex = strlen(*(char**)mular_getix(rsv->md, INDX(row, col)));
            } else if (rsv->flags & PGOBLIN_VM) {
                ifBLIN_QX0("VM instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            } else {
                ifBLIN_QX0("NOTHING instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            }
            break;
        default:
            ex = -EX_SOFTWARE;
            errno = EDOOFUS;
        }
        break;
    default:
        ex = INT_MIN;
        errno = ENOTSUP;
    }
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static char *
/**********************************************************************
 **                                                                  **/
getvalue(pgoblin_mode *rsv, int row, int col) {                     /**
 **                                                                  **
 **********************************************************************/
    char *c = NULL;

#   define blin_internal_flags (rsv->flags & BLIN_MASK)
    ifBLIN_QX2("+ %08X %d x %d", rsv->flags, row, col);
    if  (!valinfo(PGOBLIN_IsNull, rsv, row, col)) {
        if  (rsv->flags & PGOBLIN_VM) {
            ifBLIN_QX0("VM instead of MULAR");
            errno = EDOOFUS;
        } else if (rsv->flags & PGOBLIN_MULAR) {
            c = *(char**)mular_getix(rsv->md, INDX(row, col));
            ifBLIN_QX4("mular=%s~", c);
        } else {
            ifBLIN_QX0("NOTHING instead of MULAR");
            errno = EDOOFUS;
    }   }
    ifBLIN_QX2("- %s", !c ? "" : c);
    return(c);
#   undef blin_internal_flags
}

pgoblin_dbases PGOBLIN_MODULE =
{ (PGOBLIN_DB_PREPARE | PGOBLIN_DB_COPYIN | PGOBLIN_DB_COPYOUT)
, "3pGoblin-" VERS
, "sqlite2\0" VERS
, 0
, init
, fini
, shurecon
, query
, prepare
, execute
, resinfo
, subinfo
, erinfo
, valinfo
, getvalue
, getstream
, getcopy
, putcopy
, endstream
, NULL /* tarry    */
, NULL /* notifies */
, finish
, clear
, NULL
, NULL
};
