/*-
 * Copyright (C)2008..2022 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2008..2022 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: db_pgsql.c,v 1.109 2022/02/19 18:34:43 babolo Exp $"

#define BLIN_COMPAT      4
#define Bpars_COMPAT     3
#define MULAR_COMPAT     0
#define MIFE_COMPAT      4
#define PGOBLIN_COMPAT   4
#define PGOBLIN_INTERNAL 1
#define PGOBLIN_MODULE   pgoblin_module

#include <netinet/in.h>
#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <syslog.h>
#include <stdarg.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <stdint.h>
#include <errno.h>
#include <poll.h>
#include <err.h>
#include <libpq-fe.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"
#include "level.h"

#define PRINTNOTNULL(A) (!(A) ? "" : (A))
#define PRINTANDNULL(A) PRINTNOTNULL(A), ((A) ? "~" : "")
#define PQSTATNDNULL(A) PRINTANDNULL(PQparameterStatus(pcnn->pgcn, (A)))
#define ALIGN8(A)  popt[i].A ? popt[i].A : ""                                                         \
      , ( (!popt[i].A || (strlen(popt[i].A) < 8))                                                     \
        ? "		"                                                                             \
        : (strlen(popt[i].A) < 16) ? "	" : " "                                                       \
        )

#define PQEXEC(RESULT, QUERY, CONDITION, CCODE, RCONN) do {                                           \
    ifBLIN_QX4("PQexec: %s\n", (char*)(QUERY));                                                       \
    (RESULT) = PQexec(RCONN, (QUERY));                                                                \
    if  (!(RESULT)) {                                                                                 \
        ifBLIN_QX0("result is absent - fatal error.\n");                                              \
        return(CCODE);                                                                                \
    }                                                                                                 \
    ifBLIN_QX4("PQresultStatus %d", PQresultStatus(RESULT));                                          \
    if  (PQresultStatus(RESULT) != (CONDITION)) {                                                     \
        if  (PQresultStatus(RESULT) == PGRES_FATAL_ERROR) {                                           \
            ifBLIN_QX0("PQexec failed: %s", PQresultErrorMessage(RESULT));                            \
            return(CCODE);                                                                            \
        } else if ((CONDITION) == PGRES_COMMAND_OK && PQresultStatus(RESULT) == PGRES_TUPLES_OK) {    \
            ifBLIN_QX1("PQexec wait for COMMAND_OK but received TUPLES_OK");                          \
        } else if ((CONDITION) == PGRES_COMMAND_OK && PQresultStatus(RESULT) == PGRES_EMPTY_QUERY) {  \
            ifBLIN_QX1("PQexec wait for COMMAND_OK but received EMPTY_QUERY");                        \
        } else ifBLIN_QO0 {                                                                           \
            warnx("PQexec wait for %d but received %d", (CONDITION), PQresultStatus(RESULT));         \
            return(CCODE);                                                                            \
}   }   } while(0);

typedef struct pgoblin_db_conn {
    BLIN_flag     flags;
/*                PGOBLIN_DB_TYPE                                                                    */
    int           numoi;
    PGconn       *pgcn;
    u_int32_t    *toid;
    char        **tname;
    void         *dummy1;
} pgoblin_db_conn;

struct pgoblin_realpq {
    BLIN_flag        flags;
/*                   PGOBLIN_DB_TYPE                                                                 */
    PGresult        *res;
    char            *mess;
    pgoblin_db_conn *pcnn;
};

pgoblin_dbases PGOBLIN_MODULE;

static pgoblin_realpq *
/**********************************************************************
 **                                                                  **/
resv(pgoblin_rdb *rdb) {                                            /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    pgoblin_realpq    *rsv  = NULL;

    if  (!(rsv = calloc(1, sizeof(pgoblin_realpq)))) {
        ifBLIN_QW0("calloc");
    } else {
        rsv->flags = rdb->flags & (BLIN_MASK | PGOBLIN_DB_TYPE);
        rsv->pcnn = rdb->conn;
    }
    return(rsv);
#   undef blin_internal_flags
}

static void
/**********************************************************************
 **                                                                  **/
NoticeReceiver(void *rdb, const PGresult *res) {                    /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (((pgoblin_rdb *)rdb)->flags & BLIN_MASK)
    int severity; /*  ,   */

    severity = babolo_testword(&level, PQresultErrorField(res, PG_DIAG_SEVERITY_NONLOCALIZED));
    ifBLIN_QO2 {
        severity = -1;
    } else ifBLIN_QO1 {
        severity -= LOG_WARNING;
    } else ifBLIN_QO0 {
        severity -= LOG_ERR;
    }
    if  (0 > severity) {
        ifBLIN_QX0( "%s %s %s\n%s"
                  , PQresultErrorField(res, PG_DIAG_SEVERITY)
                  , PQresultErrorField(res, PG_DIAG_SQLSTATE)
                  , PQresultErrorField(res, PG_DIAG_MESSAGE_PRIMARY)
                  , PQresultErrorField(res, PG_DIAG_MESSAGE_DETAIL)
                  );
        ifBLIN_QX1( "%s\n%s\n"
                  , PQresultErrorField(res, PG_DIAG_MESSAGE_HINT)
                  , PQresultErrorField(res, PG_DIAG_STATEMENT_POSITION)
                  );
#ifdef PG_DIAG_INTERNAL_POSITION
        ifBLIN_QX1( "%s\n%s"
                  , PQresultErrorField(res, PG_DIAG_INTERNAL_QUERY)
                  , PQresultErrorField(res, PG_DIAG_INTERNAL_POSITION)
                  );
#endif
#ifdef PG_DIAG_SCHEMA_NAME
        ifBLIN_QX2( "shema=%s table=%s column=%s type=%s constraint=%s"
                  , PQresultErrorField(res, PG_DIAG_SCHEMA_NAME)
                  , PQresultErrorField(res, PG_DIAG_TABLE_NAME)
                  , PQresultErrorField(res, PG_DIAG_COLUMN_NAME)
                  , PQresultErrorField(res, PG_DIAG_DATATYPE_NAME)
                  , PQresultErrorField(res, PG_DIAG_CONSTRAINT_NAME)
                  );
#endif
        ifBLIN_QX2( "%s:%s:%s: %s"
                  , PQresultErrorField(res, PG_DIAG_SOURCE_FILE)
                  , PQresultErrorField(res, PG_DIAG_SOURCE_LINE)
                  , PQresultErrorField(res, PG_DIAG_SOURCE_FUNCTION)
                  , PQresultErrorField(res, PG_DIAG_CONTEXT)
                  );
    }
#   undef blin_internal_flags
}

#define U0 0
#define U1 1
#define U2 2
#define U3 3
#define U4 4

static u_char miss[][PGOBLIN_CR_MAX >> PGOBLIN_CR_SHIFT] =
/*  OK   TUPL OUT  IN   STR  BIN  BTUPL */
{{  U1,  U1,  U1,  U1,  U1,  U1,  U1 } /* PGRES_EMPTY_QUERY    */
,{  U0,  U2,  U2,  U2,  U2,  U2,  U2 } /* PGRES_COMMAND_OK     */
,{  U1,  U0,  U2,  U2,  U0,  U0,  U0 } /* PGRES_TUPLES_OK      */
,{  U2,  U2,  U0,  U2,  U2,  U2,  U2 } /* PGRES_COPY_OUT       */
,{  U2,  U2,  U2,  U0,  U2,  U2,  U2 } /* PGRES_COPY_IN        */
,{  U3,  U3,  U3,  U3,  U3,  U3,  U3 } /* PGRES_BAD_RESPONSE   */
,{  U3,  U3,  U3,  U3,  U3,  U3,  U3 } /* PGRES_NONFATAL_ERROR */
,{  U4,  U4,  U4,  U4,  U4,  U4,  U4 } /* PGRES_FATAL_ERROR    */
};

static ExecStatusType es[] =
{ PGRES_COMMAND_OK, PGRES_TUPLES_OK, PGRES_COPY_OUT , PGRES_COPY_IN
, PGRES_TUPLES_OK , PGRES_TUPLES_OK, PGRES_TUPLES_OK
};

static int
/*****************************************************************************
 **                                                                         **/
extatus( u_int32_t   flags                                                 /**/
       , PGresult   *res                                                   /**/
       , const char *who                                                   /**/
       , BLIN_flag   cond                                                  /**/
       , const char *func                                                  /**/
       , int         line                                                  /**/
       ) {                                                                 /**
 **                                                                         **
 *****************************************************************************/
    int            ex = 0;
    ExecStatusType st;
    u_char         cn = (cond & PGOBLIN_CR_MASK) >> PGOBLIN_CR_SHIFT;

#   define blin_internal_flags (flags & BLIN_MASK)
    st = PQresultStatus(res);
    ifBLIN_QX4("PQresultStatus %d %s, diff %d", st, PQresStatus(st), miss[st][cn]);
    switch (miss[st][cn]) {
    case U1:
        ifBLIN_QX1( "%s:%d:%s waits for %d %s but received %d %s"
                  , func
                  , line
                  , who
                  , es[cn]
                  , PQresStatus(es[cn])
                  , st
                  , PQresStatus(st)
                  );
        break;
    case U2:
        ifBLIN_QX0( "%s:%d:%s waits for %d %s but received %d %s"
                  , func
                  , line
                  , who
                  , es[cn]
                  , PQresStatus(es[cn])
                  , st
                  , PQresStatus(st)
                  );
        ex = miss[st][cn];
        break;
    case U4:
    case U3:
        ifBLIN_QX0( "%s:%d:%s failed: %s"
                  , func
                  , line
                  , who
                  , PQresultErrorMessage(res)
                  );
        /* erinfo(... ) XXXX */
        ex = miss[st][cn];
        break;
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
clear(pgoblin_exenv *exenv, pgoblin_nr nio) {                       /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio    *rio;
    pgoblin_realpq *rsv;
    int             ex = EX_OK;

    GET_RIO(rio, exenv->options, nio);
    if  (!!rio && !!rio->pq) {
        rsv = rio->pq;
        if  (rsv->res) PQclear(rsv->res);
        if  (rsv->mess) free(rsv->mess);
        free(rio->pq);
        rio->pq = NULL;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static ssize_t
/******************************************************************************
 **                                                                          **/
valinfo(pgoblin_exenv *exenv, pgoblin_nr nio, int kind, int row, int col) { /**
 **                                                                          **
 ******************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio  *rio;
    ssize_t       ex = PGOBLIN_SSIZE_MIN;

    GET_RIO(rio, exenv->options, nio);
    switch (kind) {
    case PGOBLIN_IsNull : ex = PQgetisnull(rio->pq->res, row, col); break;
    case PGOBLIN_Length : ex = PQgetlength(rio->pq->res, row, col); break;
    case PGOBLIN_BinForm: ex =   PQfformat(rio->pq->res,      col); break;
    case PGOBLIN_TypeOid: ex =     PQftype(rio->pq->res,      col); break;
    case PGOBLIN_TypeMod: ex =      PQfmod(rio->pq->res,      col); break;
    case PGOBLIN_ColSize: ex =     PQfsize(rio->pq->res,      col); break;
    default             :
        errno = EDOOFUS;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
init(pgoblin_main *options, int slot) {                             /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (options->flags & BLIN_MASK)
    int er[] = { PGRES_EMPTY_QUERY   , PGRES_COMMAND_OK  , PGRES_TUPLES_OK     , PGRES_COPY_OUT
               , PGRES_COPY_IN       , PGRES_BAD_RESPONSE, PGRES_NONFATAL_ERROR, PGRES_FATAL_ERROR
               };
    int ex = EX_OK;
    int i;

    for (i = 0; i < 8; ++i) {
        if  (er[i] != i) {
            ifBLIN_QX0("Invalid result code [%d] <> %d", er[i], i);
            ++ex;
    }   }
    if  (!ex) {
        PGOBLIN_MODULE.flags = (PGOBLIN_MODULE.flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE))
                             | (options->flags & BLIN_MASK)
                             | (slot & PGOBLIN_DB_TYPE)
        ;
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
fini(pgoblin_main *options) {                                       /**
 **                                                                  **
 **********************************************************************/
    return(0);
}

static int
/**********************************************************************
 **                                                                  **/
shurecon(pgoblin_exenv *exenv, pgoblin_nr ndb) {                    /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    PQconninfoOption *popt;
    pgoblin_db_conn  *pcnn;
    pgoblin_rdb      *rdb;
    PGresult         *res;
    int               ex = EX_OK;
    int               i;

    GET_CON(rdb, exenv->options, ndb);
    pcnn = rdb->conn;
    if  (!!pcnn && !!pcnn->pgcn && (PQstatus(pcnn->pgcn) == CONNECTION_OK)) return(0);
    if  (!rdb->conn && !(pcnn = rdb->conn = calloc(1, sizeof(pgoblin_db_conn)))) {
        ifBLIN_QX0("no MEM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    ifBLIN_QO4 {
        ifBLIN_QX4( "Connection defaults:\n"
                    "size=chr keyword	envvar		compiled val		label\n"
                  );
        popt = PQconndefaults();
        for (i = 0; popt[i].keyword; i++) {
            ifBLIN_QX4( "%4d=%s~	%s%s%s%s%s%s%s%s%s\n"
                      , popt[i].dispsize
                      , popt[i].dispchar
                      , ALIGN8(keyword)
                      , ALIGN8(envvar)
                      , PRINTNOTNULL(popt[i].compiled)
                      , !(popt[i].compiled) ? "	" : strlen(popt[i].compiled) < 7 ? "~	" : "~ "
                      , PRINTNOTNULL(popt[i].val)
                      , !(popt[i].val)
                      ? "		"
                      : strlen(popt[i].val) < 7 ? "~		" : "~	"
                      , popt[i].label
                      );
        }
        PQconninfoFree(popt);
        ifBLIN_QX4( "PGREALM=%s%s PGDATESTYLE=%s%s PGTZ=%s%s PGCLIENTENCODING=%s%s PGGEQO=%s%s\n"
                  , PRINTANDNULL(getenv("PGREALM"))
                  , PRINTANDNULL(getenv("PGDATESTYLE"))
                  , PRINTANDNULL(getenv("PGTZ"))
                  , PRINTANDNULL(getenv("PGCLIENTENCODING"))
                  , PRINTANDNULL(getenv("PGGEQO"))
                  );
    }
    MARK_R_CONN_GO(ndb);;
    ;;  if  (rdb->flags & PGOBLIN_PARMIN && rdb->dbname) {
    ;;      pcnn->pgcn = PQconnectdb(rdb->dbname);
    ;;  } else {
    ;;      pcnn->pgcn = PQsetdbLogin(rdb->host, rdb->port, NULL, NULL, rdb->dbname, rdb->username, NULL);
    ;;  }
    ;;  rdb->flags = PGOBLIN_CLOSECONN
 /* ;; */         | (rdb->flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE))
 /* ;; */         | (PGOBLIN_MODULE.flags & (BLIN_MASK | PGOBLIN_DB_TYPE))
    ;;  ;
    MARK_R_CONN_WENT(ndb);;
    ifBLIN_QX4("PQoptions=%s~\n", PQoptions(pcnn->pgcn));
    ifBLIN_QX4( "PQdb=%s~ PQuser=%s~ PQpass=%s%s "
                "PQhost=%s%s PQport=%s%s "
                "PQtty=%s%s\n"
              , PQdb(pcnn->pgcn), PQuser(pcnn->pgcn)
              , PRINTANDNULL(PQpass(pcnn->pgcn))
              , PRINTANDNULL(PQhost(pcnn->pgcn))
              , PRINTANDNULL(PQport(pcnn->pgcn))
              , PRINTANDNULL(PQtty(pcnn->pgcn))
              );
    ifBLIN_QX4( "server_version=%s%s "
                "client_encoding=%s%s "
                "is_superuser=%s%s "
                "session_authorization=%s%s "
                "DateStyle=%s%s\n"
              , PQSTATNDNULL("server_version")
              , PQSTATNDNULL("client_encoding")
              , PQSTATNDNULL("is_superuser")
              , PQSTATNDNULL("session_authorization")
              , PQSTATNDNULL("DateStyle")
              );
    ifBLIN_QX4( "PQprotocolVersion=%d PQsocket=%d PQbackendPID=%d\n"
              , PQprotocolVersion(pcnn->pgcn)
              , PQsocket(pcnn->pgcn)
              , PQbackendPID(pcnn->pgcn)
              );
    ifBLIN_QX4("PQstatus %d\n", PQstatus(pcnn->pgcn));
    (void)PQsetNoticeReceiver(pcnn->pgcn, NoticeReceiver, rdb);
    (void)PQsetErrorVerbosity(pcnn->pgcn, PQERRORS_VERBOSE);
    (void)PQsetErrorContextVisibility(pcnn->pgcn, PQSHOW_CONTEXT_ALWAYS);
    if  (CONNECTION_BAD == PQstatus(pcnn->pgcn)) {
        ex = EX_NOINPUT;
        errno = ENXIO;
        goto out;
    } else {
        if  (  rdb->flags & PGOBLIN_NEEDINTIME
            && !(rdb->flags & (PGOBLIN_INT64TIME | PGOBLIN_FLOATTIME))
            ) {
            if  (!rdb->intran) {
                PQEXEC(res, "BEGIN;", PGRES_COMMAND_OK, -1, pcnn->pgcn);
                PQclear(res);
            }
            PQEXEC( res
                  , "DECLARE pgoblin BINARY NO SCROLL CURSOR FOR"
                    " SELECT '2000-01-01 00:00:01'::timestamp"
                    ";"
                    "FETCH 1 FROM pgoblin;"
                  , PGRES_TUPLES_OK
                  , -1
                  , pcnn->pgcn
                  );
            if  (PQgetisnull(res, 0, 0) || (PQgetlength(res, 0, 0) < 8)) {
                ifBLIN_QX0("timestamp");
                ex = EX_DATAERR;
                errno = EINVAL;
            } else if (*((u_char*)(PQgetvalue(res, 0, 0))) == 077) {
                rdb->flags |= PGOBLIN_FLOATTIME;
                ifBLIN_QX2("float timestamp format");
            } else if (*((u_char*)(PQgetvalue(res, 0, 0))) == 0) {
                rdb->flags |= PGOBLIN_INT64TIME;
                ifBLIN_QX2("int64 timestamp format");
            } else {
                ifBLIN_QX0("unknown timestamp format");
                ex = EX_DATAERR;
                errno = EFTYPE;
            }
            PQclear(res);
            PQEXEC(res, "CLOSE pgoblin;", PGRES_COMMAND_OK, -1, pcnn->pgcn);
            PQclear(res);
            if  (!rdb->intran) {
                PQEXEC(res, "END;", PGRES_COMMAND_OK, -1, pcnn->pgcn);
                PQclear(res);
    }   }   }
    if  (!!ex) goto out;
    if  (CONNECTION_BAD == PQstatus(pcnn->pgcn)) {
        ex = EX_NOINPUT;
        errno = ENXIO;
        goto out;
    } else {
        const char *c;
        const char *n;
        int         i;

        if  (!rdb->intran) {
            PQEXEC(res, "BEGIN;", PGRES_COMMAND_OK, -1, pcnn->pgcn);
            PQclear(res);
        }
        PQEXEC( res
              , "DECLARE pgoblin BINARY NO SCROLL CURSOR FOR"
                " SELECT oid, typname FROM pg_catalog.pg_type ORDER BY oid"
                ";"
                "FETCH 9999 FROM pgoblin;"
              , PGRES_TUPLES_OK
              , -1
              , pcnn->pgcn
              );
        if  (!(c = PQcmdStatus(res)) || strncmp("FETCH ", c, 6)) {
            ifBLIN_QX0("PQcmdStatus pg_type");
            ex = EX_DATAERR;
            errno = EINVAL;
        } else if (!(n = PQcmdTuples(res)) || !(0 < (i = strtol(n, NULL, 10)))) {
            ifBLIN_QX0("PQcmdTuples pg_type");
            ex = EX_DATAERR;
            errno = EINVAL;
        } else if (PQntuples(res) != i) {
            ifBLIN_QX0("PQntuples <> %d", i);
            ex = EX_DATAERR;
            errno = EINVAL;
        } else if (!(pcnn->toid = calloc(i, sizeof(u_int32_t)))) {
            ifBLIN_QX0("no MEM");
            ERROUT(EX_OSERR, ENOMEM);
        } else if (!(pcnn->tname = calloc(i, sizeof(char *)))) {
            ifBLIN_QX0("no MEM");
            ERROUT(EX_OSERR, ENOMEM);
        } else {
            for (int k = 0; k < i; ++k) {
                if  (!!PQgetisnull(res, k, 0) || !!PQgetisnull(res, k, 1)) {
                    ifBLIN_QX0("PQgetisnull");
                    ex = EX_DATAERR;
                    errno = EINVAL;
                    break;
                } else if (!(pcnn->tname[k] = strndup( PQgetvalue(res, k, 1)
                                                     , PQgetlength(res, k, 1)
                          ) )                        ) {
                    ex = EX_DATAERR;
                    errno = EINVAL;
                    break;
                } else {
                    pcnn->toid[k] = ntohl(*((u_int32_t *)PQgetvalue(res, k, 0)));
                    pcnn->numoi = k;
        }   }   }
        PQclear(res);
        PQEXEC(res, "CLOSE pgoblin;", PGRES_COMMAND_OK, -1, pcnn->pgcn);
        PQclear(res);
        if  (!rdb->intran) {
            PQEXEC(res, "END;", PGRES_COMMAND_OK, -1, pcnn->pgcn);
            PQclear(res);
    }   }
out:
    if  (!!ex) ifBLIN_QX0("Connection to PostgreSQL failed: %s", PQerrorMessage(pcnn->pgcn));
    return(ex);
#   undef blin_internal_flags
}

static int
/********************************************************************************
 **                                                                            **/
query(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nct) { /**
 **                                                                            **
 ********************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    int              doend = 0;
    BLIN_flag        cond  = 0;
    pgoblin_db_conn *pcnn;
    char            *qry;
    pgoblin_rio     *rou;
    pgoblin_rio     *rct;
    PGresult        *res;
    pgoblin_rdb     *rdb;
    int              ex    = 0;
    char            *p;
    char            *q     = NULL;

    GET_CON(rdb, exenv->options, ndb);
    GET_RIO(rou, exenv->options, nou);
    GET_RIO(rct, exenv->options, nct);
    qry = rct->text;
    cond = rdb->flags & PGOBLIN_CR_MASK;
    if  (!qry) {
        ifBLIN_QX1("No query");
        goto out;
    }
    if  (!(rou->pq = resv(rdb))) {
        ifBLIN_QX0("resv");
        goto out;
    }
    rou->pq->res = NULL;
    pcnn = rdb->conn;
    switch (cond) {
    case PGOBLIN_CR_OK:
    case PGOBLIN_CR_TUPL:
    case PGOBLIN_CR_OUT:
    case PGOBLIN_CR_IN:
        ifBLIN_QX4("PQexec %d(%s)=%s~", cond, pgoblin_decond(cond), qry);
        if  (!(rou->pq->res = PQexec(pcnn->pgcn, qry))) {
            ifBLIN_QX0("PQ result is absent - fatal error %s", PQerrorMessage(pcnn->pgcn));
            ERROUT(EX_UNAVAILABLE, ENODEV);
        }
        ex = extatus(rdb->flags, rou->pq->res, "PQexec nom", cond, __func__, __LINE__);
        break;
    case PGOBLIN_CR_SSEL:
        p = "pgoblin";
        goto rest;
    case PGOBLIN_CR_BTUPL:
    case PGOBLIN_CR_BSEL:
        p = "pgoblin BINARY";
    rest:
        MARK_R_CONN_GO(ndb);;
        ;;  rdb->flags &= ~PGOBLIN_CR_MASK;
        ;;  rdb->flags |= PGOBLIN_CR_OK;
        MARK_R_CONN_WENT(ndb);;
        ifBLIN_QX4("PQexec %d=%s~", cond, "BEGIN;");
        if  (!(res = PQexec(pcnn->pgcn, "BEGIN;"))) {
            ifBLIN_QX0("PQexec result is absent: BEGIN; %s", PQerrorMessage(pcnn->pgcn));
            ERROUT(EX_SOFTWARE, ENOTRECOVERABLE);
        } else if(PGRES_COMMAND_OK != PQresultStatus(res)) {
            ex = extatus(exenv->options->flags, res, "BEGIN;", PGOBLIN_CR_OK, __func__, __LINE__);
            errno = EDOOFUS;
            goto out;
        }
        PQclear(res);
        asprintf(&q, "DECLARE %s NO SCROLL CURSOR FOR %s;", p, qry);
        if  (!q) {
            ifBLIN_QW0("asprintf #1");
            ex = EX_OSERR;
            goto out;
        }
        ifBLIN_QX4("PQexec %d=%s~", cond, q);
        if  (!(res = PQexec(pcnn->pgcn, q))) {
            ifBLIN_QX0("PQexec result is absent: %s %s", q, PQerrorMessage(pcnn->pgcn));
            ERROUT(EX_SOFTWARE, ENOTRECOVERABLE);
        } else if(PGRES_COMMAND_OK != PQresultStatus(res)) {
            ex = extatus(exenv->options->flags, res, q, PGOBLIN_CR_OK, __func__, __LINE__);
            errno = EDOOFUS;
            goto out;
        }
        PQclear(res);
        if  (PGOBLIN_CR_BTUPL == cond) {
            doend = 1;
            ifBLIN_QX4("PQexec %d=%s~", cond, "FETCH 2147483647 FROM pgoblin;");
            if  (!(rou->pq->res = PQexec(pcnn->pgcn, "FETCH 2147483647 FROM pgoblin;"))) {
                ifBLIN_QX0( "PQexec result is absent: FETCH 2147483647 FROM pgoblin; %s"
                          , PQerrorMessage(pcnn->pgcn)
                          );
                ERROUT(EX_UNAVAILABLE, ENODEV);
            } else if(PGRES_TUPLES_OK != PQresultStatus(rou->pq->res)) {
                ex = extatus( rdb->flags
                            , rou->pq->res
                            , "FETCH 2147483647 FROM pgoblin;"
                            , PGOBLIN_CR_OK
                            , __func__
                            , __LINE__
                            );
                errno = EDOOFUS;
                goto out;
            }
            ifBLIN_QX4("PQexec %d=%s~", cond, "CLOSE pgoblin;");
            if  (!(res = PQexec(pcnn->pgcn, "CLOSE pgoblin;"))) {
                ifBLIN_QX0("PQexec result is absent: CLOSE pgoblin; %s", PQerrorMessage(pcnn->pgcn));
                ERROUT(EX_UNAVAILABLE, ENODEV);
            } else if(PGRES_COMMAND_OK != PQresultStatus(res)) {
                ex = extatus(rdb->flags, res, "CLOSE pgoblin;", PGOBLIN_CR_OK, __func__, __LINE__);
                errno = EDOOFUS;
                goto out;
            }
            PQclear(res);
        }
        break;
    default:
        ifBLIN_QX0("illegal cond=%d", cond);
        ex = -1;
    }
out:
    if  (!!q) free(q);
    if  (!!doend) {
        ifBLIN_QX4("PQexec %d=%s~", cond, "END;");
        if  (!(res = PQexec(pcnn->pgcn, "END;"))) {
            ifBLIN_QX0("PQexec result is absent: END; %s", PQerrorMessage(pcnn->pgcn));
            ERROUT(EX_UNAVAILABLE, ENODEV);
        } else if(PGRES_COMMAND_OK != PQresultStatus(res)) {
            ex = extatus(rdb->flags, res, "END;", PGOBLIN_CR_OK, __func__, __LINE__);
            errno = EDOOFUS;
            goto out;
    }   }
    MARK_R_CONN_GO(ndb);;
    ;;  rdb->flags &= ~PGOBLIN_CR_MASK;
    ;;  rdb->flags |= cond;
    MARK_R_CONN_WENT(ndb);;
    return(ex);
#   undef blin_internal_flags
}

#ifdef PG_DIAG_INTERNAL_POSITION

static int
/**********************************************************************************
 **                                                                              **/
prepare(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nct) { /**
 **                                                                              **
 **********************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rou;
    pgoblin_rio     *rct;
    pgoblin_realpq **rsv = NULL;
    char            *qry;
    PGresult        *res;
    int              ex = 0;

    GET_CON(rdb, exenv->options, ndb);
    GET_RIO(rct, exenv->options, nct);
    qry = rct->text;
    GET_RIO(rou, exenv->options, nou);
    rsv = &rou->pq;
    pcnn = rdb->conn;
    res = NULL;
    ifBLIN_QX4("PQprepare =%s~", qry);
    if  (!(res = PQprepare(pcnn->pgcn, "", qry, 0, NULL))) {
        ifBLIN_QX0("PQ result is absent - fatal error %s", PQerrorMessage(pcnn->pgcn));
        ERROUT(EX_UNAVAILABLE, ENODEV);
    }
    ex = extatus(rdb->flags, res, "PQprepare", PGOBLIN_CR_OK, __func__, __LINE__);
out:
    if  (!rsv) {
        PQclear(res); /* XXXX */
    } else if (!!res) {
        if  (!(*rsv = resv(rdb))) {
            ifBLIN_QW0("resv");
            ex = EX_OSERR;
        } else {
            (*rsv)->res = res;
        }
    } else {
        *rsv = NULL;
    }
    return(ex);
#   undef blin_internal_flags
}

#endif

static int
/*****************************************************************************************************
 **                                                                                                 **/
execute(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nct, pgoblin_nr nin) {    /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    BLIN_flag        cond;
    pgoblin_realpq **rsv  = NULL;
    const char     **val  = NULL;
    PGresult        *res  = NULL;
    const char      *qry  = "";
    int             *len  = NULL;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rin;
    pgoblin_rio     *rct;
    pgoblin_rio     *rou;
    int              col;
    int              row;
    int              ex   = 0;

    GET_CON(rdb, exenv->options, ndb);
    pcnn = rdb->conn;
    cond = rdb->flags & PGOBLIN_CR_MASK;
    if  (!rdb) {
        ERROUT(EX_USAGE, EINVAL);
    }
    if  ((PGOBLIN_CR_OK != cond) && (PGOBLIN_CR_TUPL != cond)) {
        ifBLIN_QX0("execute pgsql illegal cond=%d", cond);
        ERROUT(EX_USAGE, EINVAL);
    }
    GET_RIO(rou, exenv->options, nou);
    rsv = &rou->pq;
    GET_RIO(rin, exenv->options, nin);
    col = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Nfields);
    row = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Ntuples);
    GET_RIO(rct, exenv->options, nct);
    qry = rct->text;
    ifBLIN_QX4("%d rows x %d cols %d", row, col, col);
    if  (!(val = malloc(sizeof(char *) * col))) {
        ifBLIN_QW0("pgoblin_select #1: No mem");
        ex = EX_OSERR;
        goto out;
    }
    if  (!(len = malloc(sizeof(int) * col))) {
        ifBLIN_QW0("pgoblin_select #2: No mem");
        free(val);
        val = NULL;
        ex = EX_OSERR;
        goto out;
    }
    for (int i = 0; i < row; i++) {
        for (int j = 0; j < col; j++) {
            val[j] = pgoblin_db_getvalue(exenv, nin, i, j);
            len[j] = (int)pgoblin_db_valinfo(exenv, nin, PGOBLIN_Length, i, j);
        }
        for (int j = col; j < col; j++) {
            val[j] = NULL;
            len[j] = -1;
        }
        ifBLIN_QX4("PQexecParams %d=%s~", cond, qry);
        if  (!(res = PQexecParams(pcnn->pgcn, qry, col, NULL, val, len, NULL, 0))) {
            ifBLIN_QX0("PQ result is absent - fatal error %s", PQerrorMessage(pcnn->pgcn));
            ERROUT(EX_UNAVAILABLE, ENODEV);
        }
        if  (!!(ex = extatus(rdb->flags, res, "PQexecParams", cond, __func__, __LINE__))) break;
        if  (i + 1 < row) {
            PQclear(res);
            res = NULL;
    }   }
out:
    if  (val) free(val);
    if  (len) free(len);
    if  (!!rsv) *rsv = NULL;
    if  (!!res) {
        if  (!rsv) {
            PQclear(res); /* XXXX */
            res = NULL;
        } else if (!(*rsv = resv(rdb))) {
            ifBLIN_QW0("resv");
            ex = EX_OSERR;
        } else {
            (*rsv)->res = res;
    }   }
    return(ex);
#   undef blin_internal_flags
}

static int64_t
/**********************************************************************
 **                                                                  **/
resinfo(pgoblin_exenv *exenv, pgoblin_nr nio, int kind) {           /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_realpq *rsv;
    pgoblin_rio    *rio;
    int64_t         ex = 0;
    char           *c;

    GET_RIO(rio, exenv->options, nio);
    rsv = rio->pq;
    switch (kind) {
    case PGOBLIN_Ntuples:
        ex = PQntuples(rsv->res);
        break;
    case PGOBLIN_Nfields:
        ex = PQnfields(rsv->res);
        break;
    case PGOBLIN_Nchange:
        c = PQcmdTuples(rsv->res);
        ex = *c ? strtoul(c, NULL, 10) : -1;
        break;
    case PGOBLIN_LastOid:
        ex = PQoidValue(rsv->res);
        break;
    default             :
        ex = -1;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static const char *
/**************************************************************************
 **                                                                      **/
subinfo(pgoblin_exenv *exenv, pgoblin_nr nio, int kind, int64_t sub) {  /**
 **                                                                      **
 **************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio  *rio;
    const char   *cx = NULL;
    size_t        ex;

    GET_RIO(rio, exenv->options, nio);
    switch (kind) {
    case PGOBLIN_ColName:
        cx = PQfname(rio->pq->res, sub);
        break;
    case PGOBLIN_CmdStat:
        cx = PQcmdStatus(rio->pq->res);
        break;
    case PGOBLIN_TypeNm :
        ex = valinfo(exenv, nio, PGOBLIN_TypeOid, 0, sub);
        for ( int l = 0, r = rio->pq->pcnn->numoi, m; l < r;) {
            m = (l + r) / 2;
            if  (rio->pq->pcnn->toid[m] == ex) {
                cx = rio->pq->pcnn->tname[m];
                break;
            } else if (rio->pq->pcnn->toid[m] < ex) {
                l = m;
            } else {
                r = m;
        }   }
        break;
    default             :
        cx = NULL;
    }
out:
    return(cx);
#   undef blin_internal_flags
}

static char *
/**********************************************************************
 **                                                                  **/
erinfo(pgoblin_exenv *exenv, pgoblin_nr nio) {                      /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio  *rio;
    char         *cx = NULL;

    GET_RIO(rio, exenv->options, nio);
    if  (!!rio->pq->mess) goto out;
    if  (!rio->pq->res) {
        if  (asprintf(&cx, "No results") < 0) warn("subinfo(ResMess) asprintf");
    } else ifBLIN_QO4 {
        if  ( asprintf( &cx
#ifdef PG_DIAG_INTERNAL_POSITION
                      , "%s %s %s\n%s\n%s\n%s\n%s\n%s\nfunction %s at %s line of %s file\n%s"
#else
                      , "%s %s %s\n%s\n%s\n%s\nfunction %s at %s line of %s file\n%s"
#endif
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SEVERITY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SQLSTATE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_PRIMARY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_DETAIL)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_HINT)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_STATEMENT_POSITION)
#ifdef PG_DIAG_INTERNAL_POSITION
                      , PQresultErrorField(rio->pq->res, PG_DIAG_INTERNAL_QUERY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_INTERNAL_POSITION)
#endif
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SOURCE_FUNCTION)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SOURCE_LINE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SOURCE_FILE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_CONTEXT)
                      )
              < 0
            ) warn("subinfo(ResMess) asprintf");
    } else ifBLIN_QO3 {
        if  ( asprintf( &cx
#ifdef PG_DIAG_INTERNAL_POSITION
                      , "%s %s %s\n%s\n%s\n%s\n%s\n%s\nfunction %s at %s line of %s file"
#else
                      , "%s %s %s\n%s\n%s\n%s\nfunction %s at %s line of %s file"
#endif
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SEVERITY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SQLSTATE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_PRIMARY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_DETAIL)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_HINT)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_STATEMENT_POSITION)
#ifdef PG_DIAG_INTERNAL_POSITION
                      , PQresultErrorField(rio->pq->res, PG_DIAG_INTERNAL_QUERY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_INTERNAL_POSITION)
#endif
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SOURCE_FUNCTION)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SOURCE_LINE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SOURCE_FILE)
                      )
              < 0
            ) warn("subinfo(ResMess) asprintf");
    } else ifBLIN_QO2 {
        if  ( asprintf( &cx
#ifdef PG_DIAG_INTERNAL_POSITION
                      , "%s %s %s\n%s\n%s\n%s\n%s\n%s"
#else
                      , "%s %s %s\n%s\n%s\n%s"
#endif
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SEVERITY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SQLSTATE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_PRIMARY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_DETAIL)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_HINT)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_STATEMENT_POSITION)
#ifdef PG_DIAG_INTERNAL_POSITION
                      , PQresultErrorField(rio->pq->res, PG_DIAG_INTERNAL_QUERY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_INTERNAL_POSITION)
#endif
                      )
              < 0
            ) warn("subinfo(ResMess) asprintf");
    } else ifBLIN_QO1 {
        if  ( asprintf( &cx
                      , "%s %s %s\n%s\n%s\n%s"
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SEVERITY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SQLSTATE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_PRIMARY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_DETAIL)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_HINT)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_STATEMENT_POSITION)
                      )
              < 0
            ) warn("subinfo(ResMess) asprintf");
    } else ifBLIN_QO0 {
        if  ( asprintf( &cx
                      , "%s %s %s\n%s"
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SEVERITY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_SQLSTATE)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_PRIMARY)
                      , PQresultErrorField(rio->pq->res, PG_DIAG_MESSAGE_DETAIL)
                      )
              < 0
            ) warn("subinfo(ResMess) asprintf");
    }
    rio->pq->mess = cx;
out:
    return(rio->pq->mess);
#   undef blin_internal_flags
}

static char *
/**********************************************************************
 **                                                                  **/
getvalue(pgoblin_exenv *exenv, pgoblin_nr nio, int row, int col) {  /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rio  *rio;
    char         *c = NULL;

    GET_RIO(rio, exenv->options, nio);
    if  (!!rio->pq && !valinfo(exenv, nio, PGOBLIN_IsNull, row, col)) {
        c = PQgetvalue(rio->pq->res, row, col);
    }
out:
    return(c);
#   undef blin_internal_flags
}

static int
/***********************************************************************************
 **                                                                               **/
getstream(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, int quant) {     /**
 **                                                                               **
 ***********************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    BLIN_flag        cond = 0;
    pgoblin_db_conn *pcnn;
    pgoblin_realpq **rsv;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rou;
    PGresult       **res  = NULL;
    int              ex   = EX_OK;
    char            *q    = NULL;

    GET_CON(rdb, exenv->options, ndb);
    cond = rdb->flags & PGOBLIN_CR_MASK;
    pcnn = rdb->conn;
    GET_RIO(rou, exenv->options, nou);
    if  (!(rsv = &rou->pq)) {
        ifBLIN_QX0("no rsv");
        ERROUT(EX_OSERR, ENOMEM);
    }
    if  (!!*rsv) clear(exenv, nou);
    if  (!(*rsv = resv(rdb))) {
        ifBLIN_QX0("resv");
        goto out;
    }
    res = &(*rsv)->res;
    if  (!rdb) ERROUT(EX_USAGE, EINVAL);
    if  ((cond != PGOBLIN_CR_SSEL) && (cond != PGOBLIN_CR_BSEL)) {
        ifBLIN_QX0("Illegal cond %d", cond);
        ERROUT(EX_USAGE, EINVAL);
    }
    asprintf(&q, "FETCH %d FROM pgoblin;", quant);
    if  (!q) {
        ifBLIN_QW0("asprintf #4");
        ex = EX_OSERR;
        goto out;
    }
    MARK_R_CONN_GO(ndb);;
    ;;  rdb->flags &= ~PGOBLIN_CR_MASK;
    ;;  rdb->flags |= PGOBLIN_CR_TUPL;
    MARK_R_CONN_WENT(ndb);;
    if  (!(*res = PQexec(pcnn->pgcn, q))) {
        ifBLIN_QX0("PQexec result is absent: %s %s", q, PQerrorMessage(pcnn->pgcn));
        ERROUT(EX_SOFTWARE, ENOTRECOVERABLE);
    } else if(PGRES_COMMAND_OK != PQresultStatus(*res)) {
        ex = extatus(exenv->options->flags, *res, q, PGOBLIN_CR_OK, __func__, __LINE__);
        errno = EDOOFUS;
        goto out;
    }
    MARK_IO_PQ_GO(PGOBLIN_REGSIZE - 1);;
    ;;  clear(exenv, nou);
    MARK_IO_PQ_WENT(PGOBLIN_REGSIZE - 1);;
out:
    if  (!!q) free(q);
    MARK_R_CONN_GO(ndb);;
    ;;  rdb->flags &= ~PGOBLIN_CR_MASK;
    ;;  rdb->flags |= cond;
    MARK_R_CONN_WENT(ndb);;
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
getcopy(pgoblin_exenv *exenv, pgoblin_nr ndb, void **buf) {         /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    pgoblin_rdb     *rdb;
    int              ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, ndb);
    pcnn = rdb->conn;
    ex = PQgetCopyData(pcnn->pgcn, (char**)buf, 0);
out:
    return((ex < 0) ? (ex + 1) : ex);
#   undef blin_internal_flags
}

static int
/****************************************************************************
 **                                                                        **/
putcopy(pgoblin_exenv *exenv, pgoblin_nr ndb, const void *buf, int nby) { /**
 **                                                                        **
 ****************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    pgoblin_rdb     *rdb;
    int              ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, ndb);
    pcnn = rdb->conn;
    ex = PQputCopyData(pcnn->pgcn, buf, nby);
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
endstream(pgoblin_exenv *exenv, pgoblin_nr ndb) {                   /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    BLIN_flag        cond = 0;
    PGresult        *res;
    pgoblin_rdb     *rdb;
    int              ex   = -EX_SOFTWARE;
    ExecStatusType   st;

    GET_CON(rdb, exenv->options, ndb);
    ex = EX_OK;
    pcnn = rdb->conn;
    switch ((cond = rdb->flags & PGOBLIN_CR_MASK)) {
    case PGOBLIN_CR_IN:
        if  (PQputCopyEnd(pcnn->pgcn, NULL) < 1) {
            ifBLIN_QX0("endcopy failed: %s", PQerrorMessage(pcnn->pgcn));
            ERROUT(-EX_DATAERR, EINVAL);
        }
        /* FALLTHRU */
    case PGOBLIN_CR_OUT:
        while ((res = PQgetResult(pcnn->pgcn))) {
            st = PQresultStatus(res);
            ifBLIN_QX4("PQresultStatus %d %s", st, PQresStatus(st));
            if  (st != PGRES_COMMAND_OK) {
                ifBLIN_QX0("%s", PQresultErrorMessage(res));
                ex = -EX_SOFTWARE;
                break;
            }
            PQclear(res);
            res = NULL;
        }
        break;
    case PGOBLIN_CR_SSEL:
    case PGOBLIN_CR_BSEL:
        MARK_R_CONN_GO(ndb);;
        ;;  rdb->flags &= ~PGOBLIN_CR_MASK;
        ;;  rdb->flags |= PGOBLIN_CR_OK;
        MARK_R_CONN_WENT(ndb);;
        ifBLIN_QX4("PQexec %d=%s~", cond, "CLOSE pgoblin;");
        if  (!(res = PQexec(pcnn->pgcn, "CLOSE pgoblin;"))) {
            ifBLIN_QX0("PQexec result is absent: CLOSE pgoblin; %s", PQerrorMessage(pcnn->pgcn));
            ERROUT(-EX_SOFTWARE, ENOTRECOVERABLE);
        } else if(PGRES_COMMAND_OK != PQresultStatus(res)) {
            ex = extatus(exenv->options->flags, res, "CLOSE pgoblin;", PGOBLIN_CR_OK, __func__, __LINE__);
            errno = EDOOFUS;
            goto out;
        }
        PQclear(res);
        res = NULL;
        if  (!rdb->intran) {
            ifBLIN_QX4("PQexec %d=%s~", cond, "END;");
            if  (!(res = PQexec(pcnn->pgcn, "END;"))) {
                ifBLIN_QX0("PQexec result is absent: END; %s", PQerrorMessage(pcnn->pgcn));
                ERROUT(-EX_SOFTWARE, ENOTRECOVERABLE);
            } else if(PGRES_COMMAND_OK != PQresultStatus(res)) {
                ex = extatus(exenv->options->flags, res, "END;", PGOBLIN_CR_OK, __func__, __LINE__);
                errno = EDOOFUS;
                goto out;
            }
            PQclear(res);
            res = NULL;
        }
        break;
    default:
        ifBLIN_QX0("- %d", cond);
        ex = -EX_USAGE;
        errno = EINVAL;
    }
out:
    MARK_R_CONN_GO(ndb);;
    ;;  rdb->flags &= ~PGOBLIN_CR_MASK;
    ;;  rdb->flags |= cond;
    MARK_R_CONN_WENT(ndb);;
    if  (ex) errno = EINVAL;
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
tarry(pgoblin_exenv *exenv, pgoblin_nr ndb, int timeout) {          /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    int              sock;
    pgoblin_rdb     *rdb;
    struct pollfd    pfd;
    int              i = 0;

    GET_CON(rdb, exenv->options, ndb);
    pcnn = rdb->conn;
    if  (0 > (sock = PQsocket(pcnn->pgcn))) {
        ifBLIN_QX1("PQsocket");
        i = EX_UNAVAILABLE;
        errno = EINVAL;
    } else if (timeout != 0) {
        bzero(&pfd, sizeof(pfd));
        pfd.fd = sock;
        pfd.events = POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI;
        i = poll(&pfd, 1, timeout);
    }
    if  (0 > i) {
        ifBLIN_QW1("poll");
        i = EX_IOERR;
    } else if (!i) {
        ifBLIN_QX1("Nothing to consume");
    } else if (!PQconsumeInput(pcnn->pgcn)) {
        ifBLIN_QX1("PQconsumeInput");
        errno = 0;
        i = -EX_SOFTWARE;
    } else {
        i = 1;
    }
out:
    return(i);
#   undef blin_internal_flags
}

static void *
/**********************************************************************
 **                                                                  **/
notifies(pgoblin_exenv *exenv, pgoblin_nr ndb) {                    /**
 **                                                                  **
 **********************************************************************/
    pgoblin_db_conn    *pcnn;
    pgoblin_rdb        *rdb;
    pgoblin_longnotify *c    = NULL;
    PGnotify           *n;
    char               *o    = NULL;

    GET_CON(rdb, exenv->options, ndb);
    pcnn = rdb->conn;
    n = PQnotifies(pcnn->pgcn);
    if  (!!n) {
        errno = 0;
        c = calloc(1, sizeof(pgoblin_longnotify) + strlen(n->relname) + strlen(n->extra) + 3);
        if  (!c) {
            ifBLIN_QW0("For pgsql notifies");
            goto out;
        }
        c->majorlen = strlen(n->relname) + 1;
        c->minorlen = strlen(n->extra) + 1;
        c->ident = n->be_pid;
        o = stpncpy(c->tail, n->relname, c->majorlen);
        strncpy(++o, n->extra, c->minorlen);
        PQfreemem(n);
    }
out:
    return(c);
}

static int
/**********************************************************************
 **                                                                  **/
finish(pgoblin_exenv *exenv, pgoblin_nr ndb) {                      /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    pgoblin_rdb     *rdb;
    int              ex = EX_OK;

    GET_CON(rdb, exenv->options, ndb);
    if  (!!(pcnn = rdb->conn)) {
        PQfinish(pcnn->pgcn);
        if  (pcnn->tname) for (int i = 0; i < pcnn->numoi; ++i) free(pcnn->tname[i]);
        free(pcnn->toid);
        free(rdb->conn);
        rdb->conn = NULL;
    }
out:
    return(ex);
#   undef blin_internal_flags
}

static int
/*****************************************************************************************************
 **                                                                                                 **/
prexec(pgoblin_exenv *exenv, pgoblin_nr ndb, pgoblin_nr nou, pgoblin_nr nio, pgoblin_nr nin) {     /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_db_conn *pcnn;
    BLIN_flag        cond;
    char const      *kind;
    int              imdx;
    const char     **val  = NULL;
    PGresult        *res  = NULL;
    const char      *qry  = "";
    int             *len  = NULL;
    pgoblin_realpq **rsv  = NULL;
    pgoblin_rdb     *rdb;
    pgoblin_rio     *rin;
    pgoblin_rio     *rou;
    int              col;
    int              row;
    int              sz;
    int              ex   = 0;

    GET_CON(rdb, exenv->options, ndb);
    pcnn = rdb->conn;
    cond = rdb->flags & PGOBLIN_CR_MASK;
    GET_RIO(rou, exenv->options, nou);
    rsv = &rou->pq;
    if  (!rdb) {
        ERROUT(EX_USAGE, EINVAL);
    }
    if  ((PGOBLIN_CR_OK != cond) && (PGOBLIN_CR_TUPL != cond)) {
        ifBLIN_QX0("execute pgsql illegal cond=%d", cond);
        ERROUT(EX_USAGE, EINVAL);
    }
    GET_RIO(rin, exenv->options, nin);
    col = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Nfields);
    row = pgoblin_db_resinfo(exenv, nin, PGOBLIN_Ntuples);
    kind = "PQexecPrepared";
    if  (!(res = PQdescribePrepared(pcnn->pgcn, qry))) {
        ifBLIN_QX0("PQdescribePrepared is absent - fatal error %s", PQerrorMessage(pcnn->pgcn));
        ERROUT(EX_UNAVAILABLE, ENODEV);
    }
    imdx = PQnparams(res);
    PQclear(res);
    res = NULL;
    if  (col < imdx) sz = imdx; else sz = col;
    ifBLIN_QX4("%d rows x %d cols (%d parms) %d", row, col, imdx, sz);
    if  (!(val = malloc(sizeof(char *) * sz))) {
        ifBLIN_QW0("pgoblin_select #1: No mem");
        ex = EX_OSERR;
        goto out;
    }
    if  (!(len = malloc(sizeof(int) * sz))) {
        ifBLIN_QW0("pgoblin_select #2: No mem");
        free(val);
        val = NULL;
        ex = EX_OSERR;
        goto out;
    }
    for (int i = 0; i < row; i++) {
        for (int j = 0; j < col; j++) {
            val[j] = pgoblin_db_getvalue(exenv, nin, i, j);
            len[j] = (int)pgoblin_db_valinfo(exenv, nin, PGOBLIN_Length, i, j);
        }
        for (int j = col; j < imdx; j++) {
            val[j] = NULL;
            len[j] = -1;
        }
        ifBLIN_QX4("PQexecPrepared %d=%s~", cond, qry);
        if  (!(res = PQexecPrepared(pcnn->pgcn, qry, imdx, val, len, NULL, 0))) {
            ifBLIN_QX0("PQ result is absent - fatal error %s", PQerrorMessage(pcnn->pgcn));
            ERROUT(EX_UNAVAILABLE, ENODEV);
        }
        if  (!!(ex = extatus(rdb->flags, res, "PQexecPrepared", cond, __func__, __LINE__))) break;
        if  (i + 1 < row) {
            PQclear(res);
            res = NULL;
    }   }
out:
    if  (val) free(val);
    if  (len) free(len);
    if  (!!rsv) *rsv = NULL;
    if  (!!res) {
        if  (!rsv) {
            PQclear(res); /* XXXX */
            res = NULL;
        } else if (!(*rsv = resv(rdb))) {
            ifBLIN_QW0("resv");
            ex = EX_OSERR;
        } else {
            (*rsv)->res = res;
    }   }
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
denote(pgoblin_exenv *exenv, pgoblin_nr ndb, void **note) {         /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    int ex = EX_OK;

    if  (note && *note) {
        free(*note);
        *note = NULL;
    }
    return(ex);
#   undef blin_internal_flags
}

pgoblin_dbases PGOBLIN_MODULE =
{ ( PGOBLIN_DB_COPYIN
  | PGOBLIN_DB_COPYOUT
#ifdef PG_DIAG_INTERNAL_POSITION
  | PGOBLIN_DB_PREPARE
#endif
  | PGOBLIN_LONGNOTIFY
  )
, "3pGoblin-" VERS
, "pgsql\0" VERS
, 0
, init
, fini
, shurecon
, query
#ifdef PG_DIAG_INTERNAL_POSITION
, prepare
#else
, NULL
#endif
, execute
, resinfo
, subinfo
, erinfo
, valinfo
, getvalue
, getstream
, getcopy
, putcopy
, endstream
, tarry
, notifies
, finish
, clear
, NULL /* erconn */
, prexec
, denote
, NULL /* dummy1 */
, NULL /* dummy2 */
};
