/*-
 * Copyright (C)2018 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2018 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: db_mssql.c,v 1.11 2018/09/18 23:19:34 babolo Exp $"

#define BLIN_COMPAT      3
#define Bpars_COMPAT     3
#define MULAR_COMPAT     0
#define MIFE_COMPAT      4
#define RECOBE_COMPAT    4
#define PGOBLIN_COMPAT   2
#define PGOBLIN_INTERNAL 1
#define PGOBLIN_MODULE   pgoblin_module

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sybfront.h>
#include <sysexits.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <sybdb.h>
#include <err.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <babolo/recobe.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"

#define INDX(R, C) (R * rsv->ncol + C)
#define MULAR_FLAGS (MULAR_CHAR | MULAR_ZERO | MULAR_PFRE | MULAR_STRI | MULAR_UPPE)

struct pgoblin_mode {
    BLIN_flag         flags;
/*                    PGOBLIN_DB_TYPE                                                                 */
#   define            PGOBLIN_TINIT   0x010000
#   define            PGOBLIN_BUFF    0x040000
    int               ncol;
    mular_descriptor *md;
    char            **serv;
    char             *mess;
    int              *typs;
    int              *csize;
    int              *status;
    char            **buffer;
    u_char           *reco;
    int               nrow;
};

typedef struct pgoblin_db_conn {
    BLIN_flag        flags;
/*                   PGOBLIN_DB_TYPE                                                                    */
#   define           PGOBLIN_STREAMI 0x010000
#   define           PGOBLIN_STREAMT 0x020000
#   define           PGOBLIN_STREAME 0x040000
#   define           PGOBLIN_ALLOUT  (PGOBLIN_STREAMI | PGOBLIN_STREAMT | PGOBLIN_STREAME)
#   define           PGOBLIN_COPYIN  0x100000
#   define           PGOBLIN_COPYST  0x200000
#   define           PGOBLIN_COPYEN  0x400000
#   define           PGOBLIN_ALLIN   (PGOBLIN_COPYIN | PGOBLIN_COPYST | PGOBLIN_COPYEN)
    pgoblin_mode    *res;
    LOGINREC        *login;
    DBPROCESS       *dbproc;
} pgoblin_db_conn;

static size_t  mdp[] = {1024, 512, 512};
pgoblin_dbases PGOBLIN_MODULE;

static int
/**********************************************************************
 **                                                                  **/
clear(pgoblin_mode **rsv) {                                         /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags ((*rsv)->flags & BLIN_MASK)
    if  (!!rsv && *rsv) {
        if  (!!(*rsv)->md) mular_destroy((*rsv)->md);
        if  (!!(*rsv)->serv) free((*rsv)->serv);
        if  (!!(*rsv)->typs) free((*rsv)->typs);
        if  (!!(*rsv)->csize) free((*rsv)->csize);
        if  (!!(*rsv)->buffer) {
            for (int ncol = 0; ncol < (*rsv)->ncol; ++ncol) {
                if  (!!(*rsv)->buffer[ncol]) free((*rsv)->buffer[ncol]);
            }
            free((*rsv)->buffer);
        }
        if  (!!(*rsv)->status) free((*rsv)->status);
        if  (!!(*rsv)->reco) free((*rsv)->reco);
        free(*rsv);
        *rsv = NULL;
    }
    return(ex);
#   undef blin_internal_flags
}

static pgoblin_mode *
/**********************************************************************
 **                                                                  **/
resv(pgoblin_conn *db) {                                            /**
 **                                                                  **
 **********************************************************************/
    pgoblin_mode *rsv = NULL;

    if  (!(rsv = calloc(1, sizeof(pgoblin_mode)))) {
        ifBLIN_QW0("No mem");
    } else {
        rsv->flags = db->flags & (BLIN_MASK | PGOBLIN_DB_TYPE);
    }
    return(rsv);
}

static int
/**********************************************************************
 **                                                                  **/
collect(DBPROCESS *dbproc, pgoblin_mode *res, int tupl) {           /**
 **                                                                  **
 **********************************************************************/
    RETCODE           erc;
    int               ex   = EX_OK;

    if  (!!res) {
#       define blin_internal_flags (res->flags & BLIN_MASK)
        if  (!(res->flags & PGOBLIN_TINIT)) {
            res->ncol = dbnumcols(dbproc);
            res->nrow = 0;
            res->serv = NULL;
            res->flags |= PGOBLIN_TINIT;
            if  (!tupl) goto out;
            if  (0 < res->ncol) {
                const char *column_name;
                ssize_t     nmemlen = 0;
                char       *p;

                for (int ncol = 0; ncol < res->ncol; ++ncol) {
                    if  (!!(column_name = dbcolname(dbproc, ncol + 1))) {
                        ifBLIN_QX5("%d[%d]=%s~", ncol, strlen(column_name), column_name);
                        nmemlen += strlen(column_name) + 1;
                }   }
                ++nmemlen;
                ifBLIN_QX5("[%"BLIN_D"]", BLIN_I(nmemlen));
                if  (!(res->serv = malloc(res->ncol * sizeof(char *) + nmemlen))) {
                    ifBLIN_QX0("No mem %"BLIN_D, res->ncol * sizeof(char *) + nmemlen);
                    ERROUT(-EX_SOFTWARE, EINVAL);
                }
                if  (!(res->typs = malloc(res->ncol * sizeof(int)))) {
                    ifBLIN_QX0("No mem %"BLIN_D, res->ncol * sizeof(int));
                    ERROUT(-EX_SOFTWARE, EINVAL);
                }
                if  (!(res->csize = malloc(res->ncol * sizeof(int)))) {
                    ifBLIN_QX0("No mem %"BLIN_D, res->ncol * sizeof(int));
                    ERROUT(-EX_SOFTWARE, EINVAL);
                }
                p = (char *)&res->serv[res->ncol];
                for (int ncol = 0; ncol < res->ncol; ++ncol) {
                    res->typs[ncol] = dbcoltype(dbproc, ncol + 1);
                    ifBLIN_QX5("[%d][%"BLIN_D"]", ncol, BLIN_I(nmemlen));
                    if  (0 >= nmemlen) {
                        ifBLIN_QX0("strings %"BLIN_X" overlow on %d of %d", nmemlen, ncol, res->ncol);
                        break;
                    }
                    if  (!!(column_name = dbcolname(dbproc, ncol + 1))) {
                        ifBLIN_QX5("%d %"BLIN_X"[%"BLIN_D"]=%s~", ncol, p, nmemlen, column_name);
                        res->serv[ncol] = p;
                        if  (!(p = memccpy(p, column_name, 0, nmemlen))) {
                            ifBLIN_QX0("%s > %"BLIN_D, column_name, BLIN_I(nmemlen));
                        }
                        nmemlen -= p - res->serv[ncol];
                    } else {
                        res->serv[ncol] = NULL;
                    }
                    res->csize[ncol] = dbcollen(dbproc, ncol + 1);
                    if  (  (SYBCHAR != res->typs[ncol])
                        && (255 < (res->csize[ncol] = dbprcollen(dbproc, ncol + 1)))
                        ) {
                        res->csize[ncol] = 255;
                    }
                    ifBLIN_QX5( "%d (%d)[%d]=%s~"
                              , ncol
                              , res->typs[ncol]
                              , res->csize[ncol]
                              , res->serv[ncol]
                              );
                }
                if  (!(res->status = calloc(res->ncol, sizeof(int)))) {
                    ifBLIN_QX0("no RAM %d x %d", (int)res->ncol, (int)sizeof(int));
                    ERROUT(EX_OSERR, ENOMEM);
                }
                if  (!(res->buffer = calloc(res->ncol, sizeof(char*)))) {
                    ifBLIN_QX0("no RAM %d x %d", (int)res->ncol, (int)sizeof(char*));
                    ERROUT(EX_OSERR, ENOMEM);
                }
                for (int ncol = 0; ncol < res->ncol; ncol++) {
                    if  (!(res->buffer[ncol] = calloc(1, res->csize[ncol] + 1))){
                        ifBLIN_QX0("no RAM %d", res->csize[ncol] + 1);
                        ERROUT(EX_OSERR, ENOMEM);
                    }
                    erc = dbbind( dbproc
                                , ncol + 1
                                , NTBSTRINGBIND
                                , res->csize[ncol] + 1
                                , (BYTE*)res->buffer[ncol]
                                );
                    if  (FAIL == erc) {
                        ifBLIN_QX0("dbbind");
                        ERROUT(-EX_SOFTWARE, EINVAL);
                    }
                    if  (FAIL == (erc = dbnullbind(dbproc, ncol + 1, &res->status[ncol]))) {
                        ifBLIN_QX0("dbbind");
                        ERROUT(-EX_SOFTWARE, EINVAL);
    }   }   }   }   }
out:;
    res->reco = recobe_collection.domains[1]->reconv(recobe_collection.domains[1]->convbyname("wk"));
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
msg_handler( DBPROCESS *dbproc                                      /**/
/**/       , DBINT      msgno                                       /**/
/**/       , int        msgstate                                    /**/
/**/       , int        severity                                    /**/
/**/       , char      *msgtext                                     /**/
/**/       , char      *srvname                                     /**/
/**/       , char      *procname                                    /**/
/**/       , int        line                                        /**/
/**/       ) {                                                      /**
 **                                                                  **
 **********************************************************************/
    enum {changed_database = 5701, changed_language = 5703 };

    if  (0 < msgno) {
        ifBLIN_QX1("Msg %ld, Level %d, State %d", (long)msgno, severity, msgstate);
        if  (0 < strlen(srvname)) ifBLIN_QX0("Server '%s', ", srvname);
        if  (0 < strlen(procname)) ifBLIN_QX0("Procedure '%s', ", procname);
        if  (0 < line) ifBLIN_QX0("Line %d", line);
    }
    if  (10 < severity) {
        ifBLIN_QX0("%s: severity %d > 10, exiting", msgtext, severity);
        exit(severity);
    } else {
        ifBLIN_QX1("%s", msgtext);
    }
    return(0);
}

static int
/**********************************************************************
 **                                                                  **/
err_handler( DBPROCESS *dbproc                                      /**/
/**/       , int        severity                                    /**/
/**/       , int        dberr                                       /**/
/**/       , int        oserr                                       /**/
/**/       , char      *dberrstr                                    /**/
/**/       , char      *oserrstr                                    /**/
/**/       ) {	                                                    /**
 **                                                                  **
 **********************************************************************/
    if  (!!dberr) {
        ifBLIN_QX0("Msg %d, Level %d=%s", dberr, severity, dberrstr);
    } else {
        ifBLIN_QX0("DB-LIBRARY error: %s", dberrstr);
    }
    return(INT_CANCEL);
}
	
static int
/**********************************************************************
 **                                                                  **/
init(BLIN_flag flags) {                                             /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (flags & BLIN_MASK)
    PGOBLIN_MODULE.flags = (PGOBLIN_MODULE.flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE))
                           | (flags & (BLIN_MASK | PGOBLIN_DB_TYPE));
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
fini() {                                                            /**
 **                                                                  **
 **********************************************************************/
    return(0);
}

static int
/**********************************************************************
 **                                                                  **/
shurecon(pgoblin_conn *db) {                                        /**
 **                                                                  **
 **********************************************************************/
    pgoblin_db_conn  *conn;
    int               ex   = EX_OK;
#   define blin_internal_flags (db->flags & BLIN_MASK)
    ifBLIN_QX2("+ %08X", db->flags);
    if  (db->conn != NULL) {
        ifBLIN_QX1("NULL DB conn");
        goto out;
    }
    ifBLIN_QX2("= %d", (int)sizeof(pgoblin_db_conn));
    if  (!(conn = calloc(1, sizeof(pgoblin_db_conn)))) {
        ifBLIN_QX0("no MEM");
        ERROUT(EX_OSERR, ENOMEM);
    }
    conn->flags = PGOBLIN_MODULE.flags & (BLIN_MASK | PGOBLIN_DB_TYPE);
    MARK_R_CONN_GO(db);;
    ;;  db->conn = conn;
    ;;  db->flags = PGOBLIN_CLOSECONN | (db->flags & ~(BLIN_MASK | PGOBLIN_DB_TYPE)) | conn->flags;
    MARK_R_CONN_WENT(db);;
    if  (FAIL == dbinit()) {
	ifBLIN_QX0("dbinit");
    }
    dberrhandle(err_handler);
    dbmsghandle(msg_handler);
    if  (!(conn->login = dblogin())) {
        ifBLIN_QX0("dblogin");
        ERROUT(EX_SOFTWARE, EDOOFUS);
    }
    DBSETLUSER(conn->login, db->username);
    DBSETLPWD(conn->login, db->password);
    if  (!(conn->dbproc = dbopen(conn->login, db->host))) {
        ifBLIN_QX0("dbopen");
        ERROUT(EX_SOFTWARE, EDOOFUS);
    }
    if  (db->dbname && (FAIL == dbuse(conn->dbproc, db->dbname))) {
        ifBLIN_QX0("dbuse");
        ERROUT(EX_SOFTWARE, EDOOFUS);
    }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
query(pgoblin_conn *db, pgoblin_mode **rsv, char *qry)    {         /**
 **                                                                  **
 **********************************************************************/
    int               row_code;
    BLIN_flag         cond = db->flags & PGOBLIN_CR_MASK;
    pgoblin_db_conn  *conn = db->conn;
    RETCODE           erc;
    int               idx;
    pgoblin_mode     *res;
    int               ex   = EX_OK;
    const char       *q;
    char             *p;
    static const babolo_lexor copy = {(u_char*)
    "\000..." "...."    "  | "    "  .."    "...."    "...."    "...."    "...."
    " .."    "..."    "...."    ".-F."    "IIII"    "IIII"    "II.."    "...."
    ".EEC"    "EEEE"    "EEEE"    "EEEO"    "PEEE"    "EEEE"    "EYE."    "...E"
    ".EEC"    "EEEE"    "EEEE"    "EEEO"    "PEEE"    "EEEE"    "EYE."    "...."
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"    "EEEE"
    , NULL, Bpars_AABS | Bpars_CEND, 0x1C, 0x1C, 1
    ,{/*00*/ 3, ' ', 0x00, '|', 0x00, '-', 0x05, 'C', 0x09, 0xFF
     ,/*05*/ 0, '-', 0x07, 0xFF
     ,/*07*/ 0, '|', 0x00, 0x07
     ,/*09*/ 0, 'O', 0x0B, 0xFF
     ,/*0B*/ 0, 'P', 0x0D, 0xFF
     ,/*0D*/ 0, 'Y', 0x0F, 0xFF
     ,/*0F*/ 8, '', 0xFE, ' ', 0x0F, '|', 0x0F, '-', 0x19, 'C', 0xFD
              , 'O', 0xFD, 'P', 0xFD, 'Y', 0xFD, 'E', 0xFD, 0xFF
     ,/*19*/ 0, '-', 0x1A, 0xFF
     ,/*1A*/ 0, '|', 0x0F, 0x1A
    }};
    /*****************
        * 0 ...
        *   ' --\n'...
        * 1 'COPY "'
        * 1 'COPY --\n "'
        * 1 "COPY '"
        * 1 'COPY\n"'
        * 2 'COPY x'
        * 2 'COPY --\n x'
    *****************/

#   define blin_internal_flags (db->flags & BLIN_MASK)
    ifBLIN_QX2("+ %d=%s~", cond, qry);
    if  (conn->flags & (PGOBLIN_ALLOUT | PGOBLIN_ALLIN)) {
        ifBLIN_QX0("COPY state");
        ERROUT(EX_DATAERR, ENODEV);
    }
    res = NULL;
    q = qry;
    switch (cond) {
    case PGOBLIN_CR_OUT:
        switch (babolo_goword(&copy, &q)) {
        case 1:
            if  (!*q) {
                ERROUT(EX_DATAERR, EINVAL);
            }
            q--;
            if  (*q == '\'') {
                for (idx = 1; ;) {
                    if  (q[idx++] == '\'' && q[idx] != '\'') break;
                    if  (!q[idx]) {
                        ERROUT(EX_DATAERR, EINVAL);
                }   }
            } else if (*q == '\"') {
                for (idx = 1; ;) {
                    if  (q[idx++] == '\"' && q[idx] != '\"') break;
                    if  (!q[idx]) {
                        ERROUT(EX_DATAERR, EINVAL);
                }   }
            } else {
                ERROUT(EX_DATAERR, EINVAL);
            }
            idx++;
            ifBLIN_QX4("goword quoted %d", idx);
            goto p3;
        case 2:
            q--;
            for (idx = 1; ;) if ((copy.retable[q[idx++] & 0xFF] & 0xC0) != 0x40) break;
            ifBLIN_QX4("goword direct %d", idx);
        p3: if  (!(p = strndup(q, --idx))) {
                ifBLIN_QX0("no mem #5");
                ERROUT(EX_OSERR, ENOMEM);
            }
            if  (asprintf(&qry, "SELECT * FROM %s", p) < 0) {
                ifBLIN_QX0("no mem #6");
                ERROUT(EX_OSERR, ENOMEM);
            }
            ifBLIN_QX3("qry=%s~", qry);
        }
            /* FALLTHROUGH */
    case PGOBLIN_CR_SSEL:
        if  (FAIL == dbfcmd(conn->dbproc, qry)) {
            ifBLIN_QX0("dbfcmd=%s~", qry);
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (FAIL == dbsqlexec(conn->dbproc)) {
            ifBLIN_QX0("dbsqlexec=%s~", qry);
            ERROUT(EX_DATAERR, EDOOFUS);
        }
        if  (NO_MORE_RESULTS != (erc = dbresults(conn->dbproc))) {
            if  (FAIL == erc) {
                ifBLIN_QX0("dbresults");
                ERROUT(EX_DATAERR, EDOOFUS);
            }
            if  (!!res) clear(&res);
            if  (!(res = resv(db))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            if  (0 > (collect(conn->dbproc, res, 1))) {
                ifBLIN_QX0("collect");
                goto out;
        }   }
        conn->flags |= PGOBLIN_STREAMI;
        break;
    case PGOBLIN_CR_OK:
        if  (FAIL == dbfcmd(conn->dbproc, qry)) {
            ifBLIN_QX0("dbfcmd=%s~", qry);
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (FAIL == dbsqlexec(conn->dbproc)) {
            ifBLIN_QX0("dbsqlexec=%s~", qry);
            ERROUT(EX_DATAERR, EDOOFUS);
        }
        for (int i = 0; NO_MORE_RESULTS != (erc = dbresults(conn->dbproc)); ++i) {
            if  (FAIL == erc) {
                ifBLIN_QX0("dbresults %d", i);
                ERROUT(EX_DATAERR, EDOOFUS);
            }
            if  (!!res) clear(&res);
            if  (!(res = resv(db))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            if  (0 > (collect(conn->dbproc, res, 0))) {
                ifBLIN_QX0("collect");
                goto out;
            }
            for (int j = 0; NO_MORE_ROWS != (row_code = dbnextrow(conn->dbproc)); ++j) {
                switch (row_code) {
                case REG_ROW:
                    break;
                case BUF_FULL:
                    ifBLIN_QX0("BUF_FULL res=%d row=%d", i, j);
                    break;
                case FAIL:
                    ifBLIN_QX0("FAIL res=%d row=%d", i, j);
                    ERROUT(-EX_SOFTWARE, EINVAL);
                default:
                    ifBLIN_QX0("%d ignored, res=%d row=%d", i, j);
                    break;
            }   }
            if  (0 > DBCOUNT(conn->dbproc)) {
                ifBLIN_QX1("DBCOUNT");
            } else if (res->nrow != DBCOUNT(conn->dbproc)) {
                ifBLIN_QX0("row %d <> DBCOUNT %d", res->nrow, DBCOUNT(conn->dbproc));
            }
            if  (TRUE == dbhasretstat(conn->dbproc)) {
                ifBLIN_QX1("Status returned %d", dbretstatus(conn->dbproc));
        }   }
        break;
    case PGOBLIN_CR_TUPL:
        if  (FAIL == dbfcmd(conn->dbproc, qry)) {
            ifBLIN_QX0("dbfcmd=%s~", qry);
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (FAIL == dbsqlexec(conn->dbproc)) {
            ifBLIN_QX0("dbsqlexec=%s~", qry);
            ERROUT(EX_DATAERR, EDOOFUS);
        }
        for (int i = 0; NO_MORE_RESULTS != (erc = dbresults(conn->dbproc)); ++i) {
            int          ncol;

            if  (FAIL == erc) {
                ifBLIN_QX0("dbresults %d", i);
                ERROUT(EX_DATAERR, EDOOFUS);
            }
            if  (!!res) clear(&res);
            if  (!(res = resv(db))) {
                ifBLIN_QX0("no RAM");
                ERROUT(EX_OSERR, ENOMEM);
            }
            if  (0 > (collect(conn->dbproc, res, 1))) {
                ifBLIN_QX0("collect");
                goto out;
            }
            if  (!res->md) { /* XXXX */
                res->md = mular_create( MULAR_FLAGS, 3, sizeof(char*), mdp);
                res->md->tofree = free;
            }
            for (int j = 0; NO_MORE_ROWS != (row_code = dbnextrow(conn->dbproc)); ++j) {
                switch (row_code) {
                case REG_ROW:
                    for (ncol = 0; ncol < res->ncol; ++ncol) {
                        char *t;

                        if  (0 > res->status[ncol]) {
                            ifBLIN_QX5("%d x %d NULL", res->nrow, ncol);
                            *(char**)(mular_add(res->md)) = NULL;
                        } else if (0 < res->status[ncol]) {
                            ifBLIN_QX0("%d x %d over %d", res->nrow, ncol, res->status[ncol]);
                            ERROUT(EX_SOFTWARE, ENOMEM);
                        } else if (!(t = strdup(res->buffer[ncol]))) {
                            ifBLIN_QX0("no RAM");
                            ERROUT(EX_OSERR, ENOMEM);
                        } else {
                            ifBLIN_QX5("%d x %d =%s~", res->nrow, ncol, t);
                            *(char**)(mular_add(res->md)) = t;
                    }   }
                    res->nrow += 1;
                    break;
                case BUF_FULL:
                    ifBLIN_QX0("BUF_FULL res=%d row=%d", i, j);
                    break;
                case FAIL:
                    ifBLIN_QX0("FAIL res=%d row=%d", i, j);
                    ERROUT(-EX_SOFTWARE, EINVAL);
                default:
                    ifBLIN_QX0("%d ignored, res=%d row=%d", ncol, i, j);
                    break;
            }   }
            for (ncol = 0; ncol < res->ncol; ncol++) {
                free(res->buffer[ncol]);
            }
            free(res->buffer);
            free(res->status);
            if  (0 > DBCOUNT(conn->dbproc)) {
                ifBLIN_QX1("DBCOUNT");
            } else if (res->nrow != DBCOUNT(conn->dbproc)) {
                ifBLIN_QX0("row %d <> DBCOUNT %d", res->nrow, DBCOUNT(conn->dbproc));
            }
            if  (TRUE == dbhasretstat(conn->dbproc)) {
                ifBLIN_QX1("Status returned %d", dbretstatus(conn->dbproc));
        }   }
        break;
    case PGOBLIN_CR_BSEL:
    case PGOBLIN_CR_BTUPL:
    default:
        ifBLIN_QX0("illegal cond=%d", cond);
        ex = EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
out:
    if  (rsv) *rsv = res; else if (res) clear(&res);
    conn->res = res;
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
getstream(pgoblin_conn *db, pgoblin_mode **rsv, int quant) {        /**
 **                                                                  **
 **********************************************************************/
    int               row_code;
    pgoblin_db_conn  *conn = db->conn;
    int               ex   = EX_OK;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    ifBLIN_QX2("+ %08X %d", conn->flags, quant);
    if  (!(conn->flags & PGOBLIN_STREAMI)) {
        ifBLIN_QX0("not COPYout state");
        ERROUT(-EX_DATAERR, ENODEV);
    }
    if  (!rsv) {
        ifBLIN_QX0("no res");
        ERROUT(EX_USAGE, EINVAL);
    }
    if  (!*rsv) {
        conn->flags |= PGOBLIN_STREAME;
        goto out;
    }
    (*rsv)->flags &= ~PGOBLIN_BUFF;
    (*rsv)->nrow = 0;
    if  (NO_MORE_ROWS != (row_code = dbnextrow(conn->dbproc))) {
        switch (row_code) {
        case REG_ROW:
            ifBLIN_QX3("row_code %d", row_code);
            (*rsv)->nrow = 1;
            (*rsv)->flags |= PGOBLIN_BUFF;
            break;
        case BUF_FULL:
            ifBLIN_QX0("BUF_FULL");
            break;
        case FAIL:
            ifBLIN_QX0("FAIL");
            ERROUT(-EX_SOFTWARE, EINVAL);
        default:
            ifBLIN_QX0("ignored");
            break;
       }
    } else {
        if  (conn->flags & PGOBLIN_STREAMT) conn->flags |= PGOBLIN_STREAME;
        ex = 0;
    }
    if  (!(*rsv)->nrow) conn->flags |= PGOBLIN_STREAME;
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
getcopy(pgoblin_conn *db, void **buf) {                             /**
 **                                                                  **
 **********************************************************************/
    int               row_code;
    pgoblin_db_conn  *conn = db->conn;
    int               ex   = EX_OK;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    ifBLIN_QX2("+ %08X %08X %"BLIN_X, db->flags, conn->flags, buf);
    if  (!(conn->flags & PGOBLIN_STREAMI)) {
        ifBLIN_QX0("not COPYout state");
        ERROUT(-EX_DATAERR, ENODEV);
    }
    if  (!conn->res) {
        conn->flags |= PGOBLIN_STREAME;
        ifBLIN_QX3("res out %08X", conn->flags);
        goto out;
    }
    conn->res->nrow = 0;
    if  (NO_MORE_ROWS != (row_code = dbnextrow(conn->dbproc))) {
        int          ncol;
        char        *c;
        int          l;
        int          k;

        switch (row_code) {
        case REG_ROW:
            ifBLIN_QX3("row_code %d", row_code);
            for (ncol = 0, l = 2; ncol < conn->res->ncol; ++ncol) {
                if  (!conn->res->status[ncol]) {
                    l += strlen(conn->res->buffer[ncol]) * 2 + 1;
                } else {
                    l += 3;
                    if  (0 < conn->res->status[ncol]) {
                        ifBLIN_QX0( "col %d: real %d > buff %d"
                                  , ncol
                                  , conn->res->status[ncol]
                                  , conn->res->csize[ncol]
                                  );
                        ERROUT(-EX_SOFTWARE, ENOMEM);
                }   }
                ifBLIN_QX5("[%d]l %d", ncol, l);
            }
            ifBLIN_QX4("Want malloc %d", l);
            if  (!(*buf = malloc(l))) {
                ifBLIN_QX0("no RAM");
                ERROUT(-EX_OSERR, ENOMEM);
            }
            k = 0;
            for (ncol = 0; ncol < conn->res->ncol; ++ncol) {
                ifBLIN_QO5 {
                    ifBLIN_QX5( "%d[%d]=%s~ %d[%d,%d]%d"
                              , conn->res->status[ncol]
                              , ncol
                              , dbcolname(conn->dbproc, ncol + 1)
                              , dbcoltype(conn->dbproc, ncol + 1)
                              , dbcollen(conn->dbproc, ncol + 1)
                              , dbprcollen(conn->dbproc, ncol + 1)
                              , dbdatlen(conn->dbproc, ncol + 1)
                              );
                    blin_dumb( 5
                             , dbdata(conn->dbproc, ncol + 1)
                             , dbdatlen(conn->dbproc, ncol + 1)
                             );
                }
                if  (!!conn->res->status[ncol] || !(c = conn->res->buffer[ncol])) {
                    (*(char**)buf)[k++]  = '\\';
                    (*(char**)buf)[k++]  = 'N';
                } else {
                    ifBLIN_QX3("col[%d]=%s~", ncol, c);
                    for (int i = 0; !!c[i]; ++i) {
                        switch(c[i]) {
                        case 0x08: (*(char**)buf)[k++] = '\\'; (*(char**)buf)[k++] = 'b' ; break;
                        case 0x09: (*(char**)buf)[k++] = '\\'; (*(char**)buf)[k++] = 't' ; break;
                        case 0x0A: (*(char**)buf)[k++] = '\\'; (*(char**)buf)[k++] = 'n' ; break;
                        case 0x0B: (*(char**)buf)[k++] = '\\'; (*(char**)buf)[k++] = 'v' ; break;
                        case 0x0C: (*(char**)buf)[k++] = '\\'; (*(char**)buf)[k++] = 'f' ; break;
                        case 0x0D: (*(char**)buf)[k++] = '\\'; (*(char**)buf)[k++] = 'r' ; break;
                        case '\\': (*(char**)buf)[k++] = '\\'; (*(char**)buf)[k++] = '\\'; break;
                        default:
                            if  (!conn->res->reco) {
                                (*(char**)buf)[k++] = c[i];
                            } else {
                                (*(char**)buf)[k++] = conn->res->reco[c[i] & 0x0FF];
                }   }   }   }
                (*(char**)buf)[k++] = (ncol < (conn->res->ncol - 1)) ? '\t' : '\n';
                (*(char**)buf)[k] = 0;
                ifBLIN_QX5("[%d]k %d", ncol, k);
            }
            conn->res->nrow = 1;
            ex = k;
            break;
        case BUF_FULL:
            ifBLIN_QX0("BUF_FULL");
            break;
        case FAIL:
            ifBLIN_QX0("FAIL");
            ERROUT(-EX_SOFTWARE, EINVAL);
        default:
            ifBLIN_QX0("ignored");
            break;
        }
        dbclrbuf(conn->dbproc, 999);
    } else {
        if  (conn->flags & PGOBLIN_STREAMT) conn->flags |= PGOBLIN_STREAME;
        ex = 0;
        ifBLIN_QX3("row end %08X", conn->flags);
    }
    if  (!conn->res->nrow) conn->flags |= PGOBLIN_STREAME;
out:
    ifBLIN_QX2("- %d %08X", ex, conn->flags);
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
endstream(pgoblin_conn *db) {                                       /**
 **                                                                  **
 **********************************************************************/
    BLIN_flag        cond = db->flags & PGOBLIN_CR_MASK;
    pgoblin_db_conn *conn = db->conn;
    int              ex   = 0;

#   define blin_internal_flags (db->flags & BLIN_MASK)
    switch (cond) {
    case PGOBLIN_CR_SSEL:
    case PGOBLIN_CR_OUT:
        if  (!(conn->flags & PGOBLIN_STREAMI)) {
            ifBLIN_QX0("not stream state");
            ERROUT(-EX_DATAERR, ENODEV);
        }
        if  (0 > DBCOUNT(conn->dbproc)) {
            ifBLIN_QX1("DBCOUNT");
        } else {
            ifBLIN_QX3("row %d", DBCOUNT(conn->dbproc));
        }
        if  (TRUE == dbhasretstat(conn->dbproc)) {
            ifBLIN_QX1("Status returned %d", dbretstatus(conn->dbproc));
        }
        break;
    case PGOBLIN_CR_IN:
        if  (!(conn->flags & PGOBLIN_COPYIN)) {
            ifBLIN_QX0("not COPYin state");
            ERROUT(-EX_DATAERR, ENODEV);
        }
        if  (0 > DBCOUNT(conn->dbproc)) {
            ifBLIN_QX1("DBCOUNT");
        } else {
            ifBLIN_QX2("row %d", DBCOUNT(conn->dbproc));
        }
        if  (TRUE == dbhasretstat(conn->dbproc)) {
            ifBLIN_QX1("Status returned %d", dbretstatus(conn->dbproc));
        }
        break;
    default:
        ifBLIN_QX0("- %d", cond);
        ex = -EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
out:
    conn->flags = 0;
    return(ex);
#   undef blin_internal_flags
}

static int
/**********************************************************************
 **                                                                  **/
resinfo(int kind, pgoblin_mode *rsv) {                              /**
 **                                                                  **
 **********************************************************************/
    int ex;

    switch (kind) {
    case PGOBLIN_Ntuples: ex = rsv ? rsv->nrow : 0; break;
    case PGOBLIN_Nfields: ex = rsv ? rsv->ncol : 0; break;
    default             :
        ex = -EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
    return(ex);
}

static const char *
/**********************************************************************
 **                                                                  **/
subinfo(int kind, pgoblin_mode *rsv, int sub) {                     /**
 **                                                                  **
 **********************************************************************/
    const char *cx = NULL;

    switch (kind) {
    case PGOBLIN_ColName:
        if  (!!rsv->serv && (rsv->ncol > sub)) cx = rsv->serv[sub];
        break;
    default:;
    }
    return(cx);
}

static char *
/**********************************************************************
 **                                                                  **/
erinfo(pgoblin_mode *rsv) {                                         /**
 **                                                                  **
 **********************************************************************/
    char *cx = NULL;

    return(cx);
}

static int
/**********************************************************************
 **                                                                  **/
valinfo(int kind, pgoblin_mode *rsv, int row, int col) {            /**
 **                                                                  **
 **********************************************************************/
    int ex;

#   define blin_internal_flags (rsv->flags & BLIN_MASK)
    switch (kind) {
    case PGOBLIN_IsNull:
        if  (rsv) {
            if  ((row < 0) || (row >= rsv->nrow) || (col < 0) || (col >= rsv->ncol)) {
                ifBLIN_QX1( "+ row %d %s in [0..%d], col %d %s in [0..%d]"
                          , row, (row >= 0 && row < rsv->nrow) ? "" : "NOT", rsv->nrow
                          , col, (col >= 0 && col < rsv->ncol) ? "" : "NOT", rsv->ncol
                          );
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            } else if (!!(rsv->flags & PGOBLIN_BUFF)) {
                if  (0 < rsv->status[INDX(row, col)]) {
                    ex = -EX_SOFTWARE;
                    errno = ENOMEM;
                } else {
                    ex = !!rsv->status[INDX(row, col)];
                }
            } else if (!!rsv->md) {
                ex = !*(char**)mular_getix(rsv->md, INDX(row, col));
            } else {
                ifBLIN_QX0("NOTHING instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            }
        } else if (row == 0 && col == 0) {
            ex = 1;
        } else {
            ifBLIN_QX1("row %d col %d in emty", row, col);
            ex = -EX_SOFTWARE;
            errno = EDOOFUS;
        }
        break;
    case PGOBLIN_Length:
        switch (valinfo(PGOBLIN_IsNull, rsv, row, col)) {
        case  1:
            ex = 0;
            break;
        case  0:
            if  (!!(rsv->flags & PGOBLIN_BUFF)) {
                ex = strlen(rsv->buffer[INDX(row, col)]);
            } else if (!!rsv->md) {
                ex = strlen(*(char**)mular_getix(rsv->md, INDX(row, col)));
            } else {
                ifBLIN_QX0("NOTHING instead of MULAR");
                ex = -EX_SOFTWARE;
                errno = EDOOFUS;
            }
            break;
        default:
            ex = -EX_SOFTWARE;
            errno = EDOOFUS;
        }
        break;
    default:
        ex = -EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static char *
/**********************************************************************
 **                                                                  **/
getvalue(pgoblin_mode *rsv, int row, int col) {                     /**
 **                                                                  **
 **********************************************************************/
    char *c = NULL;

    if  (!valinfo(PGOBLIN_IsNull, rsv, row, col)) {
        if  (!!(rsv->flags & PGOBLIN_BUFF)) {
            c = rsv->buffer[INDX(row, col)];
            ifBLIN_QX4("buff=%s~", c);
        } else if (!!rsv->md) {
            c = *(char**)mular_getix(rsv->md, INDX(row, col));
            ifBLIN_QX4("mular=%s~", c);
        } else {
            ifBLIN_QX0("NOTHING instead of MULAR");
            errno = EDOOFUS;
    }   }
    return(c);
}

static int
/**********************************************************************
 **                                                                  **/
finish(u_int32_t flags, int kind, void **entity) {                  /**
 **                                                                  **
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (flags & BLIN_MASK)
    switch (kind) {
    case PGOBLIN_Connect: {
            pgoblin_db_conn **conn = (pgoblin_db_conn **)entity;

            if  (conn && *conn) {
                if  (!!(*conn)->dbproc) {
                    dbclose((*conn)->dbproc);
                    (*conn)->dbproc = NULL;
                }
                free(*conn);
                *conn = NULL;
        }   }
        break;
    case PGOBLIN_FrCOPY:
        if  (entity && *entity) {
            free(*entity);
            *entity = NULL;
        }
        break;
    case PGOBLIN_FrNote:
        break;
    default:
        ex = EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
    return(ex);
#   undef blin_internal_flags
}

pgoblin_dbases PGOBLIN_MODULE =
{ 0
, "3pGoblin-" VERS
, "pgsql\0" VERS
, 0
, init
, fini
, shurecon
, query
, NULL     /* prepare */
, NULL     /* execute */
, resinfo
, subinfo
, erinfo
, valinfo
, getvalue
, getstream
, getcopy
, NULL     /* putcopy */
, endstream
, NULL     /* tarry */
, NULL     /* notifies */
, finish
, clear
, NULL
, NULL
};
