/*-
 * Copyright (C)2002..2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2002..2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: vdump.c,v 1.64 2024/07/14 00:00:02 babolo Exp $"

#define BLIN_COMPAT      4
#define Bpars_COMPAT     4
#define MULAR_COMPAT     0
#define MIFE_COMPAT      5
#define PGOBLIN_COMPAT   5
#define PGOBLIN_INTERNAL 1

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <limits.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <err.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"

const char pgoblin_regn[] =
"0123456789"
"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
"abcdefghijklmnopqrstuvwxyz"
"_-"
;

static char pm[12] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

char *
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_parmask(int n, u_int ch) {                                  /** 
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    int i = !n ? 0 : 6;

    if  (PGOBLIN_OUTSET   & ch) pm[i++] = 'o';
    if  (PGOBLIN_PQRESULT & ch) pm[i++] = 'q';
    if  (PGOBLIN_MIFEDES  & ch) pm[i++] = 'm';
    if  (PGOBLIN_TEXTPARM & ch) pm[i++] = 't';
    if  (PGOBLIN_PATHPARM & ch) pm[i++] = 'f';
    pm[i++] = '\0';
    return(&pm[!n ? 0 : 6]);
}

void
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_Vdump(pgoblin_main *options, pgoblin_prog *pgm) {           /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_tuple *command;
    u_int32_t      cmd;
    char          *lit;
    size_t         p;
    int            i;
    u_int32_t      t;

#   define blin_internal_flags (options->flags & BLIN_MASK)
    if  (pgm) {
        ifBLIN_QX3("+ pgm=%"BLIN_D, MULAR_NEXT(pgm->code));
    } else {
        ifBLIN_QX3("+ pgm=NULL");
    }
    printf("+######################################");
    for (p = 0; p < PGOBLIN_STRING_ID_LEN && pgm->id[p]; p++) printf("%c", pgm->id[p]);
    for (p = 0; p < MULAR_NEXT(pgm->code); p++) {
        if  (!(command = mular_getix(pgm->code, p))) {
            ifBLIN_QW0("mular_getix %zu", p);
            goto out;
        }
        t = command->cmd;
        cmd = t & PGOBLIN_COMMAND;
        if  (cmd == PGOBLIN_extended) {
            cmd = (command->ecmd & PGOBLIN_COMMAND) | PGOBLIN_EXTENDED;
            lit = NULL;
        } else {
            lit = command->lit;
        }
        if  (cmd >= pgoblin.max && !(~pgoblin.a[cmd].flags & PGOBLIN_ILLEGAL)) {
            printf("\n#ILLEGAL ");
        } else {
            printf("\n#%s ", pgoblin.a[cmd].str);
        }
        t >>= 2;
        for (i = 0; i < pgoblin.binparam; i++)
            printf("%c", pgoblin_regn[(t >>= PGOBLIN_BPARG) & PGOBLIN_ARGMASK]);
        printf(" ([%d]=%03X)", (int)p, cmd);
        if  (lit) printf("\n%s", command->lit);
    }
out:
    printf("\n-######################################\n");
    ifBLIN_QX3("-");
#   undef blin_internal_flags
};

void
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_VVdump() {                                                  /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    u_int j;
    int i;

    printf( "SYMREGS=%d BINREGS=%d PGOBLIN_max=%03X\n"
          , pgoblin.symparam, pgoblin.binparam, pgoblin.max
          );
    for (i = 0; i < pgoblin.binparam; i++)
        printf("ARG%dS=%d%s", i + 1, pgoblin.binparams[i], (i + 1 < pgoblin.binparam) ? " " : "\n");
    printf("sym val bin mask name\n");
    for (i = 0; i < pgoblin.symparam; i++)
        printf( " %02X  %02X %s%2d %4d %s\n"
              , pgoblin.p[i].sym
              , pgoblin.p[i].val
              , (pgoblin.p[i].bin < 0) ? "~" : " "
              , (pgoblin.p[i].bin < 0) ? ~pgoblin.p[i].bin : pgoblin.p[i].bin
              , pgoblin.p[i].mask
              , pgoblin.p[i].name
              );
    printf("      flags     car   str\n");
    for (j = 0; j < pgoblin.max; ++j) printf( "%03X %08X %" BLIN_X "%s%s%s\n"
                                            , j
                                            , pgoblin.a[j].flags
                                            , BLIN_I(pgoblin.a[j].car)
                                            , (pgoblin.a[j].str) ? " =" : ""
                                            , (pgoblin.a[j].str) ? pgoblin.a[j].str : ""
                                            , (pgoblin.a[j].str) ? "~" : ""
                                            );
}

void
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_verify(pgoblin_main *options) {                             /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_command cmd;
    u_int32_t       m  ;
    int             b  ;

    for (u_int j = 0; j <= (PGOBLIN_ANYIVL | PGOBLIN_OUTSET); j++) {
        printf( "%c%s%s%s%s%s\n"
              , pgoblin_regn[j]
              , (j & PGOBLIN_PATHPARM) ? " path"  : ""
              , (j & PGOBLIN_TEXTPARM) ? " text"  : ""
              , (j & PGOBLIN_MIFEDES ) ? " mife"  : ""
              , (j & PGOBLIN_PQRESULT) ? " pq"    : ""
              , (j & PGOBLIN_OUTSET  ) ? " out"   : ""
              );
    }
    if  (pgoblin.symparam != PGOBLIN_SYMREGS) {
        printf("!!! pgoblin.symparam=%d <> SYMREGS=%d\n", pgoblin.symparam, PGOBLIN_SYMREGS);
    }
    if  (pgoblin.binparam != PGOBLIN_BINREGS) {
        printf("!!! pgoblin.binparam=%d <> BINREGS=%d\n", pgoblin.binparam, PGOBLIN_BINREGS);
    }
    if  (pgoblin.regsize != PGOBLIN_REGSIZE) {
        printf("!!! pgoblin.regsize=%d <> REGSIZE=%d\n", pgoblin.regsize, PGOBLIN_REGSIZE);
    }
    if  (pgoblin.max != PGOBLIN_max) {
        printf("!!! pgoblin.max=%d <> PGOBLIN_max=%d\n", pgoblin.max, PGOBLIN_max);
    }
    if  (pgoblin.binparams[0] != PGOBLIN_ARG1S) {
        printf( "!!! pgoblin.binparams[0]=%d <> PGOBLIN_ARG1S=%d\n"
              , pgoblin.binparams[0], PGOBLIN_ARG1S
              );
    }
    if  (pgoblin.binparams[1] != PGOBLIN_ARG2S) {
        printf( "!!! pgoblin.binparams[1]=%d <> PGOBLIN_ARG2S=%d\n"
              , pgoblin.binparams[1], PGOBLIN_ARG2S
              );
    }
    if  (pgoblin.binparams[2] != PGOBLIN_ARG3S) {
        printf( "!!! pgoblin.binparams[2]=%d <> PGOBLIN_ARG3S=%d\n"
              , pgoblin.binparams[2], PGOBLIN_ARG3S
              );
    }
    if  (pgoblin.binparams[3] != PGOBLIN_ARG4S) {
        printf( "!!! pgoblin.binparams[3]=%d <> PGOBLIN_ARG4S=%d\n"
              , pgoblin.binparams[3], PGOBLIN_ARG4S
              );
    }
    for (pgoblin_nr j = 1; j < pgoblin.binparam; j++) { /*    */
        if  (pgoblin.binparams[j - 1] + PGOBLIN_BPARG != pgoblin.binparams[j]) {
            printf( "!!! pgoblin.binparams[%d]=%d + %d <> pgoblin.binparams[%d]=%d\n"
                  , j - 1
                  , pgoblin.binparams[j - 1]
                  , PGOBLIN_BPARG
                  , j
                  , pgoblin.binparams[j]
                  );
    }   }
    for (pgoblin_nr j = 0; j < pgoblin.regsize; j++) {
        if  (pgoblin_ctonr(pgoblin_regn[j]) != j) {
            printf( "reg[%d <> %d] name %c\n"
                  , j
                  , pgoblin_ctonr(pgoblin_regn[j])
                  , pgoblin_regn[j]
                  );
    }   }
    for (u_int j = 0; j < 256; j++) {
        if  ((pgoblin_ctonr((char)j) >= 0) && (pgoblin_regn[pgoblin_ctonr((char)j)] != (char)j)) {
            printf( "name %c <> %c reg[%d]\n"
                  , pgoblin_regn[pgoblin_ctonr((char)j)]
                  , j
                  , pgoblin_ctonr((char)j)
                  );
    }   }
    for (cmd = 1; cmd < pgoblin.max; cmd++) {
        m = 0;
        if  (!(~pgoblin.a[cmd].flags & PGOBLIN_ILLEGAL)) {
            if  (pgoblin.a[cmd].car) printf("!executor for illegal command %03X\n", cmd);
            if  (pgoblin.a[cmd].str) {
                printf("!name=%s~ for illegal command %03X\n", pgoblin.a[cmd].str, cmd);
            }
        } else {
            if  (!pgoblin.a[cmd].str) {
                printf("!! No name for legal command %03X\n", cmd);
            } else if  (babolo_testword(pgoblin.cmds, pgoblin.a[cmd].str) != cmd) {
                printf( "!!! cmd=%03X: %s=%03X\n"
                      , cmd
                      , pgoblin.a[cmd].str
                      , babolo_testword(pgoblin.cmds, pgoblin.a[cmd].str)
                      );
            }
            if  (pgoblin.a[cmd].flags & PGOBLIN_IGNORE) {
                if  (pgoblin.a[cmd].car) printf("!executor for ignored command %03X\n", cmd);
                if  (pgoblin.a[cmd].str) {
                    printf("!name=%s~ for ignored command %03X\n", pgoblin.a[cmd].str, cmd);
                }
            } else if (pgoblin.a[cmd].flags && !pgoblin.a[cmd].car) {
                printf("!! No executor for legal command %03X\n", cmd);
            }
            if  (!!(pgoblin.a[cmd].flags & PGOBLIN_NOLITER) && !(cmd & PGOBLIN_EXTENDED)) {
                printf("!short command %03X without literal\n", cmd);
                if  (pgoblin.a[cmd].str) {
                    printf("!short command without literal name=%s~\n", pgoblin.a[cmd].str);
            }   }
            if  (!(pgoblin.a[cmd].flags & PGOBLIN_NOLITER) && !!(cmd & PGOBLIN_EXTENDED)) {
                printf("!extended command %03X with literal\n", cmd);
                if  (pgoblin.a[cmd].str) {
                    printf("!extended command with literal name=%s~\n", pgoblin.a[cmd].str);
        }   }   }
        for (pgoblin_nr j = 0; j < pgoblin.symparam; j++) {
            if  (0 <= (b = pgoblin_breg(cmd, j))) {
                if  (!pgoblin.a[cmd].ch[b]) {
                    printf("!! cmd=%03X: a[cmd].flags <#> a[cmd].ch[%df=>%dl)]\n", cmd, b, j);
                    if  (pgoblin.a[cmd].str) printf("!! name=%s~\n", pgoblin.a[cmd].str);
                    ifBLIN_QX1( "pgoblin.a[%03X].flags=%08X pgoblin.p[%d].sym=%02X, bin=%02X,"
                                " mask=%02X breg(%03X,%d)=%d pgoblin.a[%03X].ch[%d]=%02X"
                              , cmd
                              , pgoblin.a[cmd].flags
                              , j
                              , pgoblin.p[j].sym
                              , pgoblin.p[j].bin & 0xFF
                              , pgoblin.p[j].mask
                              , cmd
                              , j
                              , b
                              , cmd
                              , b
                              , pgoblin.a[cmd].ch[b]
                              );
                } else {
                    if  (m & (1 << b)) {
                        printf("!!! Second hand r=%d for cmd=%03X\n", j, cmd);
                    }
                    m |= (1 << b);
        }   }   }
        for (pgoblin_nr j = 0; j < pgoblin.binparam; j++) {
            if  (!(m & (1 << j)) && pgoblin.a[cmd].ch[j]) {
                printf("!!! Unused r=%d for cmd=%03X\n", j, cmd);
                if  (pgoblin.a[cmd].str) printf("!!! Unused in name=%s~\n", pgoblin.a[cmd].str);
}   }   }   }

static int
/**********************************************************************
 **                                                                  **/
pgmdmp(pgoblin_exenv *exenv, pgoblin_prog *pgm, FILE *ouf) {        /**
 **                                                                  **
 **********************************************************************/
    pgoblin_tuple   *command;
    u_char           regs[pgoblin.symparam];
    pgoblin_command  cmd;
    char            *lit;
    int              ex     = EX_OK;
    int              k;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    fprintf( ouf
           , "PGM flags=%08X debug=%"BLIN_X" id=%.*s~\n"
           , pgm->flags
           , BLIN_I(pgm->debug)
           , PGOBLIN_STRING_ID_LEN, pgm->id
           );
    for (u_int i = 0; i < MULAR_NEXT(pgm->code); ++i) {
        if  (!(command = mular_getix(pgm->code, i))) {
            ifBLIN_QW0("mular_getix %u", i);
            ex = EX_SOFTWARE;
            goto out;
        }
        if  (PGOBLIN_extended == (cmd = command->cmd & PGOBLIN_COMMAND)) {
            cmd = (command->ecmd & PGOBLIN_COMMAND) | PGOBLIN_EXTENDED;
            lit = NULL;
        } else {
            lit = command->lit;
        }
        for (pgoblin_nr j = 0; j < pgoblin.symparam; j++) {
            int   b;

            if  (0 > (b = pgoblin_breg(cmd, j))) {
                regs[j] = '0';
            } else {
                regs[j] = (u_char)pgoblin_regn[ (command->cmd >> pgoblin.binparams[b])
                                              & PGOBLIN_ARGMASK
                                              ];
        }   }
        for (k = 6; 0 < k; --k) if ('0' != regs[k - 1]) break;
        if  (!k) {
            fprintf(ouf, "#%s\n", pgoblin.a[cmd].str);
        } else if (7 <= strlen(pgoblin.a[cmd].str)) {
            fprintf(ouf, "#%s %.*s\n", pgoblin.a[cmd].str, k, regs);
        } else {
            fprintf( ouf
                   , "#%s %.*s%.*s\n"
                   , pgoblin.a[cmd].str
                   , (int)(7 - strlen(pgoblin.a[cmd].str))
                   , "       "
                   , k
                   , regs
                   );
        }
        if  (!!lit) fprintf(ouf, "%s\n", lit);
    }
out:
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_dump(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_rio   *rou;
    FILE          *ouf    = NULL;
    int            ex     = EX_OK;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3("+ %c.....", pgoblin_regn[rn[PGO_COUT]]);
    GET_RIO(rou, exenv->options, rn[PGO_COUT]);
    if  (!(ouf = pgoblin_io_funopen(rou))) {
        ifBLIN_QW0("funopen on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
        goto out;
    }
    fprintf( ouf
           , "EXENV flags=%08X ppoint=%u ctrap=%u lexlevel=%d\n"
           , exenv->flags
           , exenv->ppoint
           , exenv->ctrap
           , exenv->lexlevel
           );
    if  (!!exenv->stat) {
        if  (exenv->stat != exenv->dyna) {
            fprintf(ouf, "stat ");
        } else {
            fprintf(ouf, "stat == dyna ");
        }
        fprintf( ouf
               , "flags=%08X ppoint=%u ctrap=%u lexlevel=%d\n"
               , exenv->stat->flags
               , exenv->stat->ppoint
               , exenv->stat->ctrap
               , exenv->stat->lexlevel
               );
    }
    if  (!!exenv->dyna && (exenv->stat != exenv->dyna)) {
        fprintf( ouf
               , "dyna flags=%08X ppoint=%u ctrap=%u lexlevel=%d\n"
               , exenv->dyna->flags
               , exenv->dyna->ppoint
               , exenv->dyna->ctrap
               , exenv->dyna->lexlevel
               );
    }
    if  (!!exenv->next) {
        fprintf( ouf
               , "dyna flags=%08X ppoint=%u ctrap=%u lexlevel=%d\n"
               , exenv->next->flags
               , exenv->next->ppoint
               , exenv->next->ctrap
               , exenv->next->lexlevel
               );
    }
    if  ((!!exenv->pgm) && !!(ex = pgmdmp(exenv, exenv->pgm, ouf))) {
        ifBLIN_QW0("pgmdmp");
        goto out;
    }
    if  (!!exenv->options) {
        fprintf( ouf
               , "OPTIONS flags=%08X vers=%02X%02X%02X%02X delim=%02X%02X%02X%02X maxtmout=%u\n"
                 "    useio=%016"BLIN_O"X usecon=%016"BLIN_O"X"
                 " usejob=%016"BLIN_O"X usesty=%016"BLIN_O"X\n"
                 "    demaskf=%016"BLIN_O"X demaskt=%016"BLIN_O"X"
                 " demaskm=%016"BLIN_O"X demaskq=%016"BLIN_O"X\n"
                 "    demasko=%016"BLIN_O"X demaskc=%016"BLIN_O"X"
                 " demaskj=%016"BLIN_O"X demasks=%016"BLIN_O"X\n"
                 "    xtrace=%d pip0=%d,%d pip1=%d,%d pip2=%d,%d\n"
               , exenv->options->flags
               , exenv->options->vers[0]
               , exenv->options->vers[1]
               , exenv->options->vers[2]
               , exenv->options->vers[3]
               , exenv->options->delim[0]
               , exenv->options->delim[1]
               , exenv->options->delim[2]
               , exenv->options->delim[3]
               , exenv->options->maxtmout
               , exenv->options->useio
               , exenv->options->usecon
               , exenv->options->usejob
               , exenv->options->usesty
               , exenv->options->demaskf
               , exenv->options->demaskt
               , exenv->options->demaskm
               , exenv->options->demaskq
               , exenv->options->demasko
               , exenv->options->demaskc
               , exenv->options->demaskj
               , exenv->options->demasks
               , exenv->options->xtrace
               , exenv->options->pip0[0]
               , exenv->options->pip0[1]
               , exenv->options->pip1[0]
               , exenv->options->pip1[1]
               , exenv->options->pip2[0]
               , exenv->options->pip2[1]
               );
        if  (!!exenv->options->ftrace) fprintf(ouf, "    ftrace=%s\n", exenv->options->ftrace);
        if  (!!exenv->options->bos) {
            fprintf( ouf
                   , "BOS flags=%08X redirenv=%u argin=%u symr=%c(%02X)\n"
                   , exenv->options->bos->flags
                   , exenv->options->bos->redirenv
                   , exenv->options->bos->argin
                   , exenv->options->bos->symr
                   , exenv->options->bos->symr & 0X0FF
                   );
        }
        for (u_int i = 0; !!exenv->options->argv[i]; ++i) {
            fprintf(ouf, "    argv[%u]=%s~\n", i, exenv->options->argv[i]);
        }
        for (u_int i = 0; !!exenv->options->envp[i]; ++i) {
            fprintf(ouf, "    envp[%u]=%s~\n", i, exenv->options->envp[i]);
        }
        for (u_int i = 0; i < PGOBLIN_SOSIZE; ++i) if (!!exenv->options->libexec[i]) {
            fprintf(ouf, "    libexec[%u]=%s~\n", i, exenv->options->libexec[i]);
        }
        if  (!!exenv->options->lvldumps) {
            pgoblin_lvldia *lvl;

            for (u_int i = 0; i < MULAR_NEXT(exenv->options->lvldumps); ++i) {
                lvl = mular_getix(exenv->options->lvldumps, i);
                fprintf( ouf
                       , "    lvldumps[%u] statmin=%u statmax=%u dynmin=%u dynmax=%u\n"
                       , i
                       , lvl->statmin
                       , lvl->statmax
                       , lvl->dynmin
                       , lvl->dynmax
                       );
        }   }
        if  (!!exenv->options->deargs) {
            const char  *o;

            for (u_int i = 0; i < MULAR_NEXT(exenv->options->deargs); ++i) {
                o = mular_getix(exenv->options->deargs, i);
                if  (!o) {
                    fprintf(ouf, "deargs[%u] NULL\n", i);
                } else {
                    fprintf(ouf, "deargs[%u]=%s~\n", i, o);
        }   }   }
        for (u_int i = 0; i < PGOBLIN_REGSIZE; ++i) {                              /************* IO */
            for (pgoblin_rio *rio = exenv->options->io[i]; !!rio; rio = rio->prev) {
                fprintf( ouf
                       , "IO[%c%c]+%u flags=%08X cortege=%zd pid=%d\n"
                       , pgoblin_regn[i]
                       , pgoblin_regn[rio->nio]
                       , rio->deep
                       , rio->flags
                       , rio->cortege
                       , rio->mpid
                       );
                if  (!!rio->oxenv) {
                    if  (!!(ex = pgmdmp(exenv, rio->oxenv->pgm, ouf))) {
                        ifBLIN_QW0("pgmdmp");
                        goto out;
                    }
                } else if (!!rio->stymd) {
                    size_t            offset;

                    for (u_int k = 0; k < MULAR_NEXT(rio->stymd); ++k) {
                        offset = *(size_t*)mular_getix(rio->stymd, k);
                        fprintf(ouf, "    style[%u]=%s~\n", k, &(((char *)(rio->text))[offset]));
                    }
                } else {
                    if  (!!rio->text) {
                        if  ((0 < rio->length) || !!rio->binparm) {
                            fprintf( ouf
                                   , "    text[%"BLIN_P"u]=%.*s~\n"
                                   , rio->length
                                   , (int)(rio->length)
                                   , rio->text
                                   );
                        } else {
                            fprintf(ouf, "    text=%s~\n", rio->text);
                    }   }
                    if  (!!rio->path) fprintf(ouf, "    path=%s~\n", rio->path);
                    if  (!!rio->pq) {
                        pgoblin_dbases *dbc;
                        ssize_t         row;
                        ssize_t         col;
                        ssize_t         w  ;
                        ssize_t         v  ;

                        dbc = pgoblin.dbasetable[(*rio->pq) & PGOBLIN_DB_TYPE];
                        if  (!dbc->resinfo) {
                            ifBLIN_QX0("No resinfo");
                            errno = ENOSYS;
                            ex = EX_UNAVAILABLE;
                            goto out;
                        }
                        if  (!dbc->valinfo) {
                            ifBLIN_QX0("No valinfo");
                            errno = ENOSYS;
                            ex = EX_UNAVAILABLE;
                            goto out;
                        }
                        if  (0 > (v = (ssize_t)dbc->resinfo(rio, PGOBLIN_Ntuples))) {
                            ifBLIN_QW0("resinfo Ntuples");
                            ex = EX_SOFTWARE;
                            goto out;
                        }
                        if  (0 > (w = (ssize_t)dbc->resinfo(rio, PGOBLIN_Nfields))) {
                            ifBLIN_QW0("resinfo Nfields");
                            ex = EX_SOFTWARE;
                            goto out;
                        }
                        fprintf(ouf, "    Pq %zd rows x %zd cols\n", v, w);
                        for (row = 0; row < v; ++row) {
                            fprintf(ouf, "    [%zd]", row);
                            for (col = 0; col < w; ++col) {
                                if  (!!dbc->valinfo(rio, PGOBLIN_IsNull, row, col)) {
                                    fprintf(ouf, "|");
                                } else {
                                    fprintf( ouf
                                           , "=%.*s|"
                                           , (int)dbc->valinfo(rio, PGOBLIN_Length, row, col)
                                           , dbc->getvalue(rio, row, col)
                                           );
                            }   }
                            fprintf(ouf, "\n");
        }   }   }   }   }
        for (u_int i = 0; i < PGOBLIN_REGSIZE; ++i) {                            /************* CONN */
            for (pgoblin_rdb *rdb = exenv->options->conn[i]; !!rdb; rdb = rdb->prev) {
                fprintf( ouf
                       , "CONN[%c%c]+%u flags=%08X intran=%u x2=%"BLIN_P"X x3=%"BLIN_P"X\n"
                         "    id=%.*s~ name=%.*s~ flags=%08X\n"
                       , pgoblin_regn[i]
                       , pgoblin_regn[rdb->ndb]
                       , exenv->options->conn[rdb->ndb]->deep - rdb->deep
                       , rdb->flags
                       , rdb->intran
                       , BLIN_I(rdb->dummy2)
                       , BLIN_I(rdb->dummy3)
                       , PGOBLIN_STRING_ID_LEN
                       , pgoblin.dbasetable[rdb->flags & PGOBLIN_DB_TYPE]->id
                       , PGOBLIN_STRING_ID_LEN
                       , pgoblin.dbasetable[rdb->flags & PGOBLIN_DB_TYPE]->name
                       , pgoblin.dbasetable[rdb->flags & PGOBLIN_DB_TYPE]->flags
                       );
                if  (!!rdb->dbname)   fprintf(ouf, " dbname=%s~"  , rdb->dbname);
                if  (!!rdb->host)     fprintf(ouf, " host=%s~"    , rdb->host);
                if  (!!rdb->port)     fprintf(ouf, " port=%s~"    , rdb->port);
                if  (!!rdb->username) fprintf(ouf, " username=%s~", rdb->username);
                if  (!!rdb->password) fprintf(ouf, " password=%s~", rdb->password);
                if  (  !!rdb->dbname
                    || !!rdb->host
                    || !!rdb->port
                    || !!rdb->username
                    || !!rdb->password
                    ) fprintf(ouf, "\n");
        }   }
        for (u_int i = 0; i < PGOBLIN_REGSIZE; ++i) {                             /************* JOB */
            pgoblin_jobl *j;

            for (pgoblin_rjb *rjb = exenv->options->job[i]; !!rjb; rjb = rjb->prev) {
                fprintf( ouf
                       , "JOB[%c%c]+%u flags=%08X kq=%d concur=%u dead=%u"
                       , pgoblin_regn[i]
                       , pgoblin_regn[rjb->njb]
                       , exenv->options->job[rjb->njb]->deep - rjb->deep
                       , rjb->flags
                       , rjb->kq
                       , rjb->concur
                       , rjb->dead
                       );
                if  (!!rjb->host) fprintf(ouf, " host=%s~", rjb->host);
                fprintf( ouf
                       , "\n    id=%.*s~ name=%.*s~ flags=%08X\n"
                       , PGOBLIN_STRING_ID_LEN
                       , pgoblin.jobtable[rjb->flags & PGOBLIN_JOB_TYPE]->id
                       , PGOBLIN_STRING_ID_LEN
                       , pgoblin.jobtable[rjb->flags & PGOBLIN_JOB_TYPE]->name
                       , pgoblin.jobtable[rjb->flags & PGOBLIN_JOB_TYPE]->flags
                       );
                for (u_int k = 0; k < MULAR_NEXT(rjb->pids); ++k) {
                    j = mular_getix(rjb->pids, k);
                    fprintf( ouf
                           , "        [%u] cmdn=%"BLIN_P"d pid=%d status=%08X\n"
                           , k
                           , j->cmdn
                           , j->pid
                           , j->status
                           );
        }   }   }
        for (u_int i = 0; i < PGOBLIN_REGSIZE; ++i) {                           /************* STYLE */
            for (pgoblin_rst *rst = exenv->options->style[i]; !!rst; rst = rst->prev) {
                fprintf( ouf
                       , "STYLE[%c%c]+%u flags=%08X dummy=%d\n"
                       , pgoblin_regn[i]
                       , pgoblin_regn[rst->nst]
                       , exenv->options->style[rst->nst]->deep - rst->deep
                       , rst->flags
                       , rst->dummy
                       );
                fprintf( ouf
                       , "    id=%.*s~ name=%.*s~ flags=%08X =%"BLIN_X
                       , PGOBLIN_STRING_ID_LEN
                       , pgoblin.styletable[rst->flags & PGOBLIN_STYLE_TYPE]->id
                       , PGOBLIN_STRING_ID_LEN
                       , pgoblin.styletable[rst->flags & PGOBLIN_STYLE_TYPE]->name
                       , pgoblin.styletable[rst->flags & PGOBLIN_STYLE_TYPE]->flags
                       , BLIN_I(rst->style)
                       );
                if  (!!rst->style) fprintf(ouf, "=%.*s~", PGOBLIN_STRING_ID_LEN, rst->style->id);
                fprintf(ouf, "\n");
                pgoblin_st_dump(rst, ouf);
                if  (ENOSYS == errno) errno = 0; /*       */
    }   }   }
    for (int i = 0; i < pgoblin.outablesz; ++i) {
        if  (!!pgoblin.outable[i]) {
            fprintf( ouf
                   , "OU[%d]  id=%.*s~ name=%.*s~\n"
                   , i
                   , PGOBLIN_STRING_ID_LEN, pgoblin.outable[i]->id
                   , PGOBLIN_STRING_ID_LEN, pgoblin.outable[i]->name
                   );
    }   }
    for (int i = 0; i < pgoblin.iotablesz; ++i) {
        if  (!!pgoblin.iotable[i]) {
            fprintf( ouf
                   , "IO[%d]  id=%.*s~ name=%.*s~\n"
                   , i
                   , PGOBLIN_STRING_ID_LEN, pgoblin.iotable[i]->id
                   , PGOBLIN_STRING_ID_LEN, pgoblin.iotable[i]->name
                   );
    }   }
    for (int i = 0; i < pgoblin.intablesz; ++i) {
        if  (!!pgoblin.intable[i]) {
            fprintf( ouf
                   , "IN[%d]  id=%.*s~ name=%.*s~\n"
                   , i
                   , PGOBLIN_STRING_ID_LEN, pgoblin.intable[i]->id
                   , PGOBLIN_STRING_ID_LEN, pgoblin.intable[i]->name
                   );
    }   }
    for (int i = 0; i < pgoblin.dbtablesz; ++i) {
        if  (!!pgoblin.dbasetable[i]) {
            fprintf( ouf
                   , "DB[%d]  id=%.*s~ name=%.*s~\n"
                   , i
                   , PGOBLIN_STRING_ID_LEN, pgoblin.dbasetable[i]->id
                   , PGOBLIN_STRING_ID_LEN, pgoblin.dbasetable[i]->name
                   );
    }   }
    for (int i = 0; i < pgoblin.jobtabsiz; ++i) {
        if  (!!pgoblin.jobtable[i]) {
            fprintf( ouf
                   , "JOB[%d] id=%.*s~ name=%.*s~\n"
                   , i
                   , PGOBLIN_STRING_ID_LEN, pgoblin.jobtable[i]->id
                   , PGOBLIN_STRING_ID_LEN, pgoblin.jobtable[i]->name
                   );
    }   }
    for (int i = 0; i < pgoblin.styletbsz; ++i) {
        if  (!!pgoblin.styletable[i]) {
            fprintf( ouf
                   , "ST[%d]  id=%.*s~ name=%.*s~\n"
                   , i
                   , PGOBLIN_STRING_ID_LEN, pgoblin.styletable[i]->id
                   , PGOBLIN_STRING_ID_LEN, pgoblin.styletable[i]->name
                   );
    }   }
out:
    if  (!!ouf) fflush(ouf);
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}
