/*-
 * Copyright (C)2017..2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2017..2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: db.c,v 1.73 2024/07/14 00:00:02 babolo Exp $"

#define BLIN_COMPAT      4
#define Bpars_COMPAT     4
#define MULAR_COMPAT     0
#define MIFE_COMPAT      5
#define PGOBLIN_COMPAT   5
#define PGOBLIN_INTERNAL 1

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <errno.h>
#include <stdio.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_db_shurecon(pgoblin_rdb *rdb) {                             /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    int          ex = EX_OK;

    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->shurecon) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->shurecon(rdb);
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*******************************************************************************************
 *******************************************************************************************
 **                                                                                       **/
pgoblin_db_query(pgoblin_exenv *exenv, pgoblin_nr *rn) {                                 /**
 **                                                                                       **
 *******************************************************************************************
 *******************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    /*    */
    pgoblin_rdb *rdb;
    int          ex = EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->query) {
        errno = ENOSYS;
        ex = EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->query(exenv, rn);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*********************************************************************************************
 *********************************************************************************************
 **                                                                                         **/
pgoblin_db_prepare(pgoblin_exenv *exenv, pgoblin_nr *rn) {                                 /**
 **                                                                                         **
 *********************************************************************************************
 *********************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rdb *rdb;
    int          ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->prepare) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->prepare(exenv, rn);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/************************************************************************************************
 ************************************************************************************************
 **                                                                                            **/
pgoblin_db_execute(pgoblin_exenv *exenv, pgoblin_nr *rn) {                                    /**
 **                                                                                            **
 ************************************************************************************************
 ************************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rdb *rdb;
    int          ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->execute) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->execute(exenv, rn);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int64_t
/*************************************************************************
 *************************************************************************
 **                                                                     **/
pgoblin_db_resinfo(pgoblin_rio *rio, int kind) {                       /**
 **                                                                     **
 *************************************************************************
 *************************************************************************/
#   define blin_internal_flags (rio->flags & BLIN_MASK)
    int64_t      ex = -EX_SOFTWARE;

    ifBLIN_QX3("+ [%d]%d", !rio->pq ? -1 : (int)(*rio->pq & PGOBLIN_DB_TYPE), kind);
    if  (!rio->pq) {
        ifBLIN_QX1("No pq");
        errno = EFAULT;
        ex = -EX_UNAVAILABLE;
    } else if (!DBASE(*rio->pq) || !DBASE(*rio->pq)->resinfo) {
        ifBLIN_QX1("No DB or resinfo");
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else if (  !!(rio->purge)
              && ((PGOBLIN_Ntuples == kind) || (PGOBLIN_Nfields == kind))
              ) {
        ex = 0;
    } else if (  (1 < (ex = DBASE(*rio->pq)->resinfo(rio, kind)))
              && (PGOBLIN_Ntuples == kind)
              && (0 <= rio->cortege)
              ) {
        ex = 1;
    }
    ifBLIN_QX3("- %"BLIN_O"d", ex);
    return(ex);
#   undef blin_internal_flags
}

const char *
/**************************************************************************************
 **************************************************************************************
 **                                                                                  **/
pgoblin_db_subinfo(pgoblin_rio *rio, int kind, int64_t sub) {                       /**
 **                                                                                  **
 **************************************************************************************
 **************************************************************************************/
#   define blin_internal_flags (rio->flags & BLIN_MASK)
    const char   *c = NULL;

    ifBLIN_QX3("+ [%d]%d %"BLIN_O"d", !rio->pq ? -1 : (int)(*rio->pq & PGOBLIN_DB_TYPE), kind, sub);
    if  (!rio->pq) {
        errno = EFAULT;
    } else if (!DBASE(*rio->pq) || !DBASE(*rio->pq)->subinfo) {
        errno = ENOSYS;
    } else if (  !!(rio->purge)
              && ((PGOBLIN_TypeNm == kind) || (PGOBLIN_ColName == kind))
              ) {
        errno = ENOENT;
    } else {
        c = DBASE(*rio->pq)->subinfo(rio, kind, sub);
    }
    ifBLIN_QX3("- %s", c);
    return(c);
#   undef blin_internal_flags
}

char *
/************************************************************************
 ************************************************************************
 **                                                                    **/
pgoblin_db_erinfo(pgoblin_rio *rio) {                                 /**
 **                                                                    **
 ************************************************************************
 ************************************************************************/
#   define blin_internal_flags (rio->flags & BLIN_MASK)
    char         *c = NULL;

    ifBLIN_QX3("+ [%d]", !rio->pq ? -1 : (int)(*rio->pq & PGOBLIN_DB_TYPE));
    if  (!rio->pq) {
        errno = EFAULT;
    } else if (!DBASE(*rio->pq) || !DBASE(*rio->pq)->erinfo) {
        errno = ENOSYS;
    } else {
        c = DBASE(*rio->pq)->erinfo(rio);
    }
    ifBLIN_QX3("- %s", c);
    return(c);
#   undef blin_internal_flags
}

ssize_t
/*******************************************************************************************
 *******************************************************************************************
 **                                                                                       **/
pgoblin_db_valinfo(pgoblin_rio *rio, int kind, ssize_t row, ssize_t col) {               /**
 **                                                                                       **
 *******************************************************************************************
 *******************************************************************************************/
#   define blin_internal_flags (rio->flags & BLIN_MASK)
    ssize_t       ex = PGOBLIN_SSIZE_MIN;

    ifBLIN_QX3("+ [%d]%d %zd %zd", !rio->pq ? -1 : (int)(*rio->pq & PGOBLIN_DB_TYPE), kind, row, col);
    if  (!rio->pq) {
        errno = EFAULT;
    } else if (!DBASE(*rio->pq) || !DBASE(*rio->pq)->valinfo) {
        errno = ENOSYS;
    } else if (!!rio->purge && (PGOBLIN_Length == kind)) {
        ex = 0;
    } else {
        ex = DBASE(*rio->pq)->valinfo(rio, kind, (0 > rio->cortege) ? row : rio->cortege, col);
    }
    ifBLIN_QX3("- %zd", ex);
    return(ex);
#   undef blin_internal_flags
}

char *
/**********************************************************************************
 **********************************************************************************
 **                                                                              **/
pgoblin_db_getvalue(pgoblin_rio *rio, ssize_t row, ssize_t col) {               /**
 **                                                                              **
 **********************************************************************************
 **********************************************************************************/
#   define blin_internal_flags (rio->flags & BLIN_MASK)
    char        *c = NULL;

    ifBLIN_QX3( "+ [%d]%zd %zd(%zd)"
              , (!rio->pq) ? -1 : (int)(*(rio->pq) & PGOBLIN_DB_TYPE)
              , row
              , col
              , rio->cortege
              );
    if  (!rio->pq) {
        errno = EFAULT;
    } else if (!DBASE(*(rio->pq)) || !DBASE(*(rio->pq))->getvalue) {
        errno = ENOSYS;
    } else if (!!rio->purge) {
        errno = ENOENT;
    } else {
        c = DBASE(*(rio->pq))->getvalue(rio, (0 > rio->cortege) ? row : rio->cortege, col);
    }
    ifBLIN_QX3("- %s", c);
    return(c);
#   undef blin_internal_flags
}

int
/**********************************************************************************************
 **********************************************************************************************
 **                                                                                          **/
pgoblin_db_getstream(pgoblin_exenv *exenv, pgoblin_nr *rn, int quant) {                     /**
 **                                                                                          **
 **********************************************************************************************
 **********************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rdb *rdb;
    int          ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]%d", rdb->flags & PGOBLIN_DB_TYPE, quant);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->getstream) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->getstream(exenv, rn, quant);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**************************************************************************
 **************************************************************************
 **                                                                      **/
pgoblin_db_getcopy(pgoblin_exenv *exenv, pgoblin_nr *rn, void **buf) {  /**
 **                                                                      **
 **************************************************************************
 **************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rdb *rdb;
    int          ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->getcopy) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->getcopy(exenv, rn, buf);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*******************************************************************************************
 *******************************************************************************************
 **                                                                                       **/
pgoblin_db_putcopy(pgoblin_exenv *exenv, pgoblin_nr *rn, const void *buf, ssize_t len) { /**
 **                                                                                       **
 *******************************************************************************************
 *******************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rdb *rdb;
    int          ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]%zd", rdb->flags & PGOBLIN_DB_TYPE, len);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->putcopy) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->putcopy(exenv, rn, buf, len);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_db_endstream(pgoblin_exenv *exenv, pgoblin_nr *rn) {        /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rdb *rdb;
    int          ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->endstream) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->endstream(exenv, rn);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/************************************************************************
 ************************************************************************
 **                                                                    **/
pgoblin_db_tarry(pgoblin_rdb *rdb, int timeout) {                     /**
 **                                                                    **
 ************************************************************************
 ************************************************************************/
#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    int          ex = 0;

    ifBLIN_QX3("+ [%d]%d", rdb->flags & PGOBLIN_DB_TYPE, timeout);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->tarry) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->tarry(rdb, timeout);
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

void *
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_db_notifies(pgoblin_rdb *rdb) {                             /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    char        *c = NULL;

    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    c = DBASE(rdb->flags)->notifies(rdb);
    ifBLIN_QX3("- %s", c);
    return(c);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_db_finish(pgoblin_rdb *rdb) {                               /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    int          ex = 0;

    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags)->finish) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->finish(rdb);
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_db_clear(pgoblin_rio *rio) {                                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (rio->flags & BLIN_MASK)
    int          ex = 0;

    ifBLIN_QX3( "+ [%d]", (!rio || !rio->pq) ? -1 : (int)(*(rio->pq) & PGOBLIN_DB_TYPE));
    if  (!!rio && !!rio->pq) {
        if  (!DBASE(*rio->pq) || !DBASE(*rio->pq)->clear) {
            errno = ENOSYS;
            ex = -EX_UNAVAILABLE;
        } else {
            ex = DBASE(*rio->pq)->clear(rio);
    }   }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

char *
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_db_erconn(pgoblin_rdb *rdb) {                               /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    char        *c = NULL;

    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags)->erconn) {
        errno = ENOSYS;
    } else {
        c = DBASE(rdb->flags)->erconn(rdb);
    }
    ifBLIN_QX3("- %s", c);
    return(c);
#   undef blin_internal_flags
}

int
/************************************************************************************************
 ************************************************************************************************
 **                                                                                            **/
pgoblin_db_prexec(pgoblin_exenv *exenv, pgoblin_nr *rn) {                                     /**
 **                                                                                            **
 ************************************************************************************************
 ************************************************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    pgoblin_rdb *rdb;
    int          ex = -EX_SOFTWARE;

    GET_CON(rdb, exenv->options, rn[PGO_CCON]);
    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags) || !DBASE(rdb->flags)->execute) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->prexec(exenv, rn);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*************************************************************************
 *************************************************************************
 **                                                                     **/
pgoblin_db_denote(pgoblin_rdb *rdb, void **note) {                     /**
 **                                                                     **
 *************************************************************************
 *************************************************************************/
#   define blin_internal_flags (rdb->flags & BLIN_MASK)
    int          ex = -EX_SOFTWARE;

    ifBLIN_QX3("+ [%d]", rdb->flags & PGOBLIN_DB_TYPE);
    if  (!DBASE(rdb->flags)->denote) {
        errno = ENOSYS;
        ex = -EX_UNAVAILABLE;
    } else {
        ex = DBASE(rdb->flags)->denote(rdb, note);
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}
