/*-
 * Copyright (C)2004..2025 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2004..2025 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: cmd_parm.c,v 1.15 2025/02/18 01:23:23 babolo Exp $"

#define BLIN_COMPAT        4
#define Bpars_COMPAT       4
#define MULAR_COMPAT       0
#define MIFE_COMPAT        5
#define PGOBLIN_COMPAT     5
#define PGOBLIN_INTERNAL

#include <sys/param.h>
#include <sys/types.h>
#include <sys/event.h>
#include <sys/user.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sysexits.h>
#include <dirent.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <stdio.h>
#include <fts.h>
#include <err.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>

#ifdef BLIN_OSLinux
# include <dirent.h>
#else
# include <sys/sysctl.h>
#endif

#include "pgoblin.h"

#ifndef BLIN_4STA24N
# define BLIN_4STO24N ((blin_statectl){{4, 0, 8, 24, '\0', 5, 1, 1, 0}})
#endif

#define Cd 0x00800000 /*        */
#define Cx 0x00400000 /*  a..f            */
#define CX 0x00200000 /*  A..F            */
#define Ci 0x00100000 /*            */
#define Cj 0x00080000 /*       */
#define Cn 0x00040000 /*             */
#define Cv 0x00020000 /*        */
#define Cb 0x00010000 /*                             */
#define Cp 0x00008000 /*                        */
#define Ct 0x00004000 /*                         */
#define Cf 0x00002000 /*                    */
#define Co 0x00001000 /*   =                   */
#define Cs 0x00000800 /*                          */
#define Ce 0x00000400 /*                                 */
#define C_state  0xFF /*                     */

typedef struct chunk {
    size_t size;
    size_t index;
    char   text[0];
} chunk;

typedef enum classes
{ nl = 0x00 /* nul       */
, no = 0x01 /*  */
, hc = 0x02 /* % 0x25    */
, hm = 0x03 /* & 0x26    */
, hq = 0x04 /* = 0x3D    */
, hl = 0x05 /* + 0x2B    */
, hd = 0x06 /* 0..9      */
, hx = 0x07 /* a..f      */
, hX = 0x08 /* A..F      */
, ne = 0x09 /* \\ 0x5C   */
, nb = 0x19 /* \b 0x62   */
, nt = 0x29 /* \t 0x74   */
, nn = 0x39 /* \n 0x6E   */
, nv = 0x49 /* \v 0x76   */
, nf = 0x59 /* \f 0x66   */
, nr = 0x69 /* \r 0x72   */
, xx = 0x0A
} classes;

typedef enum states
{ d0 /*                    */
, d1 /*    - %,  1     */
, d2 /*  2             */
, v0 /*                        */
, v1 /*    - %,  1  */
, v2 /*  2          */
, dx /*                               */
} states;

static const char *
/*****************************************************************************************************/
clasnames[] =                                                                                    /****
 *****************************************************************************************************/
{ "nl", "no", "hc", "hm", "hq", "hl", "hd", "hx", "hX", "ne", "- ", "- ", "- ", "- ", "- ", "- "
, "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "nb", "- ", "- ", "- ", "- ", "- ", "- "
, "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "nt", "- ", "- ", "- ", "- ", "- ", "- "
, "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "nn", "- ", "- ", "- ", "- ", "- ", "- "
, "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "nv", "- ", "- ", "- ", "- ", "- ", "- "
, "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "nf", "- ", "- ", "- ", "- ", "- ", "- "
, "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- ", "nr"
};

static const classes
/*****************************************************************************************************/
class[256] =                                                                                     /****
 *****************************************************************************************************/
{nl,no,no,no ,no,no,no,no ,nb,nt,nn,nv ,nf,nr,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,hc,hm,no ,no,no,no,hl ,no,no,no,no
,hd,hd,hd,hd ,hd,hd,hd,hd ,hd,hd,no,no ,no,hq,no,no

,no,hX,hX,hX ,hX,hX,hX,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,ne,no,no,no
,no,hx,hx,hx ,hx,hx,hx,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no

,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no

,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
,no,no,no,no ,no,no,no,no ,no,no,no,no ,no,no,no,no
};

static const char
/*****************************************************************************************************/
statenames[][3] = {"d0", "d1", "d2", "v0", "v1", "v2", "dx"};                                    /****
 *****************************************************************************************************/

static const char
/*****************************************************************************************************/
pn[][3] =                                                                                        /****
 *****************************************************************************************************/
{ "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- "
, "d ", "x ", "X ", "i ", "j ", "n ", "v ", "b "
, "p ", "t ", "f ", "o ", "s ", "e ", "- ", "- "
};

static const char
/*****************************************************************************************************/
decode[] = "\\\b\t\n\v\f\r\\";                                                                    /****
 *****************************************************************************************************/

static const u_int32_t
/*****************************************************************************************************/
automa[xx][dx] =                                                                                 /****
 *****************************************************************************************************/
/*     d0     d1        d2     v0     v1        v2  */
{ { Cs|dx, Ce|dx,    Ce|dx, Co|dx, Ce|dx,    Ce|dx } /* nl nul       */
, { Cn|d0, Ce|dx,    Ce|dx, Cv|v0, Ce|dx,    Ce|dx } /* no  */
, {    d1, Ce|dx,    Ce|dx,    v1, Ce|dx,    Ce|dx } /* hc % 0x25    */
, { Cs|d0, Ce|dx,    Ce|dx, Co|d0, Ce|dx,    Ce|dx } /* hm & 0x26    */
, {    v0, Ce|dx,    Ce|dx, Cv|v0, Ce|dx,    Ce|dx } /* hq = 0x3D    */
, { Cb|d0, Ce|dx,    Ce|dx, Cp|v0, Ce|dx,    Ce|dx } /* hl + 0x2B    */
, { Cn|d0, Cd|d2, Cd|Ci|d0, Cv|v0, Cd|v2, Cd|Cj|v0 } /* hd 0..9      */
, { Cn|d0, Cx|d2, Cx|Ci|d0, Cv|v0, Cx|v2, Cx|Cj|v0 } /* hx a..f      */
, { Cn|d0, CX|d2, CX|Ci|d0, Cv|v0, CX|v2, CX|Cj|v0 } /* hX A..F      */
, { Ct|d0, Ce|dx,    Ce|dx, Cf|v0, Ce|dx,    Ce|dx } /* n* \b..\\    */
};

static chunk *
/**********************************************************************
 **                                                                  **/
chput(chunk *ch, char c) {                                          /**
 **                                                                  **
 **********************************************************************/
    chunk *r = ch;

    if  (!r && !!(r = malloc(128))) {
        r->size = 128 - sizeof(chunk);
        r->index = 0;
        r->text[0] = '\0';
    }
    if  (!!r) {
        r->text[r->index++] = c;
        if  ((r->index >= r->size) && !!(r = realloc(ch, 0))) r->text[r->index] = '\0';
    }
    return(r);
}

static int
/**********************************************************************
 **                                                                  **/
cparm(BLIN_flag flags, const char *in, ssize_t len, FILE *fi) {     /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (flags & BLIN_MASK)
    char        *escaped = NULL;
    u_int32_t    control ;
    states       state   ;
    chunk       *value   = NULL;  /*     */
    chunk       *name    = NULL;  /*          */
    u_int32_t    reg     ;        /*   hex           */
    ssize_t      ixi     ;        /*     in      */
    size_t       cnt     ;        /*                     */
    classes      sym     ;        /*                */
    int          ex      = EX_OK;
    ssize_t      e       ;
    ssize_t      o       ;

    ifBLIN_QX3("+ %08X", flags);
    reg = 0;
    cnt = 0;
    blin_stateheader(BLIN_4STO24N, pn);
    for (state = !in ? dx : d0, ixi = 0; state < dx; state = control & C_state, ++ixi) {
        sym = ((0 > len) || (ixi < len)) ? (class[in[ixi] & 0xFF]) : nl;
        control = automa[sym & 0x0F][state];
        blin_statebody( BLIN_4STO24N
                      , pn
                      , statenames
                      , clasnames[sym]
                      , in
                      , (size_t)ixi
                      , control
                      , state
                      , reg
                      , (u_char)reg
                      );
        if  (Cd & control) {
            reg <<= 4;
            reg += (u_char)(in[ixi] - '0');
        }
        if  (Cx & control) {
            reg <<= 4;
            reg += (u_char)(in[ixi] - 'a' + 10);
        }
        if  (CX & control) {
            reg <<= 4;
            reg += (u_char)(in[ixi] - 'A' + 10);
        }
        if  (Ci & control) {
            if  (!(name = chput(name, (char)reg))) {
                ifBLIN_QW0("No mem");
                ex = EX_OSERR;
                goto out;
            }
            reg = 0;
        }
        if  (Cj & control) {
            if  (!(value = chput(value, (char)reg))) {
                ifBLIN_QW0("No mem");
                ex = EX_OSERR;
                goto out;
            }
            reg = 0;
        }
        if  ((Cn & control) && !(name = chput(name, in[ixi]))) {
            ifBLIN_QW0("No mem");
            ex = EX_OSERR;
            goto out;
        }
        if  ((Cv & control) && !(value = chput(value, in[ixi]))) {
            ifBLIN_QW0("No mem");
            ex = EX_OSERR;
            goto out;
        }
        if  ((Cb & control) && !(name = chput(name, ' '))) {
            ifBLIN_QW0("No mem");
            ex = EX_OSERR;
            goto out;
        }
        if  ((Cp & control) && !(value = chput(value, ' '))) {
            ifBLIN_QW0("No mem");
            ex = EX_OSERR;
            goto out;
        }
        if  (  (Ct & control)
            && ((PGOBLIN_PARMCTL & flags) || !(class[in[ixi] & 0xFF] & 0xF0))
            && !(name = chput(name, decode[class[in[ixi] & 0xFF] >> 4]))
            ) {
            ifBLIN_QW0("No mem");
            ex = EX_OSERR;
            goto out;
        }
        if  (  (Cf & control)
            && ((PGOBLIN_PARMCTL & flags) || !(class[in[ixi] & 0xFF] & 0xF0))
            && !(value = chput(value, decode[class[in[ixi] & 0xFF] >> 4]))
            ) {
            ifBLIN_QW0("No mem");
            ex = EX_OSERR;
            goto out;
        }
        if  (Co & control) {
            escaped = malloc(((!name ? 2 : name->index) + (!value ? 2 : value->index)) * 2 + 4);
            if  (!escaped) {
                ifBLIN_QW0("No mem");
                ex = EX_OSERR;
                goto out;
            }
            e = pgoblin_onecopy( escaped
                               , !name ? "" : name->text
                               , !name ? 0 : (ssize_t)name->index
                               , -1
                               );
            if  (0 > e) {
                ifBLIN_QW0("pgoblin_onecopy");
                free(escaped);
                ex = EX_SOFTWARE;
                goto out;
            }
            o = pgoblin_onecopy( &escaped[e]
                               , !value ? "" : value->text
                               , !value ? 0 : (ssize_t)value ->index
                               , 0
                               );
            if  (0 > o) {
                ifBLIN_QW0("pgoblin_onecopy");
                free(escaped);
                ex = EX_SOFTWARE;
                goto out;
            }
            escaped[e + o] = '\0';
            if  (0 > fprintf(fi, "%"BLIN_D"\t%s", cnt++, escaped)) {
                ifBLIN_QW0("fprintf fun");
                free(escaped);
                ex = EX_SOFTWARE;
                goto out;
            }
            free(escaped);
            free(name);
            name = NULL;
            free(value);
            value = NULL;
        }
        if  (Cs & control) {
            if  (!(escaped = malloc(!name ? 2 : name->index * 2 + 2))) {
                ifBLIN_QW0("No mem");
                ex = EX_OSERR;
                goto out;
            }
            e = pgoblin_onecopy( escaped
                               , !name ? "" : name->text
                               , !name ? 0 : (ssize_t)name ->index
                               , 0
                               );
            if  (0 > e) {
                ifBLIN_QW0("pgoblin_onecopy");
                free(escaped);
                ex = EX_SOFTWARE;
                goto out;
            }
            escaped[e] = '\0';
            if  (0 > fprintf(fi, "%"BLIN_D"\t\\N\t%s", cnt++, escaped)) {
                ifBLIN_QW0("fprintf fun");
                free(escaped);
                ex = EX_SOFTWARE;
                goto out;
            }
            free(escaped);
            free(value);
            value = NULL;
        }
        if  (Ce & control) {
            ifBLIN_QX0("Invalid syntax @ %"BLIN_D, cnt);
            ex = EX_DATAERR;
            errno = EILSEQ;
            goto out;
    }   }
    if  ((~flags & PGOBLIN_COPEND_NOGEN) && (0 > fprintf(fi, "\\.\n"))) {
        ifBLIN_QW0("fprintf fun");
        free(escaped);
        ex = EX_SOFTWARE;
        goto out;
    }
out:
    if  (!!name) free(name);
    if  (!!value) free(value);
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getget(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    int          ex = EX_OK;
    pgoblin_rio *rou;
    FILE        *fi    = NULL;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+ %c.....", pgoblin_regn[rn[PGO_COUT]]);
    GET_RIO(rou, exenv->options, rn[PGO_COUT]);
    if  (!(fi = pgoblin_io_funopen(rou))) {
        ifBLIN_QW0("getget funopenh");
        ex = EX_IOERR;
        goto out;
    }
    if  (!!(ex = cparm(exenv->options->flags, getenv("QUERY_STRING"), -1, fi))) {
        ifBLIN_QW0("cparm");
    }
out:
    if  (0 > pgoblin_io_funclose(rou)) {
        ifBLIN_QW0("io_funclose");
        ex = EX_OSERR;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getpost(pgoblin_exenv *exenv, pgoblin_nr *rn) {             /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    int          ex = EX_OK;
    pgoblin_rio *rou;
    pgoblin_rio *rin;
    FILE        *fi    = NULL;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+ %c.%c...", pgoblin_regn[rn[PGO_COUT]], pgoblin_regn[rn[PGO_CIN]]);
    GET_RIO(rou, exenv->options, rn[PGO_COUT]);
    if  (!(fi = pgoblin_io_funopen(rou))) {
        ifBLIN_QW0("getpost funopenh");
        ex = EX_IOERR;
        goto out;
    }
    GET_RIO(rin, exenv->options, rn[PGO_CIN]);
    if  (0 > pgoblin_io_read(rin, 0, 0)) {
        ifBLIN_QW0("pgoblin_io_read");
        return(EX_OSERR);
    }
    if  (!!(ex = cparm(exenv->options->flags, mife_pointer(rin->mife), mife_ctlsize(rin->mife), fi))) {
        ifBLIN_QW0("cparm");
    }
out:
    if  (0 > pgoblin_io_funclose(rou)) {
        ifBLIN_QW0("io_funclose");
        ex = EX_OSERR;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getjob(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 ** pgoblin_getjob()   2  ,                **
 **     ,                          **
 **     .                              **
 **  pgoblin_syntax                          **
 **   - Ij  ch[2].                              **
 **   - ,                        **
 ** (  )                    **
 **   ,   .                  **
 **   R_JOBIN,   R_JOB                       **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    int           status;
    pgoblin_rio  *rou;
    pgoblin_rjb  *rjb;
    int           exx = EX_OK;
    int           ex  = EX_OK;
    FILE         *fi;
    pgoblin_jobl *j;
    size_t        i;

    ifBLIN_QX3("+getjob %c...%c.", pgoblin_regn[rn[PGO_COUT]], pgoblin_regn[rn[PGO_CJOB]]);
    GET_JOB(rjb, exenv->options, rn[PGO_CJOB]);
    if  (rjb->pids) {
        GET_RIO(rou, exenv->options, rn[PGO_COUT]);
        if  (!(fi = pgoblin_io_funopen(rou))) {
            ifBLIN_QW0("funopen on %d", rn[PGO_CJOB]);
            ex = EX_OSERR;
        } else {
            for (i = 0; i <  MULAR_NEXT(rjb->pids); i++) {
                j = mular_getix(rjb->pids, i);
                if  (j->pid <= 0) {
                    ifBLIN_QX0("No pid in %"BLIN_D, j->cmdn);
                    errno = EINVAL;
                    ex = EX_SOFTWARE;
                } else if ((j->cmdn >= 0) && (j->status == PGOBLIN_NOSTATUS)) {
                    if  (!(ex = waitpid(j->pid, &status, WNOHANG))) {
                    } else if (ex == j->pid) {
                        j->status = status;
                        rjb->dead++;
                        ifBLIN_QU3( rjb->flags
                                  , "pgoblin_getjob st=%08X fl=%08X ex=%d exx=%d IFEX=%d EXST=%d"
                                  , status
                                  , rjb->flags
                                  , ex
                                  , exx
                                  , WIFEXITED(status), WEXITSTATUS(status)
                                  );
                        if  (rjb->flags & PGOBLIN_ERROR_SENSOR) {
                            if  (  WIFEXITED(status)
                                && WEXITSTATUS(status)
                                ) {
                                  exx = WEXITSTATUS(status);
                            } else if (WIFSIGNALED(status)) {
                                  exx = WTERMSIG(j->status) + 256;
                        }   }
                        if  ((rjb->flags & PGOBLIN_ERROR_SENSIB) && exx) break;
                        ex = EX_OK;
                    } else if (ex > 0) {
                        ifBLIN_QW0("Pid %d<>%d in %d", ex, j->pid, (int)j->cmdn);
                        ERROUT(EX_OSERR, ENOATTR);
                    } else if (errno == ECHILD || errno == EDEADLK || errno == EAGAIN) {
                        ex = EX_OK;
                    } else {
                        ifBLIN_QW0("Pid %d in %d", j->pid, (int)j->cmdn);
                        ex = EX_OSERR;
                        goto out;
                }   }
                fprintf(fi, "%d	%d	%d	%d\n", (int)i, (int)j->cmdn, j->pid, j->status);
            }
            if  (!(exenv->options->flags & PGOBLIN_COPEND_NOGEN)) fprintf(fi, "\\.\n");
            if  (0 > pgoblin_io_funclose(rou)) {
                ifBLIN_QW0("io_funclose on %d", rn[PGO_CJOB]);
                ex = EX_OSERR;
    }   }   }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex ? ex : exx);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getarg(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    size_t       nowl;
    pgoblin_rio *rou;
    pgoblin_rio *rct;
    int          ex   = EX_OK;
    FILE        *fi   = NULL;
    const char  *o;
    char        *q    = NULL;
    u_int64_t    n;
    ssize_t      l;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+getarg %c%c....", pgoblin_regn[rn[PGO_COUT]], pgoblin_regn[rn[PGO_CCTL]]);
    GET_RIO(rou, exenv->options, rn[PGO_COUT]);
    if  (!(fi = pgoblin_io_funopen(rou))) {
        ifBLIN_QW0("funopen on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
        goto out;
    }
    if  (!exenv->options->deargs) goto empty;
    GET_RIO(rct, exenv->options, rn[PGO_CCTL]);
    n = 0;
    --n;
    if  (!!rct->text) {
        errno = 0;
        n = strtoul(rct->text, NULL, 0);
        if  (!!errno) {
            ifBLIN_QW0("Invalid CTL in #getarg: %s", rct->text);
            ex = EX_DATAERR;
            goto out;
    }   }
    q = NULL;
    nowl = 0;
    for (u_int64_t i = 0; (i < n) && (i < MULAR_NEXT(exenv->options->deargs)); ++i) {
        o = *(char**)mular_getix(exenv->options->deargs, (size_t)i);
        ifBLIN_QX4("[%"BLIN_O"u]%"BLIN_X"=%s%s", i, BLIN_I(o), o ? o : "", o ? "~" : "");
        if  (nowl < (strlen(o) * 2 + 2)) {
            q = realloc(q, strlen(o) * 2 + 2);
            nowl = strlen(o) * 2 + 2;
        }
        if  (!q) {
            ifBLIN_QW0("No mem");
            ex = EX_OSERR;
            goto out;
        }
        if  (0 > (l = pgoblin_onecopy(q, o, -1, 0))) {
            ifBLIN_QW0("pgoblin_onecopy");
            ex = EX_SOFTWARE;
            goto out;
        }
        q[l] = '\0';
        if  (0 > fprintf(fi, "%"BLIN_O"u\t%s", i, q)) {
            ifBLIN_QW0("write error");
            ex = EX_IOERR;
            goto out;
    }   }
    free(q);
empty:
    if  ((~exenv->options->flags & PGOBLIN_COPEND_NOGEN) && (0 > fprintf(fi, "\\.\n"))) {
        ifBLIN_QW0("fprintf fun");
        free(q);
        ex = EX_IOERR;
        goto out;
    }
out:
    if  (0 > pgoblin_io_funclose(rou)) {
        ifBLIN_QW0("io_funclose on %d", rn[PGO_COUT]);
        ex = EX_OSERR;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getenv(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    char        *esco = NULL;
    ssize_t      een;
    size_t       len = 1024;
    pgoblin_rio *rou = NULL;
    char        *in  = NULL;
    int          ex  = EX_OK;
    FILE        *fi  = NULL;
    char        *p;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+getenv %c.....", pgoblin_regn[rn[PGO_COUT]]);
    GET_RIO(rou, exenv->options, rn[PGO_COUT]);
    if  (!(fi = pgoblin_io_funopen(rou))) {
        ifBLIN_QW0("funopen on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
        goto out;
    }
    if  (!(in = malloc(len))) {
        ifBLIN_QW0("malloc in %"BLIN_D, len);
        ex = EX_OSERR;
        goto out;
    }
    if  (!(esco = malloc(len * 2 + 2))) {
        ifBLIN_QW0("malloc esco %"BLIN_D, len * 2 + 2);
        ex = EX_OSERR;
        goto out;
    }
    for (int i = 0; !ex && !!exenv->options->envp[i]; i++) {
        if  (!(len > strlen(exenv->options->envp[i]))) {
            free(in);
            free(esco);
            len *= 2;
            if  (!(in = malloc(len))) {
                ifBLIN_QW0("malloc in %"BLIN_D, len);
                ex = EX_OSERR;
                goto out;
            }
            if  (!(esco = malloc(len * 2 + 2))) {
                ifBLIN_QW0("malloc esco %"BLIN_D, len * 2 + 2);
                ex = EX_OSERR;
                goto out;
        }   }
        strlcpy(in, exenv->options->envp[i], len);
        if  (!!(p = strchr(in, '='))) *(p++) = '\0';
        if  (0 > (een = pgoblin_onecopy(esco, in, -1, -1))) {
            ifBLIN_QW0("pgoblin_onecopy name");
            ex = (int)-een;
            goto out;
        }
        if  (0 > fprintf(fi, "%d	%.*s", i, (int)een, esco)) {
            ifBLIN_QW0("");
            ex = EX_IOERR;
            goto out;
        }
        if  (!p) {
            if  (0 > fprintf(fi, "\n")) {
                ifBLIN_QW0("");
                ex = EX_IOERR;
                goto out;
            }
        } else {
            if  (0 > (een = pgoblin_onecopy(esco, p, -1, 0))) {
                ifBLIN_QW0("pgoblin_onecopy value");
                ex = (int)-een;
                goto out;
            }
            if  (0 > fprintf(fi, "%.*s", (int)een, esco)) {
                ifBLIN_QW0("");
                ex = EX_IOERR;
                goto out;
    }   }   }
    if  ((~exenv->options->flags & PGOBLIN_COPEND_NOGEN) && (0 > fprintf(fi, "\\.\n"))) {
        ifBLIN_QW0("fprintf fun");
        free(esco);
        ex = EX_IOERR;
        goto out;
    }
out:
    free(in);
    free(esco);
    if  (0 > pgoblin_io_funclose(rou)) {
        ifBLIN_QW0("io_funclose on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

#ifdef BLIN_OSLinux

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getproc(pgoblin_exenv *exenv, pgoblin_nr *rn) {             /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
#   define LENO                4096
#   define EXEPATHLEN          32
    struct dirent     *dentry;
    DIR               *dirp  = NULL;
    char              *esco  = NULL;
    ssize_t            lee   = 0;          /* size of the esco          */
    ssize_t            een   ;
    ssize_t            ssz   ;
    pgoblin_rio       *rou   ;
    char               exe   [EXEPATHLEN]; /* /proc/NNNN for request    */
    char              *buf   = NULL;
    size_t             len   = 0;          /* size of the buf           */
    ssize_t            sz    = 0;          /* size of a path in the buf */
    mife_descriptor   *md    = NULL;
    FILE              *fi    = NULL;
    int                ex    = EX_OK;
    char              *p     ;

    ifBLIN_QX3( "+getproc %c.....", pgoblin_regn[rn[PGO_COUT]]);
    GET_RIO(rou, exenv->options, rn[PGO_COUT]);
    if  (!(fi = pgoblin_io_funopen(rou))) {
        ifBLIN_QW0("getproc funopen on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
        goto out;
    }
    if  (!(dirp = opendir("/proc"))) {
        ifBLIN_QW0("opendir /proc");
        ex = EX_IOERR;
        goto out;
    }
    if  (!(buf = malloc((len = LENO)))) {
        ifBLIN_QW0("");
        ex = EX_OSERR;
        goto out;
    }
    if  ((lee < ((2 * len) + 2)) && !(esco = malloc((lee = (2 * len) + 2)))) {
        ifBLIN_QW0("malloc =*2");
        ex = EX_OSERR;
        goto out;
    }
    for (u_int32_t i = 0; ; ++i) {
        errno = 0;
        if  (!(dentry = readdir(dirp))) {
            if  (!errno) break;
            ifBLIN_QW0("readdir %u", i);
            ex = EX_IOERR;
            goto out;
        }
        if  (!!dentry->d_name[strspn(dentry->d_name, "0123456789")]) continue;
        ex = snprintf(exe, EXEPATHLEN, "/proc/%s/exe", dentry->d_name);
        if  (0 > ex) {
            ifBLIN_QW0("snprintf %s", dentry->d_name);
            ex = EX_IOERR;
            goto out;
        }
        if  (EXEPATHLEN <= ex) {
            ifBLIN_QW0("Too long PID %s", dentry->d_name);
            ex = EX_OSERR;
            errno = ESRCH;
            goto out;
        }
        do {
            sz = readlink(exe, buf, len);
            if  (0 > sz) {
                if  (EACCES == errno) break;
                ifBLIN_QW0("readlink %s", exe);
                ex = EX_OSERR;
                goto out;
            }
            if  (sz >= len) {
                if  (!(buf = reallocf(buf, (len *= 2)))) {
                    ifBLIN_QW0("reallocf %zu", len);
                    ex = EX_OSERR;
                    goto out;
                }
                if  ((lee < ((2 * len) + 2)) && !(esco = reallocf(esco, (lee = (2 * len) + 2)))) {
                    ifBLIN_QW0("malloc =*2");
                    ex = EX_OSERR;
                    goto out;
                }
                continue;
            }
            buf[sz] = '\0';
        } while(0);
        if  (0 <= sz) {
            een = pgoblin_onecopy(esco, buf, strnlen(buf, sz), 0);
            ex = fprintf(fi, "%s	-1	%.*s", dentry->d_name, (int)een, esco);
        } else {
            ex = snprintf(exe, EXEPATHLEN, "/proc/%s/comm", dentry->d_name);
            if  (0 > ex) {
                ifBLIN_QW0("snprintf %s", dentry->d_name);
                ex = EX_IOERR;
                goto out;
            }
            if  (EXEPATHLEN <= ex) {
                ifBLIN_QW0("Too long PID %s", dentry->d_name);
                ex = EX_OSERR;
                errno = ESRCH;
                goto out;
            }
            if  (!(md = mife_init(MIFE_PIPE | MIFE_FULL))) {
                ifBLIN_QW0("mife_init");
                ex = EX_OSERR;
                goto out;
            }
            een = -1;
            do {
                if  (0 > mife_ctlfile(md, exe)) {
                    ifBLIN_QW1("mife_ctlfile");
                    break;
                }
                if  (0 > (sz = mife_read(md, 0, 0))) {
                    ifBLIN_QW1("mife_read");
                    break;
                }
                if  (!(p = mife_get(md, 0))) {
                    ifBLIN_QW1("mife_get");
                    break;
                }
                if  ('\n' == p[sz - 1]) p[--sz] = '\0';
                if  ((lee < ((2 * sz) + 2)) && !(esco = reallocf(esco, (lee = (2 * sz) + 2)))) {
                    ifBLIN_QW0("malloc =*2");
                    ex = EX_OSERR;
                    goto out;
                }
                een = pgoblin_onecopy(esco, p, sz, 0);
                ex = fprintf(fi, "%s	-1	%.*s", dentry->d_name, (int)een, esco);
            } while(0);
            if  (!!md && (0 > mife_fini(md))) ifBLIN_QW0("mife_fini");
            md = NULL;
            if  (0 > een) ex = fprintf(fi, "%s	-1	\\N\n", dentry->d_name);
        }
        if  (0 > ex) {
            ifBLIN_QW0("fprintf");
            ex = EX_IOERR;
            goto out;
        }
        ex = snprintf(exe, EXEPATHLEN, "/proc/%s/cmdline", dentry->d_name);
        if  (0 > ex) {
            ifBLIN_QW0("snprintf %s", dentry->d_name);
            ex = EX_IOERR;
            goto out;
        }
        if  (EXEPATHLEN <= ex) {
            ifBLIN_QW0("Too long PID %s", dentry->d_name);
            ex = EX_OSERR;
            errno = ESRCH;
            goto out;
        }
        if  (!(md = mife_init(MIFE_PIPE))) {
            ifBLIN_QW0("mife_init");
            ex = EX_OSERR;
            goto out;
        }
        if  (0 > mife_ctlfile(md, exe)) {
            ifBLIN_QW0("mife_ctlfile");
            ex = EX_IOERR;
            goto out;
        }
        if  (0 > (sz = mife_read(md, 0, 0))) {
            ifBLIN_QW0("mife_read");
            ex = EX_IOERR;
            goto out;
        }
        if  (!(p = mife_get(md, 0))) {
            ifBLIN_QW0("mife_get");
            ex = EX_IOERR;
            goto out;
        }
        for (ssize_t i = 0, j = 0; j < sz; ++i, ++j) {
            ssz = strnlen(&p[j], sz);
            if  ((lee < ((2 * ssz) + 2)) && !(esco = reallocf(esco, (lee = (2 * ssz) + 2)))) {
                ifBLIN_QW0("malloc =*2");
                ex = EX_OSERR;
                goto out;
            }
            een = pgoblin_onecopy(esco, &p[j], ssz, 0);
            if  (0 > fprintf(fi, "%s	%zd	%.*s", dentry->d_name, i, (int)een, esco)) {
                ifBLIN_QW0("fprintf");
                ex = EX_IOERR;
                goto out;
            }
            j += strcspn(&p[j], "");
        }
        if  (!!md && (0 > mife_fini(md))) ifBLIN_QW0("mife_fini");
        md = NULL;
    }
    ex = 0;
    if  ((~exenv->options->flags & PGOBLIN_COPEND_NOGEN) && (0 > fprintf(fi, "\\.\n"))) {
        ifBLIN_QW0("fprintf fun");
        ex = EX_IOERR;
        goto out;
    }
out:
    if  (!!md && (0 > mife_fini(md))) {
        ifBLIN_QW0("");
        ex = EX_IOERR;
    }
    free(buf);
    if  (!!dirp && (0 > closedir(dirp))) {
        ifBLIN_QW0("");
        ex = EX_IOERR;
    }
    free(esco);
    if  (0 > pgoblin_io_funclose(rou)) {
        ifBLIN_QW0("io_funclose on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

#else

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getproc(pgoblin_exenv *exenv, pgoblin_nr *rn) {             /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
#   define LENO                4096
    int                mibl  [8];    /* mib kern.proc.proc           */
    int                mibp  [8];    /* mib kern.proc.pathname       */
    int                miba  [8];    /* mib kern.proc.args           */
    char              *esco  = NULL;
    char              *syo   = NULL;
    char              *syp   = NULL;
    pgoblin_rio       *rou   ;
    size_t             len   ;
    size_t             fen   ;
    ssize_t            een   ;
    FILE              *fi    = NULL;
    int                ex    = EX_OK;
    struct kinfo_proc *kp    = NULL; /* sysctl kern.proc.proc        */
    size_t             ml    ;       /*  mib kern.proc.proc     */
    size_t             mp    ;       /*  mib kern.proc.pathname */
    size_t             ma    ;       /*  mib kern.proc.args     */
    size_t             s     ;

    ifBLIN_QX3( "+getproc %c.....", pgoblin_regn[rn[PGO_COUT]]);
    GET_RIO(rou, exenv->options, rn[PGO_COUT]);
    if  (!(fi = pgoblin_io_funopen(rou))) {
        ifBLIN_QW0("getproc funopen on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
        goto out;
    }
    ml = 8;
    if  (0 > (ex = sysctlnametomib("kern.proc.proc", mibl, &ml))) {
        ifBLIN_QW0("sysctlnametomib kern.proc.proc");
        ex = EX_OSERR;
        goto out;
    }
    ma = 8;
    if  (0 > (ex = sysctlnametomib("kern.proc.args", miba, &ma))) {
        ifBLIN_QW0("sysctlnametomib kern.proc.args");
        ex = EX_OSERR;
        goto out;
    }
    mp = 8;
    if  (0 > (ex = sysctlnametomib("kern.proc.pathname", mibp, &mp))) {
        ifBLIN_QW0("sysctlnametomib kern.proc.pathname");
        ex = EX_OSERR;
        goto out;
    }
    if  (!(syo = malloc(LENO))) {
        ifBLIN_QW0("malloc =");
        ex = EX_OSERR;
        goto out;
    }
    if  (!(esco = malloc(2 * LENO) + 2)) {
        ifBLIN_QW0("malloc =*2");
        ex = EX_OSERR;
        goto out;
    }
    len = 0;
    if  (0 > (ex = sysctl(mibl, (u_int)ml, NULL, &len, NULL, 0))) {
        ifBLIN_QW0("sysctl kern.proc.proc 0");
        ex = EX_OSERR;
        goto out;
    }
    if  (!(kp = malloc(len))) {
        ifBLIN_QW0("malloc %"BLIN_D, len);
        ex = EX_OSERR;
        goto out;
    }
    if  (0 > (ex = sysctl(mibl, (u_int)ml, kp, &len, NULL, 0))) {
        ifBLIN_QW0("sysctl kern.proc.proc %"BLIN_D, len);
        ex = EX_OSERR;
        goto out;
    }
    for (u_int i = 0; i * sizeof(struct kinfo_proc) < len; i++) {
        /*    kinfo_proc kp[] */
        mibp[mp] = kp[i].ki_pid;
        miba[ma] = kp[i].ki_pid;
        fen = LENO;
        if  (0 > (ex = sysctl(mibp, (u_int)mp + 1, syo, &fen, NULL, 0))) {
            ifBLIN_QW1("sysctl kern.proc.pathname pid=%d", kp[i].ki_pid);
            if  (0 > fprintf(fi, "%d	-1	\\N\n", kp[i].ki_pid)) {
                ifBLIN_QW0("");
                ex = EX_IOERR;
                goto out;
            }
        } else {
            s = strnlen(syo, fen);
            if  (!s) {
                syp = kp[i].ki_tdname;
                s = strnlen(syp, TDNAMLEN);
            } else {
                syp = syo;
            }
            if  (0 > (een = pgoblin_onecopy(esco, syp, (ssize_t)s, 0))) {
                ifBLIN_QW0("pgoblin_onecopy pathname");
                ex = (int)-een;
                goto out;
            }
            if  (0 > fprintf(fi, "%d	-1	%.*s", kp[i].ki_pid, (int)een, esco)) {
                ifBLIN_QW0("");
                ex = EX_IOERR;
                goto out;
        }   }
        fen = LENO;
        if  (0 > (ex = sysctl(miba, (u_int)ma + 1, syo, &fen, NULL, 0))) {
            ifBLIN_QW1("sysctl kern.proc.args pid=%d", kp[i].ki_pid);
            continue;
        }
        syp = syo;
        for (int k = 0; !!fen; ++k) {
            s = strnlen(syp, fen);
            if  (s < fen) {
                if  (0 > (een = pgoblin_onecopy(esco, syp, (ssize_t)s, 0))) {
                    ifBLIN_QW0("pgoblin_onecopy arg");
                    ex = (int)-een;
                    goto out;
                }
                if  (0 > fprintf(fi, "%d	%d	%.*s", kp[i].ki_pid, k, (int)een, esco)) {
                    ifBLIN_QW0("");
                    ex = EX_IOERR;
                    goto out;
                }
                if  (fen <= s) break;
                fen -= s + 1;
                syp += s + 1;
            } else {
                if  (0 > (een = pgoblin_onecopy(esco, syp, (ssize_t)fen, 0))) {
                    ifBLIN_QW0("pgoblin_onecopy last arg");
                    ex = (int)-een;
                    goto out;
                }
                if  (0 > fprintf(fi, "%d	%d	%.*s", kp[i].ki_pid, k, (int)een, esco)) {
                    ifBLIN_QW0("");
                    ex = EX_IOERR;
                    goto out;
                }
                break;
    }   }   }
    if  ((~exenv->options->flags & PGOBLIN_COPEND_NOGEN) && (0 > fprintf(fi, "\\.\n"))) {
        ifBLIN_QW0("fprintf fun");
        free(esco);
        ex = EX_IOERR;
        goto out;
    }
out:
    free(kp);
    free(syo);
    free(esco);
    if  (0 > pgoblin_io_funclose(rou)) {
        ifBLIN_QW0("io_funclose on %d", rn[PGO_COUT]);
        ex = EX_IOERR;
    }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

#endif
