/*-
 * Copyright (C)2002..2018 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2002..2018 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: cmd_trans.c,v 1.107 2018/12/04 09:20:26 babolo Exp $"

#define BLIN_COMPAT      3
#define Bpars_COMPAT     3
#define MULAR_COMPAT     0
#define MIFE_COMPAT      4
#define PGOBLIN_COMPAT   2
#define PGOBLIN_INTERNAL 1

#include <sys/types.h>
#include <sys/param.h>
#if defined(__FreeBSD_cc_version) && __FreeBSD_cc_version > 599999 /* I do not know exact version */
#include <arpa/inet.h>
#endif
#include <sys/wait.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sysexits.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <poll.h>
#include <err.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "pgoblin.h"

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_getjob(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 ** pgoblin_getjob()   2  ,                **
 **     ,                          **
 **     .                              **
 **  pgoblin_syntax                          **
 **   - Ij  ch[2].                              **
 **   - ,                        **
 ** (  )                    **
 **   ,   .                  **
 **   R_JOBIN,   R_JOB                       **
 **********************************************************************
 **********************************************************************/
#define R_JOBIN exenv->options->job[rn[PGO_CJOB]]
    int           status;
    int           exx = EX_OK;
    int           ex = EX_OK;
    FILE         *fi;
    pgoblin_jobl *j;
    size_t        i;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3("+getjob %c...%c.", pgoblin_regn[rn[PGO_COUT]], pgoblin_regn[rn[PGO_CJOB]]);
    if  (R_JOBIN.pids) {
        if  (!(fi = funopen( (void*)(size_t)(exenv->options->io[rn[PGO_COUT]].onu)
                           , NULL
                           , (int (*)(void*, const char*, int))mife_writ
                           , NULL
                           , NULL
            ) )            ) {
            ifBLIN_QW0("funopen on %d", rn[PGO_CJOB]);
            ex = EX_OSERR;
        } else {
            for (i = 0; i <  MULAR_NEXT(R_JOBIN.pids); i++) {
                j = mular_getix(R_JOBIN.pids, i);
                if  (j->pid <= 0) {
                    ifBLIN_QX0("No pid in %"BLIN_D, j->cmdn);
                    errno = EINVAL;
                    ex = EX_SOFTWARE;
                } else if ((j->cmdn >= 0) && (j->status == PGOBLIN_NOSTATUS)) {
                    if  (!(ex = waitpid(j->pid, &status, WNOHANG))) {
                    } else if (ex == j->pid) {
                        j->status = status;
                        R_JOBIN.dead++;
                        ifBLIN_QU3( R_JOBIN.flags
                                  , "pgoblin_getjob st=%08X fl=%08X ex=%d exx=%d IFEX=%d EXST=%d"
                                  , status, R_JOBIN.flags, ex, exx
                                  , WIFEXITED(status), WEXITSTATUS(status)
                                  );
                        if  (  (R_JOBIN.flags & PGOBLIN_ERROR_SENSOR)
                            && WIFEXITED(status)
                            && WEXITSTATUS(status)
                            ) exx = WEXITSTATUS(status);
                        if  ((R_JOBIN.flags & PGOBLIN_ERROR_SENSIB) && exx) break;
                        ex = EX_OK;
                    } else if (ex > 0) {
                        ifBLIN_QW0("Pid %d<>%d in %d", ex, j->pid, (int)j->cmdn);
                        ERROUT(EX_OSERR, ENOATTR);
                    } else if (errno == ECHILD || errno == EDEADLK || errno == EAGAIN) {
                        ex = EX_OK;
                    } else {
                        ifBLIN_QW0("Pid %d in %d", j->pid, (int)j->cmdn);
                        ex = EX_OSERR;
                        goto out;
                }   }
                fprintf(fi, "%d	%d	%d	%d\n", (int)i, (int)j->cmdn, j->pid, j->status);
            }
            fprintf(fi, "\\.\n");
            if  (fclose(fi) < 0) {
                ifBLIN_QW0("fclose on %d", rn[PGO_CJOB]);
                ex = EX_OSERR;
    }   }   }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex ? ex : exx);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_exec(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_io *ctlreg;
    char      **argv = NULL;
    int         ex = EX_OK;
    int         jn;
    int         i;
    int         g = 0;
    int         c;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+exec %c%c%c.%c."
              , pgoblin_regn[rn[PGO_COUT]]
              , pgoblin_regn[rn[PGO_CCTL]]
              , pgoblin_regn[rn[PGO_CIN]]
              , pgoblin_regn[rn[PGO_CJOB]]
              );
    ctlreg = &(exenv->options->io[rn[PGO_CCTL]]);
    jn = exenv->options->job[rn[PGO_CJOB]].flags & PGOBLIN_JOB_TYPE;
    g = pgoblin_db_resinfo(PGOBLIN_Nfields, ctlreg->pq);
    i = 0;
    if  (g && !(argv = malloc((g + 1) * sizeof(char**)))) {
        ERROUT(EX_OSERR, ENOMEM);
    }
    for (c = 0; c < g; c++) {
        if  (!(pgoblin_db_valinfo(PGOBLIN_IsNull, ctlreg->pq, exenv->sequen, c))) {
            argv[i++] = pgoblin_db_getvalue(ctlreg->pq, exenv->sequen, c);
    }   }
    if  (i) {
        int eex;
        argv[i] = NULL;
        eex = pgoblin.jobtable[jn]->jxec(exenv, rn, argv);
        if  (eex) {
            ifBLIN_QO0 {
                int z;

                warn("Exec error on %d", rn[PGO_CJOB]);
                for (z = 0; argv[z]; z++) fprintf(stderr, "[%d]=%s~\n", z, argv[z]);
            }
            if  (eex < 0) {
                if  (eex  < ex) ex = eex;
            } else if (ex >= 0) {
                if  (eex > ex) ex = eex;
    }   }   }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_fork(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    int ex = EX_OK;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+fork %c.%c.%c."
              , pgoblin_regn[rn[PGO_COUT]]
              , pgoblin_regn[rn[PGO_CIN]]
              , pgoblin_regn[rn[PGO_CJOB]]
              );
    /* if  (R_JOB) XXXX?? */
    ex = pgoblin.jobtable[exenv->options->job[rn[PGO_CJOB]].flags & PGOBLIN_DB_TYPE]->jork(exenv, rn);
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_wait(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    int             ex = EX_OK;
    pgoblin_jobreg *j;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3("+wait ....%c.", pgoblin_regn[rn[PGO_CJOB]]);
    j = &(exenv->options->job[rn[PGO_CJOB]]);
    /* if  (R_JOB) XXXX?? */
    ex = pgoblin.jobtable[j->flags & PGOBLIN_JOB_TYPE]->jait(exenv, rn);
    MARK_R_JOB_NGO(rn[PGO_CJOB]);;
    ;;  j->dead = MULAR_NEXT(j->pids);
    MARK_R_JOB_NWENT(rn[PGO_CJOB]);;
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_untrap(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    ifBLIN_QX3("+untrap ......");
    exenv->flags &= ~PGOBLIN_TRAPDEF;
    ifBLIN_QX3("- %d", EX_OK);
    return(EX_OK);
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_trap(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    ifBLIN_QX3("+trap ......");
    exenv->ctrap = exenv->ppoint + 1,
    exenv->flags |= PGOBLIN_TRAPDEF,
    exenv->lexlevel++;
    ifBLIN_QX3("- %d", EX_OK);
    return(EX_OK);
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_call(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    int ex = EX_UNAVAILABLE;

    ifBLIN_QX3( "+call ......");
    ifBLIN_QX3("- %d", ex);
    return(ex);
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_eval(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_exenv *nexenv;
    pgoblin_io    *io;
    int            ex = EX_OK;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3("+eval .%c....", pgoblin_regn[rn[PGO_CCTL]]);
    if  (!(nexenv = calloc(1, sizeof(pgoblin_exenv)))) {
        ifBLIN_QW0("no mem for nexenv");
        ex = EX_OSERR;
        goto out;
    }
    nexenv->options = exenv->options;
    nexenv->flags = exenv->flags & BLIN_MASK;
    nexenv->stat = exenv;
    nexenv->dyna = exenv;
    io = &(exenv->options->io[rn[PGO_CCTL]]);
    if  ((ex = pgoblin_parser(exenv->options, io->text, io->length, &nexenv->pgm))) {
        ifBLIN_QW0("Parser failed in #eval");
    } else {
        nexenv->pgm->flags |= io->flags & PGOBLIN_FREETEXT;
        if  (nexenv->options->flags & PGOBLIN_VERIFY) pgoblin_Vdump(nexenv->options, nexenv->pgm);
        if  ((ex = pgoblin_execute(nexenv->options, nexenv->pgm, nexenv->dyna, nexenv->stat, 0))) {
            ifBLIN_QX0( "cmd[%d]=%03X(%s %c%c%c%c%c%c) %08X Executor failed in #eval"
                      , nexenv->ppoint, nexenv->pgm->code[nexenv->ppoint].cmd
                      , pgoblin.a[nexenv->pgm->code[nexenv->ppoint].cmd & PGOBLIN_COMMAND].str
                      , pgoblin_regn[rn[0]], pgoblin_regn[rn[1]], pgoblin_regn[rn[2]]
                      , pgoblin_regn[rn[3]], pgoblin_regn[rn[4]], pgoblin_regn[rn[5]]
                      , pgoblin.a[nexenv->pgm->code[nexenv->ppoint].cmd & PGOBLIN_COMMAND].flags
                      );
    }   }
    if  (nexenv->pgm) free(nexenv->pgm);
    if  (nexenv) free(nexenv);
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_close(pgoblin_exenv *exenv, pgoblin_nr *rn) {               /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    ifBLIN_QX3( "+eval %c.%c%c.%c"
              , pgoblin_regn[rn[PGO_COUT]]
              , pgoblin_regn[rn[PGO_CIN]]
              , pgoblin_regn[rn[PGO_CCON]]
              , pgoblin_regn[rn[PGO_CSTY]]
              );
    if  (rn[PGO_CCON]) {
        MARK_R_CONN_NGO(rn[PGO_CCON]);;
        ;;  exenv->options->conn[rn[PGO_CCON]].flags &= ~PGOBLIN_CLOSECONN;
        MARK_R_CONN_NWENT(rn[PGO_CCON]);;
    }
    ifBLIN_QX3("- %d", EX_OK);
    return(EX_OK);
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_clear(pgoblin_exenv *exenv, pgoblin_nr *rn) {               /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    ifBLIN_QX3( "+eval %c...%c%c"
              , pgoblin_regn[rn[PGO_COUT]]
              , pgoblin_regn[rn[PGO_CJOB]]
              , pgoblin_regn[rn[PGO_CSTY]]
              );
    ifBLIN_QX3("- %d", EX_OK);
    return(EX_OK);
}

static void
/**********************************************************************
 **                                                                  **/
exitfin(pgoblin_exenv *exenv, pgoblin_nr *rn, int ex) {             /**
 **                                                                  **
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    int g;
    int f;
    int t;

    f = 0, t = pgoblin.regsize;
    if  (rn[PGO_CCON]) {
        f = rn[PGO_CCON];
        t = f + 1;
    }
    for (g = f; g < t; g++) {
#       define O(G) (exenv->options->conn[G])
        MARK_R_CONN_NGO(g);;
        ;;  if  (O(g).flags & PGOBLIN_CLOSECONN)
    /*  ;;  */  pgoblin_db_finish(O(g).flags, PGOBLIN_Connect, &O(g).conn);
        ;;  O(g).conn = NULL;
        ;;  if  (O(g).flags & PGOBLIN_PARMDBFREE) free((char*)O(g).dbname);
        ;;  O(g).dbname = NULL;
        ;;  O(g).flags = exenv->options->conn[0].flags & PGOBLIN_DB_TYPE;
        ;;  O(g).host = NULL;
        ;;  O(g).port = NULL;
        ;;  O(g).username = NULL;
        ;;  O(g).intran = 0;
        MARK_R_CONN_NWENT(g);;
#       undef O
    }
    ifBLIN_QX3("pgoblin exit %d", ex);
    exit(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_return(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    int ex = EX_OK;

    ifBLIN_QX3("+return ...%c..", pgoblin_regn[rn[PGO_CCON]]);
    if  (exenv->flags & PGOBLIN_FORKED) exitfin(exenv, rn, ex);
    exenv->flags |= PGOBLIN_END;
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_exit(pgoblin_exenv *exenv, pgoblin_nr *rn) {                /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    int ex = EX_OK;

    ifBLIN_QX3("+exit .%c.%c..", pgoblin_regn[rn[PGO_CCTL]], pgoblin_regn[rn[PGO_CCON]]);
    if  (exenv->options->io[rn[PGO_CCTL]].text) {
        ex = strtol(exenv->options->io[rn[PGO_CCTL]].text, NULL, 0);
    }
    exitfin(exenv, rn, ex);
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_begin(pgoblin_exenv *exenv, pgoblin_nr *rn) {               /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_conn   *conn;
    pgoblin_mode   *res;
    pgoblin_dbases *db;
    int             ex = EX_OK;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3("+begin ...%c..", pgoblin_regn[rn[PGO_CCON]]);
    conn = &(exenv->options->conn[rn[PGO_CCON]]);
    db = pgoblin.dbasetable[conn->flags & PGOBLIN_DB_TYPE];
    MARK_R_CONN_NGO(rn[PGO_CCON]);;
    ;;  conn->flags &= ~PGOBLIN_ACR_MASK;
    ;;  conn->flags |= PGOBLIN_CR_OK;
    MARK_R_CONN_NWENT(rn[PGO_CCON]);;
    if  (  !(conn->intran++)
        && !(ex = pgoblin_db_query(conn, &res, "BEGIN"))
        ) {
        pgoblin_db_clear(&res);
    }
    MARK_R_CONN_NGO(rn[PGO_CCON]);;
    ;;  conn->flags &= ~PGOBLIN_ACR_MASK;
    MARK_R_CONN_NWENT(rn[PGO_CCON]);;
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_end(pgoblin_exenv *exenv, pgoblin_nr *rn) {                 /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_conn   *conn;
    pgoblin_mode   *res;
    pgoblin_dbases *db;
    int             ex = EX_OK;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3("+end ...%c..", pgoblin_regn[rn[PGO_CCON]]);
    conn = &(exenv->options->conn[rn[PGO_CCON]]);
    db = pgoblin.dbasetable[conn->flags & PGOBLIN_DB_TYPE];
    if  (!conn->intran) {
        ifBLIN_QX0("Non transaction end");
        ERROUT(EX_DATAERR, EINVAL);
    }
    MARK_R_CONN_NGO(rn[PGO_CCON]);;
    ;;  conn->flags &= ~PGOBLIN_ACR_MASK;
    ;;  conn->flags |= PGOBLIN_CR_OK;
    MARK_R_CONN_NWENT(rn[PGO_CCON]);;
    if  (  !(--conn->intran)
        && !(ex = pgoblin_db_query(conn, &res, "END"))
        ) {
        pgoblin_db_clear(&res);
    }
out:
    MARK_R_CONN_NGO(rn[PGO_CCON]);;
    ;;  conn->flags &= ~PGOBLIN_ACR_MASK;
    MARK_R_CONN_NWENT(rn[PGO_CCON]);;
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_listen(pgoblin_exenv *exenv, pgoblin_nr *rn) {              /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    int             ex = EX_OK;
    int             timeout;
    char           *notify;
    int64_t         tout;
    pgoblin_conn   *conn;
    pgoblin_io     *ctl;
    pgoblin_io     *in;
    pgoblin_dbases *db;
    struct timeval  t;
    int             e;
    union {
        int64_t     itime;
        u_int32_t   d[2];
    }               dbtime;
    typedef union {
        u_int64_t   p;
        u_int32_t   d[2];
    } pmatsemit;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+listen .%c%c%c.."
              , pgoblin_regn[rn[PGO_CCTL]]
              , pgoblin_regn[rn[PGO_CIN]]
              , pgoblin_regn[rn[PGO_CCON]]
              );
    ctl = &(exenv->options->io[rn[PGO_CCTL]]);
    in = &(exenv->options->io[rn[PGO_CIN]]);
    if  (!ctl->text) {
        ifBLIN_QX0(" CTL[%d]: no command", rn[PGO_CCTL]);
        ex = EX_DATAERR;
        errno = EINVAL;
    } else if (!in->text || (in->length < sizeof(pmatsemit))) {
        ifBLIN_QX0(" IN[%d]: no timeout", rn[PGO_CIN]);
        ex = EX_DATAERR;
        errno = EINVAL;
    } else {
        /* Should be 64 bit time wait to in usec from 1970-01-01 00:00:00 UTC          *
         * ((date_part('epoch', t))::int8 * 1000000                                    *
         * +((date_part('second', t) - floor(date_part('second', t))) * 1000000)::int8 *
         * ) , where t - time wait to                                                  *
         *******************************************************************************/
        dbtime.d[1] = ntohl(((pmatsemit*)(in->text))->d[0]);
        dbtime.d[0] = ntohl(((pmatsemit*)(in->text))->d[1]);
        conn = &(exenv->options->conn[rn[PGO_CCON]]);
        db = pgoblin.dbasetable[conn->flags & PGOBLIN_DB_TYPE];
        for (; !ex;) {
            for (e = 0; (notify = pgoblin_db_notifies(conn));) {
                ifBLIN_QX3("Received NOTIFY=%s~", notify);
                e = !strcmp(notify, ctl->text);
                pgoblin_db_finish(conn->flags, PGOBLIN_FrNote, (void **)&notify);
            }
            if  (e) break;
            if  (gettimeofday(&t, NULL)) {
                ifBLIN_QW0("gettimeofday:");
                ex = EX_OSERR;
                break;
            }
            tout = dbtime.itime - ((int64_t)1000000LL * t.tv_sec + t.tv_usec);
            if  (tout <= 0) {
                break;
            } else if ((tout / 1000) > exenv->options->maxtmout) {
                timeout = exenv->options->maxtmout;
                dbtime.itime = ((int64_t)1000LL * exenv->options->maxtmout)
                           + ((int64_t)1000000LL * t.tv_sec)
                           + t.tv_usec
                           ;
            } else {
                timeout = tout / 1000;
            }
            ifBLIN_QX3("LISTEN=%s~ for %d ms", (char*)ctl->text, timeout);
            if  (pgoblin_db_tarry(conn, timeout) < 0) {
                ifBLIN_QW0("tarry:");
                ex = EX_OSERR;
                break;
    }   }   }
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

#define L0 0 /* \0           */
#define Ll 1 /*   */
#define Lb 2 /*        */
#define Ls 3 /*    */
#define Ld 4 /* #            */
#define Lo 5 /*     */
#define Lx 6

static const size_t msz[] = {1024, 512, 512};
static const char *clasnames[] = { "L0", "Ll", "Lb", "Ls", "Ld", "Lo", "Lx"};

static const u_char class[256] =
{ L0, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lb, Ll, Ll,  Lo, Ll, Lo, Lo
, Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo
, Lb, Lo, Lo, Ld,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo
, Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Lo, Lo,  Lo, Lo, Lo, Lo

, Lo, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls
, Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Lo,  Lo, Lo, Lo, Ls
, Lo, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls
, Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Lo,  Lo, Lo, Lo, Lo

, Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo
, Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo
, Lo, Lo, Lo, Ls,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo
, Lo, Lo, Lo, Ls,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo,  Lo, Lo, Lo, Lo

, Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls
, Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls
, Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls
, Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls,  Ls, Ls, Ls, Ls
};

static const char statenames[][3] =
{ "bg", "bs", "nq", "tc", "tb", "xx"};

enum states
{ bg /*                                           */
, bs /*                                                 */
, nq /*     ?                       */
, tc /*                           */
, tb /*        */
, xx
};
static const char pn[][3] = 
{ "- ", "- ", "- ", "- ", "- ", "- ", "- ", "- "
, "- ", "- ", "Ci", "- ", "- ", "- ", "- ", "- "
, "Co", "Cl", "Cn", "Cf", "- ", "Cx", "- ", "- "
};

#define Ci  0x200000 /*     */
#define Co  0x008000 /*             */
#define Cl  0x004000 /*                 */
#define Cn  0x002000 /*                      */
#define Cf  0x001000 /*               */
#define Cx  0x000400 /*                    */
#define C_state 0xFF /*           */

static u_int32_t automa[xx][Lx] =
/* \0 L0  \n Ll  \b Lb  Ls   # Ld   * Lo   *    @          */
{{ Cx|xx,    bg,    bs,       bs,    nq,    bs }/* bg @                   */
,{ Cx|xx,    bg,    bs,       bs,    bs,    bs }/* bs @                   */
,{ Cx|xx,    bg,    bs, Co|Cl|tc,    bs,    bs }/* nq @ 1           */
,{ Cx|xx, Cn|bs, Cf|tb,    Cl|tc, Cn|bs, Cx|xx }/* tc              */
,{ Cx|xx, Cn|bs,    tb,    Ci|bs, Cn|bs, Cn|bs }/* tb          */
};

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_style(pgoblin_exenv *exenv, pgoblin_nr *rn) {               /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    size_t          candidate = 0;
    int             lfcount;
    u_int32_t       control;
    size_t          offset;
    enum states     state;
    size_t          found     = 0;
    int             cmp       = 0;
    pgoblin_styreg *sty;
    pgoblin_io     *ctl;
    int             ex        = EX_OK;
    int             rs        = 0;
    pgoblin_io     *in;
    size_t          d         = 0;
    int             i;
    char           *q;
    u_char          p;
    u_char          s;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+style .%c%c..%c"
              , pgoblin_regn[rn[PGO_CCTL]]
              , pgoblin_regn[rn[PGO_CIN]]
              , pgoblin_regn[rn[PGO_CSTY]]
              );
    if  (!rn[PGO_CSTY]) {
        ifBLIN_QX0("Try to change style 0");
        ERROUT(EX_DATAERR, EINVAL);
    }
    ctl = &(exenv->options->io[rn[PGO_CCTL]]);
    if  (!ctl->text) {
        ifBLIN_QX0("No pgoblin style name");
        ERROUT(EX_DATAERR, EINVAL);
    }
    in = &(exenv->options->io[rn[PGO_CIN]]);
    if  (!in->text) {
        ifBLIN_QX0("No pgoblin style text");
        ERROUT(EX_DATAERR, EINVAL);
    }
    sty = &(exenv->options->style[rn[PGO_CSTY]]);
    q = in->text;
    if  (!!in->stymd) {
        for (i = 0; i < (int)MULAR_NEXT(in->stymd); ++i) {
            offset = *(size_t*)mular_getix(in->stymd, i);
            cmp = strcmp(&q[offset], (char*)ctl->text);
            ifBLIN_QX3("%s %d %s", &q[offset], cmp, (char*)ctl->text);
            if  (!cmp) {
                for (found = offset + 1 + strlen(&q[offset]); Lb == class[q[found] & 0xFF]; ++found) {}
                break;
        }   }
    } else {
        MARK_IO_TEXT_NGO(rn[PGO_CIN]);;
        ;;  in->stymd = mular_create(MULAR_ZERO, 3, sizeof(size_t), msz);
        ;;  MARK_IO_MIFE_NGO(rn[PGO_CIN]);;
        ;;  ;;  in->flags &= ~(PGOBLIN_FREETEXT | PGOBLIN_FREEMIFE); /* XXXX */
        ;;  MARK_IO_MIFE_NWENT(rn[PGO_CIN]);;
        MARK_IO_TEXT_NWENT(rn[PGO_CIN]);;
        if  (!in->stymd) {
            ifBLIN_QX0("No mem");
            ERROUT(EX_OSERR, ENOMEM);
        }
        blin_stateheader(BLIN_4STA24G, pn);
        lfcount = 1;
        for (state = bg, offset = 0, control = 0; state < xx; offset++, state = control & C_state) {
            if  (((in->flags & PGOBLIN_BINPARM) || in->length) && (offset >= in->length)) {
                p = L0;
                s = 0;
            } else {
                s = q[offset];
                p = class[s];
                if  (s == '\n') lfcount++;
            }
            control = automa[state][p];
            blin_statebody( BLIN_4STO24G
                          , pn
                          , statenames
                          , clasnames[p]
                          , q
                          , offset
                          , control
                          , state
                          , found
                          , rs
                          );
            if  ((control & Ci) && !found && !cmp) found = offset;
            if  (control & Co) {
                cmp = 0, d = 0;
                candidate = offset;
            }
            if  (control & Cl) {
                cmp |= (((char*)ctl->text)[d++] - q[offset]);
            }
            if  (control & Cn) {
                candidate = 0;
            }
            if  (control & Cf) {
                q[offset] = '\0';
                if  ((!(ctl->flags & PGOBLIN_BINPARM) && !ctl->length) || (d < ctl->length)) {
                    cmp |= ((u_char*)ctl->text)[d++];
                }
                if  (!!candidate) {
                    size_t *ix;
                    ix = mular_add(in->stymd);
                    if  (!ix) {
                        ifBLIN_QW0("mular_add");
                        ex = EX_OSERR;
                        goto out;
                    }
                    *ix = candidate;
                }
                candidate = 0;
            }
            if  ((control & Cx) && !found) {
                ifBLIN_QX0("Erroneous style %"BLIN_D, found);
                ERROUT(EX_DATAERR, EINVAL);
    }   }   }
    if  (!!found) {
        for (rs = 0, offset = found; Ls == class[q[offset] & 0xFF]; ++offset) ++rs;
        if  (0 > (i = pgoblin_load(exenv->options, 5, &q[found], rs))) {
            ifBLIN_QX0("Invalid style interpretor name=%.*s~", rs, &q[found]);
            ERROUT(EX_DATAERR, EINVAL);
        } else {
            MARK_R_STYLE_NGO(rn[PGO_CSTY]);;
            ;;  sty->path = NULL;
            ;;  sty->flags = (exenv->options->flags & BLIN_MASK) | i;
            MARK_R_STYLE_NWENT(rn[PGO_CSTY]);;
            if  ((ex = STYLS(sty->flags)->parser(sty, (u_char*)&q[found + rs]))) {
                ifBLIN_QX0("style parser fail");
            }
            if  (exenv->options->flags & PGOBLIN_VERIFY) {
                STYLS(sty->flags)->dump(sty);
            }
            goto out;
    }   }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static const babolo_lexor modsel =
/* 0(FF) - 
 * 1(FE) - 
 * 2(FD) -  
 * 3(FC) -    
 * 4(FB) -   
 */
{ (u_char*)class, NULL, Bpars_AABS, 0x07, 0x07, 1
, { /*00*/ 1, L0,0xFE, Ls,0x03, 0xFF
  , /*03*/ 2, L0,0xFD, Ls,0x03, Lb,0xFB, 0xFC
} };

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_connect(pgoblin_exenv *exenv, pgoblin_nr *rn) {             /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    pgoblin_conn *conn;
    pgoblin_io   *ctl;
    int           ex = EX_OK;
    int           i;
    char         *q;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3("+connect .%c.%c..", pgoblin_regn[rn[PGO_CCTL]], pgoblin_regn[rn[PGO_CCON]]);
    ctl = &(exenv->options->io[rn[PGO_CCTL]]);
    if  (!(q = ctl->text)) {
        ifBLIN_QX1("Empty DB #1");
        goto out;
    }
    switch(babolo_goword(&modsel, (const char**)&q)) {
    case 0 : ifBLIN_QX0("Illegal DB");
           ; ERROUT(EX_DATAERR, EINVAL);
    case 1 : ifBLIN_QX1("Empty DB #2");
           ; goto out;
    case 3 : ifBLIN_QX1("Illegal DB separator");
           ; *--q = '\0';
           ; /* FALLTHRU */
    case 2 : q = NULL;
           ; break;
    case 4 : *--q = '\0';
           ; q++;
           ; break;
    default: ifBLIN_QX0("Internal error");
           ; ERROUT(EX_SOFTWARE, EINVAL);
    }
    ifBLIN_QX3("db=%s%s%s~", (char*)ctl->text, q ? "~\nparm=" : "", q ? q : "");
    conn = &(exenv->options->conn[rn[PGO_CCON]]);
    if  (0 > (i = pgoblin_load(exenv->options, 3, ctl->text, -1))) {
        ifBLIN_QX0("Invalid DBMS name %s", (char*)ctl->text);
        ERROUT(EX_DATAERR, EINVAL);
    } else {
        u_int32_t nwflags = PGOBLIN_PARMIN;

        if  (!q) {
        } else if (!*q) {
            q = "";
        } else if (ctl->flags & PGOBLIN_FREETEXT) {
            if  (!(q = strdup(q))) {
                ifBLIN_QX0("pgoblin_connect: no mem");
                ERROUT(EX_OSERR, ENOMEM);
            }
            nwflags |= PGOBLIN_PARMDBFREE;
        }
        MARK_R_CONN_NGO(rn[PGO_CCON]);;
        ;;  conn->flags = (exenv->options->flags & BLIN_MASK) | nwflags | i;
        ;;  conn->dbname = q;
        MARK_R_CONN_NWENT(rn[PGO_CCON]);;
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/**********************************************************************
 **********************************************************************
 **                                                                  **/
pgoblin_job(pgoblin_exenv *exenv, pgoblin_nr *rn) {                 /**
 **                                                                  **
 **********************************************************************
 **********************************************************************/
    char       *jbn;
    int         ex = EX_OK;
    const char *p;
    char       *q;
    int         i;

#   define blin_internal_flags (exenv->options->flags & BLIN_MASK)
    ifBLIN_QX3( "+job .%c..%c."
              , pgoblin_regn[rn[PGO_CCTL]]
              , pgoblin_regn[rn[PGO_CJOB]]
              );
    if  (!(q = jbn = exenv->options->io[rn[PGO_CCTL]].text)) {
        ifBLIN_QX1("Empty JOB #1");
        goto out;
    }
    switch(babolo_goword(&modsel, (const char**)&q)) {
    case 0 : ifBLIN_QX0("Illegal JOB");
           ; ERROUT(EX_DATAERR, EINVAL);
    case 1 : ifBLIN_QX1("Empty JOB #2");
           ; goto out;
    case 3 : ifBLIN_QX1("Illegal JOB separator");
           ; *--q = '\0';
           ; /* FALLTHRU */
    case 2 : q = NULL;
           ; break;
    case 4 : *--q = '\0';
           ; q++;
           ; break;
    default: ifBLIN_QX0("Internal error");
           ; ERROUT(EX_SOFTWARE, EINVAL);
    }
    if  (0 > (i = pgoblin_load(exenv->options, 4, jbn, -1))) {
        ifBLIN_QX0("Invalid job module name %s", jbn);
        ERROUT(EX_DATAERR, EINVAL);
    } else {
        u_int32_t u = 0, nwflags = 0;
        babolo_opts *bos = NULL;
        pgoblin_jobreg *j;
        babolo_parm *bp;
        int c;

        ifBLIN_QX4("opt=%s~", q);
        if  (!(bp = babolo_getparm(Bpars_NOEN | Bpars_NOAL, &q, "", 0))) {
            ifBLIN_QW0("babolo_getparm");
            ex = EX_SOFTWARE;
            goto out;
        }
        if  (!(bos = babolo_openopts(exenv->options->flags & BLIN_MASK, 0))) {
            ifBLIN_QW0("babolo_openopts");
            ex = EX_SOFTWARE;
            goto out;
        }
        if  (babolo_setopts(bos, 0, bp->argc, bp->argv, "zZ")) {
            ifBLIN_QW0("babolo_setopts");
            ex = EX_SOFTWARE;
            goto out;
        }
        while (0 < (c = babolo_getopts(bos))) {
            switch (c) {
            case 'z': nwflags |= (nwflags & PGOBLIN_ERROR_SENSOR) << 1;
                    ; nwflags |= PGOBLIN_ERROR_SENSOR;
                    ; break;
            case 'Z': nwflags &= ~(PGOBLIN_ERROR_SENSIB | PGOBLIN_ERROR_SENSOR);
                    ; break;
            default : ifBLIN_QX0("Illegal job flag");
        }   }
        ifBLIN_QX3("nwflags=%08X", nwflags);
        if  ((p = babolo_getargs(bos)) && *p) {
            errno = 0;
            u = strtoul(p, NULL, 0);
            if  (errno) {
                ifBLIN_QW0("No num =%s~", p);
                u = 0;
        }   }
        j = &(exenv->options->job[rn[PGO_CJOB]]);
        if  ((j->flags & PGOBLIN_JOB_TYPE) == i) {
            u_int32_t svflags;

            svflags = j->flags & ~(BLIN_MASK | PGOBLIN_ERROR_SENSIB | PGOBLIN_ERROR_SENSOR);
            MARK_R_JOB_NGO(rn[PGO_CJOB]);;
            ;;  j->flags = (exenv->options->flags & BLIN_MASK) | svflags | nwflags;
            ;;  j->concur = u;
            MARK_R_JOB_NWENT(rn[PGO_CJOB]);;
        } else {
            ex = pgoblin.jobtable[j->flags & PGOBLIN_JOB_TYPE]->jose(exenv, rn);
            MARK_R_JOB_NGO(rn[PGO_CJOB]);;
            ;;  mular_destroy(j->pids);
            ;;  j->pids = NULL;
            ;;  j->host = NULL;
            ;;  j->concur = u;
            ;;  j->dead = 0;
            ;;  j->flags = (exenv->options->flags & BLIN_MASK) | nwflags | i;
            MARK_R_JOB_NWENT(rn[PGO_CJOB]);;
            ifBLIN_QW0("Close JOB in #job");
        }
        pgoblin.jobtable[j->flags & PGOBLIN_JOB_TYPE]->jpen(exenv, rn, bos);
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}
