/*-
 * Copyright (C)2008..2011 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] = "\
@(#)Copyright (C)2008..2011 @BABOLO http://www.babolo.ru/\n\
@(#)All rights reserved.\n";
static const char rcsid[] = "$Id: db_pgsql.c,v 1.30 2013/02/05 06:14:37 babolo Exp $";
#endif /* not lint */

#include <sys/types.h>
#include <sysexits.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <errno.h>
#include <poll.h>
#include <err.h>
#include <libpq-fe.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <mife.h>
#include "pgoblin.h"
#include "pgob.h"

#define PRINTNOTNULL(A) (!(A) ? "" : (A))
#define PRINTANDNULL(A) PRINTNOTNULL(A), ((A) ? "~" : "")
#define PQSTATNDNULL(A) PRINTANDNULL(PQparameterStatus(db->conn, (A)))
#define ALIGN8(A)  popt[i].A, (strlen(popt[i].A) < 8 ? "		" : "	")
#define ALIGN16(A) popt[i].A                                                                          \
      , ((strlen(popt[i].A) < 8) ? "		" : (strlen(popt[i].envvar) < 16) ? "	" : " ")

#define PQEXEC(RESULT, QUERY, CONDITION, CCODE, RCONN) {                                              \
    ifBLIN_QV5(db->flags) fprintf(stderr, "PQexec: %s\n", (char*)(QUERY));                            \
    (RESULT) = PQexec(RCONN, (QUERY));                                                                \
    if  (!(RESULT)) {                                                                                 \
        ifBLIN_QV1(db->flags) warnx("result is absent - fatal error.\n");                             \
        return(CCODE);                                                                                \
    }                                                                                                 \
    ifBLIN_QV5(db->flags) fprintf(stderr, "PQresultStatus %d\n", PQresultStatus(RESULT));             \
    if  (PQresultStatus(RESULT) != (CONDITION)) {                                                     \
        if  (PQresultStatus(RESULT) == PGRES_FATAL_ERROR) {                                           \
            ifBLIN_QV1(db->flags)                                                                     \
                warnx("PQexec failed: %s", PQresultErrorMessage(RESULT));                             \
            return(CCODE);                                                                            \
        } else if ((CONDITION) == PGRES_COMMAND_OK && PQresultStatus(RESULT) == PGRES_TUPLES_OK) {    \
            ifBLIN_QV2(db->flags)                                                                     \
                warnx("PQexec wait for COMMAND_OK but received TUPLES_OK");                           \
        } else if ((CONDITION) == PGRES_COMMAND_OK && PQresultStatus(RESULT) == PGRES_EMPTY_QUERY) {  \
            ifBLIN_QV2(db->flags)                                                                     \
                warnx("PQexec wait for COMMAND_OK but received EMPTY_QUERY");                         \
        } else ifBLIN_QV1(db->flags) {                                                                \
            warnx("PQexec wait for %d but received %d", (CONDITION), PQresultStatus(RESULT));         \
            return(CCODE);                                                                            \
}   }   }

typedef struct pgoblin_res {
    pgoblin_conn    *db;
    PGresult        *res;
    char            *prepar;
    u_int32_t        flags;
} pgoblin_res;

static void
NoticeProcessor(void *db, const char *message) {
    ifBLIN_QV1(((pgoblin_conn *)db)->flags) fprintf(stderr, "%s", message);
}

static int
finit(u_int32_t flags, pgoblin_dbases *this) {
    return(0);
}

static int
shurecon(pgoblin_conn *db) {
    PQconninfoOption *popt;
    int i, ex = EX_OK;
    PGresult *res;

    if  (db->conn != NULL && PQstatus(db->conn) == CONNECTION_OK) return(0);
    ifBLIN_QV5(db->flags) {
        fprintf( stderr
               , "Connection defaults:\n"
                 "size=chr keyword	envvar		compiled val		label\n"
               );
        popt = PQconndefaults();
        for (i = 0; popt[i].keyword; i++) {
            fprintf( stderr
                   , "%4d=%s~	%s%s%s%s%s%s%s%s%s\n"
                   , popt[i].dispsize
                   , popt[i].dispchar
                   , ALIGN8(keyword)
                   , ALIGN16(envvar)
                   , PRINTNOTNULL(popt[i].compiled)
                   , !(popt[i].compiled) ? "	" : strlen(popt[i].compiled) < 7 ? "~	" : "~ "
                   , PRINTNOTNULL(popt[i].val)
                   , !(popt[i].val)
                   ? "		"
                   : strlen(popt[i].val) < 7 ? "~		" : "~	"
                   , popt[i].label
                   );
        }
        PQconninfoFree(popt);
        fprintf( stderr
               , "PGREALM=%s%s PGDATESTYLE=%s%s PGTZ=%s%s PGCLIENTENCODING=%s%s PGGEQO=%s%s\n"
               , PRINTANDNULL(getenv("PGREALM"))
               , PRINTANDNULL(getenv("PGDATESTYLE"))
               , PRINTANDNULL(getenv("PGTZ"))
               , PRINTANDNULL(getenv("PGCLIENTENCODING"))
               , PRINTANDNULL(getenv("PGGEQO"))
               );
    }
    MARK_R_CONN_GO(0);;
    ;;  if  (db->flags & PGOBLIN_PARMIN && db->dbname) {
    ;;      db->conn = PQconnectdb(db->dbname);
    ;;  } else {
    ;;      db->conn = PQsetdbLogin(db->host, db->port, NULL, NULL, db->dbname, db->username, NULL);
    ;;  }
    ;;  db->flags |= PGOBLIN_CLOSECONN;
    MARK_R_CONN_WENT(0);;
    ifBLIN_QV5(db->flags) {
        fprintf(stderr, "PQoptions=%s~\n", PQoptions(db->conn));
        fprintf( stderr
               , "PQdb=%s~ PQuser=%s~ PQpass=%s%s "
                 "PQhost=%s%s PQport=%s%s "
                 "PQtty=%s%s\n"
               , PQdb(db->conn), PQuser(db->conn)
               , PRINTANDNULL(PQpass(db->conn))
               , PRINTANDNULL(PQhost(db->conn))
               , PRINTANDNULL(PQport(db->conn))
               , PRINTANDNULL(PQtty(db->conn))
               );
        fprintf( stderr
               , "server_version=%s%s "
                 "client_encoding=%s%s "
                 "is_superuser=%s%s "
                 "session_authorization=%s%s "
                 "DateStyle=%s%s\n"
               , PQSTATNDNULL("server_version")
               , PQSTATNDNULL("client_encoding")
               , PQSTATNDNULL("is_superuser")
               , PQSTATNDNULL("session_authorization")
               , PQSTATNDNULL("DateStyle")
               );
        fprintf( stderr
               , "PQprotocolVersion=%d PQsocket=%d PQbackendPID=%d\n"
               , PQprotocolVersion(db->conn)
               , PQsocket(db->conn)
               , PQbackendPID(db->conn)
               );
        fprintf(stderr, "PQstatus %d\n", PQstatus(db->conn));
    }
    PQsetNoticeProcessor(db->conn, NoticeProcessor, db);
    if  (PQstatus(db->conn) != CONNECTION_BAD) {
        if  (  db->flags & PGOBLIN_NEEDINTIME
            && !(db->flags & (PGOBLIN_INT64TIME | PGOBLIN_FLOATTIME))
            ) {
            if  (!db->intran) {
/*          if  (PQtransactionStatus(db->conn) != PQTRANS_INTRANS) { */
                PQEXEC(res, "BEGIN", PGRES_COMMAND_OK, -1, db->conn);
                PQclear(res);
            }
            PQEXEC( res
                  , "DECLARE pgoblin BINARY NO SCROLL CURSOR FOR "
                    "SELECT '2000-01-01 00:00:01'::timestamp;"
                    "FETCH 1 FROM pgoblin"
                  , PGRES_TUPLES_OK, -1
                  , db->conn
                  );
            if  (PQgetisnull(res, 0, 0) || (PQgetlength(res, 0, 0) < 8)) {
                ex = EX_DATAERR;
                errno = EINVAL;
            } else {
                if  (*((u_char*)(PQgetvalue(res, 0, 0))) == 077) {
                    db->flags |= PGOBLIN_FLOATTIME;
                    ifBLIN_QV3(db->flags) fprintf(stderr, "shurecon float timestamp format\n");
                } else if (*((u_char*)(PQgetvalue(res, 0, 0))) == 0) {
                    db->flags |= PGOBLIN_INT64TIME;
                    ifBLIN_QV3(db->flags) fprintf(stderr, "shurecon int64 timestamp format\n");
                } else {
                    ifBLIN_QV1(db->flags) fprintf(stderr, "shurecon unknown timestamp format\n");
                    ex = EX_DATAERR;
                    errno = EFTYPE;
            }   }
            PQclear(res);
            PQEXEC(res, "CLOSE pgoblin", PGRES_COMMAND_OK, -1, db->conn);
            PQclear(res);
            if  (!db->intran) {
                PQEXEC(res, "END", PGRES_COMMAND_OK, -1, db->conn);
                PQclear(res);
        }   }
        return(0);
    }
    ifBLIN_QV1(db->flags) warnx("Connection to PostgreSQL failed: %s\n", PQerrorMessage(db->conn));
    return(1);
}

static int
tarry(pgoblin_conn *db, int timeout) {
    PGconn *conn = (PGconn *)(db->conn);
    int           sock, i = 0;
    struct pollfd pfd;

    if  ((sock = PQsocket(conn)) < 0) {
        i = -2;
        errno = EINVAL;
    } else if (timeout != 0) {
        bzero(&pfd, sizeof(pfd));
        pfd.fd = sock;
        pfd.events = POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI;
        i = poll(&pfd, 1, timeout);
    }
    if  (i >= 0) {
        if  (!PQconsumeInput(conn)) i = -1;
    }
    return(i);
}

#define U0 0
#define U1 1
#define U2 2
#define U3 3
#define U4 4

static u_char miss[][PGOBLIN_CR_MAX] =
/*  OK   TUPL OUT  IN   STR  BIN   */
{{  U1,  U1,  U1,  U1,  U2,  U2 } /* PGRES_EMPTY_QUERY    */
,{  U0,  U2,  U2,  U2,  U2,  U2 } /* PGRES_COMMAND_OK     */
,{  U1,  U0,  U2,  U2,  U2,  U2 } /* PGRES_TUPLES_OK      */
,{  U2,  U2,  U0,  U2,  U2,  U2 } /* PGRES_COPY_OUT       */
,{  U2,  U2,  U2,  U0,  U2,  U2 } /* PGRES_COPY_IN        */
,{  U3,  U3,  U3,  U3,  U3,  U3 } /* PGRES_BAD_RESPONSE   */
,{  U3,  U3,  U3,  U3,  U3,  U3 } /* PGRES_NONFATAL_ERROR */
,{  U4,  U4,  U4,  U4,  U4,  U4 } /* PGRES_FATAL_ERROR    */
};

static ExecStatusType es[] = {PGRES_COMMAND_OK, PGRES_TUPLES_OK, PGRES_COPY_OUT, PGRES_COPY_IN};

static int
extatus(u_int32_t flags, PGresult *res, const char *who, u_char cond) {
    ExecStatusType st;
    int ex = 0;

    st = PQresultStatus(res);
    ifBLIN_QV5(flags)
        fprintf(stderr, "PQresultStatus %d %s, diff %d\n", st, PQresStatus(st), miss[st][cond]);
    switch (miss[st][cond]) {
    case U1:
        ifBLIN_QV2(flags) warnx( "%s waits for %d %s but received %d %s"
                               , who, es[cond], PQresStatus(es[cond])
                               , st, PQresStatus(st)
                               );
        break;
    case U2:
        ifBLIN_QV1(flags) warnx( "%s wait for %d %s but received %d %s"
                               , who, es[cond], PQresStatus(es[cond])
                               , st, PQresStatus(st)
                               );
        ex = miss[st][cond];
        break;
    case U4:
    case U3:
        ifBLIN_QV1(flags) warnx("%s failed: %s", who, PQresultErrorMessage(res));
        ex = miss[st][cond];
        break;
    }
    return(ex);
}

static int
doquery(pgoblin_conn *db, void **rsv, const char *query, u_char cond) {
    pgoblin_res **rs;
    PGresult *res;
    char *q, *p;
    int ex = 0;

    res = NULL;
    rs = (pgoblin_res **)rsv;
    if  (!db) ERROUT(EX_USAGE, EINVAL);
    switch (cond) {
    case PGOBLIN_CR_OK:
    case PGOBLIN_CR_TUPL:
    case PGOBLIN_CR_OUT:
    case PGOBLIN_CR_IN:
        ifBLIN_QV5(db->flags) fprintf(stderr, "PQexec %d=%s~\n", cond, query);
        if  (!(res = PQexec(db->conn, query))) {
            ifBLIN_QV1(db->flags)
                warnx("PQ result is absent - fatal error %s", PQerrorMessage(db->conn));
            ERROUT(EX_UNAVAILABLE, ENODEV);
        }
        ex = extatus(db->flags, res, "PQexec", cond);
        break;
    case PGOBLIN_CR_SSEL:
        p = "pgoblin";
        goto rest;
    case PGOBLIN_CR_BSEL:
        p = "pgoblin BINARY";
    rest:
        if  (!db->intran && (ex = doquery(db, NULL, "BEGIN", PGOBLIN_CR_OK))) goto out;
        asprintf(&q, "DECLARE %s NO SCROLL CURSOR FOR %s", p, query);
        if  (!q) {
            ifBLIN_QV1(db->flags) warnx("doquery: No memory #1");
            ERROUT(EX_OSERR, ENOMEM);
        }
        ex = doquery(db, rsv, q, PGOBLIN_CR_OK);
        free(q);
        break;
    default:
        ifBLIN_QV1(db->flags) warnx("doquery pgsql illegal cond=%d", cond);
        ex = -1;
    }
out:
    if  (!rsv) {
        PQclear(res); /* XXXX */
    } else if (rsv && res) {
        *rs = malloc(sizeof(pgoblin_res));
        if  (!*rs) {
            ifBLIN_QV1(db->flags) warnx("doquery: No memory #2");
            ex = EX_OSERR;
            errno = ENOMEM;
        } else {
            bzero(*rs, sizeof(pgoblin_res));
            (*rs)->res = res;
            (*rs)->db = db;
    }   }
    return(ex);
}

#ifdef PG_DIAG_INTERNAL_POSITION

static int
prepare(pgoblin_conn *db, void **rsv, const char *query) {
    pgoblin_res **rs;
    PGresult *res;
    int ex = 0;

    res = NULL;
    rs = (pgoblin_res **)rsv;
    if  (!db) ERROUT(EX_USAGE, EINVAL);
    ifBLIN_QV5(db->flags) fprintf(stderr, "PQprepare =%s~\n", query);
    if  (!(res = PQprepare(db->conn, "", query, 0, NULL))) {
        ifBLIN_QV1(db->flags) warnx("PQ result is absent - fatal error %s", PQerrorMessage(db->conn));
        ERROUT(EX_UNAVAILABLE, ENODEV);
    }
    ex = extatus(db->flags, res, "PQprepare", PGOBLIN_CR_OK);
out:
    if  (!rsv) {
        PQclear(res); /* XXXX */
    } else if (rsv && res) {
        *rs = malloc(sizeof(pgoblin_res));
        if  (!*rs) {
            ifBLIN_QV1(db->flags) warnx("doquery: No memory #2");
            ex = EX_OSERR;
            errno = ENOMEM;
        } else {
            bzero(*rs, sizeof(pgoblin_res));
            if  (!ex) (*rs)->prepar = "";
            (*rs)->res = res;
            (*rs)->db = db;
    }   }
    return(ex);
}

#endif

static int
execute( pgoblin_conn *db
       , void **rsv
       , const void *qp, int cnt, const char * const *val, const int *len
       , u_char cond
       ) {
    const char *query = "";
    char const *kind;
    pgoblin_res **rs;
    PGresult *res;
    int ex = 0;

    res = NULL;
    rs = (pgoblin_res **)rsv;
    if  (!db) ERROUT(EX_USAGE, EINVAL);
    if  (cond & PGOBLIN_ALTERQ) {
        kind = "PQexecPrepared";
    } else if (!qp) {
        ifBLIN_QV1(db->flags) warnx("NULL qp in pgsql prepare()");
        ERROUT(EX_USAGE, EINVAL);
    } else {
        kind = "PQexecParams";
        query = qp;
    }
    switch (cond & ~PGOBLIN_ALTERQ) {
    case PGOBLIN_CR_OK:
    case PGOBLIN_CR_TUPL:
        ifBLIN_QV5(db->flags) fprintf(stderr, "%s %d=%s~\n", kind, cond, query);
        if  (!(res = (cond & PGOBLIN_ALTERQ)
                   ? PQexecPrepared(db->conn, query, cnt,       val, len, NULL, 0)
                   : PQexecParams(  db->conn, query, cnt, NULL, val, len, NULL, 0)
            ) ) {
            ifBLIN_QV1(db->flags)
                warnx("PQ result is absent - fatal error %s", PQerrorMessage(db->conn));
            ERROUT(EX_UNAVAILABLE, ENODEV);
        }
        ex = extatus(db->flags, res, kind, cond & ~PGOBLIN_ALTERQ);
        break;
    default:
        ifBLIN_QV1(db->flags) warnx("execute pgsql illegal cond=%d", cond);
        ex = -1;
    }
out:
    if  (!rsv) {
        PQclear(res); /* XXXX */
    } else if (rsv && res) {
        *rs = malloc(sizeof(pgoblin_res));
        if  (!*rs) {
            ifBLIN_QV1(db->flags) warnx("doquery: No memory #2");
            ex = EX_OSERR;
            errno = ENOMEM;
        } else {
            bzero(*rs, sizeof(pgoblin_res));
            if  (!ex) (*rs)->prepar = "";
            (*rs)->res = res;
            (*rs)->db = db;
    }   }
    return(ex);
}

static char *
notifies(pgoblin_conn *db) {
    PGconn *conn = (PGconn *)(db->conn);
    PGnotify *n;
    char *c = NULL;

    n = PQnotifies(conn);
    if  (n) {
        if  (n->relname) c = strdup(n->relname);
        PQfreemem(n);
    }
    return(c);
}

static int
p1info(int kind, void *resv) {
    PGresult *res = ((pgoblin_res *)resv)->res;
    char *c;
    int ex;

    switch (kind) {
    case PGOBLIN_Ntuples: ex =   PQntuples(res); break;
    case PGOBLIN_Nfields: ex =   PQnfields(res); break;
    case PGOBLIN_Nchange:
        c = PQcmdTuples(res);
        ex = *c ? strtoul(c, NULL, 10) : -1;
        break;
    case PGOBLIN_LastOid: ex =  PQoidValue(res); break;
    default             : ex = -1;
    }
    return(ex);
}

static char *
p2info(int kind, void *resv, int sub) {
    PGresult *res = ((pgoblin_res *)resv)->res;
    char *cx;

    switch (kind) {
    case PGOBLIN_ColName: cx =     PQfname(res, sub); break;
    case PGOBLIN_CmdStat: cx = PQcmdStatus(res     ); break;
    case PGOBLIN_ResMess: {
            if  (!res) {
                if  (asprintf(&cx, "No results") < 0) warn("p2info(ResMess) asprintf");
            } else ifBLIN_QV5(sub) {
                if  ( asprintf( &cx
#ifdef PG_DIAG_INTERNAL_POSITION
                              , "%s %s %s\n%s\n%s\n%s\n%s\n%s\nfunction %s at %s line of %s file\n%s"
#else
                              , "%s %s %s\n%s\n%s\n%s\nfunction %s at %s line of %s file\n%s"
#endif
                              , PQresultErrorField(res, PG_DIAG_SEVERITY)
                              , PQresultErrorField(res, PG_DIAG_SQLSTATE)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_PRIMARY)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_DETAIL)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_HINT)
                              , PQresultErrorField(res, PG_DIAG_STATEMENT_POSITION)
#ifdef PG_DIAG_INTERNAL_POSITION
                              , PQresultErrorField(res, PG_DIAG_INTERNAL_QUERY)
                              , PQresultErrorField(res, PG_DIAG_INTERNAL_POSITION)
#endif
                              , PQresultErrorField(res, PG_DIAG_SOURCE_FUNCTION)
                              , PQresultErrorField(res, PG_DIAG_SOURCE_LINE)
                              , PQresultErrorField(res, PG_DIAG_SOURCE_FILE)
                              , PQresultErrorField(res, PG_DIAG_CONTEXT)
                              )
                      < 0
                    ) warn("p2info(ResMess) asprintf");
            } else ifBLIN_QV4(sub) {
                if  ( asprintf( &cx
#ifdef PG_DIAG_INTERNAL_POSITION
                              , "%s %s %s\n%s\n%s\n%s\n%s\n%s\nfunction %s at %s line of %s file"
#else
                              , "%s %s %s\n%s\n%s\n%s\nfunction %s at %s line of %s file"
#endif
                              , PQresultErrorField(res, PG_DIAG_SEVERITY)
                              , PQresultErrorField(res, PG_DIAG_SQLSTATE)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_PRIMARY)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_DETAIL)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_HINT)
                              , PQresultErrorField(res, PG_DIAG_STATEMENT_POSITION)
#ifdef PG_DIAG_INTERNAL_POSITION
                              , PQresultErrorField(res, PG_DIAG_INTERNAL_QUERY)
                              , PQresultErrorField(res, PG_DIAG_INTERNAL_POSITION)
#endif
                              , PQresultErrorField(res, PG_DIAG_SOURCE_FUNCTION)
                              , PQresultErrorField(res, PG_DIAG_SOURCE_LINE)
                              , PQresultErrorField(res, PG_DIAG_SOURCE_FILE)
                              )
                      < 0
                    ) warn("p2info(ResMess) asprintf");
            } else ifBLIN_QV3(sub) {
                if  ( asprintf( &cx
#ifdef PG_DIAG_INTERNAL_POSITION
                              , "%s %s %s\n%s\n%s\n%s\n%s\n%s"
#else
                              , "%s %s %s\n%s\n%s\n%s"
#endif
                              , PQresultErrorField(res, PG_DIAG_SEVERITY)
                              , PQresultErrorField(res, PG_DIAG_SQLSTATE)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_PRIMARY)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_DETAIL)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_HINT)
                              , PQresultErrorField(res, PG_DIAG_STATEMENT_POSITION)
#ifdef PG_DIAG_INTERNAL_POSITION
                              , PQresultErrorField(res, PG_DIAG_INTERNAL_QUERY)
                              , PQresultErrorField(res, PG_DIAG_INTERNAL_POSITION)
#endif
                              )
                      < 0
                    ) warn("p2info(ResMess) asprintf");
            } else ifBLIN_QV2(sub) {
                if  ( asprintf( &cx
                              , "%s %s %s\n%s\n%s\n%s"
                              , PQresultErrorField(res, PG_DIAG_SEVERITY)
                              , PQresultErrorField(res, PG_DIAG_SQLSTATE)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_PRIMARY)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_DETAIL)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_HINT)
                              , PQresultErrorField(res, PG_DIAG_STATEMENT_POSITION)
                              )
                      < 0
                    ) warn("p2info(ResMess) asprintf");
            } else ifBLIN_QV1(sub) {
                if  ( asprintf( &cx
                              , "%s %s %s\n%s"
                              , PQresultErrorField(res, PG_DIAG_SEVERITY)
                              , PQresultErrorField(res, PG_DIAG_SQLSTATE)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_PRIMARY)
                              , PQresultErrorField(res, PG_DIAG_MESSAGE_DETAIL)
                              )
                      < 0
                    ) warn("p2info(ResMess) asprintf");
        }   }
        break;
    default             : cx = NULL;
    }
    return(cx);
}

static int
p3info(int kind, void *resv, int row, int col) {
    PGresult *res = ((pgoblin_res *)resv)->res;
    int ex;

    switch (kind) {
    case PGOBLIN_IsNull : ex = PQgetisnull(res, row, col); break;
    case PGOBLIN_Length : ex = PQgetlength(res, row, col); break;
    case PGOBLIN_BinForm: ex =   PQfformat(res,      col); break;
    case PGOBLIN_TypeOid: ex =     PQftype(res,      col); break;
    case PGOBLIN_TypeMod: ex =      PQfmod(res,      col); break;
    default             : ex = -1;
    }
    return(ex);
}

static char *
getvalue(void *resv, int row, int col) {
    PGresult *res = ((pgoblin_res *)resv)->res;
    char *c = NULL;
    if  (!p3info(PGOBLIN_IsNull, resv, row, col)) {
        c = PQgetvalue(res, row, col);
    }
    return(c);
}

static int
finish(u_int32_t flags, int kind, void **entity) {
    int ex = EX_OK;

    switch (kind) {
    case PGOBLIN_Connect: {
            PGconn **conn = (PGconn **)entity;

            if  (conn && *conn) {
                PQfinish(*conn);
                *conn = NULL;
        }   }
        break;
    case PGOBLIN_FrCOPY:
    case PGOBLIN_FrNote:
        if  (entity && *entity) {
            PQfreemem(*entity);
            *entity = NULL;
        }
        break;
    case PGOBLIN_PFormed:
    case PGOBLIN_CleaRes: {
            pgoblin_res **res = (pgoblin_res **)entity;

            if  (res && *res) {
                if  ((*res)->res) PQclear((*res)->res);
                (*res)->prepar = NULL; /* XXXX */
                (*res)->db = NULL;
                free(*res);
                *res = NULL;
        }   }
        break;
    case PGOBLIN_FrMess:
        if  (entity && *entity) {
            free(*entity);
            *entity = NULL;
        }
        break;
    default:
        ex = EX_UNAVAILABLE;
        errno = ENOTSUP;
    }
    return(ex);
}

static int
getstream(pgoblin_conn *db, void **res, int quant, u_char cond) {
    int ex = EX_OK;
    char *query;

    if  (!db) ERROUT(EX_USAGE, EINVAL);
    if  ((cond != PGOBLIN_CR_SSEL) && (cond != PGOBLIN_CR_BSEL)) {
        ifBLIN_QV1(db->flags) warnx("getstream: Illegal cond %d", cond);
        ERROUT(EX_USAGE, EINVAL);
    }
    if  (asprintf(&query, "FETCH %d FROM pgoblin", quant) < 0) {
        ifBLIN_QV1(db->flags) warnx("getstream: No memory");
        ERROUT(EX_OSERR, ENOMEM);
    }
    ex = doquery(db, res, query, PGOBLIN_CR_TUPL);
    free(query);
out:
    return(ex);
}

static int
getcopy(pgoblin_conn *db, void **buf) {
    int ex;

    ex = PQgetCopyData(db->conn, (char**)buf, 0);
    return((ex < 0) ? (ex + 1) : ex);
}

static int
putcopy(pgoblin_conn *db, const void *buf, int nby) {
    return(PQputCopyData(db->conn, buf, nby));
}

static int
endstream(pgoblin_conn *db, u_char cond) {
    ExecStatusType st;
    int ex = EX_OK;
    PGresult *res;

    switch (cond) {
    case PGOBLIN_CR_IN:
        if  (PQputCopyEnd(db->conn, NULL) < 1) {
            ifBLIN_QV1(db->flags)
                warnx("endstream: endcopy failed: %s", PQerrorMessage(db->conn));
            ERROUT(-EX_DATAERR, EINVAL);
        }
        /* FALLTHRU */
    case PGOBLIN_CR_OUT:
        while ((res = PQgetResult(db->conn))) {
            st = PQresultStatus(res);
            ifBLIN_QV5(db->flags) fprintf(stderr, "PQresultStatus %d %s\n", st, PQresStatus(st));
            if  (st != PGRES_COMMAND_OK) {
                ifBLIN_QV1(db->flags) warnx("endstream pgsql: %s", PQresultErrorMessage(res));
                ex = EX_SOFTWARE;
                break;
            }
            PQclear(res);
            res = NULL;
        }
        break;
    case PGOBLIN_CR_SSEL:
    case PGOBLIN_CR_BSEL:
        ex = doquery(db, NULL, "CLOSE pgoblin", PGOBLIN_CR_OK);
        if  (!db->intran) {
            ex |= doquery(db, NULL, "END", PGOBLIN_CR_OK);
        }
        break;
    default:
        ifBLIN_QV1(db->flags) warnx("endstream pgsql %d", cond);
        ex = EX_USAGE;
    }
out:
    if  (ex) errno = EINVAL;
    return(ex);
}

pgoblin_dbases pgoblin_db_pgsql =
{ (PGOBLIN_DB_PREPARE | PGOBLIN_DB_COPYIN | PGOBLIN_DB_COPYOUT)
, "3pGoblin-" VERS
, "pgsql\0" VERS
, finit         /* init      */
, finit         /* fini      */
, shurecon      /* shurecon  */
, doquery       /* query     */
#ifdef PG_DIAG_INTERNAL_POSITION
, prepare       /* prepare   */
#else
, NULL
#endif
, execute       /* execute   */
, p1info        /* resinfo   */
, p2info        /* subinfo   */
, p3info        /* valinfo   */
, getvalue      /* getvalue  */
, getstream     /* getstream */
, getcopy       /* getcopy   */
, putcopy       /* putcopy   */
, endstream     /* endstream */
, tarry         /* tarry     */
, notifies      /* notifies  */
, finish        /* finish    */
};
