/*-
 * Copyright (C)2022 @BABOLO http://www.babolo.ru/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ident "@(#) Copyright (C)2022 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: mod_authn_sqlite.c,v 1.3 2022/05/14 03:17:28 babolo Exp $"

#include <apr_strings.h>
#include <apr_md5.h>
#include <ap_config.h>
#include <ap_provider.h>
#include <httpd.h>
#include <sqlite.h>
#include <http_config.h>
#include <http_core.h>
#include <http_log.h>
#include <http_protocol.h>
#include <http_request.h>
#include <mod_auth.h>

typedef struct {
    char *dbfile;
    char *passwd;
    char *digest;
    char *context;
    char *pop;
    char *dop;
} authn_sqlite_config_rec;

module AP_MODULE_DECLARE_DATA authn_sqlite_module;
const static char c[] = "SELECT 1 FROM \"%s\" WHERE context = ? AND login = ? AND passwd = ? LIMIT 1;";
const static char f[] = "SELECT 1 FROM \"%s\" WHERE login = ? AND passwd = ? LIMIT 1;";
const static char u[] =
    " !$%&*+,-./0123456789:=?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[]^_abcdefghijklmnopqrstuvwxyz{|}~";
const static char e[28][11] =
{ "OK"      , "ERROR"    , "INTERNAL", "PERM"      , "ABORT"   , "BUSY"  , "LOCKED"  , "NOMEM"
, "READONLY", "INTERRUPT", "IOERR"   , "CORRUPT"   , "NOTFOUND", "FULL"  , "CANTOPEN", "PROTOCOL"
, "EMPTY"   , "SCHEMA"   , "TOOBIG"  , "CONSTRAINT", "MISMATCH", "MISUSE", "NOLFS"   , "AUTH"
, "FORMAT"  , "RANGE"    , "NOTADB"  , "unkn"
};

static char *
error_string(int ex) {
    if  ((ex < 0) || (ex > 27)) ex = 27;
    return((char*)(e[ex]));
}

static void *
create_authn_sqlite_dir_config(apr_pool_t *p, char *d) {
    authn_sqlite_config_rec *conf = apr_palloc(p, sizeof(authn_sqlite_config_rec));

    bzero(conf, sizeof(authn_sqlite_config_rec));
    return(conf);
}

static const command_rec authn_sqlite_cmds[] =
{   AP_INIT_TAKE1( "Auth_SQLite_File"
                 , ap_set_file_slot
                 , (void *)APR_OFFSETOF(authn_sqlite_config_rec, dbfile)
                 , OR_AUTHCFG
                 , "SQLite DB file containing user IDs and passwords"
                 )
,   AP_INIT_TAKE1( "Auth_SQLite_Passwd"
                 , ap_set_string_slot
                 , (void *)APR_OFFSETOF(authn_sqlite_config_rec, passwd)
                 , OR_AUTHCFG
                 , "Name of passwd table"
                 )
,   AP_INIT_TAKE1( "Auth_SQLite_Digest"
                 , ap_set_string_slot
                 , (void *)APR_OFFSETOF(authn_sqlite_config_rec, digest)
                 , OR_AUTHCFG
                 , "Name of digest table"
                 )
,   AP_INIT_TAKE1( "Auth_SQLite_Context"
                 , ap_set_string_slot
                 , (void *)APR_OFFSETOF(authn_sqlite_config_rec, context)
                 , OR_AUTHCFG
                 , "Context of AAA"
                 )
,   {NULL}
};

static authn_status
check_password(request_rec *r, const char *user, const char *password) {
    char                    *errmsg = NULL;
    const char             **tuple;
    authn_sqlite_config_rec *conf   = ap_get_module_config(r->per_dir_config, &authn_sqlite_module);
    int                      ncol;
    sqlite                  *odf;
    sqlite_vm               *vm;
    int                      ex;
    int                      i;

    if  (!conf->dbfile) {
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01619) "Auth_SQLite_File not specified in the configuration"
                     );
        return(AUTH_GENERAL_ERROR);
    }
    if  (!conf->passwd) {
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01619) "Auth_SQLite_Passwd not specified in the configuration"
                     );
        return(AUTH_GENERAL_ERROR);
    }
    if  (!conf->pop) {
        if  (!!conf->passwd[strspn(conf->passwd, u)]) {
            ap_log_rerror( APLOG_MARK
                         , APLOG_ERR
                         , 0
                         , r
                         , APLOGNO(01620) "Illegal table name: %s"
                         , conf->dbfile
                         );
            return(AUTH_GENERAL_ERROR);
        }
        size_t sz = strlen((!!conf->context) ? c : f) + strlen(conf->passwd);
        conf->pop = apr_palloc(r->pool, sz);
        snprintf(conf->pop, sz, (!!conf->context) ? c : f, conf->passwd);
    }
    if  (!(odf = sqlite_open(conf->dbfile, 0, NULL))) {
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01620) "Could not open password DB: %s"
                     , conf->dbfile
                     );
        return(AUTH_GENERAL_ERROR);
    }
    if  (!!(ex = sqlite_compile(odf, conf->pop, NULL, &vm, NULL))) {
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01620) "Could not compile %d<%s>: %s"
                     , ex
                     , error_string(ex)
                     , conf->pop
                     );
        if  (!vm || !!sqlite_finalize(vm, &errmsg)) {
            ap_log_rerror( APLOG_MARK
                         , APLOG_ERR
                         , 0
                         , r
                         , APLOGNO(01620) "Could not compile: %s"
                         , !!errmsg ? errmsg : ""
                         );
        }
        return(AUTH_GENERAL_ERROR);
    }
    i = 0;
    if  (!!conf->context && !!(ex = sqlite_bind(vm, ++i, conf->context, -1, 0))) {
        sqlite_close(odf);
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01620) "Could not bind[%d] %d<%s>: %s"
                     , i
                     , ex
                     , error_string(ex)
                     , conf->context
                     );
        return(AUTH_GENERAL_ERROR);
    }
    if  (!!(ex = sqlite_bind(vm, ++i, user, -1, 0))) {
        sqlite_close(odf);
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01620) "Could not bind[%d] %d<%s>: %s"
                     , i
                     , ex
                     , error_string(ex)
                     , user
                     );
        return(AUTH_GENERAL_ERROR);
    }
    if  (!!(ex = sqlite_bind(vm, ++i, password, -1, 0))) {
        sqlite_close(odf);
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01620) "Could not bind[%d] %d<%s>: %s"
                     , i
                     , ex
                     , error_string(ex)
                     , password
                     );
        return(AUTH_GENERAL_ERROR);
    }
    switch((ex = sqlite_step(vm, &ncol, &tuple, NULL))) {
    case SQLITE_ROW:
        sqlite_finalize(vm, NULL);
        sqlite_close(odf);
        return(AUTH_GRANTED);
    case SQLITE_DONE:
        sqlite_finalize(vm, NULL);
        sqlite_close(odf);
        return(AUTH_DENIED);
    }
    sqlite_finalize(vm, NULL);
    sqlite_close(odf);
    ap_log_rerror( APLOG_MARK
                 , APLOG_ERR
                 , 0
                 , r
                 , APLOGNO(01620) "Could not SELECT %d<%s>"
                 , ex
                 , error_string(ex)
                 );
    return(AUTH_GENERAL_ERROR);
}

static authn_status
get_realm_hash(request_rec *r, const char *user, const char *realm, char **rethash) {
    char                    *sqlite_hash = NULL;
    authn_sqlite_config_rec *conf = ap_get_module_config(r->per_dir_config, &authn_sqlite_module);

    if  (!conf->dbfile) {
        ap_log_rerror( APLOG_MARK
                     , APLOG_ERR
                     , 0
                     , r
                     , APLOGNO(01621) "Auth_SQLite_File not specified in the configuration"
                     );
        return AUTH_GENERAL_ERROR;
    }




    if (!sqlite_hash) return(AUTH_USER_NOT_FOUND);
    *rethash = sqlite_hash;
    return(AUTH_USER_FOUND);
}

static const authn_provider authn_sqlite_provider =
{   &check_password
,   &get_realm_hash
};

static void
register_hooks(apr_pool_t *p) {
    ap_register_auth_provider( p
                             , AUTHN_PROVIDER_GROUP
                             , "sqlite"
                             , AUTHN_PROVIDER_VERSION
                             , &authn_sqlite_provider
                             , AP_AUTH_INTERNAL_PER_CONF
                             );
}

AP_DECLARE_MODULE(authn_sqlite) =
{   STANDARD20_MODULE_STUFF
,   create_authn_sqlite_dir_config  /* dir config creater */
,   NULL                            /* dir merger --- default is to override */
,   NULL                            /* server config */
,   NULL                            /* merge server config */
,   authn_sqlite_cmds               /* command apr_table_t */
,   register_hooks                  /* register hooks */
};
