/*-
 * Copyright (C)2023..2024 @BABOLO http://www.babolo.ru/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ident "@(#) Copyright (C)2023..2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: mod_authn_daemon.c,v 1.7 2024/03/08 23:08:43 babolo Exp $"

#include <apr_strings.h>
#include <apr_md5.h>
#include <ap_config.h>
#include <ap_provider.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <unistd.h>
#include <string.h>
#include <netdb.h>
#include <httpd.h>
#include <http_config.h>
#include <http_core.h>
#include <http_log.h>
#include <http_protocol.h>
#include <http_request.h>
#include <mod_auth.h>

#define INDTEX          "mod_authn_daemon@BABOLO "
#define MER             APLOG_MARK, APLOG_ERR, 0, r, INDTEX
#define MCO             f, l, m, APLOG_ERR, 0, r, INDTEX
#define MT              "Auth_Daemon_"
#define AAACIPA_HLEN    11       /*              */
#define MAXQSZ          255      /*     AAA   */
#define AAACIPA_MAXCRED (sizeof(int64_t) + sizeof(u_int64_t) + AAACIPA_HLEN) /*
                                       AAA                 */

typedef struct {
    char      *daemonpath;
    char      *context;
    char      *host;
    char      *port;
    int        testenv;
} authn_config_rec;

typedef struct {
    size_t     q;
    char       query[MAXQSZ + 1];
} querystr;

typedef struct {
    u_int64_t       cred;
    u_int64_t       sess;
    double          now ;
    char            hash[AAACIPA_HLEN + 1];
} aaacipa_pass;

module AP_MODULE_DECLARE_DATA authn_daemon_module;

static char toenv1[24];
static char toenv2[24];

static const command_rec authn_daemon_cmds[] =
{   AP_INIT_TAKE1( MT "Path"
                 , ap_set_file_slot
                 , (void *)APR_OFFSETOF(authn_config_rec, daemonpath)
                 , OR_AUTHCFG
                 , "Daemon file path"
                 )
,   AP_INIT_TAKE1( MT "Context"
                 , ap_set_string_slot
                 , (void *)APR_OFFSETOF(authn_config_rec, context)
                 , OR_AUTHCFG
                 , "Context of AAA"
                 )
,   AP_INIT_TAKE1( MT "Port"
                 , ap_set_string_slot
                 , (void *)APR_OFFSETOF(authn_config_rec, port)
                 , OR_AUTHCFG
                 , "AAA daemon's port"
                 )
,   AP_INIT_TAKE1( MT "Host"
                 , ap_set_string_slot
                 , (void *)APR_OFFSETOF(authn_config_rec, host)
                 , OR_AUTHCFG
                 , "AAA daemon's IP"
                 )
,   AP_INIT_TAKE1( MT "TestEnv"
                 , ap_set_int_slot
                 , (void *)APR_OFFSETOF(authn_config_rec, testenv)
                 , OR_AUTHCFG
                 , "Include TEST_* variables into environment"
                 )
,   {NULL}
};

static void
testset(request_rec *r) {
    if  (!!r->connection) {
        conn_rec *c = r->connection;

        if  (!!c->base_server) {
            server_rec *s = c->base_server;

            if  (!!s->defn_name)
                apr_table_setn(r->subprocess_env, "TEST_BDEFNAME", s->defn_name);
            if  (!!s->server_scheme)
                apr_table_setn(r->subprocess_env, "TEST_BSCHEME", s->server_scheme);
        }
        if  (!!c->client_ip)   apr_table_setn(r->subprocess_env, "TEST_CIP"  , c->client_ip  );
        if  (!!c->remote_host) apr_table_setn(r->subprocess_env, "TEST_RHOST", c->remote_host);
        if  (!!c->local_ip)    apr_table_setn(r->subprocess_env, "TEST_LIP"  , c->local_ip   );
        if  (!!c->local_host)  apr_table_setn(r->subprocess_env, "TEST_LHOST", c->local_host );
    }
    if  (!!r->server) {
        server_rec *s = r->server;

        if  (!!s->defn_name)     apr_table_setn(r->subprocess_env, "TEST_DEFNAME", s->defn_name    );
        if  (!!s->server_scheme) apr_table_setn(r->subprocess_env, "TEST_SCHEME" , s->server_scheme);
    }
    if  (!!r->uri)            apr_table_setn(r->subprocess_env, "TEST_URI"  , r->uri           );
    if  (!!r->args)           apr_table_setn(r->subprocess_env, "TEST_ARGS" , r->args          );
    if  (!!r->protocol)       apr_table_setn(r->subprocess_env, "TEST_PROTO", r->protocol      );
    if  (!!r->the_request)    apr_table_setn(r->subprocess_env, "TEST_REQ"  , r->the_request   );
    if  (!!r->hostname)       apr_table_setn(r->subprocess_env, "TEST_HOST" , r->hostname      );
    if  (!!r->method)         apr_table_setn(r->subprocess_env, "TEST_METH" , r->method        );
    if  (!!r->user)           apr_table_setn(r->subprocess_env, "TEST_USER" , r->user          );
    if  (!!r->unparsed_uri)   apr_table_setn(r->subprocess_env, "TEST_UURI" , r->unparsed_uri  );
    if  (!!r->path_info)      apr_table_setn(r->subprocess_env, "TEST_PATH" , r->path_info     );
    if  (!!r->useragent_ip)   apr_table_setn(r->subprocess_env, "TEST_UIP"  , r->useragent_ip  );
    if  (!!r->useragent_host) apr_table_setn(r->subprocess_env, "TEST_UHOST", r->useragent_host);
    if  (!!ap_http_scheme(r)) apr_table_setn(r->subprocess_env, "TEST_RSCH" , ap_http_scheme(r));
}

static void *
create_authn_daemon_dir_config(apr_pool_t *p, char *d) {
    authn_config_rec *conf;

    conf = apr_palloc(p, sizeof(authn_config_rec));
    bzero(conf, sizeof(authn_config_rec));
    return(conf);
}

static int
toquery(querystr *query, const char *c) {
    size_t len;
    int    ex = 0;

    if  (!c) {
        query->query[query->q++] = 0;
    } else {
        len = strlen(c);
        if  (len > 255) {
            ex = len;
        } else {
            query->query[query->q++] = ++len;
            query->q += strlcpy(&query->query[query->q], c, MAXQSZ - query->q);
    }   }
    if  (MAXQSZ < query->q) ex = -1;
    return(ex);
}

static authn_status
check_password(request_rec *r, const char *user, const char *password) {
    aaacipa_pass         credent;
    querystr             query  ;
    authn_config_rec    *conf   = ap_get_module_config(r->per_dir_config, &authn_daemon_module);
    struct addrinfo      hint   ;
    struct addrinfo     *res    ;
    int                  ex     ;
    int                  s      ;
    const char          *g      ;

    if  (!conf->host) {
        ap_log_rerror(MER MT "Daemon host not specified in the config");
        conf->host = "127.0.0.1";
    }
    if  (!conf->port) {
        ap_log_rerror(MER MT "Daemon port not specified in the config");
        conf->host = "1357";
    }
    query.q = 0;
    g = conf->context;
    if  (0 < (ex = toquery(&query, g))) {
        ap_log_rerror(MER MT "context too big: %d=%s", ex, g);
        return(AUTH_GENERAL_ERROR);
    } else if (0 > ex) {
        ap_log_rerror(MER MT "Query too big");
        return(AUTH_GENERAL_ERROR);
    }
    g = r->useragent_ip;
    if  (0 < (ex = toquery(&query, g))) {
        ap_log_rerror(MER MT "useragent_ip too big: %d=%s", ex, g);
        return(AUTH_GENERAL_ERROR);
    } else if (0 > ex) {
        ap_log_rerror(MER MT "Query too big");
        return(AUTH_GENERAL_ERROR);
    }
    g = user;
    if  (0 < (ex = toquery(&query, g))) {
        ap_log_rerror(MER MT "user too big: %d=%s", ex, g);
        return(AUTH_GENERAL_ERROR);
    } else if (0 > ex) {
        ap_log_rerror(MER MT "Query too big");
        return(AUTH_GENERAL_ERROR);
    }
    g = password;
    if  (0 < (ex = toquery(&query, g))) {
        ap_log_rerror(MER MT "password too big: %d=%s", ex, g);
        return(AUTH_GENERAL_ERROR);
    } else if (0 > ex) {
        ap_log_rerror(MER MT "Query too big");
        return(AUTH_GENERAL_ERROR);
    }
    g = ap_http_scheme(r);
    if  (0 < (ex = toquery(&query, g))) {
        ap_log_rerror(MER MT "scheme too big: %d=%s", ex, g);
        return(AUTH_GENERAL_ERROR);
    } else if (0 > ex) {
        ap_log_rerror(MER MT "Query too big");
        return(AUTH_GENERAL_ERROR);
    }
    g = r->the_request;
    if  (0 < (ex = toquery(&query, g))) {
        ap_log_rerror(MER MT "request too big: %d=%s", ex, g);
        return(AUTH_GENERAL_ERROR);
    }
    bzero(&hint, sizeof(hint));
    hint.ai_family = PF_INET;
    hint.ai_socktype = SOCK_STREAM;
    hint.ai_protocol = IPPROTO_TCP;
    hint.ai_flags = AI_ADDRCONFIG;
    if  ((ex = getaddrinfo(conf->host, conf->port, &hint, &res))) {
        ap_log_rerror(MER MT "Can't find AAA daemon's socket");
        return(AUTH_GENERAL_ERROR);
    }
    if  (0 > (s = socket(res->ai_family, res->ai_socktype, res->ai_protocol))) {
        ap_log_rerror(MER MT "Can't get socket");
        return(AUTH_GENERAL_ERROR);
    }
    for (int i = 0; i < 33 && (0 > (connect(s, res->ai_addr, res->ai_addrlen))); ++i) {
        int p;

        if  (!conf->daemonpath) {
            break;
        } else if (0 > (p = fork())) {
            ap_log_rerror(MER MT "Can't fork");
            return(AUTH_GENERAL_ERROR);
        } else if (!p) {
            usleep(33000);
            break;
        } else {
            execl(conf->daemonpath, NULL, NULL);
            ap_log_rerror(MER MT "Can't exec");
            return(AUTH_GENERAL_ERROR);
        }
        if  (33 < i) {
            ap_log_rerror(MER MT "No AAA daemon");
            return(AUTH_GENERAL_ERROR);
    }   }
    freeaddrinfo(res);
    if  (query.q != (ex = write(s, query.query, query.q))) {
        ap_log_rerror(MER MT "Can't write socket");
        return(AUTH_GENERAL_ERROR);
    }
    if  (0 > (ex = read(s, &credent, sizeof(credent)))) {
        ap_log_rerror(MER MT "Can't read socket");
        return(AUTH_GENERAL_ERROR);
    }
    close(s);
    if  (0 > snprintf(toenv1, 24, "%lu", credent.cred)) {
        ap_log_rerror(MER MT "snprintf cred");
        return(AUTH_GENERAL_ERROR);
    }
    apr_table_setn(r->subprocess_env, "CREDENTIALS", toenv1);
    if  (0 > snprintf(toenv2, 24, "%lu", credent.sess)) {
        ap_log_rerror(MER MT "snprintf sess");
        return(AUTH_GENERAL_ERROR);
    }
    apr_table_setn(r->subprocess_env, "SESSION", toenv2);
    credent.hash[AAACIPA_HLEN] = '\0';
    apr_table_setn(r->subprocess_env, "HASH", credent.hash);
    if  (!credent.cred) apr_table_setn(r->subprocess_env, "ALLTRYPASS", password);
    if  (!!conf->testenv) testset(r);
    return(AUTH_GRANTED);
}

static authn_status
get_realm_hash(request_rec *r, const char *user, const char *realm, char **rethash) {
    char             *daemon_hash = NULL;
    authn_config_rec *conf        = ap_get_module_config(r->per_dir_config, &authn_daemon_module);

    if  (!conf->daemonpath) {
        ap_log_rerror(MER MT "File not specified in the config");
        return AUTH_GENERAL_ERROR;
    }




    if (!daemon_hash) return(AUTH_USER_NOT_FOUND);
    *rethash = daemon_hash;
    return(AUTH_USER_FOUND);
}

static const authn_provider authn_daemon_provider =
{   &check_password
,   &get_realm_hash
};

static void
register_hooks(apr_pool_t *p) {
    ap_register_auth_provider( p
                             , AUTHN_PROVIDER_GROUP
                             , "daemon"
                             , AUTHN_PROVIDER_VERSION
                             , &authn_daemon_provider
                             , AP_AUTH_INTERNAL_PER_CONF
                             );
}

AP_DECLARE_MODULE(authn_daemon) =
{   STANDARD20_MODULE_STUFF
,   create_authn_daemon_dir_config  /* dir config creater */
,   NULL                            /* dir merger --- default is to override */
,   NULL                            /* server config */
,   NULL                            /* merge server config */
,   authn_daemon_cmds               /* command apr_table_t */
,   register_hooks                  /* register hooks */
};
