/*-
 * Copyright (C)2003..2018 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2003..2018 @BABOLO http://www.babolo.ru/\n"
#ident "@(#) $Id: open.c,v 1.39 2018/06/03 14:37:21 babolo Exp $\n"

#define BLIN_COMPAT   3
#define MIFE_COMPAT   5
#define MIFE_INTERNAL 1

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sysexits.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <babolo/BLINflag.h>
#include "mife.h"

#define TRUFLAG(A) ( (A)                                                                              \
                   & ( MIFE_PIPE                                                                      \
                     | MIFE_STRI                                                                      \
                     | MIFE_RTRY                                                                      \
                     | MIFE_BUFL                                                                      \
                     | MIFE_SLID                                                                      \
                     | MIFE_EOFL                                                                      \
                     | MIFE_FULL                                                                      \
                     | MIFE_BUFX                                                                      \
                     | MIFE_BACK                                                                      \
                     | BLIN_MASK                                                                      \
                   ) )

BLIN_flag mife_default = BLIN_BIT0;

static ssize_t
mindelta(BLIN_flag flags) {
    ssize_t s = 0;

    if  (flags & MIFE_BUFL) s = 1 << (((flags & MIFE_BUFL) >> MIFE_BUFS) + 16);
    if  (s < MINDELTA) s = MINDELTA;
    return(s);
}

mife_descriptor *
mife_init(BLIN_flag flags) {
    mife_descriptor *mife = NULL;

#   define blin_internal_flags (flags & BLIN_MASK)
    ifBLIN_QX2("+ flags=%08X", flags);
    SETBDEF(flags);
    if  (!(mife = calloc(1, sizeof(mife_descriptor)))) {
        ifBLIN_QW0("calloc(1, %"BLIN_U")", sizeof(mife_descriptor));
        errno = ENOMEM;
        goto out;
    }
    mife->flags = TRUFLAG(flags);
    mife->mindelta = mindelta(mife->flags);
    if  (!(MIFE_RTRY & flags)) mife->flags |= NRETRY00;
out:
    ifBLIN_QX2("- %"BLIN_X" flags=%08X", BLIN_I(mife), mife->flags);
    return(mife);
#   undef blin_internal_flags
}

static int
setime(mife_descriptor *mife, struct timespec *stime, time_t sec, long nsec) {
    int  ex = EX_OK;

    if  (!mife->kev) {
        if  (!(mife->kev = calloc(1, sizeof(mife_event)))) {
            ifBLIN_QX0("calloc(1, %"BLIN_U")", sizeof(mife_event));
            ERROUT(-EX_IOERR, ENOMEM);
        }
        mife->kev->kq = -1;
    }
    if  ((0 > mife->kev->kq) && (0 > (mife->kev->kq = kqueue()))) {
        ifBLIN_QW0("kqueue");
        EXOUT(-EX_IOERR);
    }
    stime->tv_sec = sec;
    stime->tv_nsec = nsec;
out:
    return(ex);
}

#define blin_internal_flags ((mife ? mife->flags : mife_default) & BLIN_MASK)

int
mife_ctlflag(mife_descriptor *mife, BLIN_flag flags) {
    int  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X" flags=%08X", BLIN_I(mife), flags);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    mife->flags = TRUFLAG(flags);
    if  (!(MIFE_RTRY & flags)) mife->flags |= NRETRY00;
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctlnull(mife_descriptor *mife) {
    int  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X, BLIN_I(mife));
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    if  (mife->kev && (0 < mife->kev->kq)) {
        close(mife->kev->kq);
        mife->kev->kq = -1;
    }
    if  (mife->buffer) {
        if  ((mife->flags & MIFE_MMAP) && !!(ex = munmap(mife->buffer, mife->buflen))) {
            ifBLIN_QW0("munmap");
        }
        if  (mife->flags & MIFE_ALLC) free(mife->buffer);
        mife->buffer = NULL;
        mife->buflen = 0;
    }
    if  ((mife->flags & MIFE_CLOS) && !!mife->a.close) {
        if  (!!(ex = mife->a.close(mife->a.ad))) {
            ifBLIN_QW0("mife->a.close");
        }
        mife->a.close = NULL;
    }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctlfile(mife_descriptor *mife, const char *s) {
    int  ex = EX_OK;
    int  i;

    ifBLIN_QX2("+ %"BLIN_X" %s", BLIN_I(mife), s);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    mife->flags = TRUFLAG(mife->flags) | MIFE_CLOS;
    if  (0 > (mife->f.fd = open(s, O_RDONLY))) {
        i = errno;
        ifBLIN_QW0("open %s", s);
        mife_ctlnull(mife);
        mife = NULL;
        ERROUT(-EX_IOERR, i);
    }
    mife->f.read = read;
    mife->f.close = close;
    mife->actbeg = 0;
    mife->actlen = 0;
    mife->offset = 0;
    mife->ofpoint = 0;
    mife->st_size = 0;
    if  (mife->eol) mife->eol->stop(mife);
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctlfdsc(mife_descriptor *mife, int fd) {
    int  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X" %d", BLIN_I(mife), fd);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    mife->flags = TRUFLAG(mife->flags);
    mife->f.fd = fd;
    mife->f.read = read;
    mife->f.close = close;
    mife->actbeg = 0;
    mife->actlen = 0;
    mife->offset = 0;
    mife->ofpoint = 0;
    mife->st_size = 0;
    if  (mife->eol) mife->eol->stop(mife);
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctlcnst(mife_descriptor *mife, const void *s, size_t size) {
    int  ex = EX_OK;
    int  i;

    ifBLIN_QX2("+ %"BLIN_X" %"BLIN_X"["BLIN_U"]", BLIN_I(mife), BLIN_I(s), size);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    mife->actlen = size;
    mife->flags = TRUFLAG(mife->flags) | MIFE_INMM | MIFE_ALLC | MIFE_EOFL;
    mife->mindelta = 0;
    mife->a.ad = (void *)-1;
    mife->ofpoint = mife->offset = 0;
    mife->actbeg = 0;
    i = (mife->flags & MIFE_FULL) ? 1 : 0;
    if  (!mife->actlen && !(mife->flags & MIFE_STRI)) mife->actlen = strlen(s);
    mife->buflen = mife->actlen + i;
    if  (!(mife->buffer = malloc(mife->buflen))) {
        ifBLIN_QX0("malloc(%"BLIN_U") MIFE_CTLCNST", mife->buflen);
        mife_ctlnull(mife);
        mife = NULL;
        ERROUT(-EX_OSERR, ENOMEM);
    }
    bcopy(s, mife->buffer, mife->actlen);
    if  (i) CHARBUF[mife->actlen] = 0;
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctlbuff(mife_descriptor *mife, void *s, size_t size) {
    int  ex = EX_OK;
    int  i;

    ifBLIN_QX2("+ %"BLIN_X" %"BLIN_X"["BLIN_U"]", BLIN_I(mife), BLIN_I(s), size);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    mife->actlen = size;
    mife->flags = TRUFLAG(mife->flags) | MIFE_INMM | MIFE_EOFL;
    mife->mindelta = 0;
    mife->a.ad = (void *)-1;
    mife->ofpoint = mife->offset = 0;
    mife->actbeg = 0;
    i = (mife->flags & MIFE_FULL) ? 1 : 0;
    if  (!mife->actlen && !(mife->flags & MIFE_STRI)) {
        mife->actlen = strlen(s);
        mife->buflen = mife->actlen + i;
        i = 0;
    } else {
        mife->buflen = mife->actlen + i;
    }
    mife->buffer = s;
    if  (i) CHARBUF[mife->actlen] = 0;
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctleoli(mife_descriptor *mife, const char *s) {
    int  ex = EX_OK;

    if  (!s) {
        ifBLIN_QX2("+ %"BLIN_X" NULL", BLIN_I(mife), !s ? "NULL" : s);
    } else {
        ifBLIN_QX2("+ %"BLIN_X" =%s~", BLIN_I(mife), s);
    }
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    if  (mife->eol) {
        if  (mife->eol->trilen && mife->eol->trix) {
            free(mife->eol->trix);
            mife->eol->trix = NULL;
        }
        free(mife->eol);
        mife->eol = NULL;
    }
    if  (!!s) {
        if  (!(mife->eol = malloc(sizeof(mife_eol)))) {
            ifBLIN_QX0("malloc(%"BLIN_U") MIFE_CTLEOLI", sizeof(mife_eol));
            ERROUT(-EX_OSERR, ENOMEM);
        }
        bcopy(&mife_eol0, mife->eol, sizeof(mife_eol));
        if  (mife->eol->init && ((ex = mife->eol->init(mife, s)) < 0)) {
            ifBLIN_QW0("eol->init MIFE_CTLEOLI");
            goto out;
    }   }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctltim0(mife_descriptor *mife, mife_tim timeout) {
    int  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X" %"BLIN_O"u", BLIN_I(mife), timeout);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    ex = setime(mife, &mife->itime, timeout / 1000000000ULL, timeout % 1000000000ULL);
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctltim1(mife_descriptor *mife, mife_tim timeout) {
    int  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X" %"BLIN_O"u", BLIN_I(mife), timeout);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    ex = setime(mife, &mife->otime, timeout / 1000000000ULL, timeout % 1000000000ULL);
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_fini(mife_descriptor *mife) {
    int  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X, BLIN_I(mife));
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    if  (!(ex = mife_ctlnull(mife))) {
        if  (mife->kev) {
            free(mife->kev);
            mife->kev = NULL;
        }
        if  (mife->eol) {
            if  (mife->eol->trilen && mife->eol->trix) {
                free(mife->eol->trix);
                mife->eol->trix = NULL;
            }
            free(mife->eol);
            mife->eol = NULL;
    }   }
    free(mife);
    mife = NULL;
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctltest(mife_descriptor *mife) {
    int  ex = EX_OK;

    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    ifBLIN_QX2( "+ %"BLIN_X"->%"BLIN_X"[%"BLIN_D"] actbeg=%"BLIN_D" actlen=%"BLIN_D
                " offset=%"BLIN_O"d ofpoint=%"BLIN_O"d"
              , BLIN_I(mife)
              , BLIN_I(mife->buffer)
              , mife->buflen
              , mife->actbeg
              , mife->actlen
              , mife->offset
              , mife->ofpoint
              )
    ;
    ex = -EX_DATAERR;
    if  (!mife->buffer) {
        ifBLIN_QX0("mife->buffer NULL");
        errno = EFAULT;
        goto out;
    }
    if  (mife->buffer == MAP_FAILED) {
        ifBLIN_QX0("mife->buffer MAP_FAILED");
        errno = EFAULT;
        goto out;
    }
    if  (!mife->buflen) {
        ifBLIN_QX0("mife->buflen 0");
        errno = EBADF;
        goto out;
    }
    if  ((mife->actbeg + mife->actlen) > mife->buflen) {
        ifBLIN_QX0( "mife->actbeg %"BLIN_U" + mife->actlen %"BLIN_U" > mife->buflen %"BLIN_U
                  , mife->actbeg
                  , mife->actlen
                  , mife->buflen
                  )
        ;
        errno = EBADF;
        goto out;
    }
    if  (mife->flags & MIFE_FULL) {
        if  ((mife->actbeg + mife->actlen) >= mife->buflen) {
            ifBLIN_QX0("no space for 0 in FULL");
            errno = EBADF;
            goto out;
        }
        if  (CHARBUF[mife->actbeg + mife->actlen]) {
            ifBLIN_QX0("no 0 in FULL");
            ex = EX_DATAERR;
            errno = EINVAL;
            goto out;
    }   }
    if  (0 > mife->offset) {
        ifBLIN_QX0("0 > mife->offset %"BLIN_O"d", mife->offset);
        errno = EBADF;
        goto out;
    }
    if  (0 > mife->ofpoint) {
        ifBLIN_QX0("0 > mife->ofpoint %"BLIN_O"d", mife->ofpoint);
        errno = EBADF;
        goto out;
    }
    if  (mife->offset > mife->ofpoint) {
        ifBLIN_QX0( "mife->offset %"BLIN_O"d > mife->ofpoint %"BLIN_O"d"
                  , mife->offset
                  , mife->ofpoint
                  )
        ;
        errno = EBADF;
        goto out;
    }
    if  ((mife->ofpoint - mife->offset) > (off_t)mife->actlen) {
        ifBLIN_QX0( "mife->thepoint %"BLIN_O"d > mife->actlen %"BLIN_U
                  , mife->ofpoint - mife->offset
                  , mife->actlen
                  )
        ;
        ex = EX_SOFTWARE;
        errno = EINVAL;
        goto out;
    }
    if  (mife->flags & MIFE_MMAP) {
        if  (0 > mife->st_size) {
            ifBLIN_QX0("0 > mife->st_size %"BLIN_O"d", mife->st_size);
            errno = EINVAL;
            goto out;
        }
        if  (mife->ofpoint > mife->st_size) {
            ifBLIN_QX0( "mife->ofpoint %"BLIN_O"d > mife->st_size %"BLIN_O"d"
                      , mife->ofpoint
                      , mife->st_size
                      )
            ;
            errno = EINVAL;
            goto out;
    }   }
    ex = EX_OK;
    if  (!(mife->flags & (MIFE_MMAP | MIFE_PIPE | MIFE_INMM))) {
        ifBLIN_QX1("Noninitialized");
    }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctlpipe(mife_descriptor *mife) {
    int   ex = EX_OK;
    int   fd[2];
    pid_t pid;

    ifBLIN_QX2("+ %"BLIN_X, BLIN_I(mife));
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    if  (0 > (ex = mife_ctltest(mife))) {
        ifBLIN_QW0("Illegal mife");
        goto out;
    } else if (!!ex) {
        ifBLIN_QX0("Incorrect mife");
        errno = EINVAL;
        ex = -ex;
        goto out;
    }
    if  (0 > pipe(fd)) {
        ifBLIN_QW0("pipe");
        ex = -EX_IOERR;
        goto out;
    }
    if  (0 > (pid = fork())) {
        ifBLIN_QW0("fork");
        ex = -EX_OSERR;
        goto out;
    }
    if  (!pid) {
        off_t   offset;
        off_t   delta;
        ssize_t len;

        for (len = 1, offset = mife->ofpoint; len > 0;) {
            if  (0 > (len = mife_read(mife, 0, offset))) {
                ifBLIN_QW0("mife_read");
                ex = -EX_IOERR;
                break;
            }
            offset += len;
            delta = mife->ofpoint - mife->offset;
            len = mife_writ(fd[1], mife->buffer + mife->actbeg + delta, mife->actlen - delta);
            if  (0 > len) {
                ifBLIN_QW0("mife_writ");
                ex = -EX_IOERR;
                break;
        }   }
        exit(ex);
    }
    ex = fd[0];
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

ssize_t
mife_ctlsize(mife_descriptor *mife) {
    ssize_t  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X, BLIN_I(mife));
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    if  (0 > (ex = mife_ctltest(mife))) {
        ifBLIN_QW0("Illegal mife");
        goto out;
    } else if (!!ex) {
        ifBLIN_QX0("Incorrect mife");
        errno = EINVAL;
        ex = -ex;
        goto out;
    }
    ex = mife->offset + mife->actlen - mife->ofpoint;
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}

int
mife_ctlofft(mife_descriptor *mife, off_t offset) {
    int  ex = EX_OK;

    ifBLIN_QX2("+ %"BLIN_X" %"BLIN_O"d", BLIN_I(mife), offset);
    if  (!mife) {
        ifBLIN_QX0("mife NULL");
        ERROUT(-EX_DATAERR, EFAULT);
    }
    ex = -EX_IOERR;
    if  (0 > offset) {
        ifBLIN_QX0("0 > offset %"BLIN_O"d", offset);
        errno = EINVAL;
        goto out;
    }
    if  (mife->offset > offset) {
        ifBLIN_QX0("mife->offset %"BLIN_O"d > offset %"BLIN_O"d", mife->offset, offset);
        errno = ESPIPE;
        goto out;
    }
    if  ((mife->flags & MIFE_MMAP) && (offset > mife->st_size)) {
        ifBLIN_QX0("offset %"BLIN_O"d > st_size %"BLIN_O"d", offset, mife->st_size);
        errno = EINVAL;
        goto out;
    }
    ex = EX_IOERR;
    if  ((offset - mife->offset) > (off_t)mife->actlen) {
        ifBLIN_QX0( "offset - mife->offset %"BLIN_O"d-%"BLIN_O"d=%"BLIN_O"d > mife->actlen %"BLIN_U
                  , offset
                  , mife->offset
                  , offset - mife->offset
                  , mife->actlen
                  )
        ;
        errno = EINVAL;
        goto out;
    }
    ex = EX_OK;
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
}
