#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/event.h>

#include <stdio.h>
#include <string.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <err.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <syslog.h>
#include <stdarg.h>
#include <time.h>

#define DEFAULT_PORT 11111
#define MAX_USERS    100
#define BUF_SIZE     1000

int failed = 0, all = 0;

void 
writeLogs(int priority, const char *message) {
       openlog("multistream",LOG_PID | LOG_NDELAY, LOG_DAEMON);
       syslog(priority, message);
       closelog(); 
}

void 
sig_handler(int signo) {
    if (signo == SIGHUP) printf("\nStatistics:\
\ttotal blocks:\t%d\t100 percent\
\twrited blocks:\t%d\t%5.2f percent\
\tfailed blocks:\t%d\t%5.2f percent\
block size:\t %d bytes\n\n",
all, all - failed, all ? (float)(all - failed) / all * 100 : all,
failed, all ? (float)failed / all * 100 : all, BUF_SIZE);
    return;
}

int
main(int ac, char **av) {
       
	int serverSocket, clientSocket, userSocket;
        struct sockaddr_in serverAddr, clientAddr, userAddr;
        int clientPort, serverPort, len, i, userCounter = 0, debug = 0;
        char *p, *msg = "Too many users\n";
        struct in_addr serverIP;
        struct hostent *he;
	int user_fd[MAX_USERS];
        
	int kq, event;
	struct kevent ke;

	char buf[BUF_SIZE];
	char readBuf[10000];
        char logMessage[256];
        
	FILE *pid = NULL;

        time_t time1 = 0, time2 = 0;

        signal(SIGPIPE, SIG_IGN);
	signal(SIGHUP, SIG_IGN);
        
	signal(SIGHUP, &sig_handler);	

        while ( (i = getopt(ac, av, "dP:")) > 0) {
                switch (i) {
                case 'd':
                       debug = 1;
                break;
		case 'P':
        		if ((pid = fopen(optarg, "w")) == NULL) 
            			err(1, "Can't create PID file %s", optarg);
        	break;
		default:
			err(1, "Unknown error");
                }   
        }

        ac -=optind;
        av +=optind;

        if (ac < 2) {
                printf("Usage: multistream [-d] [-P pidfile] client_port host:port\n");
                exit(0);
        }
        
        clientPort = atoi(av[0]);
        if (debug) 
                  printf("Port for clients:\t%d\n", clientPort);
        
        if ((p = strchr(av[1], ':')) != NULL) {
                *(p++) = '\0';
                serverPort = atoi(p);       
        } else serverPort = DEFAULT_PORT;
        
        if (!(he = gethostbyname(av[1]))) {
                printf("Unknown server name\n");
                exit(0);
        }   
     
        serverIP = *(struct in_addr *)he->h_addr_list[0];
        
        if (debug)
                   printf("Server\t%s:%d\n", inet_ntoa(serverIP), serverPort);

        /* connect to server */
        if ((serverSocket = socket(AF_INET, SOCK_STREAM, 0)) < 0) 
		err(1, "serverSocket");
        memset(&serverAddr, 0, sizeof(struct sockaddr_in));
	serverAddr.sin_family = AF_INET;
        serverAddr.sin_port = htons(serverPort);
        serverAddr.sin_addr = serverIP;
	
	if (connect(serverSocket, (struct sockaddr *)&serverAddr, sizeof serverAddr) < 0) 
		err(1, "connect");
	
	/* listening socket */
        if ((clientSocket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		err(1, "clientSocket");
	memset(&clientAddr, 0, sizeof(struct sockaddr_in));
	clientAddr.sin_family = AF_INET;
        clientAddr.sin_port = htons(clientPort);
        clientAddr.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(clientSocket, (struct sockaddr *)&clientAddr, sizeof clientAddr) < 0)
		err(1, "bind");
	
	if (listen(clientSocket, 1) < 0)	
		err(1, "listen");

	/* kqueue */
	if ((kq = kqueue()) < 0)
		err(1, "kqueue");	
	
	/* fill out the kevent struct */
	memset(&ke, 0, sizeof(struct kevent));
	EV_SET(&ke, serverSocket, EVFILT_READ, EV_ADD, 0, 5, NULL);

	/* set the event */
	if (kevent(kq, &ke, 1, NULL, 0, NULL) < 0)
		err(1, "kevent: serverSocket");
	
	/* fill out the kevent struct */
	memset(&ke, 0, sizeof(struct kevent));
	EV_SET(&ke, clientSocket, EVFILT_READ, EV_ADD, 0, 5, NULL);

	/* set the event */
	if (kevent(kq, &ke, 1, NULL, 0, NULL) < 0)
		err(1, "kevent: clientSocket");

	memset(user_fd, 0, sizeof(int) * MAX_USERS);
	
	if (pid) {
        	fprintf(pid, "%d", getpid());
        	fclose(pid);
    	}

	if (debug) 
	           time1 = time(NULL);
	while(1) {
		
		if (debug) {
		          time2 = time(NULL);  
		          if (abs(time2 - time1) >= 5) {
		                  time1 = time2;
		                  printf("\nStatistics:\
\ttotal blocks:\t%d\t100 percent\
\twrited blocks:\t%d\t%5.2f percent\
\tfailed blocks:\t%d\t%5.2f percent\
block size:\t %d bytes\n\n",
all, all - failed, all ? (float)(all - failed) / all * 100 : all,
failed, all ? (float)failed / all * 100 : all, BUF_SIZE);
		           }
		}
		memset(&ke, 0, sizeof(ke));
                  
		/* receive an event, a blocking call as timeout is NULL */
		if ((event = kevent(kq, NULL, 0, &ke, 1, NULL)) < 0) {
			if (errno == EINTR) continue;
			else err(1, "kevent!");		
		} 
                if (event == 0) continue;
		
		/* got a message from clientSocket */
		if (ke.ident == clientSocket) {
			len = (socklen_t)sizeof(userAddr);
			userSocket = accept(clientSocket, (struct sockaddr *)&userAddr, &len);
			if (userSocket == -1)
				err(1, "accept");
						
			if (userCounter == MAX_USERS) {
				write(userSocket, msg, strlen(msg));
				if (debug) 
				          printf("User limit reached\n");
				snprintf(logMessage, 256, "User limit reached");
                                writeLogs(LOG_NOTICE, logMessage);
				close(userSocket);
				continue;
			}
		
			user_fd[userCounter++] = userSocket;
			if (debug)
			         printf("New user connected:\t%s:%d\n", inet_ntoa(*(struct in_addr *)&userAddr.sin_addr.s_addr),
			                ntohs(userAddr.sin_port));
                        snprintf(logMessage, 256, "New user connected: %s:%d", inet_ntoa(*(struct in_addr *)&userAddr.sin_addr.s_addr), 
                                  ntohs(userAddr.sin_port));
                        writeLogs(LOG_DEBUG, logMessage);
						
			EV_SET(&ke, userSocket, EVFILT_READ, EV_ADD, 0, 0, NULL);
			if (kevent(kq, &ke, 1, NULL, 0, NULL) < 0)
				err(1, "kevent: add user");
		
		/* got a message from server */
		} else if (ke.ident == serverSocket) {
			memset(buf, 0, sizeof(buf));

			i = read(serverSocket, buf, sizeof(buf));
			if (i == -1)
				continue;
			if (i == 0) { /* EOF from server */
				/* connection closed by server */
				printf("Connection closed by server\n");
				snprintf(logMessage, 256, "Connection closed by server");
                                writeLogs(LOG_NOTICE, logMessage);
				close(serverSocket);
				exit(0);
			}
                        
                        for (i = 0; i < userCounter; i++) {
                                if (write(user_fd[i], buf, sizeof(buf)) < 0) 
					failed++;
				all++;		
		        }
		} else 
		        /* got a message from users */
		        for (i = 0; i < userCounter; i++) 
		                if (ke.ident == user_fd[i]) {  
		                        if (read(user_fd[i], readBuf, sizeof(readBuf)) == 0) {
		                                /* delete user */
		                                EV_SET(&ke, user_fd[i], EVFILT_READ, EV_DELETE, 0, 0, NULL);
                			        if (kevent(kq, &ke, 1, 0, 0, NULL) < 0)
                				        err(1, "rm user from kq");
	                                        user_fd[i] = user_fd[--userCounter];
	                                        if (debug) 
	                                                 printf("User <%s:%d> logged off\n",
	                                                        inet_ntoa(*(struct in_addr *)&userAddr.sin_addr.s_addr),
			                                        ntohs(userAddr.sin_port));
						snprintf(logMessage, 256, "User <%s:%d> logged off",
						         inet_ntoa(*(struct in_addr *)&userAddr.sin_addr.s_addr),
			                                 ntohs(userAddr.sin_port));
                                                writeLogs(LOG_DEBUG, logMessage);
	                                        close(user_fd[i]);
	                                 }
	                                 break;
	                        }
	} /* end of while */	
}
