/*
 * Copyright (c) 2005 Anton Antonov <aga@pikenet.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: compare_tables.c,v 1.11 2006/01/27 10:28:42 aga Exp $ 
 */

#include <syslog.h>
#include <string.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <errno.h>
#include <stdlib.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/ip_fw2.h>
#include <net/route.h> /* def. of struct route */
#include <netinet/ip_dummynet.h>
#include <stdio.h>
#include <multilar.h>
#include <babolo/BLINflag.h>

int s, kq, event, HZ;
char errMessage[256];
struct kevent ke;
ipfw_table *tbl;

int time_counter = 0;
long t = 0;
struct timespec timeout; /* timeout */

void 
writeLogs(int priority, const char *message);

int 
compare_ent(ipfw_table_entry ent1, ipfw_table_entry ent2);

static int
do_cmd(int optname, void *optval, uintptr_t optlen) {
    int i;
    /* fill out the kevent struct */
    EV_SET(&ke, s, EVFILT_WRITE, EV_ADD, 0, 0, NULL);
    /* set the event */
    event = kevent(kq, &ke, 1, NULL, 0, NULL);
    if (event == -1) {
        snprintf(errMessage, 256, "kevent: set event(do_cmd): %s", strerror(errno) );
        writeLogs(LOG_ALERT, errMessage);
        return -1;
    }

    memset(&ke, 0, sizeof(ke));
    /* receive an event */
    event = kevent(kq, NULL, 0, &ke, 1, &timeout);
    if (event < 0) {
        snprintf(errMessage,256,"kevent: receive event(do_cmd): %s", strerror(errno) );
        writeLogs(LOG_ALERT, errMessage);
        return -1;
    }
    if (ke.ident == s) {
        i = setsockopt(s, IPPROTO_IP, optname, optval, optlen);
        /* Delete event from the kqueue */
        /* fill out the kevent struct */
        EV_SET(&ke, s, EVFILT_WRITE, EV_DELETE, 0, 0, NULL);
        /* set the event */
        event = kevent(kq, &ke, 1, NULL, 0, NULL);
        if (event == -1) {
            snprintf(errMessage,256,"kevent: delete event(do_cmd): %s", strerror(errno) );
            writeLogs(LOG_ALERT, errMessage);
            return -1;
        }
        return i;
    } else {
        snprintf(errMessage, 256, "kevent: bad event(do_cmd)");
        writeLogs(LOG_ALERT, errMessage);
        return -1;
    }
}

int
get_table(int num) {
      uint32_t a;
      ipfw_table_entry ent;
      socklen_t l;
      long tick1, tick2;
      int i, len = sizeof(long);
      	   
      ent.tbl = num;
      a = ent.tbl;
      l = sizeof(a);
      if ( sysctlbyname("net.inet.ip.dummynet.curr_time", &tick1, &len, 0, 0) == -1) {
            snprintf(errMessage, 256, "sysctlbyname: net.inet.ip.dummynet.curr_time: %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -1;
      }
      i = getsockopt(s, IPPROTO_IP, IP_FW_TABLE_GETSIZE, &a, &l);
      if( sysctlbyname("net.inet.ip.dummynet.curr_time", &tick2, &len, 0, 0) == -1) {
            snprintf(errMessage, 256, "sysctlbyname: net.inet.ip.dummynet.curr_time: %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -1;
      }
      if (i < 0) {
            snprintf(errMessage,256,"get_table: getsockopt(IP_FW_TABLE_GETSIZE): %s", strerror(errno) );
            writeLogs(LOG_ALERT, errMessage);
            return -1;	
      }
      t += abs(tick2 - tick1) * HZ;          
      l = sizeof(*tbl) + a * sizeof(ipfw_table_entry);
      free(tbl);
      if ( (tbl = malloc(l)) == NULL) {
      	    snprintf(errMessage, 256, "get_table: malloc: %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -1;	          
      }
      tbl->tbl = ent.tbl;
      if ( sysctlbyname("net.inet.ip.dummynet.curr_time", &tick1, &len, 0, 0) == -1) {
            snprintf(errMessage, 256, "sysctlbyname: net.inet.ip.dummynet.curr_time: %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -1;
      }
      i = getsockopt(s, IPPROTO_IP, IP_FW_TABLE_LIST, tbl, &l);
      if (sysctlbyname("net.inet.ip.dummynet.curr_time", &tick2, &len, 0, 0) == -1) {
            snprintf(errMessage, 256, "sysctlbyname: net.inet.ip.dummynet.curr_time: %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -1;
      }
      if (i<0){
      	    snprintf(errMessage,256,"get_table: getsockopt(IP_FW_TABLE_LIST): %s", strerror(errno) );
            writeLogs(LOG_ALERT, errMessage);
            return -1;	
      }
      time_counter++;
      t += abs(tick2 - tick1) * HZ;
      /* (Average time)*100 */
      t = 100 * t / time_counter;
      if ( t < 100 * HZ ) t = 100 * HZ;
      if ( t >= 1000000) {
          timeout.tv_sec = (time_t)(t / 1000000);
          timeout.tv_nsec = (t % 1000000) * 1000;
      } else timeout.tv_nsec = t * 1000;
      return 0;
}

int
compare_tables(mular_descriptor **mular, int *counter) {
	int i, j, k, m, state, is_exist;
	ipfw_table_entry *tmp_ent;
	enum states {begin, compare, end};
	
	struct clockinfo cl_info;
	int len = sizeof(struct clockinfo);
	
	timeout.tv_sec = 1;
	timeout.tv_nsec = 0;
	/* socket */
        if ((s = socket(AF_INET, SOCK_RAW, IPPROTO_RAW)) < 0){
        	snprintf(errMessage, 256, "socket: %s", strerror(errno) );
        	writeLogs(LOG_EMERG, errMessage);
        	return -1;
    	}
        /* kqueue */
    	if ((kq = kqueue()) < 0) {
        	snprintf(errMessage, 256, "kqueue: %s", strerror(errno) );
        	writeLogs(LOG_EMERG, errMessage);
        	return -1;
        }
	if (sysctlbyname("kern.clockrate", &cl_info, &len, 0, 0) < 0) {
        	snprintf(errMessage, 256, "sysctlbyname: kern.clockrate: %s", strerror(errno) );
        	writeLogs(LOG_EMERG, errMessage);
        	return -1;
    	} else HZ = cl_info.tick;

	for(i = 0; i < 128; i++) {
		if (get_table(i) < 0) return -1;
		if (counter[i] != 0 && mular[i] != NULL) { 
			  k = 0; j = 0;
			  while (j < counter[i]) {
			  	state = 0;
				is_exist = 0;
				tmp_ent = (ipfw_table_entry *)mular_getix(mular[i], j++);
				while (state < end) {
					switch (state) {
					case begin:
						if (k < tbl->cnt) state = compare;
						else state = end;
						break;
					case compare:
						if ((m = compare_ent(*tmp_ent, (*tbl).ent[k])) == 0 && 
						     tmp_ent->value == (*tbl).ent[k].value) {
							k++;
							is_exist = 1;
							state = end;
						} else if (m <= 0) { 
							if (do_cmd(IP_FW_TABLE_DEL, &(*tbl).ent[k], sizeof((*tbl).ent[k])) < 0) {
								snprintf(errMessage, 256,"compare_tables: do_cmd(IP_FW_TABLE_DEL): %s", strerror(errno));
            							writeLogs(LOG_ALERT, errMessage);
            							return -1;
							} else {k++; state = begin;}
						}else state = end;
						break;
					}
				}
				if (!is_exist) if (do_cmd(IP_FW_TABLE_ADD, tmp_ent, sizeof(*tmp_ent)) < 0) {
                                       snprintf(errMessage, 256, "compare_tables: do_cmd(IP_FW_TABLE_ADD): %s", strerror(errno));
                                       writeLogs(LOG_ALERT, errMessage);
                                       return -1;
                               }
			}/*     */
                        for( ; k < tbl->cnt; k++)
                                if (do_cmd(IP_FW_TABLE_DEL, &(*tbl).ent[k], sizeof((*tbl).ent[k])) < 0) {
                                        snprintf(errMessage, 256, "compare_tables: do_cmd(IP_FW_TABLE_DEL): %s", strerror(errno));
                                        writeLogs(LOG_ALERT, errMessage);
                                        return -1;
                                }
		} else continue;
	}
     for (i = 0; i < 128; i++) mular_destroy(mular[i]);
     free(tbl);
     return 0;
}
