/*
 * Copyright (c) 2005 Anton Antonov <aga@pikenet.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: conrules.c,v 1.10 2006/08/10 14:27:29 aga Exp $
 */

#include <sys/param.h>
#include <sys/socket.h>

#include <ctype.h>
#include <errno.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_fw2.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

#include <syslog.h>
#include <fcntl.h>
#include "multilar.h"
#include "babolo/BLINflag.h"

#include "conrules.h"

#define FROM_STDIN 0x1
#define FROM_FILE  0x2

#define R_NUM   100      /*      */
#define BLOCK   4000000  /*      rules */
#define STR_LEN 20       /*      */

/*
 *  
 */
 
char errMessage[256], str[20];
FILE *f;
int c, lineno, br_counter;
ipfw_insn *cmd, *dst;

int 
compare_rules(mular_descriptor **, int *, void *);

/*
 *   Next command
 */
static 
ipfw_insn *next_cmd(ipfw_insn *cmd) {
	cmd += F_LEN(cmd);
	bzero(cmd, sizeof(*cmd));
	return cmd;
}

/*
 * A function to fill simple commands of size 1.
 * Existing flags are preserved.
 */
static void
fill_cmd(ipfw_insn *cmd, int flags, uint16_t arg) {
	cmd->len =  ((cmd->len | flags) & (F_NOT | F_OR)) | 1;
	cmd->arg1 = arg;
}

static int
match_token(struct _s_x *table, char *string) {
	struct _s_x *pt;
	uint i = strlen(string);

	for (pt = table ; i && pt->s != NULL ; pt++)
		if (strlen(pt->s) == i && !bcmp(string, pt->s, i))
			return pt->x;
	return -1;
};

int detect_ID(char *string, struct _s_x *h_t) {
	for(; h_t->s != NULL ;h_t++) 	
		if (strcmp(h_t->s, string) == 0)
			return h_t->x;
        return -1;
}

static int
contigmask(uint8_t *p, int len) {
	int i, n;

	for (i=0; i < len ; i++)
		if ( (p[i/8] & (1 << (7 - (i%8)))) == 0) /* first bit unset */
			break;
	for (n=i+1; n < len; n++)
		if ( (p[n/8] & (1 << (7 - (n%8)))) != 0)
			return -1; /* mask not contiguous */
	return i;
}

void 
show_usage(void) {
    fprintf(stderr,"usage:
            conrules [-f file ... ]
            The program will read the first 12 files\n");
    exit(-1);
}

void 
writeLogs(int priority, const char *message){
       openlog("conrules",LOG_PID | LOG_NDELAY,LOG_DAEMON);
       syslog(priority, message);
       closelog(); 
}

int 
check(struct ip_fw *r) {
        ipfw_insn *tmp;
        int i = 0;
        int flag = 0;
        
        if (r->rulenum < 1 || r->rulenum > 65534 || r->set > RESVD_SET) return 1;    
        
        for (tmp = r->cmd; tmp != dst; tmp += i) {
                i = F_LEN(tmp);
                switch (tmp->opcode) {
                   /*     O_PROB */
                   case O_PROB: 
                      if (flag == 0) {
                          ipfw_insn_u32 *p = (ipfw_insn_u32 *)tmp;
                          double d = 1.0 * p->d[0];
                          d = (d / 0x7fffffff);
                          if (d <= 0 || d > 1.0) return 1;
                      } else return 1;
                      flag = 1;
                   break;
                   /*     O_PROBE_STATE */
                   case O_PROBE_STATE:
                       if (flag > 1) return 1;
                       flag = 2;
                   break; 
                   /*  */
		   case O_IP_SRC_LOOKUP:
		   case O_IP_DST_LOOKUP:
			   if (((ipfw_insn_ip *)tmp)->o.arg1 > 127) return 1;
		   case O_IP_SRC:       
                   case O_IP_DST:
		   case O_IP_SRC_ME:
		   case O_IP_DST_ME:
                   case O_IP_SRC_MASK:
                   case O_IP_DST_MASK:
                   case O_IP_SRC_SET:
                   case O_IP_DST_SET: 
                      if (flag > 2) return 1;
                      flag = 2;
                   break;
                   case O_IP_SRCPORT:
                   case O_IP_DSTPORT:
                   case O_IPLEN:
                   case O_IPID:
                   case O_IPTTL:
                        if (flag > 2) return 1;
                        {
                          uint16_t *port;
                          int i;
                          port = ((ipfw_insn_u16 *)tmp)->ports;
                          for (i = F_LEN((ipfw_insn *)tmp) - 1; i > 0; i--, port += 2) 
                                if (port[0] > port[1]) return 1;
                        }
                        flag = 2;
                        break;
                   case O_PROTO: 
                        if (flag > 2) return 1;
			if (tmp->arg1 > 255 || tmp->arg1 == 0) return 1;
                        flag = 2;
                        break;
                   case O_RECV:
                   case O_XMIT:
                   case O_VIA:
                   case O_UID:
                   case O_GID:
                   case O_TCPWIN:
                   case O_TCPSEQ:
                   case O_TCPACK:
                   case O_ICMPTYPE:
                   case O_TCPOPTS:
                   case O_IPVER:        
                   case O_IPPRECEDENCE:
                   case O_IPOPT:
                   case O_IPTOS:
                   case O_TCPFLAGS:
                   case O_LAYER2:
                   case O_ESTAB:
                   case O_FRAG:
                   case O_IN:
                   case O_VERREVPATH:
                        if (flag > 2) return 1;
                        flag = 2; 
                   break;
                   /* have_state command*/
                   case O_KEEP_STATE:
                   case O_LIMIT:
		        if (flag > 2) return 1;
		        if (((ipfw_insn_limit *)cmd)->conn_limit == 0 || 
			    ((ipfw_insn_limit *)cmd)->limit_mask == 0) return 1;
                        flag = 3;
                   break;
                   /* O_LOG */
                   case O_LOG:
                        if (flag > 3) return 1;
                        flag = 4;
                   break;
                   /* actions */
                   case O_CHECK_STATE:
                   case O_ACCEPT:
                   case O_COUNT:
                   case O_DENY:
                   case O_REJECT:
                   case O_SKIPTO:
                   case O_PIPE:
                   case O_QUEUE:
                   case O_DIVERT:
                   case O_TEE:
                   case O_FORWARD_IP:
                        if (flag > 4) return 1;
                        flag = 5;
                   break;
		   default: return 1;      
                } /* end of switch */   
        } /* end of for */
        if (flag != 5) return 1;
        return 0;
}

int 
inner_automat(int id_number, int opcode){
	int i = 0, j = 0, r = 0;
	uint32_t action, state = 0;
        ipfw_insn_u16 *cmd16 = (ipfw_insn_u16 *)cmd;
        ipfw_insn_if *cmd_if = (ipfw_insn_if *)cmd ;
        uid_t uid;
        struct passwd *pwd;
        gid_t gid;
        struct group *grp;
        double match_prob;
        uint32_t *d = NULL;
        int val, masklen, len = 0;
        struct _s_x *flags = NULL;
        uint16_t *port = cmd16->ports;
        uint32_t *map = NULL;
        uint8_t type;
	uint8_t which = 0;
	char *params[256];       /*   */
	
	//printf("I'm here\n");
	/*  params */
	if (id_number < 10)
           while (state < in_end) {
		
		action = in_auto[id_number - 1][state][class[c]];
		state = action & 0xff;     /*   */
		if (action & A_rn) {
			if (j >= STR_LEN) {r = -1; break;}
                        else str[j++] = (char)c; 
		}
		if (action & A_wr) { 
			if (j >= STR_LEN) {r = -1; break;}
			else {
				if (action & DASH) str[j++] = '-';
				str[j] = '\0';
                                strncpy(params[i++] = (char *)malloc(strlen(str) + 1), str, strlen(str) + 1);
				j = 0;	
				if (action & MASKLEN) str[j++] = '/';
				else if (action & MASK) str[j++] = ':';
		}	}
		if (state < in_end) {
			c = getc(f);
			if (class[c] == ob) br_counter++;
			if (class[c] == cb) br_counter--;
		}
	} 

	if (i < 1) return -1;
	/* Print params */
	//for (j = 0; j < i ; j++)
	//	printf("Param %d = %s\n", j, params[j]);

        switch(opcode) {
	case O_LOG: 
		cmd->len = F_INSN_SIZE(ipfw_insn_log);
                ((ipfw_insn_log *)cmd)->max_log = atoi(params[0]);
                if(((ipfw_insn_log *)cmd)->max_log < 0) r = -1;
	break;
	case O_DIVERT:
	case O_TEE:
		cmd->arg1= strtoul(params[0], NULL, 0);
	break;	  
	case O_QUEUE:
	case O_PIPE:
		cmd->len = F_INSN_SIZE(ipfw_insn_pipe);
	case O_SKIPTO:
	case O_REJECT:
		cmd->arg1 = strtoul(params[0], NULL, 10);
	break;
	case O_PROTO: 
		fill_cmd(cmd, 0, atoi(params[0])); 
	break;
	case O_IPVER: 
		fill_cmd(cmd, 0, strtoul(params[0], NULL, 0));
	break;
	case O_TCPWIN:
		fill_cmd(cmd, 0, htons(strtoul(params[0], NULL, 0)));
	break;
	case O_IPPRECEDENCE:
	     	fill_cmd(cmd, 0, (strtoul(params[0], NULL, 10) & 7) << 5);
	break;
	case O_TCPSEQ:
	case O_TCPACK:
		cmd->len = F_INSN_SIZE(ipfw_insn_u32);
		((ipfw_insn_u32 *)cmd)->d[0] = htonl(strtoul(params[0], NULL, 0));
	break;
	case O_UID:
		uid = strtoul(params[0], NULL, 0);
		pwd = getpwuid(uid); 
		((ipfw_insn_u32 *)cmd)->d[0] = pwd->pw_uid;
		cmd->len = F_INSN_SIZE(ipfw_insn_u32);
	break;
	case O_GID:
		gid = strtoul(params[0], NULL, 0);
		grp = getgrgid(gid);
		((ipfw_insn_u32 *)cmd)->d[0] = grp->gr_gid;
		cmd->len = F_INSN_SIZE(ipfw_insn_u32);
	break;
	case O_PROB:
		match_prob = strtod(params[0], NULL );
		if (match_prob > 1 || match_prob < 0) match_prob = 0;
		cmd->len = 2;
		*((int32_t *)(cmd + 1)) = (int32_t)(match_prob * 0x7fffffff);
	break;
	case O_FORWARD_IP:
		cmd->len = F_INSN_SIZE(ipfw_insn_sa);
		((ipfw_insn_sa *)cmd)->sa.sin_len = sizeof(struct sockaddr_in);
		((ipfw_insn_sa *)cmd)->sa.sin_family = AF_INET;
		((ipfw_insn_sa *)cmd)->sa.sin_port = 0;
		inet_aton(params[0], &((ipfw_insn_sa *)cmd)->sa.sin_addr);
                if (i > 1) ((ipfw_insn_sa *)cmd)->sa.sin_port = atoi(params[1]);
	break;
	case O_IPID:
	case O_IPLEN:
	case O_IPTTL:
		for (j = i; j < i; j++) 
			if (params[j][strlen(params[j]) - 1] == '-') {
			   if (j + 1 < i) {
				port[0] = strtoul(params[j++], NULL, 0);
				port[1] = strtoul(params[j], NULL, 0);
				((ipfw_insn_u16 *)cmd)->o.len |= 2;
			   } else {r = -1; break;}
			} else {
				fill_cmd(cmd, 0, strtoul(params[j], NULL, 0));
			} 
	break;
	case O_IP_SRCPORT:
	case O_IP_DSTPORT:   
		for (j = 0; j < i; j++) {
			if (params[j][strlen(params[j]) - 1] == '-') {
			   if (j + 1 < i) {
				port[0] = atoi(params[j++]);
				port[1] = atoi(params[j]);
			   } else {r = -1; break;}
			} else  port[0] = port[1] = atoi(params[j]);
			if (((ipfw_insn_u16 *)cmd)->o.len + 1 > F_LEN_MASK) {r = -1; break;}
			else  {
				((ipfw_insn_u16 *)cmd)->o.len++;
				port += 2;
			}
		}
		if (r > -1) ((ipfw_insn_u16 *)cmd)->o.len++;
	break;
   	case O_ICMPTYPE:
		((ipfw_insn_u32 *)cmd)->o.len |= F_INSN_SIZE(ipfw_insn_u32);
		for (j = i; j < i; j++) 
			if ((type = strtoul(params[j], 0, 0)) > 31) {r = -1; break;}
                        else ((ipfw_insn_u32 *)cmd)->d[0] |= 1 << type; 
	break;		
	case O_TCPFLAGS: flags = f_tcpflags;
	case O_TCPOPTS: if (opcode == O_TCPOPTS) flags = f_tcpopts;
	case O_IPTOS:   if (opcode == O_IPTOS) flags = f_iptos;
	case O_IPOPT:   if (opcode == O_IPOPT) flags = f_ipopts;
		cmd->len = (cmd->len & (F_NOT | F_OR)) | 1;
		for (j = 0; j < i; j++) 
			if (*params[j] == '!') {
				val = match_token(flags, &params[j][1]);
				if (val > 0) which |= (uint8_t)val; 
                                else {r = -1; break;}
		        	cmd->arg1 |= ((which & 0xff) << 8);
		    	} else {
				val = match_token(flags, params[j]);
				if (val > 0) which |= (uint8_t)val;
                                else {r = -1; break;}; 
		        	cmd->arg1 |= (which & 0xff);
			}
	break;
	case O_RECV:
	case O_XMIT:
	case O_VIA:
		cmd_if->name[0] = '\0';
		cmd_if->o.len |= F_INSN_SIZE(ipfw_insn_if);
		if (!strcmp(params[0], "any")) cmd_if->o.len = 0;	
		else if (!isdigit(*params[0])) {
			char *q;
			strncpy(cmd_if->name, params[0], sizeof(cmd_if->name));
			cmd_if->name[sizeof(cmd_if->name) - 1] = '\0';
			for (q = cmd_if->name; *q && !isdigit(*q) && *q != '*'; q++) continue;
			cmd_if->p.unit = (*q == '*') ? -1 : atoi(q);
			*q = '\0';
		} else inet_aton(params[0], &cmd_if->p.ip);
	break;
	case O_LIMIT:
		cmd->len = F_INSN_SIZE(ipfw_insn_limit);
		((ipfw_insn_limit *)cmd)->limit_mask = 0;
                ((ipfw_insn_limit *)cmd)->conn_limit = 0;
		for (j = 0; j < i; j++)
			if (isdigit(*params[j])) ((ipfw_insn_limit *)cmd)->conn_limit = atoi(params[j]);
		    	else {
				int val = match_token(limit_masks, params[j]);
				if (val <= 0) break;
				((ipfw_insn_limit *)cmd)->limit_mask |= val;
		    	}
	break;
	case O_IP_SRC_LOOKUP:
	case O_IP_DST_LOOKUP:
		((ipfw_insn_ip *)cmd)->o.len &= ~F_LEN_MASK;
		((ipfw_insn_ip *)cmd)->o.arg1 = strtoul(params[0], NULL, 0);
		((ipfw_insn_ip *)cmd)->o.len |= F_INSN_SIZE(ipfw_insn); 
		if (i > 1) {
			((ipfw_insn_ip *)cmd)->o.len &= ~F_LEN_MASK;
			d = ((ipfw_insn_u32 *)cmd)->d;
			((ipfw_insn_ip *)cmd)->o.len |= F_INSN_SIZE(ipfw_insn_u32);
                        d[0] = strtoul(params[1], NULL, 0);
		}
	break;
	case O_IP_SRC:
	case O_IP_DST:
	case O_IP_SRC_MASK:
	case O_IP_DST_MASK:
		d = ((ipfw_insn_u32 *)cmd)->d;
		((ipfw_insn_ip *)cmd)->o.len &= ~F_LEN_MASK;	// zero len
		
		for (j = 0; j < i; j++) {
			if (!inet_aton(params[j], (struct in_addr *)&d[0])) {r = -1; break;} //ip
			if (cmd->opcode == O_IP_SRC_MASK || cmd->opcode == O_IP_DST_MASK) {
				if (j + 1 < i && *params[j + 1] == '/') {
					if ((masklen = atoi(&params[j + 1][1])) > 32) {r = -1; break;}
					else if (masklen == 32 && i == 2) {
						i = 1;
						if (opcode == O_IP_SRC_MASK)  cmd->opcode = O_IP_SRC;
						if (opcode == O_IP_DST_MASK)  cmd->opcode = O_IP_DST;
					} else if (masklen == 0) d[1] = htonl(0);	// mask 
                                	else d[1] = htonl(~0 << (32 - masklen));
					j++;
				} else if (j + 1 < i && *params[j + 1] == ':') {
					if (!inet_aton(&params[j + 1][1], (struct in_addr *)&d[1])) {r = -1; break;}
					j++;
				} else if (i > 1) {
					d[1] = htonl(~0);
				} else {
					if (cmd->opcode == O_IP_SRC_MASK)  cmd->opcode = O_IP_SRC;
					else if (cmd->opcode == O_IP_DST_MASK)  cmd->opcode = O_IP_DST;
					else {r = -1; break;}
			}	}
			if (cmd->opcode == O_IP_SRC || cmd->opcode == O_IP_DST) {
				d[1] = htonl(~0);
				d[0] &= d[1];
			        if (i == 1) {
					((ipfw_insn_ip *)cmd)->o.len |= F_INSN_SIZE(ipfw_insn_u32);
					break;
			}       }
			len += 2;	/* two words... */
			d += 2;
		}
		if (i > 1) ((ipfw_insn_ip *)cmd)->o.len |= len + 1;
	break;
	case O_IP_SRC_SET:
	case O_IP_DST_SET: 
		d = ((ipfw_insn_u32 *)cmd)->d;
		((ipfw_insn_u32 *)cmd)->o.len &= ~F_LEN_MASK;
		if (!inet_aton(params[0],(struct in_addr *)&d[0])) {r = -1; break;} 
		d[1] = htonl(~0 << (32 - 24));
		if (i > 1) {
			if (*params[1] == '/') masklen = atoi(&params[1][1]);
			else masklen = 24;
			d[1] = htonl(~0 << (32 - masklen));
		}
		map = (uint32_t *)&((ipfw_insn_ip *)cmd)->mask;
		j = contigmask((uint8_t *)&(d[1]), 32);
		if (j < 24 || j > 31) return -1;
		((ipfw_insn_ip *)cmd)->o.arg1 = 1 << (32 - j);   // map length 
                d[0] = ntohl(d[0]);
		((ipfw_insn_ip *)cmd)->o.len |= F_INSN_SIZE(ipfw_insn_u32) + (((ipfw_insn_ip *)cmd)->o.arg1 + 31) / 32;
		for (j = 0; j < (((ipfw_insn_ip *)cmd)->o.arg1 + 31) / 32 ; j++)
			map[j] = 0;     // clear map
		if (*params[1] == '/') j = 2; 
		else j = 1;
		for (; j < i; j++) 
                        map[strtol(params[j], 0, 0)/32] |= 1<<(strtol(params[j], 0, 0) & 31);
	break;
	} // end of switch
	
	for (j = 0; j < i; j ++) free(params[j]);

	return r;
}  // end of inner_automat


int 
rules_reader(int ac, char **av) {
    
    int input = FROM_FILE, fd = -1, lineno_begin, i = 0, id_number, opcode, j;  
    uint32_t state, action, old_state;
    struct ip_fw *rule; /*    */
    ipfw_insn *prev = NULL;
    long int *tmp_ptr;
    int act_flag = -1;       /*    rule->act_ofs */
    //int opt_flag = 0;
    void *rules, *rules_begin;
    int counter[65535];
    mular_descriptor *mular[65535];
    int size[1] = {R_NUM};
    int mem_size = 0;
    int mem_counter = 0;
    
    if (ac == 1) {
        input = FROM_STDIN;
        if ((fd = open("/dev/fd/0", O_RDONLY)) < 0) {
            snprintf(errMessage, 256, "open: /dev/fd/0: %s", strerror(errno));
            writeLogs(LOG_EMERG, errMessage);
            exit(-1);
        } 
    } else {
        while ((i = getopt(ac, av, "f")) > 0) {
                switch (i) {
                case 'f':
                        break;
                default:
                        show_usage();
                }   
        }
        ac -= optind;
        av += optind;
	if (ac == 0) show_usage();
    }
    
    if ((rule = (struct ip_fw *)malloc(256)) == NULL) {  /*    */
        snprintf(errMessage, 256, "malloc: rule: %s", strerror(errno));
        writeLogs(LOG_EMERG, errMessage);
        return -1;
    }
    /*     */
    if ((rules_begin = rules = malloc(BLOCK)) == NULL) {
	snprintf(errMessage, 256, "malloc: rules: %s", strerror(errno));
        writeLogs(LOG_EMERG,errMessage);
        return -1;
     }	
    
    for (j = 0; j < 65535; j++) counter[j] = 0;
    memset(mular, 0, 65535 * 4);
    
next_file:
    while (ac > 0) {
        if (input & FROM_STDIN) {
            if ((f = fdopen(fd, "r")) < 0) {
                snprintf(errMessage, 256, "fdopen: %s", strerror(errno));
                writeLogs(LOG_EMERG, errMessage);
                exit(-1);
            } 
        } else if (input & FROM_FILE)
                if ((f = fopen(*av, "r")) == NULL) { 
                        snprintf(errMessage, 256, "fopen: %s : %s", *av, strerror(errno));
                        writeLogs(LOG_ALERT, errMessage);
                        ac--;
                        av++;
                        goto next_file;
                }
        ac--;
        av++;
	
	state = 0;
        action = 0;
        lineno = 1;
        i = 0; c = 0;
	br_counter = 0;
	
	while (!feof(f)) {
                if (state == er){
			//printf("c=%c\n", c);
                        snprintf(errMessage, 256, "Syntax error: %s: Line number: %d", *(av-1), lineno);
                        writeLogs(LOG_ALERT, errMessage);
                        i = 0;
                        while (c != EOF && br_counter != 0) {
				c = getc(f);
				//printf("c=%c br_counter = %d\n", c, br_counter);
				if (class[c] == ob) br_counter++;
				if (class[c] == cb) br_counter--;
				if (c == 10 || c == 13) lineno++;
			}	
                        if (c == EOF) break;
			state = w0;
                }
                old_state = state;
                c = getc(f);
		if (c == EOF) break;
		if (class[c] == ob) br_counter++;
		if (class[c] == cb) br_counter--;
		if (c == 10 || c == 13) lineno++;
		/* comments */
		if (c == 35) {
			while ((c = getc(f)) != 10 && c != 13 && c != EOF);
			lineno++;
			continue;
		}
                action = automat[state][class[c]];
                state = action & 0xff;     /*   */
                if (action & A_cr){
			act_flag = 1;
			//opt_flag = 0;
			//rule->_pad = 1;
			bzero(rule, 256);
			cmd = (ipfw_insn *)rule->cmd;
			lineno_begin = lineno;/*     */
			continue; } 
		if (action & A_rn) {
                        if (i > 18) state = er;
                        else str[i++] = (char)c;
                        continue;}
		if (action & A_wd) {
			if (i > 18) state = er;
			else {
				str[i] = '\0';
				if (old_state == r_num) rule->rulenum = atoi(str);
				else if (old_state == s_num) rule->set = atoi(str);
				i = 0;
			}}
		if (action & A_wn) {
			cmd->len |= F_NOT;
			continue;}
		if (action & A_wo) {
			prev->len |= F_OR;
			continue;}
		if (action & A_cX) {
			if (i > 18) { state = er; continue;}
			else str[i] = '\0';
			//printf("rules_reader: str = %s\n", str);
			/*       */
			id_number = detect_ID(str, id_state_table);
			if (id_number < 1) state = er; /*      */ 
			else { /* */
			       /*    */
			        opcode = match_token(identifiers, str);
				cmd->opcode = opcode;
				if (act_flag) /*  rule->act_ofs */
					if ((opcode > O_CHECK_STATE && opcode < O_FORWARD_MAC) || opcode == O_LOG) {
                                                rule->act_ofs = cmd - rule->cmd; act_flag = 0;
					}
				/*if (!opt_flag) 
					if ((opcode > O_NOP && opcode < O_MACADDR2) || (opcode > O_FORWARD_MAC && opcode < O_LAST_OPCODE)) { 
						rule->_pad = 0;
						opt_flag = 1;
			}*/ 
				if (opcode > O_PROB && opcode < O_IPSEC) cmd->len = 1;/* for actions default */
				if (id_number == 10) { /*    */
					switch(opcode) {
					case O_CHECK_STATE:
					case O_ESTAB:
					case O_VERREVPATH:
					case O_PROBE_STATE:
					case O_KEEP_STATE:
					case O_LAYER2:
					case O_FRAG:
					case O_IN:
						fill_cmd(cmd, 0, 0);
					break;
					case O_DENY:
						cmd->arg1 = 0;
					break;
					case O_IP_SRC_ME:
					case O_IP_DST_ME:
						((ipfw_insn_ip *)cmd)->o.len &= ~F_LEN_MASK;     /* zero len */
        					((ipfw_insn_ip *)cmd)->o.len |= F_INSN_SIZE(ipfw_insn);
					break; }
				} else if (inner_automat(id_number, opcode) < 0) { /*     */
						state = er; continue;
				}			
				prev = cmd;
				cmd = next_cmd(cmd); 
			}
			i = 0;
		}
		if (action & A_RC) {
			dst = cmd;
			rule->cmd_len = (uint32_t *)cmd - (uint32_t *)(rule->cmd);
			if (check(rule)) {
                                snprintf(errMessage, 256, "rules_reader: bad rule structure");
                                writeLogs(LOG_ALERT, errMessage);
                        } else {
				if (mular[rule->rulenum - 1] == NULL)
                                        if (!(mular[rule->rulenum - 1] = mular_create(NULL, 1, sizeof(long int), size))) {
                                                snprintf(errMessage, 256, "mular[%d] not create: %s", rule->rulenum - 1, strerror(errno));
                                                writeLogs(LOG_EMERG, errMessage);
                                                return -1;
                                        }
                                if ( (tmp_ptr = (long int *)mular_insert(mular[rule->rulenum - 1], counter[rule->rulenum - 1])) ) {
                                        mem_size += RULESIZE(rule);
                                        if (mem_size >= BLOCK) {
                                                mem_size -= BLOCK;
                                                mem_counter++;
                                                rules -= (u_int32_t)rules_begin;
                                                if ((rules_begin = realloc(rules_begin, mem_counter * BLOCK)) == NULL) {
                                                        snprintf(errMessage, 256, "realloc: %s", strerror(errno));
                                                        writeLogs(LOG_EMERG, errMessage);
                                                        return -1;
                                                } else rules += (u_int32_t)rules_begin;
                                        }
                                        bcopy(rule, rules, RULESIZE(rule));
                                        *tmp_ptr = (long int) (rules - rules_begin);
                                        rules += RULESIZE(rule);
                                        counter[rule->rulenum - 1]++;
				} else {
                                        snprintf(errMessage, 256, "mular_insert: rulenum  %d: ruleptr is not inserted", rule->rulenum);
                                        writeLogs(LOG_EMERG, errMessage);
                                        return -1;
                                }
			} 
			//printf("check(rule) = %d  rule->rulenum = %d\n",  check(rule), rule->rulenum);
		}
	} /* end of while */
    } /* end of while */
    
    /* Just for test */
    /*for (i = 0; i < 65535; i++)
        for (j=0; j < counter[i]; j++) {
                tmp_ptr = mular_getix(mular[i], j);
                //rule = (struct ip_fw *)(rules_begin + (u_int32_t) *tmp_ptr);
                printf("rulenum=%d\n", ((struct ip_fw *)(rules_begin + (u_int32_t) *tmp_ptr))->rulenum);
        }        
    */
    free(rule);
    
    i = 0;
    for(j = 0; j < 65535; j++)
	 if (counter[j] > 0) {  
	//	snprintf(errMessage, 256, "rules_reader: before compare_rules");
	//	writeLogs(LOG_NOTICE, errMessage); 
    		while ((i = compare_rules(mular, counter, rules_begin)) == -1) {
            		snprintf(errMessage, 256, "rules_reader: restart compare_rules");
            		writeLogs(LOG_ALERT, errMessage);
		}
		break;
	 }     		  
    free(rules_begin);

    return i;
}/* end of rules_reader */

int 
main(int ac, char **av) {
    int i;
    char errMessage[256];
    /* snprintf(errMessage, 256, "main: start");
    writeLogs(LOG_NOTICE, errMessage); */

    while ((i = rules_reader(ac,av)) != 0) {
        snprintf(errMessage, 256, "main: restart from the beginning");
        writeLogs(LOG_ALERT, errMessage);
    }
    /* snprintf(errMessage, 256, "main: end");
    writeLogs(LOG_NOTICE, errMessage); */

    return 0;
}

