/*
 * Copyright (c) 2005 Anton Antonov <aga@pikenet.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: compare_pipes.c,v 1.6 2005/08/02 18:33:47 aga Exp $ 
 */

#include <syslog.h>
#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/sysctl.h>
#include <errno.h>
#include <stdlib.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/ip_fw2.h>
#include <net/route.h> /* def. of struct route */
#include <netinet/ip_dummynet.h>
#include <stdio.h>
#include <string.h>
#include <multilar.h>
#include <babolo/BLINflag.h>

int kq, event, nbytes = 0, HZ;
struct kevent ke;
struct timespec timeout; /* timeout */
int s = -1;  /* the socket */
char errMessage[256];
void *data = NULL;
struct dn_pipe tmp_pipe;

void 
writeLogs(int priority, const char *message);

/*        */
enum states_pipe {begin,number,fields,next_p,end};
/* begin  - ,     pipe  queue */
/* number -       */
/* fields -    */
/* next_p -    */
/* end -  */

/*  */
#define REPLACE  0x10
#define IS_PIPE  0x20
#define IS_QUEUE 0x40

enum states_pipe com_pipes[end][2] =
/*                  true            false    */
/* begin    */ {{ number,         end            },
/* number   */  { fields,         next_p         },
/* fields   */  { next_p,         REPLACE | next_p },
/* next_p   */  { begin,          end            }};

static int
do_cmd(int optname, void *optval, uintptr_t optlen) {
    int i;
    /* fill out the kevent struct */
    EV_SET(&ke, s, EVFILT_WRITE, EV_ADD, 0, 0, NULL);
    /* set the event */
    event = kevent(kq, &ke, 1, NULL, 0, NULL);
    if (event == -1) {
        snprintf(errMessage,256,"kevent: set event(do_cmd): %s", strerror(errno) );
        writeLogs(LOG_ALERT, errMessage);
        return -1;
    }

    memset(&ke, 0, sizeof(ke));
    /* receive an event */
    event = kevent(kq, NULL, 0, &ke, 1, &timeout);
    if (event < 0) {
        snprintf(errMessage,256,"kevent: receive event(do_cmd): %s", strerror(errno) );
        writeLogs(LOG_ALERT, errMessage);
        return -1;
    }
    if (ke.ident == s) {
        i = setsockopt(s, IPPROTO_IP, optname, optval, optlen);
        /* Delete event from the kqueue */
        /* fill out the kevent struct */
        EV_SET(&ke, s, EVFILT_WRITE, EV_DELETE, 0, 0, NULL);
        /* set the event */
        event = kevent(kq, &ke, 1, NULL, 0, NULL);
        if (event == -1) {
            snprintf(errMessage,256,"kevent: delete event(do_cmd): %s", strerror(errno) );
            writeLogs(LOG_ALERT, errMessage);
            return -1;
        }
        return i;
    } else {
        snprintf(errMessage, 256, "kevent: bad event(do_cmd)");
        writeLogs(LOG_ALERT, errMessage);
        return -1;
    }
}

int 
get_pipes(void) {
    long tick1,tick2;
    int len = sizeof(long);
    int counter = 0;
    int i;
    long time = 0;
    int nalloc = 1024;	/* start somewhere... */

    /* get pipes from kernel, resizing array as necessary */
    nbytes = nalloc;

    /* fill out the kevent struct */
    EV_SET(&ke, s, EVFILT_WRITE, EV_ADD, 0, 0, NULL);
    /* set the event */
    event = kevent(kq, &ke, 1, NULL, 0, NULL);
    if (event == -1) {
        snprintf(errMessage, 256, "kevent: set event(get_pipes): %s", strerror(errno) );
        writeLogs(LOG_ALERT, errMessage);
        return -1;
    }
    while (nbytes >= nalloc) {
        nalloc = nalloc * 2 + 200;
        nbytes = nalloc;

        if ((data = realloc(data, nbytes)) == NULL) {
            snprintf(errMessage, 256, "realloc: data(get_pipes): %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -2;
        }
        memset(&ke, 0, sizeof(ke));
        counter++;
        if (sysctlbyname("net.inet.ip.dummynet.curr_time", &tick1, &len, 0, 0) == -1) {
            snprintf(errMessage, 256, "sysctlbyname: net.inet.ip.dummynet.curr_time: %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -1;
        }
        i = getsockopt(s, IPPROTO_IP, IP_DUMMYNET_GET , data, (socklen_t *)&nbytes);
        if (sysctlbyname("net.inet.ip.dummynet.curr_time", &tick2, &len, 0, 0) == -1) {
            snprintf(errMessage, 256,"sysctlbyname: net.inet.ip.dummynet.curr_time: %s", strerror(errno) );
            writeLogs(LOG_EMERG, errMessage);
            return -1;
        }
        time += abs(tick2 - tick1) * HZ;
        if (i < 0) {
            snprintf(errMessage,256,"getsockopt(IP_DUMMYNET_GET): %s", strerror(errno) );
            writeLogs(LOG_ALERT, errMessage);
            return -1;
        }
    }
    /* (Average time)*100 */
    time = 100 * time / counter;
    if (time < 100 * HZ) time = 100 * HZ;
    if (time >= 1000000) {
        timeout.tv_sec = (time_t)(time / 1000000);
        timeout.tv_nsec = (time % 1000000) * 1000;
    } else timeout.tv_nsec = time * 1000;
    return 0;
}

int 
check_fs(struct dn_flow_set *object, struct dn_flow_set *kern_object) {
    if (object->qsize == kern_object->qsize &&
       object->plr == kern_object->plr && object->rq_size == kern_object->rq_size &&
       object->flow_mask.dst_ip == kern_object->flow_mask.dst_ip &&
       object->flow_mask.src_ip == kern_object->flow_mask.src_ip &&
       object->flow_mask.dst_port == kern_object->flow_mask.dst_port &&
       object->flow_mask.src_port == kern_object->flow_mask.src_port &&
       object->flow_mask.proto == kern_object->flow_mask.proto &&
       object->flags_fs == object->flags_fs) {
                    if (((object->flags_fs&DN_IS_RED) == DN_IS_RED) || ((object->flags_fs & DN_IS_GENTLE_RED) == DN_IS_GENTLE_RED)) {
                        if (object->w_q == kern_object->w_q && object->min_th == kern_object->min_th &&
                           object->max_th == kern_object->max_th && object->max_p == kern_object->max_p) return next_p; /*   */
                        else return REPLACE | next_p;
                    } else return next_p;                  
     } else return REPLACE | next_p;
}

int 
addObjects(uint32_t *objectArray, int addCounter, int objectID) { 
        int k;
        struct dn_pipe *tmp_p;
        struct dn_flow_set *tmp_q;
        memset(&tmp_pipe, 0, sizeof tmp_pipe);
        for (k = 0; k < addCounter; k++) {
            if (objectID & IS_PIPE) { tmp_p = (struct dn_pipe *)objectArray[k]; tmp_pipe = *tmp_p; }                      
            else if (objectID & IS_QUEUE) { tmp_q = (struct dn_flow_set *)objectArray[k]; tmp_pipe.fs = *tmp_q; }
            if (do_cmd(IP_DUMMYNET_CONFIGURE, &tmp_pipe, sizeof tmp_pipe) != 0) {
                snprintf(errMessage, 256, "setsockopt: add %s %d: %s", objectID&IS_PIPE ? "pipe" : "queue",
                        objectID&IS_PIPE ? tmp_pipe.pipe_nr : tmp_pipe.fs.fs_nr, strerror(errno) );
                writeLogs(LOG_ALERT, errMessage);
                return -1;
            }
        }
        return 0;
}

int
delObjects(int *objectArray, int delCounter, int objectID){
        int k;
        memset(&tmp_pipe, 0, sizeof tmp_pipe);
        for (k = 0; k < delCounter; k++) {
            if (objectID & IS_PIPE) tmp_pipe.pipe_nr = objectArray[k];
            else if (objectID & IS_QUEUE) tmp_pipe.fs.fs_nr = objectArray[k];
            if (do_cmd(IP_DUMMYNET_DEL, &tmp_pipe, sizeof tmp_pipe)) {
                snprintf(errMessage, 256, "setsockopt: delete %s %d: %s",objectID&IS_PIPE ? "pipe" : "queue",
                        objectID&IS_PIPE ? tmp_pipe.pipe_nr : tmp_pipe.fs.fs_nr, strerror(errno) );
                writeLogs(LOG_ALERT, errMessage);
                return -1;
            }
        }
        return 0;
}

int 
compare_pipes(mular_descriptor *mularP, int *maskP,mular_descriptor *mularQ, int *maskQ) {
    
    struct dn_pipe *kern_pipe, *pipe;
    struct dn_flow_set *kern_queue = NULL, *queue;
    void *next = NULL;
    struct dn_flow_queue *q;

    int del_pipes[65535], del_queues[65535];    /*numbers of pipes/queues to delete */
    int del_Pcounter = 0, del_Qcounter = 0; /*  /   */
    
    struct dn_pipe *add_pipes[65535]; /* pipes are to add */
    struct dn_flow_set *add_queues[65535]; /* queues are to add */
    int add_Pcounter = 0, add_Qcounter = 0; /*  /   */
    
    int is_exist; /*         */
    int l;        /*     */
    int i = 0; 
    u_int8_t state; /* c */
    
    int len = sizeof(int);
    struct clockinfo cl_info;

    /* Socket */
    if (s == -1) s = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
    if (s < 0) {
        snprintf(errMessage,256,"socket: %s", strerror(errno) );
        writeLogs(LOG_EMERG, errMessage);
        return -1;
    }

    /* kqueue */
    kq = kqueue();
    if (kq == -1) {
        snprintf(errMessage, 256, "kqueue: %s", strerror(errno) );
        writeLogs(LOG_EMERG, errMessage);
        return -1;
    }

    len = sizeof(struct clockinfo);
    if (sysctlbyname("kern.clockrate", &cl_info, &len, 0, 0) < 0) {
        snprintf(errMessage, 256, "sysctlbyname: kern.clockrate: %s", strerror(errno) );
        writeLogs(LOG_EMERG, errMessage);
        return -1;
    } else HZ = cl_info.tick;

    /*      */
    if (get_pipes() < 0) return -1;
    kern_pipe = data; 
    while (i < 65535) { /*         */  
        if (maskP[i] > 0) {      /*   ( ) */
        	if ( !(pipe = (struct dn_pipe *)mular_getix(mularP, maskP[i] - 1)) ) {
			snprintf(errMessage, 256, "mular_getix: can't get index %d", maskP[i] - 1);
                        writeLogs(LOG_EMERG, errMessage);
			return -1;
		}
	} else if (i < 65534) { i++; continue;}
	       else { /*        */
                 while (kern_pipe->next == (struct dn_pipe *)DN_IS_PIPE && nbytes > 0) {
                        del_pipes[del_Pcounter++] = kern_pipe->pipe_nr;
                        l = sizeof(*kern_pipe) + kern_pipe->fs.rq_elements * sizeof(*q);
                        next = (char *)kern_pipe + l;
                        q = (struct dn_flow_queue *)(kern_pipe + 1);
                        kern_pipe = (struct dn_pipe *)next;
                        nbytes -= l;
                 } 
		 break;
              }
        is_exist = 0;  /* ,             */
	state = 0;
        while ((state&0xf) < end) { /*         */
            switch (state & 0xf) {
            case begin:
                if (nbytes > 0) if (kern_pipe->next == (struct dn_pipe *)DN_IS_PIPE) {
                                state=com_pipes[state][0];
                                break;
                              }
		state = end;  
                break;
            case number:
                if (pipe->pipe_nr == kern_pipe->pipe_nr) {
                    is_exist = 1;
                    state = com_pipes[state][0];
                } else if (pipe->pipe_nr > kern_pipe->pipe_nr) {
                    del_pipes[del_Pcounter++] = kern_pipe->pipe_nr;
                    state = com_pipes[state][1];
                } else state = end;
                break;
            case fields:
                if(pipe->bandwidth != kern_pipe->bandwidth || pipe->delay != kern_pipe->delay) {
                    state = com_pipes[state][1];
                    break;
                }
                state = check_fs(&pipe->fs, &kern_pipe->fs);
                break;
            case next_p:
                if (state & REPLACE) {
                    /* */
                    if (do_cmd(IP_DUMMYNET_CONFIGURE, pipe, sizeof (struct dn_pipe)) != 0) {
                        snprintf(errMessage, 256, "setsockopt: add pipe %d: %s", pipe->pipe_nr, strerror(errno) );
                        writeLogs(LOG_ALERT, errMessage);
                        add_pipes[add_Pcounter++] = pipe;
                        del_pipes[del_Pcounter++] = pipe->pipe_nr;
                    }
                }
                l = sizeof(*kern_pipe) + kern_pipe->fs.rq_elements * sizeof(*q);
                next = (char *)kern_pipe + l;
                q = (struct dn_flow_queue *)(kern_pipe + 1);
                kern_pipe = (struct dn_pipe *)next;

                nbytes -= l;
                if (nbytes > 0 && !(state & REPLACE)) state = com_pipes[state & 0xf][0];
                else state = com_pipes[state][1];
                state &= 0xf;
                break;
            } /* end of switch */
        } /* end of while */
        if (!is_exist)
            if (do_cmd(IP_DUMMYNET_CONFIGURE, pipe, sizeof(struct dn_pipe)) != 0) {
                snprintf(errMessage, 256, "setsockopt: add pipe %d: %s",pipe->pipe_nr, strerror(errno) );
                writeLogs(LOG_ALERT, errMessage);
                add_pipes[add_Pcounter++] = pipe;
            }
	i++;
    }
    
    if ((i = delObjects(del_pipes, del_Pcounter, IS_PIPE)) != 0) return i; /*    */
    if((i = addObjects((uint32_t *)add_pipes, add_Pcounter, IS_PIPE)) != 0) return i; /*  ,      */
    
    /*    */
    if (next != NULL) kern_queue = (struct dn_flow_set *)next;
    else kern_queue = (struct dn_flow_set *)data;
    i = 0;
    while (i < 65535) {          /*         */   
        if (maskQ[i] > 0) {      /*   ( ) */
        	if ( !(queue = (struct dn_flow_set *)mular_getix(mularQ, maskQ[i] - 1)) ) {
			snprintf(errMessage, 256, "mular_getix: can't get index %d", maskQ[i] - 1);
                        writeLogs(LOG_EMERG, errMessage);
			return -1;
		}
	} else if (i < 65534) { i++; continue;}
	     else {  /*       */
		while (kern_queue->next == (struct dn_flow_set *)DN_IS_QUEUE && nbytes > 0) { 
                        del_queues[del_Qcounter++] = kern_queue->fs_nr;
                        l = sizeof(*kern_queue) + kern_queue->rq_elements * sizeof(*q);
                        next = (char *)kern_queue + l;
                        q = (struct dn_flow_queue *)(kern_queue + 1);
                        kern_queue = (struct dn_flow_set *)next;
                        nbytes -= l;
                }
                break;
             }
	is_exist = 0;  /* ,           */
	state = 0;
        while ((state & 0xf) < end) { /*         */
            switch(state & 0xf) {
            case begin:
                if (nbytes > 0) if (kern_queue->next == (struct dn_flow_set *)DN_IS_QUEUE) {
                        state = com_pipes[state][0];
                        break;
                      }
                state = end; 
                break;
            case number:
                if (queue->fs_nr == kern_queue->fs_nr) {
                    is_exist = 1;
                    state = com_pipes[state][0];
                    break;
                } else if (queue->fs_nr > kern_queue->fs_nr) {
                    del_queues[del_Qcounter++] = kern_queue->fs_nr;
                    state = com_pipes[state][1];
                    break;
                } else state = end;
                break;
            case fields:
                if(queue->parent_nr != kern_queue->parent_nr || queue->weight != kern_queue->weight) {
                    state = com_pipes[state][1];
                    break;
                }
                state = check_fs(queue, kern_queue);
                break;
            case next_p:
                if (state & REPLACE) {
                    memset(&tmp_pipe, 0, sizeof tmp_pipe);
                    tmp_pipe.fs = *queue;
                    if (do_cmd(IP_DUMMYNET_CONFIGURE, &tmp_pipe, sizeof tmp_pipe) != 0) {
                        snprintf(errMessage, 256, "setsockopt: add queue %d: %s", tmp_pipe.fs.fs_nr, strerror(errno) );
                        writeLogs(LOG_ALERT, errMessage);
                        add_queues[add_Qcounter++] = queue;
                        del_queues[del_Qcounter++] = tmp_pipe.fs.fs_nr;
                    }
                }
                l = sizeof(*kern_queue) + kern_queue->rq_elements * sizeof(*q);
                next = (char *)kern_queue + l;
                q = (struct dn_flow_queue *)(kern_queue+1);
                kern_queue = (struct dn_flow_set *)next;

                nbytes -= l;
                if (nbytes > 0 && !(state & REPLACE)) state = com_pipes[state & 0xf][0];
                else state = com_pipes[state][1];
                state &= 0xf;
                break;
            } /* end of switch */
        } /* end of while */
        if (!is_exist) {
            memset(&tmp_pipe, 0, sizeof tmp_pipe);
            tmp_pipe.fs = *queue;
            if(do_cmd(IP_DUMMYNET_CONFIGURE, &tmp_pipe, sizeof tmp_pipe) != 0) {
                snprintf(errMessage, 256, "setsockopt: add queue %d: %s", tmp_pipe.fs.fs_nr, strerror(errno) );
                writeLogs(LOG_ALERT, errMessage);
                add_queues[add_Qcounter++] = queue;
            }
        }
	i++;
    }

    if (del_Qcounter) 
	 if((i = delObjects(del_queues, del_Qcounter, IS_QUEUE)) != 0) return i; /*    */
    if (add_Qcounter)
	 if((i = addObjects((uint32_t *)add_queues, add_Qcounter, IS_QUEUE)) != 0) return i; /*  ,      */
    
    mular_destroy(mularP);
    mular_destroy(mularQ);
    free(data);
    return 0;
}




