/*-
 * Copyright (C)2002..2018 @BABOLO http://www.babolo.ru/
 * Copyright (C) 2005 Michael Deryugin <dem@pikenet.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] = "\
@(#)Copyright (C) @BABOLO, Michael Deryugin\n\
@(#)All rights reserved.\n";
static const char rcsid[] = "$Id: pglib_ip.c,v 1.10 2018/08/10 17:18:51 babolo Exp $";
#endif /* not lint */

#include <sys/param.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <postgres.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <fmgr.h>
#include <funcapi.h>
#include <utils/inet.h>
#include <access/tuptoaster.h>

#define ALLOC(ptr) palloc((ptr))
#define FREE(ptr)  pfree((ptr))

enum qchars {Q, T, F, SP};

typedef struct {
    int32_t length;
    char data[1];
} ipaggr;

struct qstring {
    int32_t position;
    int32_t qlength;
    int32_t alength;
    ipaggr *qstring_data;
#define qdata qstring_data->data
};

struct qstate {
    u_int32_t addr;
    int32_t mask;
    enum qchars qc;
    int32_t state;
    struct qstring qstr;
#define STACK_SIZE 32
    int32_t sp;
    int32_t stack[STACK_SIZE];
};

/* Error */
#define ESYM (-1)

/* Actions of qwalk */
#define  SEARCH_T    1
#define  SEARCH_F    2
#define  SEARCH_ADDR 3
#define  SET_BLANK   4
#define  AGGREGATE   5

#ifdef PG_MODULE_MAGIC
    PG_MODULE_MAGIC;
#endif

#if (PGSQL_VER > 8)
#   define NOTYPE 1
#   define INET_MEMSIZE(a)      (VARHDRSZ + offsetof(inet_struct, ipaddr) + ip_addrsize(a))
#elif ((PGSQL_VER == 8) && (PGSQL_MOD > 1))
#   define NOTYPE 1
#   define ip_family(inetptr)   (((inet_struct *)VARDATA_ANY(inetptr))->family)
#   define ip_bits(inetptr)     (((inet_struct *)VARDATA_ANY(inetptr))->bits)
#   define ip_addr(inetptr)     (((inet_struct *)VARDATA_ANY(inetptr))->ipaddr)
#   define ip_maxbits(inetptr)  (ip_family(inetptr) == PGSQL_AF_INET ? 32 : 128)
#   define INET_MEMSIZE(a)      (VARHDRSZ + offsetof(inet_struct, ipaddr) + ip_addrsize(a))
#else
#   define ip_bits(inetptr)     (((inet_struct *)VARDATA(inetptr))->bits)
#   define ip_family(inetptr)   (((inet_struct *)VARDATA(inetptr))->family)
#   define ip_type(inetptr)     (((inet_struct *)VARDATA(inetptr))->type)
#   define ip_maxbits(inetptr)  (ip_family(inetptr) == PGSQL_AF_INET ? 32 : 128)
#   define INET_MEMSIZE(a)      (VARHDRSZ + ((char*)ip_addr(a) - (char*)VARDATA(a)) + ip_addrsize(a))

#   if PGSQL_VER > 7  /*  PG8 ip_addr  ipaddr */
#       define ip_addr(inetptr) ((inet_struct *)VARDATA(inetptr))->ipaddr
#   else
#       define ip_addr(inetptr) ((inet_struct *)VARDATA(inetptr))->ip_addr
#   endif
#endif

#if (PGSQL_VER > 8) || ((PGSQL_VER == 8) && (PGSQL_MOD > 2))
#   define varattrib struct varlena
#endif

#define fctx                ((results*)funcctx->user_fctx)

typedef struct {
    u_int32_t current;
    u_int32_t brodcast;
    int32_t   masklen;
} results;

#if (PGSQL_VER < 9)
static int
ip_addrsize(inet *inetptr) {
    switch (ip_family(inetptr)) {
    case PGSQL_AF_INET:  return(4);
    case PGSQL_AF_INET6: return(16);
    default:             return(-1);
}   }
#endif

u_int32_t
ip_v4addr(inet *inetptr) {
    int i;
    u_int32_t a;
    u_char *p;

    for (p = ip_addr(inetptr), a = 0, i = 0; i < ip_addrsize(inetptr); i++) {
        a = (a << 8) | p[i];
    }
    return(a);
}

void
toip_v4addr(inet *inetptr, u_int32_t a) {
    int i;
    u_char *p;

    for (p = ip_addr(inetptr), i = ip_addrsize(inetptr) - 1; i >= 0; i--) {
        p[i] = a;
        a >>= 8;
}   }

PG_FUNCTION_INFO_V1(ip_set);
Datum
ip_set(PG_FUNCTION_ARGS) {
    u_int32_t mask;
    inet *src;
    FuncCallContext  *funcctx;
    MemoryContext oldcontext;

    if  (SRF_IS_FIRSTCALL()) {
        funcctx = SRF_FIRSTCALL_INIT();
        oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
        if  (PG_ARGISNULL(0) || PG_ARGISNULL(1)) { SRF_RETURN_DONE(funcctx); }
        src = PG_GETARG_INET_P(0);
        if  (ip_family(src) == PGSQL_AF_INET6)
            ereport(ERROR, ( errcode(ERRCODE_FEATURE_NOT_SUPPORTED)
                           , errmsg("ip_set: address family AF_INET6 not supported yet")
                   )       );
        if  (ip_family(src) != PGSQL_AF_INET)
                                    elog(ERROR, "ip_set: unknown address family (%d)", ip_family(src));
        funcctx->user_fctx = (results*)palloc(sizeof(results));
        if  (fctx == NULL)
                     ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ip_set: out of memory")));
        MemSet(fctx, 0, sizeof(results));
        fctx->masklen = PG_GETARG_INT32(1);
        if  (fctx->masklen > ip_maxbits(src) || fctx->masklen < 0)
            ereport(ERROR, ( errcode(ERRCODE_INVALID_PARAMETER_VALUE)
                           , errmsg("ip_set: Out of range")
                           , errdetail( "Parameter $2 = %d out of range (0..%d)"
                                      , fctx->masklen, ip_maxbits(src)
                   )       )          );
        if  (ip_bits(src) > fctx->masklen) { SRF_RETURN_DONE(funcctx); }
        mask = 0xFFFFFFFF << (ip_maxbits(src) - ip_bits(src));
        funcctx->max_calls = 1 << (fctx->masklen - ip_bits(src));     /* BEWARE!!! Overflow possible */
        fctx->current = mask & ip_v4addr(src);
        fctx->brodcast = ~mask | ip_v4addr(src);
        MemoryContextSwitchTo(oldcontext);
    }
    funcctx = SRF_PERCALL_SETUP();
    if  (funcctx->call_cntr < funcctx->max_calls) {
        src = (inet*)palloc(sizeof(inet));
        if  (src == NULL)
                     ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ip_set: out of memory")));
        MemSet(src, 0, sizeof(inet));
        ip_family(src) = PGSQL_AF_INET;
        ip_bits(src) = fctx->masklen;
#ifndef NOTYPE
        ip_type(src) = 0;
#endif
        toip_v4addr(src, fctx->current);
#ifdef VARATT_SIZEP
        VARATT_SIZEP(src) = INET_MEMSIZE(src);
#else
        SET_VARSIZE(src, INET_MEMSIZE(src));
#endif
        fctx->current += (1 << (ip_maxbits(src) - fctx->masklen));
        SRF_RETURN_NEXT(funcctx, InetPGetDatum(src));
    } else {
        SRF_RETURN_DONE(funcctx);
}   }

PG_FUNCTION_INFO_V1(ip_supernet);
Datum
ip_supernet(PG_FUNCTION_ARGS) {
    inet *src;
    FuncCallContext  *funcctx;
    MemoryContext oldcontext;

    if  (SRF_IS_FIRSTCALL()) {
        funcctx = SRF_FIRSTCALL_INIT();
        oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
        if  (PG_ARGISNULL(0)) { SRF_RETURN_DONE(funcctx); }
        src = PG_GETARG_INET_P(0);
        if  (ip_family(src) == PGSQL_AF_INET6)
            ereport(ERROR, ( errcode(ERRCODE_FEATURE_NOT_SUPPORTED)
                           , errmsg("ip_supernet: address family AF_INET6 not supported yet")
                   )       );
        if  (ip_family(src) != PGSQL_AF_INET)
                               elog(ERROR, "ip_supernet: unknown address family (%d)", ip_family(src));
        funcctx->user_fctx = (results*)palloc(sizeof(results));
        if  (fctx == NULL)
                ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ip_supernet: out of memory")));
        MemSet(fctx, 0, sizeof(results));
        funcctx->max_calls = fctx->masklen = ip_bits(src);
        fctx->current = ip_v4addr(src);
        MemoryContextSwitchTo(oldcontext);
    }
    funcctx = SRF_PERCALL_SETUP();
    if  (funcctx->call_cntr < funcctx->max_calls) {
        src = (inet*)palloc(sizeof(inet));
        if  (src == NULL)
                ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ip_supernet: out of memory")));
        MemSet(src, 0, sizeof(inet));
        ip_family(src) = PGSQL_AF_INET;
        ip_bits(src) = --(fctx->masklen);
#ifndef NOTYPE
        ip_type(src) = 0;
#endif
        toip_v4addr(src, fctx->current);
#ifdef VARATT_SIZEP
        VARATT_SIZEP(src) = INET_MEMSIZE(src);
#else
        SET_VARSIZE(src, INET_MEMSIZE(src));
#endif
        SRF_RETURN_NEXT(funcctx, InetPGetDatum(src));
    } else {
        SRF_RETURN_DONE(funcctx);
}   }

PG_FUNCTION_INFO_V1(int4_inet);
Datum
int4_inet(PG_FUNCTION_ARGS) {
    inet *dst;

    if  (PG_ARGISNULL(0)) PG_RETURN_NULL();
    dst = (inet*)palloc(sizeof(inet));
    if  (dst == NULL)
                  ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("int4_inet: out of memory")));
    MemSet(dst, 0, sizeof(inet));
    ip_family(dst) = PGSQL_AF_INET;
    ip_bits(dst) = 32;
#ifndef NOTYPE
    ip_type(dst) = 0;
#endif
    toip_v4addr(dst, (u_long)PG_GETARG_INT32(0));
#ifdef VARATT_SIZEP
    VARATT_SIZEP(dst) = INET_MEMSIZE(dst);
#else
    SET_VARSIZE(dst, INET_MEMSIZE(dst));
#endif
    PG_RETURN_INET_P(dst);
};

PG_FUNCTION_INFO_V1(int8_inet);
Datum
int8_inet(PG_FUNCTION_ARGS) {
    inet *dst;

    if  (PG_ARGISNULL(0)) PG_RETURN_NULL();
    dst = (inet*)palloc(sizeof(inet));
    if  (dst == NULL)
                  ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("int8_inet: out of memory")));
    MemSet(dst, 0, sizeof(inet));
    ip_family(dst) = PGSQL_AF_INET;
    ip_bits(dst) = 32;
#ifndef NOTYPE
    ip_type(dst) = 0;
#endif
    toip_v4addr(dst, (u_long)PG_GETARG_INT64(0));
#ifdef VARATT_SIZEP
    VARATT_SIZEP(dst) = INET_MEMSIZE(dst);
#else
    SET_VARSIZE(dst, INET_MEMSIZE(dst));
#endif
    PG_RETURN_INET_P(dst);
};

PG_FUNCTION_INFO_V1(inet_int4);
Datum
inet_int4(PG_FUNCTION_ARGS) {
    inet *src;

    if  (PG_ARGISNULL(0)) PG_RETURN_NULL();
    src = PG_GETARG_INET_P(0);
        switch (ip_family(src)) {
    case PGSQL_AF_INET:  PG_RETURN_INT32(ip_v4addr(src));
    case PGSQL_AF_INET6:
        ereport(ERROR, ( errcode(ERRCODE_FEATURE_NOT_SUPPORTED)
                       , errmsg("address AF_INET6 too big for int4")
               )       );
    default:
        elog(ERROR, "unknown address family (%d)", ip_family(src));
    }
    PG_RETURN_NULL();
}

PG_FUNCTION_INFO_V1(inet_int8);
Datum
inet_int8(PG_FUNCTION_ARGS) {
    inet *src;

    if  (PG_ARGISNULL(0)) PG_RETURN_NULL();
    src = PG_GETARG_INET_P(0);
    switch (ip_family(src)) {
    case PGSQL_AF_INET:  PG_RETURN_INT64(ip_v4addr(src));
    case PGSQL_AF_INET6:
        ereport(ERROR, ( errcode(ERRCODE_FEATURE_NOT_SUPPORTED)
                       , errmsg("address AF_INET6 too big for int8")
               )       );
    default:
        elog(ERROR, "unknown address family (%d)", ip_family(src));
    }
    PG_RETURN_NULL();
}

#define SETMASK(x, y)   ((u_int32_t)((y) == 0 ? 0 : (x) & 0xffffffff & (0xffffffff << (32 - (y)))))
#define GETBIT(x, y)    ((u_int32_t)(((y) >> (32 - (x))) & 0x1))
#define SETBIT_0(x, y)  ((u_int32_t)((x) & 0xffffffff & ~(1 << (32 - (y)))))
#define SETBIT_1(x, y)  ((u_int32_t)((x) | (1 << (32 - (y)))))

int
qget(struct qstring *qstr) {
    switch (qstr->qdata[qstr->position]) {
        case 'q': return(Q);
        case 't': return(T);
        case 'f': return(F);
        case ' ': return(SP);
        default:  return(ESYM);
}   }

int
qgetnext(struct qstring *qstr) {
    int qchr;

    while (qstr->position + 1 < qstr->qlength) {
        qstr->position++;
        qchr = qget(qstr);
        if  (qchr == SP) {
            continue;
        } else {
            return(qchr);
    }   }
    return(-1);
}

int
qset(struct qstring *qstr, enum qchars qchr) {
    char *ch;

    if  (qstr->position > -1 && qstr->position < qstr->qlength) {
        ch = &qstr->qdata[qstr->position];
        if  (  *ch != 'q'
            && *ch != 'f'
            && *ch != 't'
            && (qchr == Q || qchr == T || qchr == F)
            ) qstr->alength++;
        switch (qchr) {
        case Q: *ch = 'q'; break;
        case T: *ch = 't'; break;
        case F: *ch = 'f'; break;
        case SP: if  (*ch != ' ') qstr->alength--;
                 *ch = ' ';
                 break;
        }
        return(1);
    } else {
        return(-1);
}   }

int 
qsetprev(struct qstring *qstr, enum qchars qchr) {
    while (qstr->position > 0 && qget(qstr) == SP) qstr->position--;
    if  (qget(qstr) != SP) return(qset(qstr, qchr));
      else return(-1);
}

int
qputnext(struct qstring *qstr, enum qchars qchr) {
    if  (qstr->position + 1 < qstr->qlength) {
        qstr->position++;
        return(qset(qstr, qchr));
    } else {
        return(-1);
}   }

int 
qput(struct qstring *qstr, enum qchars qchr) {
    qstr->position++;
    if  (qstr->position < qstr->qlength) {
        switch (qchr) {
        case Q: qstr->qdata[qstr->position] = 'q'; break;
        case T: qstr->qdata[qstr->position] = 't'; break;
        case F: qstr->qdata[qstr->position] = 'f'; break;
        case SP: break;
        }
    } else {
        return(-1);
    }
    return(1);
}

int
qreplace(struct qstring *qstr, ipaggr *aa) {
    int nlength;
    ipaggr *new;
    void *p;

    nlength = qstr->qlength + VARSIZE(aa) - VARHDRSZ - 1;
    new = (ipaggr *)ALLOC(nlength + VARHDRSZ);
    if  (new == NULL) {
        ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ipaggr: out of memory")));
        return(-1);
    }
#ifdef VARATT_SIZEP
    VARATT_SIZEP(new) = nlength + VARHDRSZ;
#else
    SET_VARSIZE(new, nlength + VARHDRSZ);
#endif
    p = new->data;
    memcpy(p, qstr->qdata, (int32) qstr->position); 
    p += qstr->position;
    memcpy(p, aa->data, VARSIZE(aa) - VARHDRSZ); 
    p += VARSIZE(aa) - VARHDRSZ;
    memcpy(p, qstr->qdata + qstr->position + 1, qstr->qlength - qstr->position - 1);
    qstr->qstring_data = new;
    qstr->qlength = VARSIZE(new) - VARHDRSZ;
    return(1);
}

int
qdelblank(struct qstring *qstr) {
    ipaggr *new;
    int32 i, j;
	
    new = (ipaggr *)ALLOC(qstr->alength + VARHDRSZ);
    if  (new == NULL) {
        ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ipaggr: out of memory")));
        return(-1);
    }
#ifdef VARATT_SIZEP
    VARATT_SIZEP(new) = qstr->alength + VARHDRSZ;
#else
    SET_VARSIZE(new, qstr->alength + VARHDRSZ);
#endif
    for (j = 0, i = 0; j < qstr->alength; j++, i++) {
        while (  qstr->qdata[i] != 'q'
              && qstr->qdata[i] != 't'
              && qstr->qdata[i] != 'f'
              ) i++;	
        new->data[j] = qstr->qdata[i];
    }
    qstr->qstring_data = new;
    qstr->qlength = VARSIZE(new) - VARHDRSZ;
    return(1);
}

#define C_DB	0x80000000
#define C_S0	0x40000000
#define C_S1	0x20000000
#define C_MASK	0x10000000
#define C_PUSH	0x08000000
#define C_POP	0x04000000
#define C_STATE	0x000000FF

enum states {lq, lt, lf, rq, rt, rf, fin};

/* States of Q-tree:
 * lq - node Q is on left subtree 
 * lt - leaf T is on left subtree
 * lf - leaf F is on left subtree
 * rq - node Q is on right subtree
 * rt - leaf T is on right subtree
 * rf - leaf F is on right subtree
 * 
 *   parent node -> Q - X <- left {node,leaf}
 *                    \
 *                      Y <- right {node,leaf}
 */

static const u_int32_t automat[fin+1][5] =
{/*          Q                  T                F                 state */
 {C_PUSH|C_MASK|C_S0|lq , C_MASK|C_S0|lt , C_MASK|C_S0|lf , fin} /*  lq  */
,{              C_S1|rq ,   C_DB|C_S1|fin,        C_S1|rf , fin} /*  lt  */
,{              C_S1|rq ,        C_S1|rt ,   C_DB|C_S1|fin, fin} /*  lf  */
,{C_PUSH|C_MASK|C_S0|lq , C_MASK|C_S0|lt , C_MASK|C_S0|lf , fin} /*  rq  */
,{        C_POP|C_S1|rq ,  C_POP|C_S1|rt ,  C_POP|C_S1|rf , fin} /*  rt  */
,{        C_POP|C_S1|rq ,  C_POP|C_S1|rt ,  C_POP|C_S1|rf , fin} /*  rf  */
,{                   fin,             fin,             fin, fin} /*  fin */
};

void
qstate_init(ipaggr *aa, struct qstate *it) {
    struct qstring *qstr = &it->qstr;
	
    qstr->position = -1;
    it->addr = 0x00000000;
    it->mask = 0xFFFFFFFF;
    it->state = lq;
    it->sp = 0;
    if  (aa != NULL) {
        qstr->qstring_data = aa;
        qstr->qlength = VARSIZE(aa) - VARHDRSZ;
        qstr->alength = qstr->qlength;
}   }

int 
qwalk(struct qstate *it, int action, int32_t s_mask, u_int32_t s_addr) {
    struct qstring *qstr;
    u_int32_t control = 0;
    int i;

retry:
    if  (action == AGGREGATE) {
        qstate_init(NULL, it);
    }
    qstr = &(it->qstr);
    while (it->state < fin) {
        if  ((i = qgetnext(qstr)) < 0) {
            ereport( ERROR
                   , ( errcode(ERRCODE_INVALID_TEXT_REPRESENTATION)
                     , errmsg("qwalk: unexpected end of qstring")
                   ) );
            return(-1);
        }; 
        it->qc = i; 
        control = automat[it->state][it->qc];

        if  (control & C_MASK) {
            if  (it->mask < 32) {
                it->mask++;
            } else {
                ereport( ERROR
                       , ( errcode(ERRCODE_INVALID_TEXT_REPRESENTATION)
                         , errmsg("qwalk: depth of qstring must be <= 32")
                       ) );
                return(-1);
        }   }
        if  ((control & C_PUSH) && it->mask > 0 && it->sp < STACK_SIZE) it->stack[it->sp++] = it->mask;
        if  ((control & C_POP) && it->sp >= 0) {
            it->mask = it->stack[--it->sp];
            it->addr = SETMASK(it->addr, it->mask);
        }
        if  ((control & C_S0) && it->mask > 0) it->addr = SETBIT_0(it->addr, it->mask);
        if  ((control & C_S1) && it->mask > 0) it->addr = SETBIT_1(it->addr, it->mask);
        if  ((control & C_DB) && (action != AGGREGATE)) {
            ereport( ERROR
                   , ( errcode(ERRCODE_INVALID_TEXT_REPRESENTATION)
                     , errmsg("qwalk: not aggregated qstring")
                   ) );
            return(-1);
        }
        it->state = control & C_STATE;
        if  (it->qc != Q) {
            if  (it->addr == SETMASK(0xffffffff, it->mask)) it->state = fin;
            if  (it->mask == 0) it->state = fin;
        }
        switch (action) {
        case SEARCH_T:
            if  (it->qc == T) return(1);
            break;
        case SEARCH_F:
            if  (it->qc == F) return(1);
            break;
        case SEARCH_ADDR:
            if  (it->qc != Q && it->mask == 0) return(1);
            if  (s_addr == it->addr && s_mask == it->mask) return(1);
            if  (it->qc != Q && it->addr == SETMASK(s_addr, it->mask)) return(1);
            break;
        case AGGREGATE:
            if  (control & C_DB) {
                enum qchars cur;
                cur = it->qc;
                qset(&it->qstr, SP);
                qsetprev(&it->qstr, SP);
                qsetprev(&it->qstr, cur);
                goto retry;
            }
            break;
        case SET_BLANK:
            if  (it->mask > s_mask) qset(&it->qstr, SP);
              else  return(1);
break;
    }   }
    return(0);
}

ipaggr *
qgenerate(int level, int mask, u_int32_t addr, enum qchars type) {
    struct qstring qstr;
    ipaggr *aa;
    int32 length;
    int i = level;
    int pos = 0;
    enum qchars invtype = T;

    if  (type == T) invtype = F;
    if  (level > mask) {
        ereport( ERROR
               , ( errcode(ERRCODE_INVALID_TEXT_REPRESENTATION)
                 , errmsg("qgenerate: level must be <= mask")
               ) );
        return(NULL);
    }
    if  (mask > 32) {
        ereport( ERROR
               , ( errcode(ERRCODE_INVALID_TEXT_REPRESENTATION)
                 , errmsg("qgenerate: mask must be <= 32")
               ) );
        return(NULL);
    }
    length = 1 + (mask - level) * 2;
    aa = (ipaggr *)ALLOC(length + VARHDRSZ);
#ifdef VARATT_SIZEP
    VARATT_SIZEP(aa) = length + VARHDRSZ;
#else
    SET_VARSIZE(aa, length + VARHDRSZ);
#endif
    if  (aa == NULL) {
        ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ipaggr: out of memory")));
        return(NULL);
    }

    qstr.qstring_data = aa;
    qstr.qlength = length;
    qstr.alength = length;
    qstr.position = -1;
	
    if  (i == mask) {
        qputnext(&qstr, type);
        return(aa);
    } else {
        qputnext(&qstr, Q);
    }
    pos++;
    i++;
    for (; i < mask; i++) {
        if  (GETBIT(i, addr) == 0) {
            qputnext(&qstr, Q);
            pos++;
        } else {
            qputnext(&qstr, invtype);
            qputnext(&qstr, Q);
            pos += 2;
    }   }
    if  (GETBIT(i, addr) == 0) {
        qputnext(&qstr, type);
        qputnext(&qstr, invtype);
    } else {
        qputnext(&qstr, invtype);
        qputnext(&qstr, type);
    }
    pos += 2;
    for (; pos < length; pos++) qputnext(&qstr, invtype);
    return(aa);
}

ipaggr *
qcomp(ipaggr *aa, int mask, u_int32_t addr, enum qchars type) {
    struct qstate qu;
    struct qstate *qs = &qu;
    enum qchars invtype = T;

    if  (type == T) invtype = F;
    if  (aa == NULL) {
        if  (type == T) return(qgenerate(0, mask, addr, T));
          else return(NULL);
    }
    qstate_init(aa, qs);
    if  (qwalk(qs, SEARCH_ADDR, mask, addr) == 1) {
        if  (qs->qc == type) return(aa);
        if  (qs->mask != mask) {
            ipaggr *naa;
            naa = qgenerate(qs->mask, mask, addr, type);
            if  (naa == NULL) return(NULL);
            if  (qreplace(&qs->qstr, naa) != 1) {
                FREE(naa);
                return(NULL);
            } else {
                FREE(naa);
                return(qs->qstr.qstring_data);
            }
            /* NOTREACHED */
        } else if (qs->qc == invtype) {
            qset(&qs->qstr, type);
        } else if (qs->qc == Q) {
            qset(&qs->qstr, type);
            qwalk(qs, SET_BLANK, qs->mask, 0);
        }
        qwalk(qs, AGGREGATE, 0, 0);
        if  (qs->qstr.alength != qs->qstr.qlength) qdelblank(&qs->qstr);
        return(qs->qstr.qstring_data);
    }
    return(NULL);
} 

ipaggr *
qadd(ipaggr *aa, int mask, u_int32_t addr) {
    return(qcomp(aa, mask, addr, T));
}

ipaggr *
qsub(ipaggr *aa, int mask, u_int32_t addr) {
    return(qcomp(aa, mask, addr, F));
}

ipaggr *
qintersect(ipaggr *aa, ipaggr *bb) {
    struct qstate qu,qy;
    struct qstate *qs = &qu, *qp = &qy;
    ipaggr *res = NULL, *old = NULL;	

    if  (aa == NULL || bb == NULL) return(NULL);
    qstate_init(aa, qs);
    qstate_init(bb, qp);
	
    if  (qwalk(qp, SEARCH_T, 0, 0) <= 0 || qwalk(qs, SEARCH_T, 0, 0) <= 0) goto end;
    while (1) {
        while (qs->mask <= qp->mask &&
		    qs->addr == SETMASK(qp->addr, qs->mask) && 
		    qs->qc == T && qp->qc == T) {
			res = qadd(res, qp->mask, qp->addr);
			if  (old != res && old != NULL)
				FREE(old);
			old = res;
			if  (qwalk(qp, SEARCH_T, 0, 0) <= 0)
				goto end;
        }
        while (  qp->mask <= qs->mask
              && qp->addr == SETMASK(qs->addr, qp->mask)
              && qs->qc == T && qp->qc == T
              ) {
            res = qadd(res, qs->mask, qs->addr);
            if  (old != res && old != NULL) FREE(old);
            old = res;
            if  (qwalk(qs, SEARCH_T, 0, 0) <= 0) goto end;
        }
        if  (qs->addr > qp->addr) {
            if  (qwalk(qp, SEARCH_T, 0, 0) <= 0) goto end;
        } else {
            if  (qwalk(qs, SEARCH_T, 0, 0) <= 0) goto end;
    }   }
end:
    if  (res == NULL) res = qgenerate(0, 0, 0, F);
    return(res);
}

int
qcheck(ipaggr *aa) {
    struct qstate qu;
    struct qstate *qs = &qu;

    qstate_init(aa, qs);
    /* FIX magic number */
    if  (qwalk(qs, SEARCH_ADDR, 32, 0xffffffff) > 0 &&  qs->qstr.position == (qs->qstr.qlength - 1)) {
        return(1);
    } else {
        ereport( ERROR
               , ( errcode(ERRCODE_INVALID_TEXT_REPRESENTATION)
                 , errmsg("ipaggr: full aggregate is smaller than given")
               ) );
        return(0);
}   }

PG_FUNCTION_INFO_V1(ipaggr_in);
Datum
ipaggr_in(PG_FUNCTION_ARGS) {
    char *cstr = (char *)PG_GETARG_CSTRING(0);
    char *p = cstr;
    int32 len = 0;
    ipaggr *result;

    len = strlen(cstr);
    while (*p) {
		switch (*p) {
		case 'q':
		case 't':
		case 'f':
			break;
		default:	
			ereport(ERROR,
			    (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
			    errmsg("invalid input syntax for ipaggr")));
		}
		p++;
    }
    result = (ipaggr *)ALLOC(len + VARHDRSZ);
    if  (result == NULL)
        ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ipaggr: out of memory")));
    memcpy(VARDATA(result), cstr, len);
#ifdef VARATT_SIZEP
    VARATT_SIZEP(result) = len + VARHDRSZ;
#else
    SET_VARSIZE(result, len + VARHDRSZ);
#endif
    if  (qcheck(result) == 1) PG_RETURN_POINTER(result);
    PG_RETURN_NULL();
}

PG_FUNCTION_INFO_V1(ipaggr_out);
Datum
ipaggr_out(PG_FUNCTION_ARGS) {
    ipaggr *aggr;
    char *cstr;
    int32 len;

    aggr = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(0));
    len = VARSIZE(aggr) - VARHDRSZ;
    cstr = (char *)ALLOC(len + 1);
    if  (cstr == NULL)
        ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("ipaggr: out of memory")));
    memcpy(cstr, VARDATA(aggr), len);
    cstr[len] = '\0';
    PG_RETURN_CSTRING(cstr);
}

PG_FUNCTION_INFO_V1(check_net);
Datum
check_net(PG_FUNCTION_ARGS) {
    ipaggr *aggr;
    inet *net;
    u_int32_t s_addr;
    int s_mask;
    struct qstate qu;
    struct qstate *qs = &qu;

    if  (PG_ARGISNULL(0) || PG_ARGISNULL(1)) PG_RETURN_NULL();
    aggr = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(0));
    qstate_init(aggr, qs);

    net = PG_GETARG_INET_P(1);
    if  (ip_family(net) == PGSQL_AF_INET6)
        ereport( ERROR
               , ( errcode(ERRCODE_FEATURE_NOT_SUPPORTED)
                 , errmsg("add_net: address family AF_INET6 not supported yet")
               ) );
	s_mask = ip_bits(net);
	s_addr = ip_v4addr(net);

	if  (qwalk(qs, SEARCH_ADDR, s_mask, s_addr) > 0 && qs->qc == T) PG_RETURN_BOOL(true);
	  else PG_RETURN_BOOL(false);
}

PG_FUNCTION_INFO_V1(add_net);
Datum
add_net(PG_FUNCTION_ARGS) {
    ipaggr *aggr, *newaggr;
    inet *net;
    u_long s_addr;
    int s_mask;

    if  (PG_ARGISNULL(0) || PG_ARGISNULL(1)) PG_RETURN_NULL();
    aggr = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(0));
    net = PG_GETARG_INET_P(1);
    if  (ip_family(net) == PGSQL_AF_INET6)
        ereport( ERROR
               , ( errcode(ERRCODE_FEATURE_NOT_SUPPORTED)
                 , errmsg("add_net: address family AF_INET6 not supported yet")
               ) );
    s_mask = ip_bits(net);
    s_addr = ip_v4addr(net);
    newaggr = qadd(aggr, s_mask, s_addr);
    PG_RETURN_POINTER(newaggr);
}

PG_FUNCTION_INFO_V1(sub_net);
Datum
sub_net(PG_FUNCTION_ARGS) {
    ipaggr *aggr, *newaggr;
    inet *net;
    u_int32_t s_addr;
    int s_mask;
	
    if  (PG_ARGISNULL(0) || PG_ARGISNULL(1)) PG_RETURN_NULL();
    aggr = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(0));
    net = PG_GETARG_INET_P(1);
    if  (ip_family(net) == PGSQL_AF_INET6)
        ereport( ERROR
               , ( errcode(ERRCODE_FEATURE_NOT_SUPPORTED)
                 , errmsg("add_net: address family AF_INET6 not supported yet")
               ) );
    s_mask = ip_bits(net);
    s_addr = ip_v4addr(net);
    newaggr = qsub(aggr, s_mask, s_addr);
    PG_RETURN_POINTER(newaggr);
}

PG_FUNCTION_INFO_V1(add_ipaggr);
Datum
add_ipaggr(PG_FUNCTION_ARGS) {
    ipaggr *aggr1, *aggr2, *new, *old;
    struct qstate qu;
    struct qstate *qs = &qu;

    if  (PG_ARGISNULL(0) || PG_ARGISNULL(1)) PG_RETURN_NULL();
    aggr1 = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(0));
    aggr2 = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(1));
    qstate_init(aggr2, qs);
    new = aggr1;
    old = aggr1;
    while (qwalk(qs, SEARCH_T, 0, 0) > 0) {
        new = qadd(new, qs->mask, qs->addr);
        if  (old != aggr1 && old != new) FREE(old);
        old = new;
    }
    PG_RETURN_POINTER(new);
}

PG_FUNCTION_INFO_V1(sub_ipaggr);
Datum
sub_ipaggr(PG_FUNCTION_ARGS) {
    ipaggr *aggr1, *aggr2, *new, *old;
    struct qstate qu;
    struct qstate *qs = &qu;

    if  (PG_ARGISNULL(0) || PG_ARGISNULL(1)) PG_RETURN_NULL();
    aggr1 = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(0));
    aggr2 = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(1));
    qstate_init(aggr2, qs);
    new = aggr1;
    old = aggr1;
    while (qwalk(qs, SEARCH_T, 0, 0) > 0) {
        new = qsub(new, qs->mask, qs->addr);
        if  (old != aggr1 && old != new) FREE(old);
        old = new;
    }
    PG_RETURN_POINTER(new);
}

PG_FUNCTION_INFO_V1(inter_ipaggr);
Datum
inter_ipaggr(PG_FUNCTION_ARGS) {
    ipaggr *aggr1, *aggr2, *new;

    if  (PG_ARGISNULL(0) || PG_ARGISNULL(1)) PG_RETURN_NULL();
    aggr1 = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(0));
    aggr2 = (ipaggr *)heap_tuple_untoast_attr((varattrib *)PG_GETARG_POINTER(1));
    new = qintersect(aggr1, aggr2);
    PG_RETURN_POINTER(new);
}

PG_FUNCTION_INFO_V1(show_net);
Datum
show_net(PG_FUNCTION_ARGS) {
    inet *net;
    ipaggr *aggr;
    struct qstate *qs;
    FuncCallContext  *funcctx;

    if  (SRF_IS_FIRSTCALL()) {
        MemoryContext oldcontext;
        funcctx = SRF_FIRSTCALL_INIT();
        oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
        if  (PG_ARGISNULL(0)) SRF_RETURN_DONE(funcctx);
        aggr = (ipaggr *)PG_GETARG_POINTER(0);
        aggr = (ipaggr *)heap_tuple_untoast_attr((varattrib *)aggr);
        qs = (struct qstate *)ALLOC(sizeof(struct qstate));
        if  (qs == NULL)
            ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("show_net: out of memory")));
        MemSet(qs, 0, sizeof(struct qstate));
        qstate_init(aggr, qs);
        funcctx->user_fctx = (void *)qs;
        MemoryContextSwitchTo(oldcontext);
    }
    funcctx = SRF_PERCALL_SETUP();
    qs = (struct qstate *)funcctx->user_fctx;
    aggr = qs->qstr.qstring_data;
    if  (qwalk(qs, SEARCH_T, 0, 0) > 0) {
        net = (inet *)ALLOC(sizeof(inet));
        if  (net == NULL)
            ereport(ERROR, (errcode(ERRCODE_OUT_OF_MEMORY), errmsg("show_net: out of memory")));
        MemSet(net, 0, sizeof(inet));
        ip_family(net) = PGSQL_AF_INET;
        ip_bits(net) = qs->mask;
#ifndef NOTYPE
        ip_type(net) = 0;
#endif
        toip_v4addr(net, qs->addr);
#ifdef VARATT_SIZEP
        VARATT_SIZEP(net) = sizeof(inet_struct) + VARHDRSZ;
#else
        SET_VARSIZE(net, sizeof(inet_struct) + VARHDRSZ);
#endif
        funcctx->user_fctx = (void *)qs;
        SRF_RETURN_NEXT(funcctx, InetPGetDatum(net));
    } else {
        SRF_RETURN_DONE(funcctx);
}   }
