/*-
 * Copyright (C)2010..2022 @BABOLO http://www.babolo.ru/
 * PKG = babolo-libmake
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

static const char copyright[] = "\
@(#)Copyright (C)2010..2022 @BABOLO http://www.babolo.ru/\n\
@(#)All rights reserved.\n\
@(#)V.M "VERS;
static const char rcsid[] = "$Id: blin.c,v 1.59 2022/08/28 15:21:28 babolo Exp $";

#define BLIN_COMPAT VMAJOR
#define BLIN_INTERNAL

#include <sys/types.h>
#include <sys/file.h>
#include <sysexits.h>
#include <strings.h>
#include <sys/uio.h>
#include <string.h>
#include <syslog.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <err.h>
#include "../BLINflag.h"

#define ERROUT(A, B)    do {ex = (A); errno = (B); goto out;} while(0)

blin_pamparam blin_pamparam_internal =
{ 0x00000000, 0
, { {NULL, BLIN_MODWARN | BLIN_MODEOAU, LOG_LOCAL7 | LOG_ERR    }
  , {NULL, BLIN_MODWARN | BLIN_MODEOAU, LOG_LOCAL7 | LOG_WARNING}
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  }
, -1
};

__weak_reference(blin_pamparam_internal, blin_internal_flags);
__weak_reference(blin_pamparam_internal, blin_direct_flags);

static const char * const
fac[] =
{ "KERN"  , "USER"  , "MAIL"    , "DAEMON", "AUTH"  , "SYSLOG"  , "LPR"    , "NEWS"
, "UUCP"  , "CRON"  , "AUTHPRIV", "FTP"   , "NTP"   , "SECURITY", "CONSOLE", "reserved"
, "LOCAL0", "LOCAL1", "LOCAL2"  , "LOCAL3", "LOCAL4", "LOCAL5"  , "LOCAL6" , "LOCAL7"
};

static const char * const
pri[] = {"EMERG", "ALERT", "CRIT", "ERR", "WARNING", "NOTICE", "INFO", "DEBUG"};

static const char * const
blinf[] = 
{ "MAININI", NULL  , NULL  , NULL  , NULL  , NULL  , NULL  , NULL
, NULL     , NULL  , NULL  , NULL  , NULL  , NULL  , NULL  , NULL
, NULL     , NULL  , NULL  , NULL  , NULL  , NULL  , NULL  , "TRAC"
, "BIT0"   , "BIT1", "BIT2", "BIT3", "BIT4", "BIT5", "BIT6", "BIT7"
};

static const char * const
whatf[] = 
{ NULL     , NULL     , NULL     , NULL  , "MODDTXT", NULL     , NULL     , NULL
, NULL     , NULL     , NULL     , NULL  , "MODDSTR", "MODDNON", "MODDREL", "MODTIME"
, "MODEOLF", "MODEOCR", "MODEOAU", NULL  , NULL     , "MODLOGR", "MODPRNT", "MODWARN"
, NULL     , NULL     , NULL     , NULL  , NULL     , NULL     , NULL     , NULL
};

#define SPC0 64
static const char *
spaces = "                                                                "; /* SPC0 * ' ' */

size_t strnlen __P((const char *s, size_t maxlen));

size_t
blin_internal_strnlen(const char *s, size_t maxlen) {
    size_t len;

    for (len = 0; len < maxlen; len++, s++) {
        if  (!*s) break;
    }
    return (len);
}
__weak_reference(blin_internal_strnlen, strnlen);

static void
/*****************************************************************************************************/
blin_init() {                                                                                    /****
 *****************************************************************************************************/
    int i;

    if  (!(blin_pamparam_internal.flags & BLIN_MAININI)) {
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: +blin_init %08X\n"
                   , blin_pamparam_internal.flags
                   )
            ;
        }
        for (i = 0; i < 8; i++) blin_pamparam_internal.w[i].file = stderr;
        blin_pamparam_internal.flags |= BLIN_MAININI;
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: -blin_init %08X\n"
                   , blin_pamparam_internal.flags
                   )
            ;
}   }   }

static u_int32_t
/*****************************************************************************************************/
blin_do(u_int32_t mode, u_int32_t w, u_int32_t legal, u_int32_t *f) {                            /****
 *****************************************************************************************************/
    u_int32_t r = *f;              /*   */

    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf( blin_pamparam_internal.w[0].file
               , "BLIN: +blin_do %08X %08X %08X %08X\n"
               , mode, w, legal, r
               )
        ;
    }
    if  (mode & BLIN_CTL_RISE) {
        w = (((((mode & BLIN_CTL_NPRM) ? r : w) << 1) & BLIN_MASK) | BLIN_BIT0);
    } else if (mode & BLIN_CTL_NPRM) {
        w = 0;
    }
    if  (mode & BLIN_CTL_FNEG) w = ~w;
    w &= legal;
    switch (mode & (BLIN_CTL_NONE | BLIN_CTL_FAFM)) {
    case BLIN_CTL_FSET: *f |= w;                    break;
    case BLIN_CTL_FRST: *f &= ~w;                   break;
    case BLIN_CTL_FXOR: *f ^= w;                    break;
    case BLIN_CTL_FEQU: *f = (*f & ~legal) | w;     break;
    }
    if  (mode & BLIN_CTL_NPRE) r = *f;
    if  (mode & BLIN_CTL_RMSK) r &= legal;
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_do %08X\n", r);
    }
    return(r);
}

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_64(u_int64_t flags, const char * const *nm, int x, FILE *fo) {                              /****
 *****************************************************************************************************
 *****************************************************************************************************/
    int       ex = 0;
    u_int64_t m;
    int       i;
    int       k;
    int       l;

    if  (x > 64) x = 64;
    if  (x < -64) x = -64;
    l = (0 > x) ? x : -x;
    for (i = l; ~i; i >>= 1) l &= i;
    l = -l;
    switch(l) {
    case 0:
    case 1:
    case 2:
    case 4:
        if  (!(flags & 0xFFFFFFFFFFFFFFF0ULL)) {
            ex += fprintf(fo, "%01X", (u_int8_t)flags);
            break;
        }
        /* FALLTHROUGH */
    case 8:
        if  (!(flags & 0xFFFFFFFFFFFFFF00ULL)) {
            ex += fprintf(fo, "%02X", (u_int8_t)flags);
            break;
        }
        /* FALLTHROUGH */
    case 16:
        if  (!(flags & 0xFFFFFFFFFFFF0000ULL)) {
            ex += fprintf(fo, "%04X", (u_int16_t)flags);
            break;
        }
        /* FALLTHROUGH */
    case 32:
        if  (!(flags & 0xFFFFFFFF00000000ULL)) {
            ex += fprintf(fo, "%08X", (u_int32_t)flags);
            break;
        }
        /* FALLTHROUGH */
    default:
        ex += fprintf(fo, "%016"BLIN_O"X", flags);
    }
    k = 0;
    if  (0 > x) {
        for (i = 0, m = 1ULL << (l - 1); i < -x; ++i, m >>= 1) {
//printf("%d %016llX %016llX %016llX\n",i,m,flags,flags&m);
            if  ((flags & m) && !nm[i]) {
                ex += fprintf(fo, "%sunkn", k++ ? "," : "<");
                break;
        }   }
        for (i = 0, m = 1ULL << (l - 1); i < -x; ++i, m >>= 1) {
            if  ((flags & m) && !!nm[i]) {
                ex += fprintf(fo, "%s%s", k++ ? "," : "<", nm[i]);
        }   }
    } else if (0 < x) {
        for (i = 0, m = 1; i < x; ++i, m <<= 1) {
            if  ((flags & m) && !nm[i]) {
                ex += fprintf(fo, "%sunkn", k++ ? "," : "<");
                break;
        }   }
        for (i = 0, m = 1; i < x; ++i, m <<= 1) {
            if  ((flags & m) && !!nm[i]) {
                ex += fprintf(fo, "%s%s", k++ ? "," : "<", nm[i]);
    }   }   }
    if  (!!k) ex += fprintf(fo, ">");
    return(ex);
}

void *
/*****************************************************************************************************
 *****************************************************************************************************/
blin_ctl(u_int32_t mode, ...) {                                                                  /****
 *****************************************************************************************************
 *****************************************************************************************************/
    void     *res = NULL;               /*   */
    va_list   ap;                       /*   va_arg */
    u_int32_t m;                        /*  */
    u_int     i;                        /*    */
    int       d;                        /*  */

    va_start(ap, mode);
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_ctl %08X\n", mode);
    }
    blin_init();
    switch (mode & BLIN_CTL_OPER) {
    case BLIN_CTL_FLAG: {
            u_int32_t w = va_arg(ap, u_int32_t);

            res = (void *)(size_t)blin_do(mode, w, BLIN_FLEGAL, &blin_pamparam_internal.flags);
        }
        break;
    case BLIN_CTL_WHAT: {
            u_int32_t w = va_arg(ap, u_int32_t);

            for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                if  (mode & m) {
                    res = (void *)(size_t)blin_do(mode, w, ~0, &blin_pamparam_internal.w[i].what);
        }   }   }
        break;
    case BLIN_CTL_FNAM: {
            char *f = va_arg(ap, char *);
            FILE *w = fopen(f, "a");

            if  (w == NULL) {
                fprintf(blin_pamparam_internal.w[0].file, "fopen=%s:%s\n", f, sys_errlist[errno]);
            } else {
                setvbuf(w, NULL, _IONBF, 0);
                for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                    if  (mode & m) {
                        res = blin_pamparam_internal.w[i].file;
                        if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].file = w;
                        if  (mode & BLIN_CTL_NPRE) res = blin_pamparam_internal.w[i].file;
        }   }   }   }
        break;
    case BLIN_CTL_FDES: {
            int f = va_arg(ap, int);
            FILE *w = fdopen(f, "a");

            if  (w == NULL) {
                fprintf(blin_pamparam_internal.w[0].file, "fdopen=%d:%s\n", f, sys_errlist[errno]);
            } else {
                setvbuf(w, NULL, _IONBF, 0);
                for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                    if  (mode & m) {
                        res = blin_pamparam_internal.w[i].file;
                        if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].file = w;
                        if  (mode & BLIN_CTL_NPRE) res = blin_pamparam_internal.w[i].file;
        }   }   }   }
        break;
    case BLIN_CTL_FILE: {
            void *w = va_arg(ap, void *);

            for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                if  (mode & m) {
                    res = blin_pamparam_internal.w[i].file;
                    if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].file = w;
                    if  (mode & BLIN_CTL_NPRE) res = blin_pamparam_internal.w[i].file;
        }   }   }
        break;
    case BLIN_CTL_PRIO: {
            int32_t w = va_arg(ap, int32_t);

            for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                if  (mode & m) {
                    res = (void *)(ssize_t)(blin_pamparam_internal.w[i].prio);
                    if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].prio = w;
                    if  (mode & BLIN_CTL_NPRE) {
                        res = (void *)(ssize_t)(blin_pamparam_internal.w[i].prio);
        }   }   }   }
        break;
    case BLIN_CTL_TCFN: {
            char *f = va_arg(ap, char *);
            int w = open(f, O_CREAT | O_WRONLY | O_TRUNC | O_APPEND, 0666);

            if  (w < 0) {
                fprintf( blin_pamparam_internal.w[0].file
                       , "%s:%s:%d: open(%s): %s\n"
                       , __FILE__
                       , __func__
                       , __LINE__
                       , f
                       , sys_errlist[errno]
                       )
                ;
            } else {
                res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
                if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.tracfile = w;
                if  (!(mode & BLIN_CTL_NPRE)) {
                    res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
        }   }   }
        break;
    case BLIN_CTL_TCFD: {
            int w = va_arg(ap, int);

            res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
            if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.tracfile = w;
            if  (!(mode & BLIN_CTL_NPRE)) {
                res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
        }   }
        break;
    case BLIN_CTL_TCFL: {
            FILE *f = va_arg(ap, void *);
            int w = fileno(f);

            if  (w < 0) {
                fprintf( blin_pamparam_internal.w[0].file
                       , "%s:%s:%d: fileno: %s\n"
                       , __FILE__
                       , __func__
                       , __LINE__
                       , sys_errlist[errno]
                       )
                ;
            } else {
                res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
                if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.tracfile = w;
                if  (!(mode & BLIN_CTL_NPRE)) {
                    res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
        }   }   }
        break;
    case BLIN_CTL_CPRI:
        res = (void *)copyright;
        break;
    case BLIN_CTL_RCSI:
        res = (void *)rcsid;
        break;
    case BLIN_CTL_VERS:
        res = (void *)VERS;
        break;
    case BLIN_CTL_DUMP: {
            int k;

        ;   fprintf( blin_pamparam_internal.w[0].file
                   , "blin_pamparam_internal @ %"BLIN_X" "
                   , BLIN_I(&blin_pamparam_internal)
                   )
        ;   ;
        ;   blin_64(blin_pamparam_internal.flags, blinf, 32, blin_pamparam_internal.w[0].file);
        ;   if  (blin_pamparam_internal.err) {
        ;       fprintf( blin_pamparam_internal.w[0].file
                       , " !!! %d !!!\n"
                       , blin_pamparam_internal.err
                       )
                ;
        ;   } else {
        ;       fprintf(blin_pamparam_internal.w[0].file, "\n");
        ;   }
        ;   for (i = 0; i < 8; ++i) {
#               define  SPC1 40
#               define  SPC2 24

        ;       fprintf(blin_pamparam_internal.w[0].file, "  %u ", i);
        ;       d = blin_64( blin_pamparam_internal.w[i].what
                           , whatf
                           , 32
                           , blin_pamparam_internal.w[0].file
                           )
                ;
        ;       if  (SPC1 > d) {
        ;           fprintf(blin_pamparam_internal.w[0].file, "%s", &spaces[SPC0 - SPC1 + d]);
        ;           d = 0;
        ;       } else {
        ;           d -= SPC1;
        ;       }
        ;       k = 0;
        ;       m = blin_pamparam_internal.w[i].prio;
        ;       d += fprintf(blin_pamparam_internal.w[0].file, " %08X", m);
        ;       if  (m & ~(LOG_PRIMASK | LOG_FACMASK)) {
        ;           d += fprintf(blin_pamparam_internal.w[0].file, "%sunkn", k++ ? "," : "<");
        ;       }
        ;       if  (LOG_FAC(m) < LOG_NFACILITIES) {
        ;           d += fprintf( blin_pamparam_internal.w[0].file
                                , "%s%s"
                                , k++ ? "," : "<"
                                , fac[(m & LOG_FACMASK) >> 3]
                                )
        ;           ;
        ;       } else {
        ;           d += fprintf( blin_pamparam_internal.w[0].file
                                , "%s*%02X*"
                                , k++ ? "," : "<"
                                , LOG_FAC(m)
                                )
        ;           ;
        ;       }
        ;       d += fprintf( blin_pamparam_internal.w[0].file
                            , "%s%s"
                            , k++ ? "," : "<"
                            , pri[m & LOG_PRIMASK]
                            )
        ;       ;
        ;       d += fprintf(blin_pamparam_internal.w[0].file, ">");
        ;       if  (SPC2 > d) {
        ;           fprintf(blin_pamparam_internal.w[0].file, "%s", &spaces[SPC0 - SPC2 + d]);
        ;       }
        ;       if  (blin_pamparam_internal.w[i].file == stdin) {
        ;           fprintf(blin_pamparam_internal.w[0].file, " file=stdin\n");
        ;       } else if (blin_pamparam_internal.w[i].file == stdout) {
        ;           fprintf(blin_pamparam_internal.w[0].file, " file=stdout\n");
        ;       } else if (blin_pamparam_internal.w[i].file == stderr) {
        ;           fprintf(blin_pamparam_internal.w[0].file, " file=stderr\n");
        ;       } else {
        ;           fprintf( blin_pamparam_internal.w[0].file
                           , " file=%"BLIN_X"\n"
                           , BLIN_I(blin_pamparam_internal.w[i].file)
                           )
        ;           ;
        ;   }   }
            fprintf(blin_pamparam_internal.w[0].file, " track=%d\n", blin_pamparam_internal.tracfile);
    }   }
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_ctl %"BLIN_X"\n", BLIN_I(res));
    }
    va_end(ap);
    return(res);
}

static void
/*****************************************************************************************************/
doout(int level, char *text, const char *file, const char *func, int line, int er) {             /****
 *****************************************************************************************************/
    u_int32_t       ctl = blin_pamparam_internal.w[level & 7].what;
    struct timespec tp;                      /*      */

    clock_gettime(CLOCK_REALTIME, &tp);
    if  ((ctl & BLIN_MODWARN) && (level & BLIN_DOCRY)) {
        if  (ctl & BLIN_MODTIME) {
            warnx("%10"BLIN_D".%09ld %s:%s:%d: %s", tp.tv_sec, tp.tv_nsec, file, func, line, text);
        } else {
            warnx("%s:%s:%d: %s", file, func, line, text);
    }   }
    if  ((ctl & BLIN_MODPRNT) && (level & BLIN_DOCRY)) {
        if  (ctl & BLIN_MODTIME) {
            fprintf( blin_pamparam_internal.w[level & 7].file
                   , "%10"BLIN_D".%09ld %s:%s:%d: %s"
                   , tp.tv_sec, tp.tv_nsec, file, func, line, text
                   );
        } else {
            fprintf(blin_pamparam_internal.w[level & 7].file, "%s:%s:%d: %s", file, func, line, text);
    }   }
    if  ((ctl & BLIN_MODLOGR) && (level & BLIN_DOCRY)) {
        if  (ctl & BLIN_MODTIME) {
            syslog( blin_pamparam_internal.w[level & 7].prio
                  , "%10"BLIN_D".%09ld %s:%s:%d: %s"
                  , tp.tv_sec, tp.tv_nsec, file, func, line, text
                  );
        } else {
            syslog(blin_pamparam_internal.w[level & 7].prio, "%s:%s:%d: %s", file, func, line, text);
    }   }
    if  (blin_pamparam_internal.tracfile >= 0) {
        BLIN_trace        *tracinf;
        const char * const prog = getprogname();
        char              *textpoint;
        size_t             textlen = strlen(file) + strlen(func) + strlen(prog) + strlen(text);

        tracinf = malloc(offsetof(BLIN_trace, string) + textlen + 7);
        if  (!tracinf) {
            return; /* errprint */
        }
        tracinf->len    = (offsetof(BLIN_trace, string) + textlen + 7) & ~((u_int32_t)07U);
        tracinf->erno   = er;
        tracinf->pid    = getpid();
        tracinf->time   = tp.tv_sec * 1000000000LLU + tp.tv_nsec;
        tracinf->level  = level & 7;
        tracinf->line   = line;
        tracinf->progsz = strlen(prog);
        tracinf->filesz = strlen(file);
        tracinf->funcsz = strlen(func);
        textpoint = tracinf->string;
        textpoint = stpncpy(textpoint, prog, (size_t) tracinf->progsz);
        textpoint = stpncpy(textpoint, file, (size_t) tracinf->filesz);
        textpoint = stpncpy(textpoint, func, (size_t) tracinf->funcsz);
        textpoint = stpncpy(textpoint, text, strlen(text));
        bzero(textpoint, 7);
        flock(blin_pamparam_internal.tracfile, LOCK_EX);
        write(blin_pamparam_internal.tracfile, tracinf, (size_t) tracinf->len & ~((size_t)07U));
        flock(blin_pamparam_internal.tracfile, LOCK_UN);
        free(tracinf);
}   }

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_cry4(int ctl, const char *file, const char *func, int nline, const char *format, ...) {     /****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t control;                      /* what    blin_pamparam_internal */
    ssize_t   avail;                        /*                              */
    ssize_t   need = 0;                     /*                                 */
    char     *buf  = NULL;                  /*                                                 */
    char     *ptr  = NULL;                  /*  */
    char      qq   [BLIN_MAXCRY1 + 4];      /*                                      */
    int       er   = errno;                 /*  errno                                      */
    int       ex   = EX_OK;                 /*                                              */
    va_list   ap;                           /*   va_arg                               */
    ssize_t   t;                            /*  */

    blin_init();
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_cry4 %08X\n", ctl);
    }
    control = blin_pamparam_internal.w[ctl & 7].what;
    need = 0;
    if  (!(ctl & BLIN_DOCRY) && (blin_pamparam_internal.tracfile < 0)) goto out;
    do {
        va_start(ap, format);
        if  (!need) {
            buf = qq;
            avail = BLIN_MAXCRY4;
        } else {
            buf = malloc(need + 5);
            avail = need + 1;
            need = 0;
            if  (!buf) {
                ifBLIN_QX0("malloc");
                ERROUT(EX_OSERR, ENOMEM);
        }   }
        ptr = buf;
        
        t = vsnprintf(ptr, avail + 1, format, ap);
        need += t;
        if  (0 >= (avail -= t)) avail = 0; else ptr += t;
        if  (ctl & 8) {
#           define UPREFIX         "Unknown error"
#           define EBUFSIZE        11

            if  (avail-- > 0) *(ptr++) = ':';
            if  (avail-- > 0) *(ptr++) = ' ';
            need += 2;
            if  ((er <= sys_nerr) && (er >= 0)) {
                for (t = 0; sys_errlist[er][t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = sys_errlist[er][t];
                        --avail;
                    }
                    ++need;
                }
            } else if (avail < (EBUFSIZE + strlen(UPREFIX))) {
                need += EBUFSIZE + strlen(UPREFIX);
                avail = 0;
                if  (er < 0) {
                    ++need;
                    --avail;
                }
            } else {
                char *e;
                int s;

                for (t = 0; UPREFIX[t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = UPREFIX[t];
                        --avail;
                    }
                    ++need;
                }
                e = ptr + EBUFSIZE;
                need += EBUFSIZE;
                avail -= EBUFSIZE;
                if  (er < 0) {
                    s = -er;
                    ++e;
                    ++need;
                    --avail;
                } else {
                    s = er;
                }
                do *--e = "0123456789"[s % 10]; while (s /= 10);
                if  (er < 0) *--e = '-';
                while (e > ptr) *--e = ' ';
                ptr += EBUFSIZE;
                if  (er < 0) ++ptr;
        }   }
        if  (control & BLIN_MODEOAU) {
            if  ((ptr <= buf) || (ptr[-1] != '\n')) control |= BLIN_MODEOLF;
        }
        if  (!(~control & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(ptr++) = '\r';
            --avail;
            ++need;
        }
        if  (!(~control & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(ptr++) = '\n';
            --avail;
            ++need;
        }
        va_end(ap);
    } while (avail <= 0);
    *(ptr++) = '\0';
    doout(ctl & (BLIN_DOCRY | 7), buf, file, func, nline, er);
out:
    if  (buf != qq) free(buf);
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_cry4 %d %"BLIN_D"\n", ex, need);
    }
    blin_pamparam_internal.err = errno;
    errno = er;
    return(ex);
}
