/*-
 * Copyright (C)2020..2021 @BABOLO http://www.babolo.ru/
 * PKG = babolo-libmake
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2020..2021 @BABOLO http://www.babolo.ru/\n"
#ident "@(#) $Id: blinread.c,v 1.55 2021/10/25 15:04:44 cuhegh Exp $\n"

#define Bpars_COMPAT  4
#define MULAR_COMPAT  0
#define MIFE_COMPAT   5
#define BLIN_COMPAT   VMAJOR
#define BLIN_INTERNAL

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sysexits.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <limits.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <err.h>
#include "../BLINflag.h"
#include <babolo/parser.h>
#include <multilar.h>
#include <mife.h>
#include "blinread.h"

static const size_t msz[] = {1024, 512, 512};

static void
/*****************************************************************************************************/
usage(int ex) {                                                                                  /****
 *****************************************************************************************************/
    fprintf( stderr
           , PROG" @BABOLO V.M "VERS" http://www.babolo.ru/\n"
             "Usage: "PROG" [-options]  tracfile\n"
             "   -c TEXT         read from TEXT\n"
             "   -Q              give information about tracefile\n"
             "   -l MASK         filter by message level\n"
             "   -d NUM[-NUM]    filter by message data,\n"
             "                     where NUM is <YYYY-MM-DD[_hh:mm:ss[.nsec]]>\n"
             "   -t NUM[-NUM]    filter by relative message time, where NUM is <sec[.nsec]>\n"
             "   -i NUM          filter by pid\n"
             "   -r TEXT         filter by program name\n"
             "   -s TEXT         filter by file name\n"
             "   -u TEXT         filter by procedure name\n"
             "   -n NUM[-NUM]    filter by line num\n"
             "   -LATIRSUN       invert term of its kind\n"
             "   -x              erase terms\n"
             "   -e              end of expression\n"
             "   -p <options>    What part of message to print\n"
             "          h            head\n"
             "          i            pid\n"
             "          d            data\n"
             "          t            relative time\n"
             "          r            program name\n"
             "          s            file name\n"
             "          u            procedure name\n"
             "          n            line num\n"
             "          l            level\n"
             "          e            errno\n"
             "          x            text only\n"
             "   -v[v[v[...]]]   verbose\n"
             "   -q              silent\n"
             "   -= MODE         internal diagnostic\n"
             "   -?              this text\n"
           )
    ;
    ifBLIN_QX2("exit");
    exit(ex);
}

int
main(int argc, char *argv[]) {
#   define isJX(EXCODE, FORMAT, ...) \
        do { ifBLIN_QX0("!!" FORMAT, ## __VA_ARGS__); ex = EXCODE; goto out; }while(0)
#   define isJW(EXCODE, FORMAT, ...) \
        do { ifBLIN_QW0("!!" FORMAT, ## __VA_ARGS__); ex = EXCODE; goto out; }while(0)
#   define blin_internal_flags (cfg->flags & BLIN_MASK)
    const char *const opts = "c:Dd:eIi:Ll:Nn:p:QqRr:Ss:Tt:Uu:vx?=:";
    babolo_opts      *bos = NULL;              /*             */
    char              fl;                      /*                                       */
    const char       *optargs;                 /*                              */
    int               ex = EX_OK;              /*                                */
    BLINr_fraze      *curfraze;                /*                           */
    const char       *cname = NULL;            /*                             */
    mife_descriptor  *cfile = NULL;            /*                                 */
    char             *cchar = NULL;            /*                       */
    babolo_parm      *bp;                      /*   -                          */
    const char       *tname;                   /*  tracfile                              */
    BLIN_trace       *twind;                   /*   tracfile                           */
    u_int64_t         m;                       /* ޣ,              */
    BLINr_cfg        *cfg;

    if  (!(cfg = calloc(1, sizeof(BLINr_cfg)))) isJW(EX_OSERR, "cfg alloc");
    blin_ctl(BLIN_CTL_FLAG | BLIN_CTL_FSET, (cfg->flags = BLIN_BIT0));
    cfg->flags = BLINREAD_P_H | BLINREAD_P_L | BLINREAD_P_D | BLINREAD_P_I;
    cfg->flags |= BLINREAD_P_R | BLINREAD_P_S | BLINREAD_P_U | BLINREAD_P_N;
    if  (!(curfraze = cfg->filt = malloc(sizeof(BLINr_fraze)))) isJW(EX_OSERR, "curfraze malloc");
    cfg->time = mular_create(MULAR_OBJ | MULAR_STRI, 3, sizeof(BLINr_time), msz);
    if  (!cfg->time) isJW(EX_OSERR, "mular_create time");
    cfg->pid = mular_create(MULAR_OBJ | MULAR_STRI, 3, sizeof(u_int32_t) , msz);
    if  (!cfg->pid) isJW(EX_OSERR, "mular_create pid");
    cfg->prog = mular_create(MULAR_OBJ | MULAR_STRI | MULAR_PFRE, 3, sizeof(char*), msz);
    if  (!cfg->prog) isJW(EX_OSERR, "mular_create prog");
    cfg->file = mular_create(MULAR_OBJ | MULAR_STRI | MULAR_PFRE, 3, sizeof(char*), msz);
    if  (!cfg->file) isJW(EX_OSERR, "mular_create file");
    cfg->func = mular_create(MULAR_OBJ | MULAR_STRI | MULAR_PFRE, 3, sizeof(char*), msz);
    if  (!cfg->func) isJW(EX_OSERR, "mular_create func");
    cfg->line = mular_create(MULAR_OBJ | MULAR_STRI, 3, sizeof(BLINr_line), msz);
    if  (!cfg->line) isJW(EX_OSERR, "mular_create line");
    if  (!(cfg->tfile = mife_init(MIFE_FULL))) isJW(EX_OSERR, "mife_init");
    if  (!(bos = babolo_openopts(Bpars_SYME | Bpars_ESYM, 'Z'))) isJW(EX_OSERR, "babolo_openopts");
    if  (babolo_setopts(bos, Bpars_NONU, argc, argv, opts))    isJW(EX_OSERR, "babolo_setopts");
    *curfraze = (BLINr_fraze){NULL, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, 0, 0, BLINREAD_X};
    cfg->prog->tofree = cfg->file->tofree = cfg->func->tofree = free; /* free() */
    while(!!(fl = babolo_getopts(bos))) {
        ifBLIN_QX2("flag %02X -%c", fl, fl);
        switch(fl) {
        case 'c':
            cfile = mife_init(MIFE_FULL);
            cname = babolo_getoptsarg(bos);
            if  (!cfile) isJW(EX_OSERR, "flag %c mife_init", fl);
            if  (!cname) isJW(EX_DATAERR, "flag %c args", fl);
            if  (0 > mife_ctlfile(cfile, cname)) isJW(EX_IOERR, "flag %c mife_ctl %s", fl, cname);
            if  (0 > mife_read(cfile, 0, 0)) isJW(EX_IOERR, "flag %c mife_read", fl);
            cchar = mife_window(cfile, 0, 0);
            if  (!cchar) isJW(EX_IOERR, "flag %c mife_window", fl);
            if  (!(cfile->flags & MIFE_EOFL)) isJW(EX_USAGE, "flag %c file too big", fl);
            bp = babolo_getparm(Bpars_NOEN | Bpars_NONU, &cchar, "", 0);
            if  (!bp) isJW(EX_DATAERR, "flag %c no parm %s", fl, cchar);
            mife_fini(cfile);
            if  (babolo_setparm(bos, 0, bp, NULL)) isJX(EX_USAGE, "flag %c no opts %s", fl, cname);
            break;
        case 'D':
            curfraze->flags ^= BLINREAD_T;
            break;
        case 'd': {
                BLINr_time *time = mular_insert(cfg->time, curfraze->time.to++);

                if  (!time) isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->time.to - 1);
                if  (!(optargs = babolo_getoptsarg(bos)))    isJW(EX_DATAERR, "flag %c args", fl);
                time->abs = 1;
                if  (!!tim2i(cfg, time, optargs)) isJX(EX_USAGE, "flag %c tim2i", fl);
                errno = 0;
            }
            break;
        case 'e':
            if  (!curfraze->levfl) curfraze->level = 0xFFU;
            if  (!(curfraze->next = malloc(sizeof(BLINr_fraze)))) isJW(EX_OSERR, "flag %c malloc", fl);
            curfraze->next->next = NULL;
            curfraze->next->time.from = curfraze->next->time.to = curfraze->time.to;
            curfraze->next->pid.from  = curfraze->next->pid.to  = curfraze->pid.to ;
            curfraze->next->prog.from = curfraze->next->prog.to = curfraze->prog.to;
            curfraze->next->file.from = curfraze->next->file.to = curfraze->file.to;
            curfraze->next->func.from = curfraze->next->func.to = curfraze->func.to;
            curfraze->next->line.from = curfraze->next->line.to = curfraze->line.to;
            curfraze->next->level = 0x00;
            curfraze->next->levfl = 0;
            curfraze->next->flags = 0x00;
            curfraze = curfraze->next;
            break;
        case 'I':
            curfraze->flags ^= BLINREAD_I;
            break;
        case 'i': {
                u_int32_t *pid = mular_insert(cfg->pid, curfraze->pid.to++);

                if  (!pid) isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->pid.to - 1);
                *pid = babolo_getoptunum(bos, 10);
                if  (errno) isJW(EX_USAGE, "flag %c wrong arg", fl);
            }
            break;
        case 'L':
            curfraze->flags ^= BLINREAD_L;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'l': {
                u_int8_t optunum = babolo_getoptunum(bos, 0);

                if  (errno) isJW(EX_USAGE, "flag %c wrong arg", fl);
                curfraze->level |= optunum;
                curfraze->levfl = 1;
            }
            break;
        case 'N':
            curfraze->flags ^= BLINREAD_N;
            break;
        case 'n': {
                BLINr_line *line = mular_insert(cfg->line, curfraze->line.to++);

                if  (!line) isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->line.to - 1);
                if  (!(optargs = babolo_getoptsarg(bos))) isJW(EX_USAGE, "flag %c args", fl);
                line->from = strtoul(optargs, (char**) &optargs, 0);
                if  (*optargs == '-') {
                    line->nrng = 0;
                    optargs++;
                    line->to = strtoul(optargs, NULL, 0);
                } else {
                    line->nrng = 1;
            }   }
            break;
        case 'p': {
                m = 0;
                if  (!(optargs = babolo_getoptsarg(bos))) isJW(EX_USAGE, "flag %c args", fl);
                for (u_int8_t i = 0; i < strlen(optargs); i++) {
                    switch(optargs[i]) {
                    case 'h': m |= BLINREAD_P_H; break;
                    case 'l': m |= BLINREAD_P_L; break;
                    case 'd': m |= BLINREAD_P_D; break;
                    case 't': m |= BLINREAD_P_T; break;
                    case 'i': m |= BLINREAD_P_I; break;
                    case 'r': m |= BLINREAD_P_R; break;
                    case 's': m |= BLINREAD_P_S; break;
                    case 'u': m |= BLINREAD_P_U; break;
                    case 'n': m |= BLINREAD_P_N; break;
                    case 'e': m |= BLINREAD_P_E; break;
                    case 'x': m = 0; i = strlen(optargs); break;
                    default: isJX(EX_USAGE, "flag %c wrong arg", fl);
                }   }
                cfg->flags = (cfg->flags & ~BLINREAD_PMASC) | m;
            }
            break;
        case 'Q':
            cfg->flags |= BLINREAD_Q;
            break;
        case 'q': BLIN_QUIET(cfg->flags);
            break;
        case 'R':
            curfraze->flags ^= BLINREAD_R;
            break;
        case 'r': {
                const char **prog = mular_insert(cfg->prog, curfraze->prog.to++);

                if  (!prog) isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->prog.to - 1);
                if  (!(*prog = babolo_getoptsarg(bos))) isJW(EX_USAGE, "flag %c args", fl);
                if  (!(*prog = strdup(*prog))) isJW(EX_OSERR, "flag %c strdup", fl);
            }
            break;
        case 'S':
            curfraze->flags ^= BLINREAD_S;
            break;
        case 's': {
                const char **file = mular_insert(cfg->file, curfraze->file.to++);

                if  (!file) isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->file.to - 1);
                if  (!(*file = babolo_getoptsarg(bos))) isJW(EX_USAGE, "flag %c args", fl);
                if  (!(*file = strdup(*file))) isJW(EX_OSERR, "flag %c strdup", fl);
            }
            break;
        case 'T':
            curfraze->flags ^= BLINREAD_T;
            break;
        case 't': {
                BLINr_time *time = mular_insert(cfg->time, curfraze->time.to++);

                if  (!time) isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->time.to - 1);
                if  (!(optargs = babolo_getoptsarg(bos))) isJW(EX_USAGE, "flag %c args", fl);
                time->abs = 0;
                if  (!!tim2i(cfg, time, optargs)) isJX(EX_USAGE, "flag %c tim2i", fl);
                errno = 0;
            }
            break;
        case 'U':
            curfraze->flags ^= BLINREAD_U;
            break;
        case 'u': {
                const char **func = mular_insert(cfg->func, curfraze->func.to++);

                if  (!func) isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->func.to - 1);
                if  (!(*func = babolo_getoptsarg(bos))) isJW(EX_USAGE, "flag %c args", fl);
                if  (!(*func = strdup(*func))) isJW(EX_OSERR, "flag %c strdup", fl);
            }
            break;
        case 'v': BLIN_VERBOSE(cfg->flags);
            break;
        case 'x':
            curfraze->time.from = curfraze->time.to;
            curfraze->pid.from  = curfraze->pid.to;
            curfraze->prog.from = curfraze->prog.to;
            curfraze->file.from = curfraze->file.to;
            curfraze->func.from = curfraze->func.to;
            curfraze->line.from = curfraze->line.to;
            curfraze->level     = 0x00;
            curfraze->levfl     = 0;
            curfraze->flags     = BLINREAD_X;
            break;
        case '?':
            usage(EX_OK);
        case '=':
            switch((optargs = babolo_getoptsarg(bos))[0]) {
            case '0': blin_ctl(BLIN_CTL_DUMP);     exit(EX_OK);
            case '1': babolo_optest(opts);         exit(EX_OK);
            case '2': babolo_optext(bos);          exit(EX_OK);
            case '3': babolo_dumpopts(bos);        exit(EX_OK);
            case 'T': blin_ctl(BLIN_CTL_TCFN, ++optargs); break;
            default : usage(EX_USAGE);
            }
            break;
        default: usage(EX_USAGE);
        }
        ifBLIN_QX2("flag -%c", fl);
    }
    if  (!curfraze->levfl) curfraze->level = 0xFFU;
    if  (!(tname = babolo_getargs(bos))) isJX(EX_USAGE, "no tracefile");
    if  (0 > mife_ctlfile(cfg->tfile, tname)) isJX(EX_USAGE, "no such tracefile");
    twind = BLINr_next_wind(cfg);
    if  (!twind) isJW(EX_IOERR, "tracefile window %"BLIN_U, cfg->tfile->ofpoint);
    if  (cfg->flags & BLINREAD_P_H) {
        if  (cfg->flags & BLINREAD_P_I) printf("pid ");
        if  (cfg->flags & BLINREAD_P_T) {
            printf("relative_time ");
        } else if (cfg->flags & BLINREAD_P_D) {
            printf("date ");
        }
        if  (cfg->flags & BLINREAD_P_L) printf("lev ");
        if  (cfg->flags & BLINREAD_P_E) printf("erno ");
        if  (cfg->flags & BLINREAD_P_R) printf("prog:");
        if  (cfg->flags & BLINREAD_P_S) printf("file:");
        if  (cfg->flags & BLINREAD_P_U) printf("func:");
        if  (cfg->flags & BLINREAD_P_N) printf("line:");
        printf(" text\n");
    }
    if  (cfg->flags & BLINREAD_Q) {
        ex = BLINr_rq(cfg, twind);
    } else {
        ex = BLINr_rp(cfg, twind);
    }
out:
    mife_fini(cfg->tfile);
    babolo_closeopts(bos);
    if  (!!cfg->time) mular_destroy(cfg->time);
    if  (!!cfg->pid)  mular_destroy(cfg->pid);
    if  (!!cfg->prog) mular_destroy(cfg->prog);
    if  (!!cfg->file) mular_destroy(cfg->file);
    if  (!!cfg->func) mular_destroy(cfg->func);
    if  (!!cfg->line) mular_destroy(cfg->line);
    if  (!!cfg->qbuf) {
        if  (!!cfg->qbuf->pid)  mular_destroy(cfg->qbuf->pid);
        if  (!!cfg->qbuf->prog) mular_destroy(cfg->qbuf->prog);
        if  (!!cfg->qbuf->sun)  mular_destroy(cfg->qbuf->sun);
        free(cfg->qbuf);
    }
    /* free(< >) */
    ifBLIN_QX2("exit");
    exit(ex);
#   undef isJX
#   undef isJW
#   undef blin_internal_flags
}
