/*-
 * Copyright (C)2020 @BABOLO http://www.babolo.ru/
 * PKG = babolo-libmake
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: blin3.c,v 1.3 2020/12/01 18:42:39 nirvana Exp $
 */

#define BLIN_COMPAT VMAJOR
#define BLIN_INTERNAL

#include <sys/types.h>
#include <sys/file.h>
#include <sysexits.h>
#include <strings.h>
#include <sys/uio.h>
#include <string.h>
#include <syslog.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <err.h>
#include "../BLINflag.h"

#define ERROUT(A, B)    do {ex = (A); errno = (B); goto out;} while(0)

__weak_reference(blin_pamparam_internal, blin_internal_flags);
__weak_reference(blin_pamparam_internal, blin_direct_flags);

extern blin_pamparam blin_pamparam_internal;

static void
/*****************************************************************************************************/
blin_init() {                                                                                    /****
 *****************************************************************************************************/
    int i;

    if  (!(blin_pamparam_internal.flags & BLIN_MAININI)) {
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: +blin_init %08X\n"
                   , blin_pamparam_internal.flags
                   )
            ;
        }
        for (i = 0; i < 8; i++) blin_pamparam_internal.w[i].file = stderr;
        blin_pamparam_internal.flags |= BLIN_MAININI;
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: -blin_init %08X\n"
                   , blin_pamparam_internal.flags
                   )
            ;
}   }   }

static void
/*****************************************************************************************************/
doout(int level, char *q) {                                                                      /****
 *****************************************************************************************************/
    u_int32_t ctl = blin_pamparam_internal.w[level & 7].what;

    if  (ctl & BLIN_MODWARN) {
        warnx("%s", q);
    }
    if  (ctl & BLIN_MODPRNT) {
        fprintf(blin_pamparam_internal.w[level & 7].file, "%s", q);
    }
    if  (ctl & BLIN_MODLOGR) {
        syslog(blin_pamparam_internal.w[level & 7].prio, "%s", q);
}   }

static void
/*****************************************************************************************************/
dotime(int level) {                                                                              /****
 *****************************************************************************************************/
    u_int32_t ctl = blin_pamparam_internal.w[level].what;
    struct timespec tp;                              /*      */

    if  (ctl & BLIN_MODTIME) {
        if  (0 > clock_gettime(CLOCK_REALTIME, &tp)) {
            ctl &= ~BLIN_MODTIME;
        } else {
            if  (ctl & BLIN_MODWARN) {
                warnx("%10"BLIN_D".%09ld", tp.tv_sec, tp.tv_nsec);
            }
            if  (ctl & BLIN_MODPRNT) {
                fprintf( blin_pamparam_internal.w[level].file, "%10"BLIN_D".%09ld\n"
                       , tp.tv_sec, tp.tv_nsec
                       );
            }
            if  (ctl & BLIN_MODLOGR) {
                syslog( blin_pamparam_internal.w[level].prio, "%10"BLIN_D".%09ld"
                      , tp.tv_sec, tp.tv_nsec
                      );
}   }   }   }

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_cry1(int ctl, const char *format, ...) {                                                    /****
 *****************************************************************************************************
 *****************************************************************************************************/
    char qq[BLIN_MAXCRY1 + 4];              /*                                      */
    char *buf;                              /*                                                 */
    char *ptr;                              /*  */
    ssize_t avail;                          /*                              */
    ssize_t t;                              /*  */
    ssize_t need = 0;                       /*                                 */
    struct timespec tp;                     /*               */
    u_int32_t control;                      /* what    blin_pamparam_internal */
    int er;                                 /*  errno                                      */
    int ex = EX_OK;                         /*                                              */
    va_list ap;                             /*   va_arg                               */

    er = errno;
    blin_init();
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_cry1 %08X\n", ctl);
    }
    control = blin_pamparam_internal.w[ctl & 7].what;
    if  (control & BLIN_MODTIME) {
        if  (0 > clock_gettime(CLOCK_REALTIME, &tp)) control &= ~BLIN_MODTIME;
    }
    need = 0;
    do {
        va_start(ap, format);
        if  (!need) {
            buf = qq;
            avail = BLIN_MAXCRY1;
        } else {
            buf = malloc(need + 5);
            avail = need + 1;
            need = 0;
            if  (!buf) {
                ifBLIN_QX0("malloc");
                ERROUT(EX_OSERR, ENOMEM);
        }   }
        ptr = buf;
        if  (control & BLIN_MODTIME) {
            t = snprintf(ptr, avail + 1, "%10"BLIN_D".%09ld ", tp.tv_sec, tp.tv_nsec);
            need += t;
            if  (0 > (avail -= t)) avail = 0; else ptr += t;
        }
        t = vsnprintf(ptr, avail + 1, format, ap);
        need += t;
        if  (0 >= (avail -= t)) avail = 0; else ptr += t;
        if  (ctl & 8) {
#           define UPREFIX         "Unknown error"
#           define EBUFSIZE        11

            if  (avail-- > 0) *(ptr++) = ':';
            if  (avail-- > 0) *(ptr++) = ' ';
            need += 2;
            if  ((er <= sys_nerr) && (er >= 0)) {
                for (t = 0; sys_errlist[er][t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = sys_errlist[er][t];
                        --avail;
                    }
                    ++need;
                }
            } else if (avail < (EBUFSIZE + strlen(UPREFIX))) {
                need += EBUFSIZE + strlen(UPREFIX);
                avail = 0;
                if  (er < 0) {
                    ++need;
                    --avail;
                }
            } else {
                char *e;
                int s;

                for (t = 0; UPREFIX[t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = UPREFIX[t];
                        --avail;
                    }
                    ++need;
                }
                e = ptr + EBUFSIZE;
                need += EBUFSIZE;
                avail -= EBUFSIZE;
                if  (er < 0) {
                    s = -er;
                    ++e;
                    ++need;
                    --avail;
                } else {
                    s = er;
                }
                do *--e = "0123456789"[s % 10]; while (s /= 10);
                if (er < 0) *--e = '-';
                while (e > ptr) *--e = ' ';
                ptr += EBUFSIZE;
                if  (er < 0) ++ptr;
        }   }
        if  (control & BLIN_MODEOAU) {
            if  (ptr[-1] != '\n') control |= BLIN_MODEOLF;
        }
        if  (!(~control & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(ptr++) = '\r';
            --avail;
            ++need;
        }
        if  (!(~control & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(ptr++) = '\n';
            --avail;
            ++need;
        }
        va_end(ap);
    } while (avail <= 0);
    *(ptr++) = '\0';
    doout(ctl & 7, buf);
out:
    if  (buf != qq) free(buf);
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_cry1 %d %"BLIN_D"\n", ex, need);
    }
    blin_pamparam_internal.err = errno;
    errno = er;
    return(ex);
}

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_stateheader(blin_statectl staflag, const char actnames[][3]) {                              /****
 *****************************************************************************************************
 *****************************************************************************************************/
    int i, ex = EX_OK;
    u_int32_t ctl;
    ssize_t l, t;
    char *q, *p;
#ifdef __amd64__
    const char *header = "nst rs cl st --rg32-- ----offs----";
#else
    const char *header = "nst rs cl st --rg32-- --offs--";
#endif

    blin_init();
    if  (blin_pamparam_internal.flags & (BLIN_BIT0 << staflag.level)) {
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: +blin_stateheader %08X\n"
                   , staflag.statectl
                   )
            ;
        }
        dotime(staflag.level);
        ctl = blin_pamparam_internal.w[staflag.level].what;
        l = strlen("control-");
        if  (!staflag.allign) {
            for (i = 0; i < 24; ++i) if (actnames[i][0] != '-') l += 3;
        } else {
            l += 3 * staflag.allign;
        }
        l += strlen(header);
        l += 3; /* for \r\n\0 */
            /*     "---"   */
        if  (!(q = calloc(1, l + 3))) {
            ifBLIN_QX0("malloc");
            ERROUT(EX_OSERR, ENOMEM);
        }
        t = l;
        strncpy(q, "control-", t);
            /*  q  '\0'  strncpy()   calloc() */
        p = index(q, 0);
        if  (0 >= (t = q + l - p)) {
            ifBLIN_QX0("strcat #0");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (!staflag.allign) {
            for (i = 0; i < 24; ++i) {
                if  (actnames[i][0] != '-') {
                    *(p++) = ' ';
                    *(p++) = actnames[i][0];
                    *(p++) = actnames[i][1];
                    if  (0 >= (t -= 3)) {
                        ifBLIN_QX0("strcat #1 %d", i);
                        ERROUT(EX_SOFTWARE, EDOOFUS);
            }   }   }
        } else  {
            for (i = 0; i < staflag.allign; ++i) {
                *(p++) = ' ';
                *(p++) = '-';
                *(p++) = '-';
                if  (0 >= (t -= 3)) {
                    ifBLIN_QX0("strcat #2 %d", i);
                    ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }   }
            /*    (++t)      */
        strncat(q, header, ++t);
        p = index(q, 0);
        if  (0 > (t = q + l - p)) {
            ifBLIN_QX0("strcat #3 real %"BLIN_D" > %"BLIN_D" alloc", BLIN_I(p - q), BLIN_I(l));
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (ctl & BLIN_MODEOAU) {
            if  (p[-1] != '\n') ctl |= BLIN_MODEOLF;
        }
        if  (!(~ctl & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(p++) = '\r';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #4");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (!(~ctl & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(p++) = '\n';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #5");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        *(p++) = '\0';
        doout(staflag.level, q);
out:
        if  (q) free(q);
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_stateheader %d\n", ex);
    }   }
    return(ex);
}

int
/*****************************************************************************************************/
/*****************************************************************************************************/
blin_statebody( blin_statectl  staflag         /* level gtequ staten actinn bordsy allign revert     */
              , const char     actnames[][3]   /*                               */
              , const char     statenames[][3] /*                              */
              , const char     clasnname[3]    /*                            */
              , const char    *in              /*                        */
              , size_t         offset          /* 32/48 , ,    */
              , u_int64_t      control         /*                            */
              , u_int32_t      state           /*                                   */
              , u_int32_t      reg             /*   32                  */
              , u_char         byte            /*   8                   */
              ) {                                                                                   /*
 *****************************************************************************************************/
/*****************************************************************************************************/
    u_int64_t C_state = BLIN_MIMASK(staflag.staten);
    u_int64_t flymask, contrana;
    int i, k, ex = EX_OK;
    u_int32_t ctl;
    ssize_t l, t;
    char *q, *p;

    blin_init();
    if  (blin_pamparam_internal.flags & (BLIN_BIT0 << staflag.level)) {
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: +blin_statebody %08X\n"
                   , staflag.statectl
                   )
            ;
        }
        if  (staflag.revert) {
            flymask = (u_int64_t)1 << (staflag.staten + staflag.actinn - 1);
        } else {
            flymask = (u_int64_t)1 << (staflag.staten);
        }
        ctl = blin_pamparam_internal.w[staflag.level].what;
        l = 9;
        if  (!staflag.allign) {
            for (i = 0; i < 24; ++i) if (actnames[i][0] != '-') l += 3;
        } else {
            l += 3 * staflag.allign;
        }
        l += 3 + 3 + 3 + 3 + 9;
#       ifdef __amd64__
        l += 13;
#       else
        l += 9;
#       endif
        l += BLIN_MAXTXTSTA + 3;
        l += 3; /* for \r\n\0 */
        if  (!(q = calloc(1, l + 3))) {
            ifBLIN_QX0("malloc");
            ERROUT(EX_OSERR, ENOMEM);
        }
        p = q + snprintf(q, l, "%08X ", (u_int32_t)control);
        if  (0 >= (t = q + l - p)) {
            ifBLIN_QX0("strcat #0");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (!staflag.allign) {
            for (i = 0, contrana = control, k = 0; i < staflag.actinn; ++i) {
                if  (actnames[i][0] == '-') {
                    if  (contrana & flymask) {
                        ++k;
                    }
                } else {
                    if  (contrana & flymask) {
                        *(p++) = actnames[i][0];
                        *(p++) = actnames[i][1];
                    } else {
                        *(p++) = ' ';
                        *(p++) = ' ';
                    }
                    if  (k && (i == (staflag.actinn - 1))) {
                        *(p++) = '-';
                    } else {
                        *(p++) = ' ';
                    }
                    if  (0 >= (t -= 3)) {
                        ifBLIN_QX0("strcat #1 %d", i);
                        ERROUT(EX_SOFTWARE, EDOOFUS);
                }   }
                if  (staflag.revert) contrana <<= 1; else contrana >>= 1;
            }
        } else {
            for (i = 0, contrana = control, k = 0; i < staflag.actinn; ++i) {
                if  (contrana & flymask) {
                    *(p++) = actnames[i][0];
                    *(p++) = actnames[i][1];
                    *(p++) = ' ';
                    if  (0 >= (t -= 3)) {
                        ifBLIN_QX0("strcat #1 %d", i);
                        ERROUT(EX_SOFTWARE, EDOOFUS);
                    }
                    k++;
                }
                if  (staflag.revert) contrana <<= 1; else contrana >>= 1;
            }
            for (; k < staflag.allign; ++k) {
                *(p++) = ' ';
                *(p++) = ' ';
                *(p++) = ' ';
                if  (0 >= (t -= 3)) {
                    ifBLIN_QX0("strcat #2 %d", i);
                    ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }   }
        p += snprintf( p, t, "%s %02X %s %s %08X %"BLIN_X" ~"
                     , statenames[control & C_state]
                     , byte & 0x00FF
                     , clasnname
                     , statenames[state]
                     , reg
                     , BLIN_I(offset)
                     );
        if  (0 >= (t = q + l - p)) {
            ifBLIN_QX0("strcat #3");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        for (i = 0; i <= BLIN_MAXTXTSTA; ++i) {
            char c;

            if  (staflag.offmax && (i >= offset)) break;
            c = in[i + (staflag.offsus ? offset : 0)];
            if  (!c) break;
            if  (staflag.gtequ ? ((c & 0xFF) < staflag.bordsy) : ((c & 0xFF) == staflag.bordsy)) break;
            *(p++) = c;
            if  (0 > (--t)) {
                ifBLIN_QX0("strcat #4");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (i <= BLIN_MAXTXTSTA) {
            *(p++) = '~';
            --t;
        } else if (0 > t) {
            ifBLIN_QX0("strcat #5");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (ctl & BLIN_MODEOAU) {
            if  (p[-1] != '\n') ctl |= BLIN_MODEOLF;
        }
        if  (!(~ctl & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(p++) = '\r';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #6");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (!(~ctl & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(p++) = '\n';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #7");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        *(p++) = '\0';
        doout(staflag.level, q);
out:
        if  (q) free(q);
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_statebody %d\n", ex);
    }   }
    return(ex);
}

static char
z(u_char u) {
    char s;

    if  (u < ' ') s = '.';
    else if ((u < 0x7F) || (u >= (u_char)'') || (u == (u_char)'') || (u == (u_char)'')) s = u;
    else s = '.';
    return(s);
}

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_dumb(int mode, const void *buf, ssize_t len, ...) {                                         /****
 *****************************************************************************************************
 *****************************************************************************************************/
    char q[BLIN_MAXTXTDUM + BLIN_MAXSTRDUM + 3], *p, *s;
    int i, ex = EX_OK;
    u_int32_t control;
    const u_char *c;
    va_list ap;
    ssize_t t;
    size_t l;

    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_dumb %08X %"BLIN_D"\n", mode, len);
    }
    c = buf;
    dotime(mode & 7);
    control = blin_pamparam_internal.w[mode & 7].what;
    while (len > 0) {
        p = q;
        if  (control & BLIN_MODDSTR) {
            va_start(ap, len);
            s = va_arg(ap, char *);
            if  (BLIN_MAXSTRDUM < (l = strnlen(s, BLIN_MAXSTRDUM + 3))) {
                ifBLIN_QX0("Long label %u > %u", l, BLIN_MAXSTRDUM);
                ERROUT(EX_USAGE, EMSGSIZE);
            }
            bcopy(s, p, l);
            p += l;
            va_end(ap);
        }
        if  (0 >= (t = q + BLIN_MAXTXTDUM - p)) {
            ifBLIN_QX0("strcat label");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (control & BLIN_MODDNON) {
        } else if (control & BLIN_MODDREL) {
            p += snprintf(p, BLIN_MAXTXTDUM, "%04X", (int)(c - (u_char*)buf));
        } else {
            p += snprintf(p, BLIN_MAXTXTDUM, "%"BLIN_X, BLIN_I(c));
        }
        if  (0 >= (t = q + BLIN_MAXTXTDUM - p)) {
            ifBLIN_QX0("strcat #0");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        for (i = 0; i < 16; ++i) {
            if  (i && !(i & 3)) {
                *(p++) = ' ';
                t--;
                if  (0 > t) {
                    ifBLIN_QX0("strcat #1 %d", i);
                    ERROUT(EX_SOFTWARE, EDOOFUS);
            }   }
            if  (i < len) {
                p += snprintf(p, t, " %02X", c[i]);
            } else if ((mode & BLIN_MODDTXT) || (control & BLIN_MODDTXT)) {
                *(p++) = ' ';
                *(p++) = ' ';
                *(p++) = ' ';
            }
            if  (0 >= (t = q + BLIN_MAXTXTDUM - p)) {
                ifBLIN_QX0("strcat #2 %d", i);
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  ((mode & BLIN_MODDTXT) || (control & BLIN_MODDTXT)) {
            *(p++) = ' ';
            *(p++) = '|';
            if  (0 >= (t -= 2)) {
                ifBLIN_QX0("strcat #3");
                ERROUT(EX_SOFTWARE, EDOOFUS);
            }
            for (i = 0; i < 16; ++i) {
                if  (i < len) {
                    *(p++) = z(c[i]);
                    if  (0 >= (--t)) {
                        ifBLIN_QX0("strcat #4");
                        ERROUT(EX_SOFTWARE, EDOOFUS);
            }   }   }
            *(p++) = '|';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #5");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (control & BLIN_MODEOAU) control |= BLIN_MODEOLF;
        if  (!(~control & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(p++) = '\r';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #6");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (!(~control & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(p++) = '\n';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #7");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        *(p++) = '\0';
        --t;
        doout(mode & 7, q);
        if  (len >= 16) {
            c += 16;
            len -= 16;
        } else {
            c += len;
            len = 0;
    }   }
out:
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_dumb %d\n", ex);
    }
    return(ex);
}
