/*-
 * Copyright (C)2020..2021 @BABOLO http://www.babolo.ru/
 * PKG = babolo-libmake
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: blinread.c,v 1.38 2021/07/26 14:00:41 cuhegh Exp $
 */

#define Bpars_COMPAT 4
#define MULAR_COMPAT 0
#define MIFE_COMPAT 5
#define BLIN_COMPAT VMAJOR
#define BLIN_INTERNAL
#define BLIN_INTERNAL

#include <sys/time.h>
#include <sys/event.h>
#include <sys/types.h>
#include <sys/file.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <limits.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <err.h>
#include <sysexits.h>
#include "../BLINflag.h"
#include <babolo/parser.h>
#include <mife.h>
#include <multilar.h>

#define SPASES "                                "  "                                "                 \
               "                                "  "                                "                 \
               "                                "  "                                "                 \
               "                                "  "                                "    /* 256 */

#define BLINr_mursiz (&((size_t[]){1024, 512}[0]))

typedef struct {
    struct {
        u_int64_t from : 63
                , abs  : 1;
    };
    struct {
        u_int64_t to   : 63
                , nrng : 1;
    };
} BLINr_time;

typedef struct {
    u_int32_t     lnfr;
    struct {
        u_int32_t lnto : 31
                , nrng  : 1;
    };
} BLINr_line;

typedef struct {
    u_int32_t from;
    u_int32_t to;
} BLINr_RANGE;

typedef struct BLINr_fraze BLINr_fraze;

struct BLINr_fraze{
    BLINr_fraze *next;
    BLINr_RANGE  time;
    BLINr_RANGE  pid;
    BLINr_RANGE  prog;
    BLINr_RANGE  file;
    BLINr_RANGE  func;
    BLINr_RANGE  line;
    u_int8_t     level;
    u_int8_t     flags;
#   define BLINREAD_X   0x80U    /*               */
#   define BLINREAD_L   0x40U    /*              */
#   define BLINREAD_A   0x20U    /*             */
#   define BLINREAD_I   0x10U    /*    pid'           */
#   define BLINREAD_R   0x08U    /*      */
#   define BLINREAD_S   0x04U    /*          */
#   define BLINREAD_U   0x02U    /*      */
#   define BLINREAD_N   0x01U    /*        */
#   define BLINREAD_M   0x7FU    /*                           */
};

typedef struct BLINr_cfg {
    BLIN_flag  flags;
#   define BLINREAD_P_H   0x00000200U    /*                 ('-p' h) */
#   define BLINREAD_P_L   0x00000100U    /*           ('-p' l) */
#   define BLINREAD_P_A   0x00000080U    /*        ('-p' a) */
#   define BLINREAD_P_T   0x00000040U    /*     ('-p' t) */
#   define BLINREAD_P_I   0x00000020U    /*   pid'                  ('-p' i) */
#   define BLINREAD_P_R   0x00000010U    /*            ('-p' r) */
#   define BLINREAD_P_S   0x00000008U    /*                ('-p' s) */
#   define BLINREAD_P_U   0x00000004U    /*            ('-p' u) */
#   define BLINREAD_P_N   0x00000002U    /*              ('-p' n) */
#   define BLINREAD_P_E   0x00000001U    /*   errno                  ('-p' e) */
#   define BLINREAD_PMASC 0x000003FFU
#   define BLINREAD_Q     0x00800000U    /*      tracfile (-Q) */
#   define BLINREAD_HL    0x00010000U    /*    (   ) */

    mife_descriptor  *tfile;    /* tracfile */
    BLINr_fraze *filt;          /*  */
    mular_descriptor *time;     /*  */
    mular_descriptor *pid;      /*  */
    mular_descriptor *prog;     /*  */
    mular_descriptor *file;     /*  */
    mular_descriptor *func;     /*  */
    mular_descriptor *line;     /*  */
} BLINr_cfg;

typedef struct BLIN_qbuf_list BLIN_qbuf_list;

struct BLIN_qbuf_list{  /*****/
    BLIN_qbuf_list *next;
    void           *list;
    const char     *name;
};

typedef struct BLIN_list_pc BLIN_list_pc;

struct BLIN_list_pc{  /*****/
    BLIN_list_pc *next;
    const char     *name;
};

typedef struct BLIN_list_u4 BLIN_list_u4;

struct BLIN_list_u4{  /*****/
    BLIN_list_u4 *next;
    u_int32_t    num;
};

struct {  /*****/
    u_int64_t       time1;
    u_int64_t       time2;
    BLIN_list_u4   *pid;
    BLIN_list_pc   *prog;
    BLIN_qbuf_list *file;
    u_int8_t        level;
} qbuf = {0, 0, NULL, NULL, NULL, 0};

u_int64_t pow10(int8_t num) {
    u_int64_t ret;
    for (ret = 1; num > 0; num--) ret *= 10;
    return(ret);
}

static void
/*****************************************************************************************************
                                                                                                  ****/
usage(int ex) {                                                                                  /****
                                                                                                  ****
 *****************************************************************************************************/
    fprintf( stderr
           , PROG" @BABOLO V.M "VERS" http://www.babolo.ru/\n"
             "Usage: "PROG" [-options]  tracfile\n"
             "   -c TEXT         read from TEXT\n"
             "   -Q              give information about tracefile\n"
             "   -l MASK         filter by message level\n"
             "   -a NUM[-NUM]    filter by message time\n"
             "   -t NUM[-NUM]    filter by relative message time\n"
             "                    also in -at instead of int NUM could use float NUM (sec.nsec)\n"
             "   -i NUM          filter by pid\n"
             "   -r TEXT         filter by program name\n"
             "   -s TEXT         filter by file name\n"
             "   -u TEXT         filter by procedure name\n"
             "   -n NUM[-NUM]    filter by line num\n"
             "   -LATIRSUN       invert term of its kind\n"
             "   -x              erase term\n"
             "   -e              end of expression\n"
             "   -p <options>    What part of message to print\n"
             "          h            head\n"
             "          l            level\n"
             "          a            absolute time\n"
             "          t            relative time\n"
             "          i            pid\n"
             "          r            program name\n"
             "          s            file name\n"
             "          u            procedure name\n"
             "          n            line num\n"
             "          e            errno\n"
             "          x            text only\n"
             "   -v[v[v...]]     verbose\n"
             "   -q              silent\n"
             "   -= MODE         internal diagnostic\n"
             "   -?              this text\n"
           )
    ;
    ifBLIN_QX2("exit");
    exit(ex);
}

BLIN_trace* next_wind(BLINr_cfg *cfg) {
#   define isJX(FORMAT, ...) do { ifBLIN_QX0("!!" FORMAT, ## __VA_ARGS__); return(NULL); }while(0)
#   define isJW(FORMAT, ...) do { ifBLIN_QW0("!!" FORMAT, ## __VA_ARGS__); return(NULL); }while(0)
#   define blin_internal_flags (cfg->flags & BLIN_MASK)
    off_t       offset = cfg->tfile->ofpoint;           /*                */
    BLIN_trace *twind = mife_pointer(cfg->tfile);       /*   tracfile                 */
    u_int32_t   windsize = (twind ? twind->len : 0);    /*     tracfile */

    if  (!windsize) {
        twind = mife_window(cfg->tfile, 0, sizeof(u_int32_t));
        if  (!twind)    isJW("mife_window");
    }
    offset += windsize;
    windsize = *((u_int32_t*) ((char*)twind + windsize));
    twind = mife_window(cfg->tfile, offset, windsize + sizeof(u_int32_t));
    if  (!twind)    isJW("mife_window");
    if  (mife_ctlsize(cfg->tfile) < windsize) {
        ifBLIN_QX1("!less than asked");
        return(NULL);
    }
    return(twind);
#   undef isJX
#   undef isJW
#   undef blin_internal_flags
}

int main(int argc, char *argv[]) {
#   define isJX(EXCODE, FORMAT, ...) \
        do { ifBLIN_QX0("!!" FORMAT, ## __VA_ARGS__); ex = EXCODE; goto out; }while(0)
#   define isJW(EXCODE, FORMAT, ...) \
        do { ifBLIN_QW0("!!" FORMAT, ## __VA_ARGS__); ex = EXCODE; goto out; }while(0)
#   define blin_internal_flags (cfg->flags & BLIN_MASK)
    const char *const opts = "Aa:c:eIi:Ll:Nn:p:QqRr:Ss:Tt:Uu:vx?=:";
    babolo_opts      *bos = NULL;              /*             */
    char              fl;                      /*                                       */
    const char       *optargs;                 /*                              */
    int               ex = EX_OK;              /*                                */
    BLINr_fraze      *curfraze;                /*                           */
    const char       *cname = NULL;            /*                             */
    mife_descriptor  *cfile = NULL;            /*                                 */
    char             *cchar = NULL;            /*                       */
    babolo_parm      *bp;                      /*   -                          */
    const char       *tname;                   /*  tracfile                              */
    BLIN_trace       *twind;                   /*   tracfile                           */
    off_t             offset = 0;              /*                          */
    u_int64_t         fstime;                  /*    tracfile                   */
    u_int64_t         m;                       /* ޣ,              */
    u_int32_t         p;                       /* ޣ,              */
    time_t            hhtime;                  /*    tm                    */
    struct tm        *htime = NULL;            /*       ISO */
    BLINr_cfg        *cfg = calloc(1, sizeof(BLINr_cfg));

    cfg->flags = BLIN_BIT0 | BLINREAD_P_H | BLINREAD_P_L | BLINREAD_P_A | BLINREAD_P_I
               | BLINREAD_P_R | BLINREAD_P_S | BLINREAD_P_U | BLINREAD_P_N;
    curfraze = cfg->filt = malloc(sizeof(BLINr_fraze));
    cfg->time = mular_create(MULAR_OBJ | MULAR_STRI, 2, sizeof(BLINr_time), BLINr_mursiz);
    cfg->pid  = mular_create(MULAR_OBJ | MULAR_STRI, 2, sizeof(u_int32_t) , BLINr_mursiz);
    cfg->prog = mular_create(MULAR_OBJ | MULAR_STRI | MULAR_PFRE, 2, sizeof(char*), BLINr_mursiz);
    cfg->file = mular_create(MULAR_OBJ | MULAR_STRI | MULAR_PFRE, 2, sizeof(char*), BLINr_mursiz);
    cfg->func = mular_create(MULAR_OBJ | MULAR_STRI | MULAR_PFRE, 2, sizeof(char*), BLINr_mursiz);
    cfg->line = mular_create(MULAR_OBJ | MULAR_STRI, 2, sizeof(BLINr_line), BLINr_mursiz);
    cfg->tfile = mife_init(MIFE_FULL);
    bos = babolo_openopts(Bpars_SYME | Bpars_ESYM, 'Z');
    if  (!curfraze)   isJW(EX_OSERR, "curfraze malloc");
    if  (!cfg->time)  isJW(EX_OSERR, "mular_create time");
    if  (!cfg->pid)   isJW(EX_OSERR, "mular_create pid");
    if  (!cfg->prog)  isJW(EX_OSERR, "mular_create prog");
    if  (!cfg->file)  isJW(EX_OSERR, "mular_create file");
    if  (!cfg->func)  isJW(EX_OSERR, "mular_create func");
    if  (!cfg->line)  isJW(EX_OSERR, "mular_create line");
    if  (!cfg->tfile) isJW(EX_OSERR, "mife_init");
    if  (!bos)        isJW(EX_OSERR, "babolo_openopts");
    if  (babolo_setopts(bos, Bpars_NONU, argc, argv, opts))    isJW(EX_OSERR, "babolo_setopts");
    *curfraze = (BLINr_fraze){NULL, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, 0, BLINREAD_X};
    cfg->time->tofree = cfg->pid->tofree = cfg->prog->tofree =
    cfg->file->tofree = cfg->func->tofree = cfg->line->tofree = free; /* func */
    while(!!(fl = babolo_getopts(bos))) {
        ifBLIN_QX3("flag %02X", fl);
        switch(fl) {
        case 'A':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_A;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'a': {
                ifBLIN_QX2("+flag %c", fl);
                const char *mem;
                BLINr_time *time = mular_insert(cfg->time, curfraze->time.to++);

                m = 0;
                if  (!time)    isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->time.to - 1);
                if  (!(optargs = babolo_getoptsarg(bos)))    isJW(EX_DATAERR, "flag %c args", fl);
                time->abs = 1;
                m = strtoul(optargs, (char**) &optargs, 0);
                if  (*optargs == '.') {
                    mem = ++optargs;
                    time->from = m * pow10(9);
                    time->from += strtoul(optargs, (char**) &optargs, 10) * pow10(9 - (optargs - mem));
                } else {
                    time->from = m;
                }
                if  (*optargs == '-') {
                    time->nrng = 0;
                    optargs++;
                    m = strtoul(optargs, (char**) &optargs, 0);
                    if  (*optargs == '.') {
                        mem = ++optargs;
                        time->to = m * pow10(9);
                        time->to += strtoul(optargs, (char**)&optargs, 10) * pow10(9 -(optargs - mem));
                    } else {
                        time->to = m;
                    }
                } else {
                    time->nrng = 1;
                }
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'c':
            ifBLIN_QX2("+flag %c", fl);
            cfile = mife_init(MIFE_FULL);
            cname = babolo_getoptsarg(bos);
            if  (!cfile)    isJW(EX_OSERR, "flag %c mife_init", fl);
            if  (!cname)    isJW(EX_DATAERR, "flag %c args", fl);
            if  (0 > mife_ctlfile(cfile, cname))    isJW(EX_IOERR, "flag %c mife_ctl %s", fl, cname);
            if  (0 > mife_read(cfile, 0, 0))    isJW(EX_IOERR, "flag %c mife_read", fl);
            cchar = mife_window(cfile, 0, 0);
            if  (!cchar)    isJW(EX_IOERR, "flag %c mife_window", fl);
            if  (!(cfile->flags & MIFE_EOFL))    isJW(EX_USAGE, "flag %c file too big", fl);
            bp = babolo_getparm(Bpars_NOEN | Bpars_NONU, &cchar, "", 0);
            if  (!bp)    isJW(EX_DATAERR, "flag %c no parm %s", fl, cchar);
            mife_fini(cfile);
            if  (babolo_setparm(bos, 0, bp, NULL))   isJX(EX_USAGE, "flag %c no opts %s", fl, cname);
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'e':
            ifBLIN_QX2("+flag %c", fl);
            if  (!(curfraze->flags & BLINREAD_HL)) curfraze->level = ~(u_int8_t)0;
            curfraze->flags &= ~BLINREAD_HL;
            if  (!(curfraze->next = malloc(sizeof(BLINr_fraze))))  isJW(EX_OSERR, "flag %c malloc", fl);
            curfraze->next->next = NULL;
            curfraze->next->time.from = curfraze->next->time.to = curfraze->time.to;
            curfraze->next->pid.from  = curfraze->next->pid.to  = curfraze->pid.to ;
            curfraze->next->prog.from = curfraze->next->prog.to = curfraze->prog.to;
            curfraze->next->file.from = curfraze->next->file.to = curfraze->file.to;
            curfraze->next->func.from = curfraze->next->func.to = curfraze->func.to;
            curfraze->next->line.from = curfraze->next->line.to = curfraze->line.to;
            curfraze->next->level = 0;
            curfraze->next->flags = 0;
            curfraze = curfraze->next;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'I':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_I;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'i': {
                ifBLIN_QX2("+flag %c", fl);
                u_int32_t *pid = mular_insert(cfg->pid, curfraze->pid.to++);

                if  (!pid)    isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->pid.to - 1);
                *pid = babolo_getoptunum(bos, 10);
                if  (errno)    isJW(EX_USAGE, "flag %c wrong arg", fl);
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'L':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_L;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'l': {
                ifBLIN_QX2("+flag %c", fl);
                u_int8_t optunum = babolo_getoptunum(bos, 16);

                if  (errno)    isJW(EX_USAGE, "flag %c wrong arg", fl);
                curfraze->level |= optunum;
                curfraze->flags |= BLINREAD_HL;
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'N':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_N;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'n': {
                ifBLIN_QX2("+flag %c", fl);
                BLINr_line *line = mular_insert(cfg->line, curfraze->line.to++);

                if  (!line)    isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->line.to - 1);
                if  (!(optargs = babolo_getoptsarg(bos)))    isJW(EX_USAGE, "flag %c args", fl);
                line->lnfr = strtoul(optargs, (char**) &optargs, 0);
                if  (*optargs == '-') {
                    line->nrng = 0;
                    optargs++;
                    line->lnto = strtoul(optargs, NULL, 0);
                } else {
                    line->nrng = 1;
                }
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'p': {
                ifBLIN_QX2("+flag %c", fl);
                m = 0;
                if  (!(optargs = babolo_getoptsarg(bos)))    isJW(EX_USAGE, "flag %c args", fl);
                for (u_int8_t i = 0; i < strlen(optargs); i++) {
                    switch(optargs[i]) {
                    case 'h': m |= BLINREAD_P_H; break;
                    case 'l': m |= BLINREAD_P_L; break;
                    case 'a': m |= BLINREAD_P_A; break;
                    case 't': m |= BLINREAD_P_T; break;
                    case 'i': m |= BLINREAD_P_I; break;
                    case 'r': m |= BLINREAD_P_R; break;
                    case 's': m |= BLINREAD_P_S; break;
                    case 'u': m |= BLINREAD_P_U; break;
                    case 'n': m |= BLINREAD_P_N; break;
                    case 'e': m |= BLINREAD_P_E; break;
                    case 'x': m = 0; i = strlen(optargs); break;
                    default:    isJX(EX_USAGE, "flag %c wrong arg", fl);
                }   }
                cfg->flags = (cfg->flags & ~BLINREAD_PMASC) | m;
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'Q':
            ifBLIN_QX2("+flag %c", fl);
            cfg->flags |= BLINREAD_Q;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'q': BLIN_QUIET(cfg->flags);
            break;
        case 'R':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_R;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'r': {
                ifBLIN_QX2("+flag %c", fl);
                const char **prog = mular_insert(cfg->prog, curfraze->prog.to++);

                if  (!prog)    isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->prog.to - 1);
                if  (!(*prog = babolo_getoptsarg(bos)))    isJW(EX_USAGE, "flag %c args", fl);
                if  (!(*prog = strdup(*prog)))    isJW(EX_OSERR, "flag %c strdup", fl);
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'S':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_S;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 's': {
                ifBLIN_QX2("+flag %c", fl);
                const char **file = mular_insert(cfg->file, curfraze->file.to++);

                if  (!file)    isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->file.to - 1);
                if  (!(*file = babolo_getoptsarg(bos)))    isJW(EX_USAGE, "flag %c args", fl);
                if  (!(*file = strdup(*file)))    isJW(EX_OSERR, "flag %c strdup", fl);
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'T':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_A;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 't': {
                ifBLIN_QX2("+flag %c", fl);
                const char *mem;
                BLINr_time *time = mular_insert(cfg->time, curfraze->time.to++);

                m = 0;
                if  (!time)    isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->time.to - 1);
                if  (!(optargs = babolo_getoptsarg(bos)))    isJW(EX_USAGE, "flag %c args", fl);
                time->abs = 0;
                m = strtoul(optargs, (char**) &optargs, 0);
                if  (*optargs == '.') {
                    mem = ++optargs;
                    time->from = m * pow10(9);
                    time->from += strtoul(optargs, (char**) &optargs, 10) * pow10(9 - (optargs - mem));
                } else {
                    time->from = m;
                }
                if  (*optargs == '-') {
                    time->nrng = 0;
                    optargs++;
                    m = strtoul(optargs, (char**) &optargs, 0);
                    if  (*optargs == '.') {
                        mem = ++optargs;
                        time->to = m * pow10(9);
                        time->to += strtoul(optargs, (char**)&optargs, 10) * pow10(9 -(optargs - mem));
                    } else {
                        time->to = m;
                    }
                } else {
                    time->nrng = 1;
                }
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'U':
            ifBLIN_QX2("+flag %c", fl);
            curfraze->flags ^= BLINREAD_U;
            ifBLIN_QX2("-flag %c", fl);
            break;
        case 'u': {
                ifBLIN_QX2("+flag %c", fl);
                const char **func = mular_insert(cfg->func, curfraze->func.to++);

                if  (!func)    isJW(EX_OSERR, "flag %c mular_insert %u", fl, curfraze->func.to - 1);
                if  (!(*func = babolo_getoptsarg(bos)))    isJW(EX_USAGE, "flag %c args", fl);
                if  (!(*func = strdup(*func)))    isJW(EX_OSERR, "flag %c strdup", fl);
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case 'v': BLIN_VERBOSE(cfg->flags);
            break;
        case 'x': {
                ifBLIN_QX2("+flag %c", fl);
                curfraze->time.from = curfraze->time.to;
                curfraze->pid.from  = curfraze->pid.to;
                curfraze->prog.from = curfraze->prog.to;
                curfraze->file.from = curfraze->file.to;
                curfraze->func.from = curfraze->func.to;
                curfraze->line.from = curfraze->line.to;
                curfraze->level     = 0x00;
                curfraze->flags     = BLINREAD_X;
                ifBLIN_QX2("-flag %c", fl);
            }
            break;
        case '?':
            usage(EX_OK);
        case '=':
            ifBLIN_QX2("+flag %c", fl);
            switch((optargs = babolo_getoptsarg(bos))[0]) {
            case '0': blin_ctl(BLIN_CTL_DUMP);     exit(EX_OK);
            case '1': babolo_optest(opts);         exit(EX_OK);
            case '2': babolo_optext(bos);          exit(EX_OK);
            case '3': babolo_dumpopts(bos);        exit(EX_OK);
            case 'T': blin_ctl(BLIN_CTL_TCFN, ++optargs);    break;
            default : usage(EX_USAGE);
            }
            ifBLIN_QX2("-flag %c", fl);
            break;
        default: usage(EX_USAGE);
    }   }
    if  (!(curfraze->flags & BLINREAD_HL)) curfraze->level = ~(u_int8_t)0;
    curfraze->flags &= ~BLINREAD_HL;
    if  (!(tname = babolo_getargs(bos)))    isJX(EX_USAGE, "no tracefile");
    if  (0 > mife_ctlfile(cfg->tfile, tname))    isJX(EX_USAGE, "no such tracefile");
    twind = next_wind(cfg);
    fstime = twind->time;
    if  (cfg->flags & BLINREAD_P_H) {
        if  (cfg->flags & BLINREAD_P_I) printf("pid ");
        if  (cfg->flags & BLINREAD_P_T) printf("relative_time ");
        else if  (cfg->flags & BLINREAD_P_A) printf("absolute_time ");
        if  (cfg->flags & BLINREAD_P_L) printf("lev ");
        if  (cfg->flags & BLINREAD_P_E) printf("erno ");
        if  (cfg->flags & BLINREAD_P_R) printf("prog:");
        if  (cfg->flags & BLINREAD_P_S) printf("file:");
        if  (cfg->flags & BLINREAD_P_U) printf("func:");
        if  (cfg->flags & BLINREAD_P_N) printf("line:");
        printf(" text\n");
    }
    for (u_int8_t truvar, termvar; 1; twind = next_wind(cfg)) {  /*****/
        if  (!twind)    isJW(EX_IOERR, "tracefile mife_window %"BLIN_U, cfg->tfile->ofpoint);
        ifBLIN_QX3( "twind={ %u\n"
                    "        %u\n"
                    "        %u\n"
                    "        %"BLIN_U"\n"
                    "        %u\n"
                    "        %u\n"
                    "        %u\n"
                    "        %u\n"
                    "        %u\n"
                    "        %.*s\n}\n"
                  , twind->len
                  , twind->level
                  , twind->pid
                  , twind->time
                  , twind->erno
                  , twind->line
                  , twind->snprog
                  , twind->snfile
                  , twind->snfunc
                  , (int) (twind->len - offsetof(BLIN_trace, string))
                  , twind->string
                  )
        ;
        if  (cfg->flags & BLINREAD_Q) {
            if  (cfg->flags & BLINREAD_P_L) {
                qbuf.level |= (0x01 << twind->level);
                ifBLIN_QX3("qbuf.level=%02x", qbuf.level);
            }
            if  (cfg->flags & BLINREAD_P_A) {
                if  (twind->time < qbuf.time1) qbuf.time1 = twind->time;
                else if(qbuf.time2 < twind->time) qbuf.time2 = twind->time;
                ifBLIN_QX3("qbuf.time={%"BLIN_U" %"BLIN_U"}", qbuf.time1, qbuf.time2);
            }
            if  (qbuf.pid) {
                for (BLIN_list_u4 *i = qbuf.pid; i->num != twind->pid; i = i->next) {
                    if  (!i->next) {
                        i->next = malloc(sizeof(BLIN_list_u4));
                        if  (!i->next)    isJW(1, "qbuf pid malloc");
                        i->next->num = twind->pid;
                        i->next->next = 0;
                        ifBLIN_QX3("qbuf new pid %u", i->next->num);
                        break;
                }   }
            }
            if  (qbuf.prog) {
                for ( BLIN_list_pc *i = qbuf.prog
                    ; strncmp(i->name, &twind->string[0], twind->snprog)
                    ; i = i->next
                    )
                {
                    if  (!i->next) {
                        i->next = malloc(sizeof(BLIN_list_pc));
                        if  (!i->next)    isJW(EX_OSERR, "qbuf prog malloc");
                        i->next->name = strndup(&twind->string[0], twind->snprog);
                        if  (!i->next->name)    isJW(EX_OSERR, "qbuf prog strndup");
                        i->next->next = 0;
                        ifBLIN_QX3("qbuf new prog %s", i->next->name);
                        break;
                }   }
            }
            if  (qbuf.file) {  /*****/
                BLIN_qbuf_list *file;
                BLIN_qbuf_list *func;
                BLIN_list_u4   *line;
                BLIN_qbuf_list *qa;
                BLIN_qbuf_list *qam;
                BLIN_list_u4   *qb;
                BLIN_list_u4   *qbm;

                file = malloc(sizeof(BLIN_qbuf_list));
                if  (!file)    isJW(EX_OSERR, "qbuf init file malloc");
                file->next = NULL;
                file->name = strndup(&twind->string[twind->snprog], twind->snfile);
                if  (!file->name)    isJW(EX_OSERR, "qbuf init file strndup");
                file->list = func = malloc(sizeof(BLIN_qbuf_list));
                if  (!func)    isJW(EX_OSERR, "qbuf init func malloc");
                func->next = NULL;
                func->name = strndup(&twind->string[twind->snprog + twind->snfile], twind->snfunc);
                if  (!func->name)    isJW(EX_OSERR, "qbuf init func strndup");
                func->list = line = malloc(sizeof(BLIN_list_u4));
                if  (!line)    isJW(EX_OSERR, "qbuf init line malloc");
                line->next = NULL;
                line->num  = twind->line;
                for ( qam = qa = qbuf.file
                    ; (qa->next) && (strncmp(qa->name, file->name, twind->snfile) < 0)
                    ; qa = (qam = qa)->next
                    ) {}
                ifBLIN_QX3( "strncmp(%s, %s, %u) == %d"
                          , qa->name
                          , file->name
                          , twind->snfile
                          , strncmp(qa->name, file->name, twind->snfile)
                          )
                ;
                if  (strncmp(qa->name, file->name, twind->snfile)) {
                    if  (qa->next) {
                        file->next = qam->next;
                        qam->next = file;
                    } else qa->next = file;
                    ifBLIN_QX3("qbuf new file %s", file->name);
                    goto fqba;
                }
                for ( qam = qa = qa->list
                    ; (qa->next) && (strncmp(qa->name, func->name, twind->snfunc) < 0)
                    ; qa = (qam = qa)->next
                    ) {}
                if  (strncmp(qa->name, func->name, twind->snfunc)) {
                    if  (qa->next) {
                        func->next = qam->next;
                        qam->next = func;
                    } else qa->next = func;
                    ifBLIN_QX3("qbuf new func %s", func->name);
                    goto fqba;
                }
                for (qbm = qb = qa->list; (qb->next) && (qb->num < line->num); qb = (qbm = qb)->next){}
                if  (qb->num != line->num) {
                    if  (qb->next) {
                        line->next = qbm->next;
                        qbm->next = line;
                    } else qb->next = line;
                    ifBLIN_QX3("qbuf new line %u", line->num);
                    goto fqba;
                }
                free(file);
                free(func);
                free(line);
                fqba:;
            }
            if  (offset == 0) {
                if  (cfg->flags & (BLINREAD_P_A | BLINREAD_P_T)) {
                    qbuf.time1 = qbuf.time2 = twind->time;
                    ifBLIN_QX3("qbuf.time={%"BLIN_U" %"BLIN_U"}", qbuf.time1, qbuf.time2);
                }
                if  (cfg->flags & BLINREAD_P_I) {
                    qbuf.pid = malloc(sizeof(BLIN_list_u4));
                    if  (!qbuf.pid)    isJW(EX_OSERR, "qbuf init pid malloc");
                    qbuf.pid->next = NULL;
                    qbuf.pid->num = twind->pid;
                    ifBLIN_QX3("qbuf.pid=%u", qbuf.pid->num);
                }
                if  (cfg->flags & BLINREAD_P_R) {
                    qbuf.prog = malloc(sizeof(BLIN_list_pc));
                    if  (!qbuf.prog)    isJW(EX_OSERR, "qbuf init prog malloc");
                    qbuf.prog->next = NULL;
                    qbuf.prog->name = strndup(&twind->string[0], twind->snprog);
                    if  (!qbuf.prog->name)    isJW(EX_OSERR, "qbuf init prog strndup");
                    ifBLIN_QX3("qbuf.prog=%s", qbuf.prog->name);
                }
                if  (cfg->flags & (BLINREAD_P_S | BLINREAD_P_U | BLINREAD_P_N)) {
                    BLIN_qbuf_list *func;
                    BLIN_list_u4   *line;

                    qbuf.file = malloc(sizeof(BLIN_qbuf_list));
                    if  (!qbuf.file)    isJW(EX_OSERR, "qbuf init file malloc");
                    qbuf.file->next = NULL;
                    qbuf.file->name = strndup(&twind->string[twind->snprog], twind->snfile);
                    if  (!qbuf.file->name)    isJW(EX_OSERR, "qbuf init file strndup");
                    qbuf.file->list = func = malloc(sizeof(BLIN_qbuf_list));
                    if  (!func)    isJW(EX_OSERR, "qbuf init func malloc");
                    func->next = NULL;
                    func->name = strndup(&twind->string[twind->snprog + twind->snfile], twind->snfunc);
                    if  (!func->name)    isJW(EX_OSERR, "qbuf init func strndup");
                    func->list = line = malloc(sizeof(BLIN_list_u4));
                    if  (!line)    isJW(EX_OSERR, "qbuf init line malloc");
                    line->next = NULL;
                    line->num  = twind->line;
                    ifBLIN_QX3("qbuf.file={%s %s %u}", qbuf.file->name, func->name, line->num);
            }   }
        } else {
            for (curfraze = cfg->filt; curfraze; curfraze = curfraze->next) {
                ifBLIN_QX3("curfraze3 %X", curfraze);
                if  (curfraze->flags & BLINREAD_X) {
                    truvar = BLINREAD_M;
                    termvar = 0;
                }
                if  (curfraze->time.from == curfraze->time.to) {
                    if  (!(termvar & BLINREAD_A)) truvar |= BLINREAD_A;
                } else {
                    truvar &= ~BLINREAD_A;
                    termvar |= BLINREAD_A;
                    p = 0;
                    for ( BLINr_time *i = mular_getix(cfg->time, p++)
                        ; p <= curfraze->time.to
                        ; i = mular_getix(cfg->time, p++)
                        )
                    {
                        if  (!i)    isJW(1, "check time mular_getix");
                        if  ( i->nrng
                            ? (i->from + (i->abs ? 0 : fstime)) == twind->time
                            :   ((i->from + (i->abs ? 0 : fstime)) <= twind->time)
                              && (twind->time <= (i->to + (i->abs ? 0 : fstime)))
                            )
                        {
                            truvar |= BLINREAD_A;
                            break;
                }   }   }
                if  (curfraze->pid.from == curfraze->pid.to) {
                    if  (!(termvar & BLINREAD_I)) truvar |= BLINREAD_I;
                } else {
                    truvar &= ~BLINREAD_I;
                    termvar |= BLINREAD_I;
                    p = 0;
                    for ( u_int32_t *i = mular_getix(cfg->pid, p++)
                        ; p <= curfraze->pid.to
                        ; i = mular_getix(cfg->pid, p++)
                        )
                    {
                        if  (!i)    isJW(1, "check pid mular_getix");
                        if  (*i == twind->pid) {
                            truvar |= BLINREAD_I;
                            break;
                }   }   }
                if  (curfraze->prog.from == curfraze->prog.to) {
                    if  (!(termvar & BLINREAD_R)) truvar |= BLINREAD_R;
                } else {
                    truvar &= ~BLINREAD_R;
                    termvar |= BLINREAD_R;
                    p = 0;
                    for ( const char **i = mular_getix(cfg->prog, p++)
                        ; p <= curfraze->prog.to
                        ; i = mular_getix(cfg->prog, p++)
                        )
                    {
                        if  (!i)    isJW(1, "check prog mular_getix");
                        if  (  (strlen(*i) == twind->snprog)
                            && !strncmp(*i, &twind->string[0], twind->snprog)
                            )
                        {
                            truvar |= BLINREAD_R;
                            break;
                }   }   }
                if  (curfraze->file.from == curfraze->file.to) {
                    if  (!(termvar & BLINREAD_S)) truvar |= BLINREAD_S;
                } else {
                    truvar &= ~BLINREAD_S;
                    termvar |= BLINREAD_S;
                    ifBLIN_QX3( "%u %.*s"
                              , twind->snfile
                              , (int) twind->snfile
                              , &twind->string[twind->snprog]
                              )
                    ;
                    p = 0;
                    for ( const char **i = mular_getix(cfg->file, p++)
                        ; p <= curfraze->file.to
                        ; i = mular_getix(cfg->file, p++)
                        )
                    {
                        if  (!i)    isJW(1, "check file mular_getix");
                        ifBLIN_QX3("check file mular %zd %s", strlen(*i), *i);
                        if  (  (strlen(*i) == twind->snfile)
                            && (strncmp(*i, &twind->string[twind->snprog], twind->snfile) != 0)
                            )
                        {
                            truvar |= BLINREAD_S;
                            break;
                }   }   }
                if  (curfraze->func.from == curfraze->func.to) {
                    if  (!(termvar & BLINREAD_U)) truvar |= BLINREAD_U;
                } else {
                    truvar &= ~BLINREAD_U;
                    termvar |= BLINREAD_U;
                    p = 0;
                    for ( const char **i = mular_getix(cfg->func, p++)
                        ; p <= curfraze->func.to
                        ; i = mular_getix(cfg->func, p++)
                        )
                    {
                        if  (!i)    isJW(1, "check func mular_getix");
                        if  (  (strlen(*i) == twind->snfunc)
                            && !strncmp( *i
                                       , &twind->string[twind->snprog + twind->snfile]
                                       , twind->snfunc
                            )          )
                        {
                            truvar |= BLINREAD_U;
                            break;
                }   }   }
                if  (curfraze->line.from == curfraze->line.to) {
                    if  (!(termvar & BLINREAD_N)) truvar |= BLINREAD_N;
                } else {
                    truvar &= ~BLINREAD_N;
                    termvar |= BLINREAD_N;
                    p = 0;
                    for ( BLINr_line *i = mular_getix(cfg->line, p++)
                        ; p <= curfraze->line.to
                        ; i = mular_getix(cfg->line, p++)
                        )
                    {
                        if  (!i)    isJW(1, "check line mular_getix");
                        if  ( i->nrng
                            ? i->lnfr == twind->line
                            : (i->lnfr <= twind->line) && (twind->line <= i->lnto)
                            )
                        {
                            truvar |= BLINREAD_N;
                            break;
                }   }   }
                if  (curfraze->level & (0x01 << twind->level)) truvar |= BLINREAD_L;
                truvar ^= curfraze->flags & BLINREAD_M;
                if  (truvar == BLINREAD_M) break;
            }
            if  (truvar == BLINREAD_M) {
                hhtime = twind->time / pow10(9);
                htime = localtime((const time_t*)&hhtime);

                if  (cfg->flags & BLINREAD_P_I) printf("%u ", twind->pid);
                if  (cfg->flags & BLINREAD_P_T) {
                    printf( "%04"BLIN_O"u.%09"BLIN_O"u "
                          , (twind->time - fstime) / pow10(9)
                          , (twind->time - fstime) % pow10(9)
                          )
                    ;
                } else if  (cfg->flags & BLINREAD_P_A) {
                    printf( "%u-%02u-%02u %02u:%02u:%02u "
                          , htime->tm_year + 1900
                          , htime->tm_mon + 1
                          , htime->tm_mday
                          , htime->tm_hour
                          , htime->tm_min
                          , htime->tm_sec
                          )
                    ;
                }
                if  (cfg->flags & BLINREAD_P_L) printf("%u ", twind->level);
                if  (cfg->flags & BLINREAD_P_E) printf("%u ", twind->erno);
                if  (cfg->flags & BLINREAD_P_R) printf("%.*s:", twind->snprog, &twind->string[0]);
                if  (cfg->flags & BLINREAD_P_S) {
                    printf("%.*s:", twind->snfile, &twind->string[twind->snprog]);
                }
                if  (cfg->flags & BLINREAD_P_U) {
                    printf( "%.*s:"
                          , twind->snfunc
                          , &twind->string[twind->snprog + twind->snfile]
                          )
                    ;
                }
                if  (cfg->flags & BLINREAD_P_N) printf("%u:", twind->line);
                for (p = 0
                    ; twind->string[strnlen( &twind->string[0]
                                           , twind->len - offsetof(BLIN_trace, string)
                                           )
                                   - 1
                                   - p]
                    == '\n'
                    ; p++
                    ) {}
                printf( " %.*s\n"
                      , (int) ( strnlen(&twind->string[0], twind->len - offsetof(BLIN_trace, string))
                              - twind->snprog - twind->snfile - twind->snfunc - p)
                      , &twind->string[twind->snprog + twind->snfile + twind->snfunc]
                      )
                ;
        }   }
        if  (mife_ctlsize(cfg->tfile) == twind->len) break;
    }
    if  (cfg->flags & BLINREAD_Q) {  /*****/
        if  (cfg->flags & (BLINREAD_P_A | BLINREAD_P_T)) {
            printf("time ");
                       /* for (BLIN_list_u8 *i = qbuf.time; i; i = i->next) printf("%BLIN_Ox ", i->num); */
            if  (qbuf.time1 == qbuf.time2) {
                printf( "%"BLIN_U"\n", qbuf.time1 - (cfg->flags & BLINREAD_P_T ? fstime : 0));
            }
            else {
                printf( "from %"BLIN_O"x to %"BLIN_O"x\n"
                      , qbuf.time1 - (cfg->flags & BLINREAD_P_T ? fstime : 0)
                      , qbuf.time2 - (cfg->flags & BLINREAD_P_T ? fstime : 0)
                      )
                ;
            }
        }
        if  (cfg->flags & BLINREAD_P_I) {
            printf("\npid:\n");
            for (BLIN_list_u4 *i = qbuf.pid ; i; i = i->next) printf("%u ", i->num);
            printf("\n");
        }
        if  (cfg->flags & BLINREAD_P_L) {
            printf("\nlevel %02X\n", qbuf.level);
        }
        if  (cfg->flags & BLINREAD_P_R) {
            printf("\nprog:\n");
            for (BLIN_list_pc  *i = qbuf.prog; i; i = i->next) printf("%s ", i->name);
            printf("\n");
        }
        if  (cfg->flags & (BLINREAD_P_S | BLINREAD_P_U | BLINREAD_P_N)) {
            printf("\n");
            p = 0;
            for (BLIN_qbuf_list *file = qbuf.file; file; file = file->next) {
                p = 0;
                for (BLIN_qbuf_list *func = file->list; func; func = func->next) {
                    p &= ~(u_int32_t)2;
                    for (BLIN_list_u4 *line = func->list; line; line = line->next) {
                        if  (cfg->flags & BLINREAD_P_S) {
                            if  (!(cfg->flags & (BLINREAD_P_U | BLINREAD_P_N))) {
                                printf("%s\n", file->name);
                                break;
                            } else if  (p & 1) {
                                printf("%.*s ", (int)strlen(file->name), SPASES);
                            } else {
                                printf("%s:", file->name);
                                p |= 1;
                        }   }
                        if  (cfg->flags & BLINREAD_P_U) {
                            if  (!(cfg->flags & BLINREAD_P_N)) {
                                printf("%s\n", func->name);
                                break;
                            } else if  (p & 2) {
                                printf("%.*s ", (int)strlen(func->name), SPASES);
                            } else {
                                printf("%s:", func->name);
                                p |= 2;
                        }   }
                        if  (cfg->flags & BLINREAD_P_N) printf("%u\n", line->num);
    }   }   }   }   }
out:
    mife_fini(cfg->tfile);
    babolo_closeopts(bos);
    mular_destroy(cfg->time);
    mular_destroy(cfg->pid);
    mular_destroy(cfg->prog);
    mular_destroy(cfg->file);
    mular_destroy(cfg->func);
    mular_destroy(cfg->line);
    /* free(< >) */
    ifBLIN_QX2("exit");
    exit(ex);
#   undef isJX
#   undef isJW
#   undef blin_internal_flags
}
