/*-
 * Copyright (C)2023..2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2023..2024 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: aaacipa.c,v 1.51 2024/05/26 18:45:15 babolo Exp $"

#define BLIN_COMPAT      4
#define Bpars_COMPAT     4
#define MIFE_COMPAT      5

#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/event.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sysexits.h>
#include <setjmp.h>
#include <string.h>
#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <netdb.h>
#include <openssl/des.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <mife.h>
#include "aaacipa.h"

static volatile int   traped = 0;
static char         **args   ;
static jmp_buf        env    ;
static BLIN_flag     *toflags;

static void
/*****************************************************************************************************
 **                                                                                                 **/
upv(int sig) {                                                                                     /**
 **                                                                                                 **
 *****************************************************************************************************/
    (void)sig;
    BLIN_VERBOSE(*toflags);
    ifBLIN_QX1("[%d] SIGUSR2 %02X", getpid(), (*toflags >> 24) & 0x0FF);
}

static void
/*****************************************************************************************************
 **                                                                                                 **/
downv(int sig) {                                                                                   /**
 **                                                                                                 **
 *****************************************************************************************************/
    (void)sig;
    ifBLIN_QX1("[%d] SIGUSR1", getpid());
    blin_ctl(BLIN_CTL_FLAG | BLIN_CTL_FEQU, (*toflags &= ~BLIN_MASK));
}

static void
/*****************************************************************************************************
 **                                                                                                 **/
upsig(int sig) {                                                                                   /**
 **                                                                                                 **
 *****************************************************************************************************/
    ifBLIN_QX1("[%d] up trap %d", getpid(), sig);
    traped = sig;
}

static void __attribute__((__noreturn__))
/*****************************************************************************************************
 **                                                                                                 **/
exsig(int sig) {                                                                                   /**
 **                                                                                                 **
 *****************************************************************************************************/
    ifBLIN_QX1("ex trap %d", sig);
    exit(sig);
}

static void __attribute__((__noreturn__))
qlong(int s) {
    (void)s;
    ifBLIN_QX0("query timed out");
    longjmp(env, EX_TEMPFAIL);
}

static int
/*****************************************************************************************************
 **                                                                                                 **/
fromq(aaacipa_cf *cf, querystr *query, char **c) {                                                 /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    int   ex = EX_OK;
    int   p;
    int   d  = -1;

    ifBLIN_QX2( "+ %zu=[%u]%.*s~ %zu"
              , query->q
              , query->query[query->p] & 0x0FF
              , (int)(query->query[query->p] & 0x0FF) - 1, &query->query[query->p + 1]
              , query->p
              );
    if  (query->q <= query->p) {
        ex = -2;
        goto out;
    }
    p = (d = query->query[query->p++] & 0x0FF) - 1;
    ifBLIN_QX2("d=%u", d);
    if  (!d) {
        if  (!!*c) ifBLIN_QX0("Unstable and leak");
        *c = NULL;
        goto out;
    }
    if  (query->q < (query->p + (size_t)p)) {
        ifBLIN_QX2("q=%zu < p=%zu + p=%d", query->q, query->p, p);
        ex = -1;
        goto out;
    }
    if  (!*c && !(*c = strndup(&query->query[query->p], (size_t)p))) {
        ifBLIN_QW0("No mem %u", p);
        ex = EX_OSERR;
        goto out;
    }
    query->p += (size_t)p;
out:
    if  (!*c) {
        ifBLIN_QX2("- %d %u %zu %zu", ex, d, query->q, query->p);
    } else {
        ifBLIN_QX2("- %d %u=%s~ %zu %zu", ex, d, *c, query->q, query->p);
    }
    return(ex);
#   undef blin_internal_flags
}

static int
/*****************************************************************************************************
 **                                                                                                 **/
doquery(aaacipa_cf *cf) {                                                                          /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    querytype_t    querytype;
    aaacipa_pass   pass     ;
    char          *nick     = NULL;
    struct timeval tp       ;
    int            ex       = EX_OK;
    ssize_t        r        ;
    size_t         l        ;

    ifBLIN_QX2("+");
    bzero(&pass, sizeof(pass));
    for (; ;) {
        if  (AAACIPA_MAXQSZ <= pass.query.q) {
            ifBLIN_QX0("No room for query");
            ex = EX_USAGE;
            errno = EBADMSG;
            goto out;
        }
        l = (AAACIPA_MAXQSZ + 1) - pass.query.q;
        if  (0 > (r = read(cf->fd, &pass.query.query[pass.query.q], l))) {
            ifBLIN_QW0("read");
            ex = EX_IOERR;
            goto out;
        }
        if  (!r) {
            ifBLIN_QX0("Unexpected EOF");
            ex = EX_USAGE;
            errno = EPROTO;
            goto out;
        }
        ifBLIN_QX3("read %zd", r);
        pass.query.q += (size_t)r;
        pass.query.p = 0;
        ifBLIN_QO4 blin_dumb(4, pass.query.query, (ssize_t)pass.query.q);
        querytype = (querytype_t)pass.query.query[pass.query.p++];
        ifBLIN_QX2("querytype=%d", querytype);
        switch(querytype) {
        case qryaaa:
            if  (0 > (ex = fromq(cf, &pass.query, &pass.context))) continue;
            if  (!pass.context) {
                ifBLIN_QX0("NULL context fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (!!ex) {
                ifBLIN_QW0("No context fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (0 > (ex = fromq(cf, &pass.query, &pass.srcip))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No srcip fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!pass.srcip) {
                ifBLIN_QX1("NULL srcip fromq");
                pass.srcip = strdup("");
            }
            if  (0 > (ex = fromq(cf, &pass.query, &pass.user))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No user fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!pass.user) {
                ifBLIN_QX0("No user fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (0 > (ex = fromq(cf, &pass.query, &pass.password))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No password fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (0 > (ex = fromq(cf, &pass.query, &pass.cheme))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No cheme fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!pass.cheme) {
                ifBLIN_QX1("NULL cheme fromq");
                pass.crypt = 0;
            } else if (!strcmp(pass.cheme, "https")) {
                pass.crypt = 1;
            } else if (!strcmp(pass.cheme, "https")) {
                pass.crypt = 0;
            } else {
                ifBLIN_QX1("cheme=%s~", pass.cheme);
                pass.crypt = 0;
            }
            if  (0 > (ex = fromq(cf, &pass.query, &pass.request))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No request fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!pass.request) {
                ifBLIN_QX1("NULL request fromq");
                pass.request = strdup("");
            }
            break;
        case qrycre:
            pass.pflags = ((querytabs_t *)(pass.query.query))->flags; /* 8 alligned */
            pass.query.p = offsetof(querytabs_t, cred);
            if  (0 > (ex = fromq(cf, &pass.query, &pass.context))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No context fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!pass.context || !*pass.context) {
                ifBLIN_QX0("ontext required by fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (0 > (ex = fromq(cf, &pass.query, &pass.user))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No user fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!pass.user || !*pass.user) {
                ifBLIN_QX0("User required by fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (0 > (ex = fromq(cf, &pass.query, &pass.password))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No password fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!pass.password || !*pass.password) {
                ifBLIN_QX0("Password required by fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (0 > (ex = fromq(cf, &pass.query, &nick))) continue;
            if  (!!ex) {
                ifBLIN_QW0("No nick fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (!nick || !*nick) {
                ifBLIN_QX0("Nick required by fromq");
                ex = EX_NOINPUT;
                goto out;
            } else if (AAACIPA_NICKLEN < strlen(nick)) {
                ifBLIN_QX0("To long nick ");
                ex = EX_DATAERR;
                goto out;
            }
            strlcpy(pass.nick, nick, AAACIPA_NICKLEN + 1);
            break;
        case qrydel:
            /* FALLTHROUGH */
        case qrypas:
            /* FALLTHROUGH */
        case qrypasc:
            /* FALLTHROUGH */
        case qryses:
            /* FALLTHROUGH */
        case qrysesc:
            /* FALLTHROUGH */
        case qryseb:
            /* FALLTHROUGH */
        case qrysebc:
            /* FALLTHROUGH */
        case qryflo:
            /* FALLTHROUGH */
        case qryfla:
            /* FALLTHROUGH */
        case qryslo:
            pass.cred = ((querytabs_t *)(pass.query.query))->cred; /* 8 alligned */
            pass.sess = ((querytabs_t *)(pass.query.query))->sess; /* 8 alligned */
            pass.query.p = offsetof(querytabs_t, context);
            if  (0 > (ex = fromq(cf, &pass.query, &pass.context))) continue;
            if  (!pass.context) {
                ifBLIN_QX0("NULL context fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            if  (!!ex) {
                ifBLIN_QW0("No context fromq");
                ex = EX_NOINPUT;
                goto out;
            }
            break;
        }
        break;
    }
    if  (!!gettimeofday(&tp, NULL)) {
        ifBLIN_QW0("gettimeofday");
        ex = EX_OSERR;
        goto out;
    }
    pass.now = 0.000001 * tp.tv_usec + tp.tv_sec;
    if  (!!(ex = setjmp(env))) {
        ifBLIN_QX0("[%d] Q timed out", getpid());
        errno = ETIMEDOUT;
        goto out;
    }
    if  (SIG_ERR == signal(SIGALRM, qlong)) {
        ifBLIN_QW0("signal ALRM");
        ex = EX_OSERR;
        goto out;
    }
    alarm(3);
    switch(querytype) {
    case qryaaa:
        do {
            char      *o;
            char      *q;

            if  (!strncmp(pass.request, "POST /?", 7)) {
                o = &pass.request[7];
            } else if (!strncmp(pass.request, "GET /?", 6)) {
                o = &pass.request[6];
            } else if (!strncmp(pass.request, "/?", 2)) {
                o = &pass.request[2];
            } else {
                break;
            }
            if  (!(q = strchr(o, '='))) break;
            *q = 0;
            if  (AAACIPA_HLEN != strlen(o)) break;
            if  (!!(ex = aaacipa_dedes(cf, &pass.sess, o))) {
                ifBLIN_QW0("aaacipa_dedes");
                break;
            }
        } while(0);
        ifBLIN_QX1( "[%d->%d] ctx=%s~ IP=%s U=%s~ P=%s~ sch=%s~ R=%s(%lu)"
                  , cf->pid
                  , getpid()
                  , pass.context
                  , pass.srcip
                  , pass.user
                  , pass.password
                  , pass.cheme
                  , pass.request
                  , pass.sess
                  );
        if  (!!pass.context != !!cf->fieldcontext) {
            ifBLIN_QX0("context & context");
            ex = EX_USAGE;
            errno = EINVAL;
            goto out;
        }
        if  (!!(ex = aaacipa_passq(cf, &pass))) {
            ifBLIN_QW0("aaacipa_passq");
            goto out;
        }
        alarm(0);
        if  (SIG_ERR == signal(SIGALRM, SIG_DFL)) ifBLIN_QW0("signal ALRM");
        if  (!!(ex = aaacipa_endes(cf, pass.sess, pass.hash))) {
            ifBLIN_QW0("aaacipa_endes");
            goto out;
        }
        ifBLIN_QX1("C=%lu S=%lu", pass.cred, pass.sess);
        if  (0 > (ex = (int)write(cf->fd, &pass, AAACIPA_MAXCRED + strlen(pass.nick)))) {
            ifBLIN_QW0("write AAACIPA_MAXCRED");
            ex = EX_IOERR;
            goto out;
        }
        ifBLIN_QX2( "%zu + %zu = %zu # %d"
                  , AAACIPA_MAXCRED
                  , strlen(pass.nick)
                  , AAACIPA_MAXCRED + strlen(pass.nick)
                  , ex
                  );
        if  ((AAACIPA_MAXCRED + strlen(pass.nick)) != (u_int)ex) {
            ifBLIN_QX0("write %d", ex);
            ex = EX_IOERR;
            errno = ENOMSG;
            goto out;
        }
        ex = 0;
        ifBLIN_QX2( "- %d %lu=%ld %19.6f %.*s"
                  , ex
                  , pass.sess
                  , pass.cred
                  , pass.now
                  , AAACIPA_HLEN, pass.hash
                  );
        break;
    case qrycre:
        if  (!!(ex = aaacipa_ucreq(cf, &pass))) ifBLIN_QW0("aaacipa_ucreq");
        pass.ex = ex;
        pass.errnom = errno;
        alarm(0);
        if  (!!ex) pass.cred = 0;
        if  (SIG_ERR == signal(SIGALRM, SIG_DFL)) ifBLIN_QW0("signal ALRM");
        if  (0 > (ex = (int)write(cf->fd, &pass, AAACIPA_CRELEN))) {
            ifBLIN_QW0("write cred + err");
            ex = EX_IOERR;
            goto out;
        }
        if  (AAACIPA_CRELEN != (u_int)ex) {
            ifBLIN_QX0("write %d", ex);
            ex = EX_IOERR;
            errno = ENOMSG;
            goto out;
        }
        ex = 0;
        break;
    case qrydel:
        /* FALLTHROUGH */
    case qrypas:
        /* FALLTHROUGH */
    case qrypasc:
        /* FALLTHROUGH */
    case qryses:
        /* FALLTHROUGH */
    case qrysesc:
        /* FALLTHROUGH */
    case qryseb:
        /* FALLTHROUGH */
    case qrysebc:
        /* FALLTHROUGH */
    case qryflo:
        /* FALLTHROUGH */
    case qryfla:
        /* FALLTHROUGH */
    case qryslo:
        if  (!!(ex = aaacipa_tabsq(cf, &pass, querytype))) {
            ifBLIN_QW0("aaacipa_tabsq");
            goto out;
        }
        alarm(0);
        if  (SIG_ERR == signal(SIGALRM, SIG_DFL)) ifBLIN_QW0("signal ALRM");
        break;
    }
out:
    if  (0 > close(cf->fd)) ifBLIN_QW0("close");
    free(pass.context);
    free(pass.srcip);
    free(pass.user);
    free(pass.password);
    free(pass.cheme);
    free(pass.request);
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static void __attribute__((__noreturn__))
usage(int ex) {
    fprintf( stderr
           , "aaacipa @BABOLO V.M "VERS"  "DATE" http://www.babolo.ru/\n"
             "    -c name   - AAA config table name\n"
             "    -d path   - AAA DB file\n"
             "    -D        - do not daemonize\n"
             "    -f path   - config file\n"
             "    -h host   - IP or path for (TCP or local) socket\n"
             "    -L        - socket is local\n"
             "    -N C name   - context field name\n"
             "        name   - session create time field name\n"
             "       d name   - session delete time field name\n"
             "       F name   - user flags field name\n"
             "       f name   - session flags field name\n"
             "       I name   - srcip field name\n"
             "       K name   - hash key field name\n"
             "       N name   - nickname field name\n"
             "       O name   - credential field name\n"
             "       R name   - reorder vector fields name\n"
             "       r name   - remark field name\n"
             "       T name   - TTL fields name\n"
             "       u name   - session update time field name\n"
             "       U name   - user field name\n"
             "       V name   - hash initialization vector field name\n"
             "       W name   - password field name\n"
             "       Z name   - TZ shift field name\n"
             "    -p num    - port number\n"
             "    -P path   - pid file\n"
             "    -q        - quiet\n"
             "    -s name   - sessions table name\n"
             "    -t name   - user table name\n"
             "    -v        - verbose\n"
             "    -# text   - NOOP for comments\n"
             "    -?        - this text\n"
           );
    exit(ex);
}

static int
/*****************************************************************************************************
 **                                                                                                 **/
aaacipa_parsin(aaacipa_cf *cf, int argc, char *argv[]) {                                           /**
 **                                                                                                 **
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    const char      *flar = "c:d:Df:h:LN::p:P:qs:t:v#:?=:";
    babolo_opts     *bos  = NULL;
    int              ex   = EX_OK;
    const char      *o    ;
    int              c    ;

    ifBLIN_QX2("+ %d", argc);
    if  (!(bos = babolo_openopts(BLIN_BIT0 | Bpars_NONU | Bpars_SYME | Bpars_ESYM, 'Z'))) {
        ifBLIN_QW0("babolo_openopts");
        ex = EX_SOFTWARE;
        goto out;
    }
    if  (babolo_setopts(bos, (AAACIPA_CFILE & cf->flags) ? 0 : Bpars_NONU, argc, argv, flar)) {
        ifBLIN_QW0("babolo_setopts");
        ex = EX_SOFTWARE;
        goto out;
    }
    while (0 < (c = babolo_getopts(bos))) {
        switch (c) {
        case 'c': if  (!!cf->cipaconf) {
                ;     ifBLIN_QX1("Can't redefine cipaconf");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->cipaconf = o;
                ; }
                ; break;
        case 'd': if  (!!cf->dbfile) {
                ;     ifBLIN_QX1("Can't redefine dbfile");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->dbfile = o;
                ; }
                ; break;
        case 'D': cf->flags |= AAACIPA_NODEMON;
                ; break;
        case 'f': if  (!!cf->conf) {
                ;     ifBLIN_QX1("Can't redefine conf");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->conf = o;
                ; }
                ; break;
        case 'h': if  (!!cf->host) {
                ;     ifBLIN_QX1("Can't redefine host");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->host = o;
                ; }
                ; break;
        case 'L': cf->flags |= AAACIPA_LOCAL;
                ; break;
        case 'N': if  (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; }
                ; c = o[0];
                ; if  (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -N second arg");
                ;     ex = EX_USAGE;
                ;     goto out;
                ; }
                ; switch(c) {
                  case 'c': if  (!!cf->fieldcre) {
                          ;     ifBLIN_QX1("Can't redefine fieldcre");
                          ; } else {
                          ;     cf->fieldcre = o;
                          ; }
                          ; break;
                  case 'C': if  (!!cf->fieldcontext) {
                          ;     ifBLIN_QX1("Can't redefine fieldcontext");
                          ; } else {
                          ;     cf->fieldcontext = o;
                          ; }
                          ; break;
                  case 'd': if  (!!cf->fielddel) {
                          ;     ifBLIN_QX1("Can't redefine fielddel");
                          ; } else {
                          ;     cf->fielddel = o;
                          ; }
                          ; break;
                  case 'F': if  (!!cf->fieldpflags) {
                          ;     ifBLIN_QX1("Can't redefine fieldpflags");
                          ; } else {
                          ;     cf->fieldpflags = o;
                          ; }
                          ; break;
                  case 'f': if  (!!cf->fieldsflags) {
                          ;     ifBLIN_QX1("Can't redefine fieldsflags");
                          ; } else {
                          ;     cf->fieldsflags = o;
                          ; }
                          ; break;
                  case 'I': if  (!!cf->fieldsrcip) {
                          ;     ifBLIN_QX1("Can't redefine fieldsrcip");
                          ; } else {
                          ;     cf->fieldsrcip = o;
                          ; }
                          ; break;
                  case 'K': if  (!!cf->fieldkey) {
                          ;     ifBLIN_QX1("Can't redefine fieldkey");
                          ; } else {
                          ;     cf->fieldkey = o;
                          ; }
                          ; break;
                  case 'N': if  (!!cf->fieldnick) {
                          ;     ifBLIN_QX1("Can't redefine fieldnick");
                          ; } else {
                          ;     cf->fieldnick = o;
                          ; }
                          ; break;
                  case 'O': if  (!!cf->fieldcred) {
                          ;     ifBLIN_QX1("Can't redefine fieldcred");
                          ; } else {
                          ;     cf->fieldcred = o;
                          ; }
                          ; break;
                  case 'r': if  (!!cf->fieldremark) {
                          ;     ifBLIN_QX1("Can't redefine fieldremark");
                          ; } else {
                          ;     cf->fieldremark = o;
                          ; }
                          ; break;
                  case 'R': if  (!!cf->fieldreo) {
                          ;     ifBLIN_QX1("Can't redefine fieldreo");
                          ; } else {
                          ;     cf->fieldreo = o;
                          ; }
                          ; break;
                  case 'S': if  (!!cf->fieldsess) {
                          ;     ifBLIN_QX1("Can't redefine fieldsess");
                          ; } else {
                          ;     cf->fieldsess = o;
                          ; }
                          ; break;
                  case 'T': if  (!!cf->fieldttl) {
                          ;     ifBLIN_QX1("Can't redefine fieldttl");
                          ; } else {
                          ;     cf->fieldttl = o;
                          ; }
                          ; break;
                  case 'u': if  (!!cf->fieldupd) {
                          ;     ifBLIN_QX1("Can't redefine fieldupd");
                          ; } else {
                          ;     cf->fieldupd = o;
                          ; }
                          ; break;
                  case 'U': if  (!!cf->fieldlogin) {
                          ;     ifBLIN_QX1("Can't redefine fieldlogin");
                          ; } else {
                          ;     cf->fieldlogin = o;
                          ; }
                          ; break;
                  case 'V': if  (!!cf->fieldvinit) {
                          ;     ifBLIN_QX1("Can't redefine fieldvinit");
                          ; } else {
                          ;     cf->fieldvinit = o;
                          ; }
                          ; break;
                  case 'W': if  (!!cf->fieldpass) {
                          ;     ifBLIN_QX1("Can't redefine fieldpasswd");
                          ; } else {
                          ;     cf->fieldpass = o;
                          ; }
                          ; break;
                  case 'Z': if  (!!cf->fieldmzone) {
                          ;     ifBLIN_QX1("Can't redefine fieldmzone");
                          ; } else {
                          ;     cf->fieldmzone = o;
                          ; }
                          ; break;
                  default : ifBLIN_QW0("Ill flag -%c", bos->c);
                          ; usage(EX_USAGE);
                ; }
                ; break;
        case 'p': if  (!!cf->cport) {
                ;     ifBLIN_QX1("Can't redefine cport");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->cport = o;
                ; }
                ; break;
        case 'P': if  (!!cf->pidf) {
                ;     ifBLIN_QX1("Can't redefine pidf");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->pidf = o;
                ; }
                ; break;
        case 'q': BLIN_QUIET(cf->flags);
                ; break;
        case 's': if  (!!cf->sessions) {
                ;     ifBLIN_QX1("Can't redefine sessions");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->sessions = o;
                ; }
                ; break;
        case 't': if  (!!cf->passwd) {
                ;     ifBLIN_QX1("Can't redefine passwd");
                ; } else if (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; } else {
                ;     cf->passwd = o;
                ; }
                ; break;
        case 'v': BLIN_VERBOSE(cf->flags);
                ; break;
        case '#': break;
        case '?': usage(EX_OK);
        case '=': if  (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; }
                ; switch(o[0]) {
                  case '0': blin_ctl(BLIN_CTL_DUMP);           exit(EX_OK);
                  case '1': babolo_optest(flar);               exit(EX_OK);
                  case '2': babolo_optext(bos);                exit(EX_OK);
                  case '3': babolo_dumpopts(bos);              exit(EX_OK);
                  }
                ; /* FALLTHROUGH */
        default : ifBLIN_QW0("Ill flag -%c", bos->c);
                ; usage(EX_USAGE);
    }   }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****                                                                                             ****/
main(int argc, char *argv[]) {                                                                   /****
 ****                                                                                             ****
 *****************************************************************************************************
 *****************************************************************************************************/
    const int    interrupts[] = { SIGCHLD
                                , SIGHUP
                                , SIGINT
                                , SIGPIPE
                                , SIGTERM
                                , SIGXCPU
                                , SIGXFSZ
                                , SIGLIBRT
                                , 0
                                };
    babolo_parm *parms    ;
    char        *cfmap    ;
    addrinfo    *hint     ;
    aaacipa_cf  *cf       ;
    int          ex       = EX_OK;
    int          k        ;

    args = argv;
    if  (!(cf = calloc(1, sizeof(aaacipa_cf)))) {
        ifBLIN_QW0("No mem %"BLIN_X, sizeof(aaacipa_cf));
        ex = EX_OSERR;
        goto out;
    }
    blin_ctl(BLIN_CTL_FLAG | BLIN_CTL_FSET, (cf->flags = BLIN_BIT0));
#   define blin_internal_flags (cf->flags & BLIN_MASK)
    if  (0 > (cf->curdir = open(".", O_RDONLY | O_DIRECTORY))) {
        ifBLIN_QW0("open . DIR");
        ex = EX_OSERR;
        goto out;
    }
    if  (!!(ex = aaacipa_parsin(cf, argc, argv))) {
        ifBLIN_QW0("parsin");
        ex = EX_USAGE;
        goto out;
    }
    if  (!cf->conf) cf->conf = PREFIX "/etc/aaacipa.cf";
    if  (0 > (cf->confd = open(cf->conf, O_RDONLY | O_CLOEXEC))) {
        ifBLIN_QW1("Conf file absent");
    } else {
        cf->flags |= AAACIPA_CFILE;
        if  (!(cf->md = mife_init(MIFE_FULL | 1))) {
            ifBLIN_QW0("mife_init");
            ex = EX_OSERR;
            goto out;
        }
        if  (0 > mife_ctlfdsc(cf->md, cf->confd)) {
            ifBLIN_QW0("mife_ctlfdsc");
            ex = EX_IOERR;
            goto out;
        }
        if  (!(cfmap = mife_window(cf->md, 0, 0))) {
            ifBLIN_QW0("mife_window");
            ex = EX_IOERR;
            goto out;
        }
        if  (!(parms = babolo_getparm(Bpars_NOEN | Bpars_NOAL, &cfmap, NULL, 0))) {
            ifBLIN_QW0("babolo_getparm");
            ex = EX_IOERR;
            goto out;
        }
        if  (!!(ex = aaacipa_parsin(cf, parms->argc, parms->argv))) {
            ifBLIN_QW0("parsin cf");
            ex = EX_IOERR;
            goto out;
    }   }
    if  (!cf->dbfile) {
        ifBLIN_QX0("DB not defined");
        ex = EX_USAGE;
        goto out;
    }
    if  (!cf->host) {
        if  (AAACIPA_LOCAL & cf->flags) {
                            cf->host         = "/tmp/.s.aaacipa";
        } else {
                            cf->host         = "127.0.0.1";
    }   }
    if  ('/' == cf->host[0]) cf->flags |= AAACIPA_LOCAL;
    if  (!cf->cport)        cf->cport        = "1357";
    if  (!cf->cipaconf)     cf->cipaconf     = "cipaconf";
    if  (!cf->passwd)       cf->passwd       = "pass";
    if  (!cf->sessions)     cf->sessions     = "sess";
    if  (!cf->sessionb)     cf->sessionb     = "sessb";
    if  (!cf->fieldcontext) cf->fieldcontext = "context";
    if  (!cf->fieldcred)    cf->fieldcred    = "cred";
    if  (!cf->fieldlogin)   cf->fieldlogin   = "login";
    if  (!cf->fieldpass)    cf->fieldpass    = "pass";
    if  (!cf->fieldsrcip)   cf->fieldsrcip   = "srcip";
    if  (!cf->fieldttl)     cf->fieldttl     = "ttl";
    if  (!cf->fieldreo)     cf->fieldreo     = "reo";
    if  (!cf->fieldkey)     cf->fieldkey     = "key";
    if  (!cf->fieldvinit)   cf->fieldvinit   = "init";
    if  (!cf->fieldsess)    cf->fieldsess    = "sess";
    if  (!cf->fieldhash)    cf->fieldhash    = "hash";
    if  (!cf->fieldcnt)     cf->fieldcnt     = "cnt";
    if  (!cf->fieldsessip)  cf->fieldsessip  = "srcip";
    if  (!cf->fieldflags)   cf->fieldflags   = "flags";
    if  (!cf->fieldcreat)   cf->fieldcreat   = "created";
    if  (!cf->fieldupdt)    cf->fieldupdt    = "updated";
    if  (!cf->fieldmzone)   cf->fieldmzone   = "mzone";
    if  (!cf->fieldnick)    cf->fieldnick    = "nickname";
    if  (!cf->fieldpflags)  cf->fieldpflags  = "flags";
    if  (!cf->fieldsflags)  cf->fieldsflags  = "flags";
    if  (!cf->fieldremark)  cf->fieldremark  = "remark";
    if  (!cf->fieldcre)     cf->fieldcre     = "created";
    if  (!cf->fieldupd)     cf->fieldupd     = "updated";
    if  (!cf->fielddel)     cf->fielddel     = "removed";
    if  (!cf->pidf)         cf->pidf         = "/var/run/aaacipa.pid";
#if defined(USEBDB)
    {   char *repare[] = {NULL, NULL, NULL};
        int   status;

        if  (!(repare[0] = strdup(USEBDB "/db_recover"))) {
            ifBLIN_QW0("strdup");
            ex = EX_OSERR;
            goto out;
        }
        if  (0 > asprintf(&repare[1], "-h%s-journal", cf->dbfile)) {
            ifBLIN_QW0("asprintf repare");
            ex = EX_OSERR;
            goto out;
        }
        if  (0 > (ex = fork())) {
            ifBLIN_QW0("fork repare");
            ex = EX_OSERR;
            goto out;
        } else if (!ex) {
            if  (0 > (ex = execv(repare[0], repare))) {
                ifBLIN_QW0("execv %s(%s)", repare[0], repare[1]);
                ex = EX_OSERR;
                goto out;
            }
        } else {
            if  (0 > (ex = waitpid(ex, &status, WEXITED))) {
                ifBLIN_QW0("waitpid repare");
                ex = EX_OSERR;
                goto out;
            } else if (!WIFEXITED(status)) {
                ifBLIN_QW0("repare terminate");
                ex = EX_OSERR;
                goto out;
            } else if (!!WEXITSTATUS(status)) {
                ifBLIN_QW0("repare error %d", WEXITSTATUS(status));
                ex = WEXITSTATUS(status);
                goto out;
        }   }
        free(repare[1]);
        free(repare[0]);
    }
#endif
    if  (!!(ex = aaacipa_cook(cf))) {
        ifBLIN_QW0("aaacipa_cook");
        ex = EX_OSERR;
        goto out;
    }
    if  (!cf->key) ifBLIN_QX1("Key not initialized");
    if  (!cf->iv) ifBLIN_QX1("Ivector not initialized");
    if  (!!(ex = aaacipa_prep(cf))) {
        ifBLIN_QW0("aaacipa_prep");
        goto out;
    }
    ifBLIN_QX4( "%s %s conf=%s host=%s cport=%s pidf=%s "
                "dbfile=%s passwd=%s cf.cred=%s cf.context=%s cf.login=%s cf.pass=%s cf.srcip=%s"
              , (AAACIPA_LOCAL & cf->flags) ? "local" : "inet"
              , (AAACIPA_NODEMON & cf->flags) ? "" : "daemon"
              , cf->conf
              , cf->host
              , cf->cport
              , cf->pidf
              , cf->dbfile
              , cf->passwd
              , cf->fieldcred
              , cf->fieldcontext
              , cf->fieldlogin
              , cf->fieldpass
              , cf->fieldsrcip
              );
    if  (!(AAACIPA_NODEMON & cf->flags)) {
        blin_ctl(BLIN_CTL_WHAT | BLIN_CTL_FEQU | BLIN_GEN7, BLIN_MODEOAU | BLIN_MODLOGR);
        if  (daemon(0, 0)) {
            ifBLIN_QW0("Daemonize fail");
            ex = EX_OSERR;
            goto out;
        }
        cf->pid = getpid();
        ifBLIN_QX1("started "VERS" @ "DATE" pid %d", cf->pid);
        do {
            cf->pidd = open( cf->pidf
                           , O_RDWR | O_CREAT | O_TRUNC | O_EXLOCK | O_NONBLOCK | O_CLOEXEC
                           , 0666
                           );
            if  (0 <= cf->pidd) break;
            ifBLIN_QW0("[%d] open pid file=%s~", getpid(), cf->pidf);
            if  (EAGAIN != errno) {
                ex = EX_IOERR;
                goto out;
            }
            sleep(1);
        } while(1);
        if  (0 > ftruncate(cf->pidd, sizeof(aaacipa_pid))) {
            ifBLIN_QW0("[%d] ftruncate pid", getpid());
            ex = EX_IOERR;
            goto out;
        }
        cf->pidp = mmap( NULL
                       , sizeof(aaacipa_pid)
                       , PROT_WRITE | PROT_READ, MAP_NOSYNC | MAP_SHARED
                       , cf->pidd
                       , 0
                       );
        if  (MAP_FAILED == cf->pidp) {
            ifBLIN_QW0("[%d] mmap pid", getpid());
            ex = EX_IOERR;
            goto out;
        }
        bzero(cf->pidp->pid, 8);
        k = snprintf(cf->pidp->pid, 8, "%d\n", cf->pid);
        if  (0 > k) {
            ifBLIN_QW0("[%d] snprintf pid", getpid());
            ex = EX_IOERR;
            goto out;
        }
        if  (k > 7) {
            ifBLIN_QX0("[%d] snprintf pid big", getpid());
            ex = EX_SOFTWARE;
            errno = EOVERFLOW;
            goto out;
    }   }
    if  (!(hint = calloc(1, sizeof(*hint)))) {
        ifBLIN_QW0("[%d] No mem %"BLIN_X, getpid(), sizeof(*hint));
        ex = EX_OSERR;
        goto out;
    }
    if  (AAACIPA_LOCAL & cf->flags) {
        hint->ai_family = PF_LOCAL;
        hint->ai_protocol = PF_UNSPEC;
    } else {
        hint->ai_family = PF_INET;
        hint->ai_protocol = IPPROTO_TCP;
    }
    hint->ai_socktype = SOCK_STREAM;
    hint->ai_flags = AI_NUMERICSERV | AI_PASSIVE | AI_ADDRCONFIG;
    if  ((ex = getaddrinfo(cf->host, cf->cport, hint, &cf->ai))) {
        ifBLIN_QX0("[%d] getaddrinfo: %s:%s %s", getpid(), cf->host, cf->cport, gai_strerror(ex));
        ex = EX_OSERR;
        goto out;
    }
    if  (0 > (cf->bd = socket(cf->ai->ai_family, cf->ai->ai_socktype, cf->ai->ai_protocol))) {
        ifBLIN_QW0("[%d] socket", getpid());
        ex = EX_IOERR;
        goto out;
    }
    freeaddrinfo(cf->ai);
    if  (!(AAACIPA_LOCAL & cf->flags)) {
        k = 1;
        if  (0 > setsockopt(cf->bd, IPPROTO_TCP, TCP_NODELAY, &k, sizeof(k))) {
            ifBLIN_QW0("[%d] setsockopt TCP_NODELAY %d", getpid(), k);
            ex = EX_IOERR;
            goto out;
    }   }
    k = 1;
    if  (0 > setsockopt(cf->bd, SOL_SOCKET, SO_REUSEPORT, &k, sizeof(k))) {
        ifBLIN_QW0("[%d] setsockopt SO_REUSEPORT %d", getpid(), k);
        ex = EX_IOERR;
        goto out;
    }
    if  (0 > bind(cf->bd, cf->ai->ai_addr, cf->ai->ai_addrlen)) {
        int s;
        int e;

        e = errno;
        ifBLIN_QX1("[%d] Socket %d used", getpid(), cf->bd);
        if  (  (0 > (s = socket(cf->ai->ai_family, cf->ai->ai_socktype, cf->ai->ai_protocol)))
            || !(AAACIPA_LOCAL & cf->flags)
            || !(EADDRINUSE == errno)
            || !(0 > connect(s, cf->ai->ai_addr, cf->ai->ai_addrlen))
            || !(ECONNREFUSED == errno)
            || (0 > close(s))
            || (0 > unlink(cf->host))
            || (0 > bind(cf->bd, cf->ai->ai_addr, cf->ai->ai_addrlen))
            ) {
            errno = e;
            ifBLIN_QW0("[%d] bind", getpid());
            ex = EX_IOERR;
            goto out;
    }   }
    ifBLIN_QX4("[%d] listen %d", getpid(), cf->bd);
    if  (0 > listen(cf->bd, 128)) {
        ifBLIN_QW0("[%d] listen", getpid());
        ex = EX_IOERR;
        goto out;
    }
    toflags = &cf->flags;
    for (int i = 0; !!interrupts[i]; ++i) {
        if  (SIG_ERR == signal(interrupts[i], upsig)) {
            ifBLIN_QW0("[%d] signal %d", getpid(), interrupts[i]);
            ex = EX_OSERR;
            goto out;
        }
        if  (0 > siginterrupt(interrupts[i], 1)) {
            ifBLIN_QW0("[%d] siginterrupt %d", getpid(), interrupts[i]);
            ex = EX_OSERR;
            goto out;
    }   }
    if  (SIG_ERR == signal(SIGUSR1, downv)) {
        ifBLIN_QW0("[%d] signal SIGUSR1", getpid());
        ex = EX_OSERR;
        goto out;
    }
    if  (SIG_ERR == signal(SIGUSR2, upv)) {
        ifBLIN_QW0("[%d] signal SIGUSR2", getpid());
        ex = EX_OSERR;
        goto out;
    }
    while (1) {
        if  (0 > (cf->fd = accept(cf->bd, NULL, 0))) {
            int status;

            ifBLIN_QX4("[%d] accept %d", getpid(), cf->fd);
            if  (!!traped && (SIGCHLD != traped)) {
                ifBLIN_QW0("[%d] signal %d", getpid(), traped);
                if  (0 > unlink(cf->pidf)) ifBLIN_QW0("[%d] unlink pidf=%s~", getpid(), cf->pidf);
                if  ((AAACIPA_LOCAL & cf->flags) && (0 > unlink(cf->host))) {
                    ifBLIN_QW0("[%d] unlink host=%s~", getpid(), cf->host);
                }
                if  (0 > close(cf->bd)) ifBLIN_QW0("[%d] close", getpid());
                goto out;
            }
            ifBLIN_QW1("[%d] accept", getpid());
            if  (0 < (ex = waitpid(-1, &status, WNOHANG | WEXITED))) {
                if  (WIFEXITED(status) && !WEXITSTATUS(status)) ex = 0;
            } else if (0 > ex) {
                ifBLIN_QW0("[%d] waitpid", getpid());
            }
            if  (!!ex) {
                if  (0 > fchdir(cf->curdir)) ifBLIN_QW0("[%d] fchdir", getpid());
                sleep(1);
                if  (0 > close(cf->curdir)) ifBLIN_QW0("[%d] close .", getpid());;
                if  (0 > close(cf->bd)) ifBLIN_QW0("[%d] close socket", getpid());
                if  (0 > (k = fork())) {
                    ifBLIN_QW0("[%d] fork", getpid());
                } else if (!!k) {
                    ifBLIN_QX0("Exit %d", getpid());
                    ex = 0;
                    goto out;
                } else if (0 > execvp(args[0], args)) {
                    ifBLIN_QW0("[%d] execvp %s ...", getpid(), args[0]);
                }
                ex = EX_OSERR;
                goto out;
            }
        } else if (0 > (ex = fcntl(cf->fd, F_SETFD, FD_CLOEXEC))) {
            ifBLIN_QW0("[%d] fcntl", getpid());
        } else if (0 > (k = fork())) {
            ifBLIN_QW0("[%d] fork", getpid());
        } else if (!k) {
            if  (0 > close(cf->bd)) ifBLIN_QW0("close");
            if  (SIG_ERR == signal(SIGCHLD, SIG_DFL)) {
                ifBLIN_QW0("[%d] signal CHLD", getpid());
                ex = EX_OSERR;
                goto out;
            }
            for (int i = 1; !!interrupts[i]; ++i) {
                if  (SIG_ERR == signal(interrupts[i], exsig)) {
                    ifBLIN_QW0("[%d] signal %d", getpid(), interrupts[i]);
                    ex = EX_OSERR;
                    goto out;
                }
                if  (0 > siginterrupt(interrupts[i], 1)) {
                    ifBLIN_QW0("[%d] siginterrupt %d", getpid(), interrupts[i]);
                    ex = EX_OSERR;
                    goto out;
            }   }
            if  (!!(ex = doquery(cf))) ifBLIN_QW0("[%d] doquery", getpid());
            goto out;
        }
        if  ((0 <= cf->fd) && (0 > close(cf->fd))) ifBLIN_QW0("[%d] close", getpid());
    }
out:
    if  (0 > close(cf->bd)) ifBLIN_QW1("close");
    ifBLIN_QX2("- [%d] %d", getpid(), ex);
    exit(ex);
#   undef blin_internal_flags
}
